package com.baijia.tianxiao.sal.marketing.vote.service.impl;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.apache.poi.hssf.usermodel.HSSFCell;
import org.apache.poi.hssf.usermodel.HSSFRow;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.baijia.commons.lang.utils.collection.CollectionUtils;
import com.baijia.tianxiao.dal.activity.constants.CommonConstant;
import com.baijia.tianxiao.dal.activity.dao.ActivityAccessLogDao;
import com.baijia.tianxiao.dal.activity.dao.TxActivityCommonDao;
import com.baijia.tianxiao.dal.activity.dao.draw.ActivityUserDao;
import com.baijia.tianxiao.dal.activity.dao.vote.VoteInfoDao;
import com.baijia.tianxiao.dal.activity.dao.vote.VoteOptionDao;
import com.baijia.tianxiao.dal.activity.dto.PageInfo;
import com.baijia.tianxiao.dal.activity.po.DairyCountStatistics;
import com.baijia.tianxiao.dal.activity.po.TxActivityCommon;
import com.baijia.tianxiao.dal.activity.po.draw.ActivityUser;
import com.baijia.tianxiao.dal.activity.po.vote.VoteInfo;
import com.baijia.tianxiao.dal.activity.po.vote.VoteOption;
import com.baijia.tianxiao.dal.org.constant.DeleteStatus;
import com.baijia.tianxiao.sal.marketing.commons.enums.TemplateTypeCategory;
import com.baijia.tianxiao.sal.marketing.commons.service.RedisService;
import com.baijia.tianxiao.sal.marketing.commons.service.TxActivityCommonService;
import com.baijia.tianxiao.sal.marketing.commons.utils.ExcelMailSender;
import com.baijia.tianxiao.sal.marketing.commons.utils.ExcelUtil;
import com.baijia.tianxiao.sal.marketing.export.dto.ExportVoteDetail;
import com.baijia.tianxiao.sal.marketing.vote.dto.VoteCacheDto;
import com.baijia.tianxiao.sal.marketing.vote.dto.VoteInfoRequest;
import com.baijia.tianxiao.sal.marketing.vote.dto.VoteInfoResponse;
import com.baijia.tianxiao.sal.marketing.vote.dto.VoteListDto;
import com.baijia.tianxiao.sal.marketing.vote.dto.VoteOptionDto;
import com.baijia.tianxiao.sal.marketing.vote.dto.VoteRenderDto;
import com.baijia.tianxiao.sal.marketing.vote.dto.VoteStatistics;
import com.baijia.tianxiao.sal.marketing.vote.service.VoteOptionService;
import com.baijia.tianxiao.sal.marketing.vote.service.VoteService;
import com.baijia.tianxiao.sal.marketing.vote.utils.VoteOptionUtil;
import com.baijia.tianxiao.util.GenericsUtils;
import com.baijia.tianxiao.util.date.DateUtil;

import lombok.extern.slf4j.Slf4j;

@Service
@Slf4j
public class VoteServiceImpl implements VoteService {

    public static final int VOTE_ACTIVITY_CATEGORY = TemplateTypeCategory.VOTE_TYPE.getType();

    @Autowired
    private VoteInfoDao voteInfoDao;
    @Autowired
    private VoteOptionDao optionDao;
    @Autowired
    private ActivityUserDao activityUserDao;
    @Autowired
    private ActivityAccessLogDao activityAccessLogDao;
    @Autowired
    private VoteOptionService voteOptionService;
    @Autowired
    private TxActivityCommonDao txActivityCommonDao;
    @Autowired
    private TxActivityCommonService txActivityCommonService;
    @Autowired
    private RedisService redisService;

    @Override
    public Map<Long, ActivityUser> getVoteResultByWechatOpenId(Long activityId, String wechatOpenId, int isVotedByDay) {
        String date = null;
        if (isVotedByDay == 1) {
            date = DateUtil.getStrByDateFormate(new Date(), "yyyy-MM-dd");
        }
        if (StringUtils.isBlank(wechatOpenId)) {
            return Collections.EMPTY_MAP;
        }
        List<ActivityUser> users = activityUserDao.selectActivityUsers(wechatOpenId, date, activityId,
            TemplateTypeCategory.VOTE_TYPE.getType());

        if (users != null && users.size() > 0) {
            Map<Long, ActivityUser> extractMap =
                CollectionUtils.extractMap(users, new CollectionUtils.Extracter<Long, ActivityUser>() {
                    @Override
                    public Long extract(ActivityUser paramE) {
                        return paramE.getOptionsId();
                    }
                });
            return extractMap;
        } else {
            return Collections.emptyMap();
        }
    }

    @Override
    @Transactional("yunyingTransactionManager")
    public VoteInfoResponse modifyVoteInfo(VoteInfoRequest request, Long orgId) {
        VoteInfo info = request.getVoteInfo();
        info.setUpdateTime(new Timestamp(new Date().getTime()));
        voteInfoDao.updateVoteActivity(info);
        this.reSetVoteCache(info.getActivityId());

        // 同步到txActivityCommon中
        TxActivityCommon conf = txActivityCommonDao.getTxActivityCommon(orgId,
            TemplateTypeCategory.VOTE_TYPE.getTypeId(), info.getActivityId());
        if (conf == null || conf.getDelStatus() == DeleteStatus.DELETED.getValue()) {
            // 提示该活动已经被删除或者不存在
            log.error("[Vote] this vote is note exit activityId:{}, conf param:{}", info.getActivityId(), conf);
        } else {
            log.info("[Vote] conf:{}, VoteInfo param:{}", conf, info);
            txActivityCommonService.saveOrUpdateTxActivityCommon(conf, TemplateTypeCategory.VOTE_TYPE, info, null);
        }

        List<VoteOption> optionList = VoteOptionUtil.parseOptionList(request);
        if (optionList != null) {
            List<Long> ids = new ArrayList<>(optionList.size());
            List<VoteOption> newOptions = new ArrayList<>();
            for (VoteOption option : optionList) {
                log.info("[Vote] OptionId=" + option.getId());
                if (option.getId() != null && option.getId() != 0) {
                    ids.add(option.getId());
                    option.setUpdateTime(new Date());
                    option.setStatus(1);
                    optionDao.updateVoteOption(option);
                } else {
                    option.setCreateTime(new Date());
                    option.setActivityId(info.getActivityId());
                    option.setUpdateTime(new Date());
                    option.setStatus(1);
                    newOptions.add(option);
                }
            }
            log.info("[Ids are : ] {} ", ids);
            optionDao.deleteVoteOptions(request.getActivityId(), ids);
            log.info("[Vote] add new option:" + newOptions);
            optionDao.batchInsertVoteOptions(newOptions);
        }
        VoteInfoResponse response = VoteInfoResponse.getVoteInfoResponse(info);
        response.setVoteOptions(voteOptionService.getOptionList(info.getActivityId()));
        return response;
    }

    @Override
    @Transactional("yunyingTransactionManager")
    public void modifyVoteInfoStatus(Long voteId, Long orgId, int status) {
        VoteInfo info = voteInfoDao.selectVoteInfo(voteId, orgId);
        if (info != null) {
            VoteInfo updateInfo = new VoteInfo();
            updateInfo.setActivityId(voteId);
            updateInfo.setStatus(status);
            updateInfo.setUpdateTime(new Timestamp(new Date().getTime()));
            voteInfoDao.updateVoteActivity(updateInfo);

            // 同步到tx_activity_conf
            TxActivityCommon conf =
                txActivityCommonDao.getTxActivityCommon(orgId, info.getTemplateTypeId(), info.getActivityId());
            conf.setStatus(status);
            conf.setUpdateTime(new Date());
            txActivityCommonService.saveOrUpdateTxActivityCommon(conf, null, null, null, "status", "updateTime");
        } else {
            log.warn("[Vote] vote is not exist.voteId=" + voteId);
        }
    }

    @Override
    public VoteInfoResponse getVoteInfoById(long voteId, long orgId) {
        VoteInfo vote = voteInfoDao.selectVoteInfo(voteId, orgId);
        if (vote != null) {
            VoteInfoResponse response = VoteInfoResponse.getVoteInfoResponse(vote);
            response.setVoteOptions(voteOptionService.getOptionList(voteId));
            List<VoteInfoResponse> list = new ArrayList<>(1);
            list.add(response);
            setStatistics(list);
            return response;
        } else {
            return null;
        }
    }

    /**
     * 根据查询到的投票活动进行统计数据填充
     *
     * @param voteBaseInfo
     */
    private void setStatistics(List<? extends VoteStatistics> voteBaseInfo) {
        if (GenericsUtils.isNullOrEmpty(voteBaseInfo)) {
            return;
        }

        Set<Long> activityIds = getIds(voteBaseInfo);

        Map<Long, Integer> accessUserCountResult =
            this.activityAccessLogDao.selectTotalByActivityId(new ArrayList<>(activityIds), VOTE_ACTIVITY_CATEGORY);
        Map<Long, Integer> partakeUserCountResult = getUserCount(activityIds);
        Map<Long, Map<Long, Integer>> voteUserResult =
            this.activityUserDao.statisticalVotes(new ArrayList<>(activityIds), VOTE_ACTIVITY_CATEGORY);

        for (VoteStatistics vbi : voteBaseInfo) {
            Integer browseCount = accessUserCountResult.get(vbi.getActivityId());
            Integer partakeCount = partakeUserCountResult.get(vbi.getActivityId());
            vbi.setBrowseCount(browseCount == null ? 0 : browseCount);
            vbi.setUserCount(partakeCount == null ? 0 : partakeCount);
            Map<Long, Integer> votedCountResult = voteUserResult.get(vbi.getActivityId());
            List<VoteOption> options = optionDao.selectVoteOptionsByVoteId(vbi.getActivityId());
            setVoteCount(vbi, options, votedCountResult);
            log.info("activity_id : {} , browseCount : {} , partakeCount : {} , votedCount : {} ", vbi.getActivityId(),
                browseCount, partakeCount, vbi.getVoteCount());
        }
    }

    private Set<Long> getIds(List<? extends VoteStatistics> statisticsList) {
        Set<Long> activityIds = new HashSet<>();
        for (VoteStatistics statistics : statisticsList) {
            activityIds.add(statistics.getActivityId());
        }
        return activityIds;
    }

    private Map<Long, Integer> getUserCount(Set<Long> activityIds) {
        Map<Long, Integer> retMap = new HashMap<>();
        for (Long activityId : activityIds) {
            Integer count = 0;
            List<DairyCountStatistics> list =
                activityUserDao.selectDairyCountStatisticsWithSingle(activityId, VOTE_ACTIVITY_CATEGORY);
            if (GenericsUtils.notNullAndEmpty(list)) {
                for (DairyCountStatistics statistics : list) {
                    count += statistics.getCount();
                }
            }
            retMap.put(activityId, count);
        }

        return retMap;
    }

    private int setVoteCount(VoteStatistics vbi, List<VoteOption> options, Map<Long, Integer> votedCountResult) {
        List<Long> optionIds = CollectionUtils.extractList(options, new CollectionUtils.Extracter<Long, VoteOption>() {
            @Override
            public Long extract(VoteOption arg0) {
                return arg0.getId();
            }
        });
        int votedCount = 0;
        if (GenericsUtils.notNullAndEmpty(votedCountResult)) {
            for (Map.Entry<Long, Integer> entry : votedCountResult.entrySet()) {
                Long optionId = entry.getKey();
                if (optionIds.contains(optionId)) {
                    if (entry.getValue() != 0) {
                        votedCount += entry.getValue();
                    }
                }
            }
        }
        vbi.setVoteCount(votedCount);
        return votedCount;
    }

    @Override
    public VoteRenderDto getVoteRenderDtoById(long voteId, String wechatOpenId) {
        long startTime = System.currentTimeMillis();
        VoteInfo info = voteInfoDao.selectVoteInfo(voteId, null, null);
        if (info == null) {
            return null;
        }
        VoteRenderDto vote = VoteRenderDto.getInstance(info);
        vote.setVoteOptions(voteOptionService.getOptionList(voteId));
        if (StringUtils.isBlank(vote.getTopPic())) {
            vote.setTopPic(CommonConstant.VOTE_DEFAULT_TOP_PIC);
        }
        log.info("填充活动详情 costTime:" + (System.currentTimeMillis() - startTime));

        Map<Long, ActivityUser> voteResultMap =
            getVoteResultByWechatOpenId(voteId, wechatOpenId, info.getIsVotedByDay());
        log.info("[Vote] Vote status=" + voteResultMap);
        if (vote.getVoteOptions() != null) {
            Map<Long, Map<Long, Integer>> voteUserResult =
                this.activityUserDao.statisticalVotes(Arrays.asList(voteId), VOTE_ACTIVITY_CATEGORY);
            Map<Long, Integer> optionVoteCounts = null;
            if (GenericsUtils.notNullAndEmpty(voteUserResult)) {
                optionVoteCounts = voteUserResult.get(voteId);
            } else {
                optionVoteCounts = Collections.emptyMap();
            }
            for (VoteOptionDto dto : vote.getVoteOptions()) {
                Integer count = optionVoteCounts.get(dto.getId());
                count = count == null ? 0 : count;
                dto.setVoteNum(count);
                if (voteResultMap.get(dto.getId()) == null) {
                    if (voteResultMap.size() >= info.getCountLimit()) {
                        dto.setStatus(0);
                    } else {
                        dto.setStatus(1);
                    }
                } else {
                    dto.setStatus(0);
                }
            }
        }

        if (info.getIsRankSort() != null && info.getIsRankSort() == 1) {
            Collections.sort(vote.getVoteOptions(), new Comparator<VoteOptionDto>() {
                @Override
                public int compare(VoteOptionDto o1, VoteOptionDto o2) {
                    int num1 = o1.getVoteNum() == null ? 0 : o1.getVoteNum();
                    int num2 = o2.getVoteNum() == null ? 0 : o2.getVoteNum();
                    return num2 - num1;
                }
            });
        }
        log.info("填充投票人详情 costTime:" + (System.currentTimeMillis() - startTime));
        return vote;
    }

    @Override
    public List<VoteListDto> getVoteList(int status, long orgId, PageInfo pageInfo) {
        List<VoteInfo> infoList = voteInfoDao.selectVoteInfoList(status, orgId, pageInfo);

        if (infoList != null) {
            List<VoteListDto> dtos = new ArrayList<>(infoList.size());
            for (VoteInfo info : infoList) {
                dtos.add(VoteListDto.getInstance(info));
            }
            setStatistics(dtos);
            return dtos;
        }

        return Collections.emptyList();
    }

    @Override
    public void exportVoteDetail(long activityId) {
        log.info("[ExportVote] start---------");
        long start = System.currentTimeMillis();
        List<ActivityUser> users = activityUserDao.selectAllUsers(activityId, VOTE_ACTIVITY_CATEGORY);
        VoteInfo info = voteInfoDao.selectVoteInfo(activityId, null);
        List<VoteOption> optionList = optionDao.selectVoteOptionsByVoteId(activityId);
        Map<Long, VoteOption> map =
            CollectionUtils.extractMap(optionList, new CollectionUtils.Extracter<Long, VoteOption>() {
                @Override
                public Long extract(VoteOption option) {
                    return option.getId();
                }
            });
        List<ExportVoteDetail> resultList = new ArrayList<>(users.size());
        if (users != null) {
            for (ActivityUser user : users) {
                ExportVoteDetail detail = new ExportVoteDetail();
                detail.setVoteDate(user.getDoneDate());
                detail.setWechatOpenId(user.getWechatOpenId());
                detail.setActivityName(info.getName());
                detail.setOptionName(map.get(user.getOptionsId()).getName());
                detail.setIntroduction(map.get(user.getOptionsId()).getIntroduction());
                resultList.add(detail);
            }
        }

        File file = new File("vote_detail_" + activityId + ".xls");
        if (file.exists()) {
            file.delete();
        }
        FileOutputStream fis = null;
        try {
            file.createNewFile();
            fis = new FileOutputStream(file);
            exportActivityList(resultList, fis);
            fis.flush();
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            try {
                fis.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        log.info("Path=" + file.getAbsolutePath());
        ExcelMailSender.sendMail("liuxiaopeng@baijiahulian.com", "投票明细", "", new File[] { file });
        // file.delete();
        log.info("[ExportVote] End ---------.Cost = " + (System.currentTimeMillis() - start));
    }

    public void exportActivityList(List<ExportVoteDetail> list, OutputStream out) {
        HSSFWorkbook workbook = new HSSFWorkbook();
        HSSFSheet sheet = workbook.createSheet();
        String[] TITLE = { "活动名称", "投票项目名称", "投票选项简介", "投票时间", "微信id" };
        for (int i = 0; i < TITLE.length; i++) {
            sheet.setColumnWidth(i, 20 * 256);
        }

        HSSFRow row = sheet.createRow(0);
        for (int i = 0; i < TITLE.length; i++) {
            HSSFCell cell = row.createCell(i);
            cell.setCellValue(TITLE[i]);
        }

        if (list != null && list.size() > 0) {
            for (int i = 0; i < list.size(); i++) {
                ExportVoteDetail info = list.get(i);
                row = sheet.createRow(i + 1);

                ExcelUtil.createCell(row, 0, info.getActivityName());
                ExcelUtil.createCell(row, 1, info.getOptionName());
                ExcelUtil.createCell(row, 2, info.getIntroduction());
                ExcelUtil.createCell(row, 3, info.getVoteDate());
                ExcelUtil.createCell(row, 4, info.getWechatOpenId());
            }
        } else {
            log.warn("[Export] No Activity.");
        }
        try {
            workbook.write(out);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    public VoteCacheDto reSetVoteCache(Long activityId) {
        VoteInfo voteInfo = voteInfoDao.selectVoteInfo(activityId, null);
        VoteCacheDto voteCacheDto = VoteCacheDto.getInstanceByVoteInfo(voteInfo);
        this.redisService.setVoteActivityBase(activityId, voteCacheDto);
        return voteCacheDto;
    }

}
