package com.baijia.tianxiao.sal.marketing.referral.service.impl;

import java.sql.Timestamp;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.baijia.tianxiao.dal.activity.constants.CommonConstant;
import com.baijia.tianxiao.dal.activity.dao.referral.GiftInfoDao;
import com.baijia.tianxiao.dal.activity.dao.referral.ReferralInfoDao;
import com.baijia.tianxiao.dal.activity.dao.referral.ReferralRecordDao;
import com.baijia.tianxiao.dal.activity.dto.PageInfo;
import com.baijia.tianxiao.dal.activity.po.referral.GiftInfo;
import com.baijia.tianxiao.dal.activity.po.referral.ReferralInfo;
import com.baijia.tianxiao.dal.activity.po.referral.ReferralRecord;
import com.baijia.tianxiao.sal.marketing.activity.dto.ResultWrapper;
import com.baijia.tianxiao.sal.marketing.commons.enums.ResultType;
import com.baijia.tianxiao.sal.marketing.commons.utils.TupleUtil;
import com.baijia.tianxiao.sal.marketing.commons.utils.TwoTuple;
import com.baijia.tianxiao.sal.marketing.referral.dto.BrokerDto;
import com.baijia.tianxiao.sal.marketing.referral.dto.CommissionDetail;
import com.baijia.tianxiao.sal.marketing.referral.dto.CommissionDto;
import com.baijia.tianxiao.sal.marketing.referral.dto.CommissionListDto;
import com.baijia.tianxiao.sal.marketing.referral.dto.ReferralCustomer;
import com.baijia.tianxiao.sal.marketing.referral.enums.ReferralCode;
import com.baijia.tianxiao.sal.marketing.referral.service.ReferralRecordService;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.GenericsUtils;
import com.baijia.tianxiao.util.date.DateUtil;

import lombok.extern.slf4j.Slf4j;

/**
 * Created by liuxp on 16/3/4.
 * <p>
 * 转介绍记录服务
 */
@Service
@Slf4j
public class ReferralRecordServiceImpl implements ReferralRecordService {

    @Autowired
    private ReferralRecordDao recordDao;
    @Autowired
    private ReferralInfoDao infoDao;
    @Autowired
    private GiftInfoDao giftInfoDao;

    @Override
    public ReferralCode addReferralRecord(ReferralRecord record) {
        ReferralInfo info = infoDao.getReferralInfo(record.getActivityId(), null);
        if (info == null) {
            log.warn("[Referral] Add record fail.info is not exit.activityId={}", record.getActivityId());
            return ReferralCode.FAIL;
        }

        // 校验活动的有效性
        ReferralCode validate = validate(info);
        if (validate != null) {
            return validate;
        }

        ReferralRecord existRecord = recordDao.selectByReferralPhone(record.getReferralPhone());
        if (existRecord != null) {
            log.info("[Referral] Referral phone is exit.phone=" + record.getReferralPhone());
            return ReferralCode.REPEAT;
        }

        Date today = new Date();
        Integer count = recordDao.selectCountByBrokerAndTime(record.getActivityId(), record.getBrokerPhone(),
            DateUtil.getStartOfDay(today), DateUtil.getEndOfDay(today));

        if (count != null && count >= CommonConstant.REFERRAL_COUNT_LIMIT) {
            return ReferralCode.EVIL;
        }

        record.setCreateTime(new Timestamp(today.getTime()));
        record.setUpdateTime(new Timestamp(today.getTime()));
        recordDao.addReferralRecord(record);

        return ReferralCode.SUCCESS;
    }

    private ReferralCode validate(ReferralInfo info) {
        if (info.getStatus() == 0) {
            return ReferralCode.CLOSED;
        }
        Date current = new Date();
        Date begin = info.getStartTime();
        Date end = info.getEndTime();
        if (current.before(begin)) {
            return ReferralCode.NOT_START;
        } else if (current.after(end)) {
            return ReferralCode.IS_END;
        }
        return null;
    }

    @Override
    public List<ReferralCustomer> getCustomerListByActivityId(long activityId, PageInfo pageInfo) {
        List<ReferralRecord> records = recordDao.selectRecordsByPage(activityId, pageInfo);
        if (records == null || records.size() < 1) {
            return Collections.emptyList();
        } else {
            List<ReferralCustomer> customers = new ArrayList<>(records.size());
            for (ReferralRecord record : records) {
                customers.add(ReferralCustomer.getInstanceFromRecord(record));
            }
            return customers;
        }
    }

    @Override
    public List<BrokerDto> getBrokerList(long activityId, PageInfo page) {

        List<ReferralRecord> brokers = recordDao.selectBrokersByActivityId(activityId, page); // 从数据库中获取经纪人信息

        if (brokers == null) {
            return Collections.emptyList();
        }

        List<BrokerDto> dtos = new ArrayList<>(brokers.size());

        for (ReferralRecord broker : brokers) {
            dtos.add(BrokerDto.getInstance(broker));
        }
        return dtos;
    }

    @Override
    public ResultWrapper<ReferralRecord> setBrokerGift(long recordId, long giftId, long orgId) {
        ResultWrapper<ReferralRecord> result = new ResultWrapper<>();
        ReferralRecord record = recordDao.selectRecordById(recordId);
        if (record == null) {
            result.setRetType(ResultType.FAIL);
            result.setRetDesc("标记失败");
            log.warn("[Referral] mark gift fail.record is not exit.recordId=" + recordId);
            return result;
        }

        if (record.getMarkTime() != null && record.getSettlementStatus() == 1) {
            result.setRetDesc("该记录已结算，不能再标记");
            result.setRetType(ResultType.FAIL);
            return result;
        }
        ReferralInfo info = infoDao.getReferralInfo(record.getActivityId(), orgId);

        if (info == null) {
            result.setRetType(ResultType.FAIL);
            result.setRetDesc("标记失败");
            log.warn("[Referral] mark gift fail.info is not exit.activityId={},orgId={}", record.getActivityId(),
                orgId);
            return result;
        }

        GiftInfo gift = giftInfoDao.selectGiftById(giftId);
        if (gift == null) {
            result.setRetType(ResultType.FAIL);
            result.setRetDesc("标记失败");
            log.warn("[Referral] mark gift fail.Gift is not exit.giftId=" + giftId);
            return result;
        }

        record.setGiftId(giftId);
        record.setGiftDesc(gift.getGiftDesc());
        record.setIntroduceWayDesc(gift.getIntroduceWayDesc());
        record.setMarkTime(new Timestamp(new Date().getTime()));
        record.setType(gift.getType());

        recordDao.updateReferralRecord(record);

        result.setRetType(ResultType.SUCC);
        result.setData(record);

        return result;
    }

    /**
     * 获取待付佣金列表里单条记录的详情
     * 
     * @return
     */
    @Override
    public List<CommissionDetail> referralRecordDetail(long activityId, long orgId, String brokerPhone, String month,
        Integer status, PageInfo pageInfo) {

        ReferralInfo ri = this.infoDao.getReferralInfo(activityId, orgId);
        if (ri == null) {
            log.info("can not find any activity with id : {} and orgId {} ", activityId, orgId);
            return Collections.emptyList();
        }

        TwoTuple<Date, Date> timeRange = findStartAndEnd(month);

        List<ReferralRecord> referralRecords = this.recordDao.selectReferralRecordDetail(activityId, brokerPhone,
            status, timeRange.first, timeRange.second, pageInfo);

        List<CommissionDetail> cds = new ArrayList<>();

        // if (GenericsUtils.notNullAndEmpty(referralRecords)) {
        // List<Long> giftIds =
        // CollectionUtils.extractList(referralRecords, new CollectionUtils.Extracter<Long, ReferralRecord>() {
        // @Override
        // public Long extract(ReferralRecord arg0) {
        // return arg0.getGiftId();
        // }
        // });
        //
        // Map<Long, GiftInfo> giftInfos = this.giftInfoDao.selectGiftByIds(giftIds);

        for (ReferralRecord rr : referralRecords) {
            GiftInfo gift = getGift(rr);
            CommissionDetail cd = CommissionDetail.buildDto(rr, gift);
            cds.add(cd);
        }
        return cds;
    }

    private GiftInfo getGift(ReferralRecord rr) {
        Long id = rr.getGiftId();
        GiftInfo giftInfo = null;
        if (id != null && id != -1L) {
            giftInfo = new GiftInfo();
            String giftDesc = rr.getGiftDesc();
            String introduceWayDesc = rr.getIntroduceWayDesc();
            Integer type = rr.getType();
            giftInfo.setId(id);
            giftInfo.setGiftDesc(giftDesc);
            giftInfo.setIntroduceWayDesc(introduceWayDesc);
            giftInfo.setType(type);
        }
        return giftInfo;
    }

    /**
     * 按照月份对待付佣金进行结算
     */
    @Override
    public void settleReferralByMonth(long activityId, String brokerPhone, String month) {

        TwoTuple<Date, Date> findStartAndEnd = findStartAndEnd(month);

        synchronized (this) {
            List<ReferralRecord> records =
                recordDao.selectBrokersByTime(activityId, brokerPhone, 2, findStartAndEnd.first, findStartAndEnd.second);
            if (records != null && records.size() > 0) {
                for (ReferralRecord record : records) {
                    record.setUpdateTime(new Timestamp(new Date().getTime()));
                    record.setSettlementStatus(1);
                    recordDao.updateReferralRecord(record);
                }
            }
        }
    }

    public TwoTuple<Date, Date> findStartAndEnd(String month) {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM");
        Date startTime = new Date();
        try {
            startTime = sdf.parse(month);
        } catch (ParseException e) {
            log.warn("[Referral] Param error.month=" + month);
        }
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(startTime);
        calendar.set(Calendar.DAY_OF_MONTH, 1);
        startTime = DateUtil.getStartOfDay(calendar.getTime());

        calendar.add(Calendar.MONTH, 1);
        Date endTime = DateUtil.getStartOfDay(calendar.getTime());
        return TupleUtil.tuple(startTime, endTime);
    }

    @Override
    public CommissionListDto getCommissionListByMonth(long activityId, String month, final int status,
        PageDto pageDto) {
        CommissionListDto listDto = new CommissionListDto();
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM");
        Date startTime = new Date();
        try {
            startTime = sdf.parse(month);
        } catch (ParseException e) {
            log.warn("[Referral] Param error.month=" + month);
        }

        Calendar calendar = Calendar.getInstance();
        calendar.setTime(startTime);
        calendar.set(Calendar.DAY_OF_MONTH, 1);
        startTime = DateUtil.getStartOfDay(calendar.getTime());

        calendar.add(Calendar.MONTH, 1);
        Date endTime = DateUtil.getStartOfDay(calendar.getTime());

        List<ReferralRecord> records = recordDao.selectBrokersByTime(activityId,"", status, startTime, endTime);

        log.info("referralRecords are :{}", records);

        Map<String, CommissionDto> map = new HashMap<>();
        if (records != null && records.size() > 0) {
            listDto.setCount(records.size());
            for (ReferralRecord record : records) {
                log.info("record is : {} ", record);
                CommissionDto dto = map.get(record.getBrokerPhone());
                boolean isInit = false;
                if (dto == null) {
                    dto = CommissionDto.getInstanceFromRecord(record);
                    isInit = true;
                } else {
                    if (status == 2 && dto.getLastMarkTime() < record.getMarkTime().getTime()) {
                        dto.setBrokerName(record.getBrokerName());
                        dto.setLastMarkTime(record.getMarkTime().getTime());
                    }
                    if (status == 1 && dto.getSettleTime() < record.getUpdateTime().getTime()) {
                        dto.setBrokerName(record.getBrokerName());
                        dto.setSettleTime(record.getUpdateTime().getTime());
                    }
                }
                if (record.getType() == 1 && !isInit) {
                    if (isDigits(dto.getAmount())) {
                        log.info(" isDigits {} ", dto.getAmount());
                        String amount = dto.getAmount();
                        amount = removeDot(amount);
                        Integer value = Integer.parseInt(amount);
                        String giftValue = record.getGiftDesc();
                        giftValue = removeDot(giftValue);
                        value += Integer.parseInt(giftValue);
                        dto.setAmount(value.toString());
                    }
                }
                map.put(record.getBrokerPhone(), dto);
            }
        } else {
            listDto.setHasMore(0);
            return listDto;
        }

        List<CommissionDto> list = new ArrayList<>(map.values());

        log.info("record list is : {} ", list);

        Collections.sort(list, new Comparator<CommissionDto>() {
            @Override
            public int compare(CommissionDto o1, CommissionDto o2) {
                if (status == 2) {
                    return (int) (o2.getLastMarkTime() - o1.getLastMarkTime());
                } else {
                    return (int) (o2.getSettleTime() - o1.getSettleTime());
                }
            }
        });

        Integer totalAmount = 0;
        for (CommissionDto cd : list) {
            String amount = cd.getAmount().trim();
            log.info("amount is :{} ", totalAmount);
            if (GenericsUtils.notNullAndEmpty(amount) && cd.getGiftType() == 1) {
                try {
                    Integer value = Integer.parseInt(amount);
                    totalAmount += value;
                } catch (NumberFormatException e) {
                }
            }
        }
        int startIndex = (pageDto.getPageNum() - 1) * pageDto.getPageSize();
        listDto.setTotalAmount(totalAmount);
        if (GenericsUtils.isNullOrEmpty(list) || (startIndex > list.size())) {
            listDto.setList(Collections.<CommissionDto> emptyList());
            listDto.setHasMore(0);
            return listDto;
        }
        int endIndex = Math.min(startIndex + pageDto.getPageSize(), list.size());

        listDto.setList(list.subList(startIndex, endIndex));

        return listDto;
    }

    private String removeDot(String number) {
        return number.replaceAll("([0-9]+)\\.?[\\d]*", "$1");
    }

    private static boolean isDigits(String amount) {
        String regex = "[1-9][0-9]*\\.?[\\d]*";
        return Pattern.matches(regex, amount.trim());
    }

}
