package com.baijia.tianxiao.sal.marketing.commons.service.impl;

import java.util.Date;

import org.apache.commons.lang3.time.DateFormatUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.dao.DataAccessException;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.stereotype.Service;

import com.baijia.tianxiao.constants.TianXiaoConstant;
import com.baijia.tianxiao.redis.AbstractBaseRedisDao;
import com.baijia.tianxiao.sal.marketing.commons.enums.ConstantEnums;
import com.baijia.tianxiao.sal.marketing.commons.enums.EmailType;
import com.baijia.tianxiao.sal.marketing.commons.service.RedisService;
import com.baijia.tianxiao.sal.marketing.draw.dto.DrawCacheDto;
import com.baijia.tianxiao.sal.marketing.vote.dto.VoteCacheDto;
import com.baijia.tianxiao.util.GenericsUtils;

import lombok.extern.slf4j.Slf4j;

/**
 * Created by liuxp on 16/1/20.
 */
@Service
@Slf4j
public class RedisServiceImpl extends AbstractBaseRedisDao<String, Object> implements RedisService {

    public final Logger logger = LoggerFactory.getLogger(getClass());

    public final static String emailCountKey = "TX#ACTIVITY#EMAIL#COUNT";
    public final static String DRAW_ACTIVITY_KEY = "TX#ACTIVITY#DRAW";
    public final static String VOTE_ACTIVITY_KEY = "TX#ACTIVITY#VOTE";
    public final static String ACTIVITY_VISITY_COUNT_KEY = "TX#ACTIVITY#VISITY#COUNT";
    public final static String ACTIVITY_VISITY_STATUS_KEY = "TX#ACTIVITY#STATUS_NEW";

    private final static int EMAIL_COUNT_LIMIT;

    static {
        if (GenericsUtils.isNullOrEmpty(ConstantEnums.EMAIL_COUNT_LIMIT.value())) {
            EMAIL_COUNT_LIMIT = Integer.parseInt(ConstantEnums.EMAIL_COUNT_LIMIT.value());
        } else {
            EMAIL_COUNT_LIMIT = 5;
        }
    }

    /**
     * 统计某个openId访问某个活动的次数
     * 
     * @param wechatOpenId
     * @param activityId
     * @param activityTemplateId
     * @return
     */
    @Override
    public Integer visitCount(String wechatOpenId, Long orgId) {
        final String fieldStr = getAcitivityVisitCountKey(wechatOpenId, orgId);
        return redisTemplate.execute(new RedisCallback<Integer>() {
            @Override
            public Integer doInRedis(RedisConnection connection) throws DataAccessException {
                connection.select(TianXiaoConstant.TX_PV_REDIS_DB);
                RedisSerializer<String> serializer = getRedisSerializer();
                byte[] keyBytes = serializer.serialize(ACTIVITY_VISITY_COUNT_KEY);
                byte[] field = serializer.serialize(fieldStr);
                RedisSerializer<?> redisSerializer = redisTemplate.getDefaultSerializer();
                byte[] value = connection.hGet(keyBytes, field);
                return (Integer) redisSerializer.deserialize(value);
            }
        });
    }

    private String getAcitivityVisitCountKey(String wechatOpenId, long orgId) {
        String format = "%s_%s";
        final String fieldStr = String.format(format, orgId, wechatOpenId);
        return fieldStr;
    }

    /**
     * 设置某个类型的活动的某个微信用户的访问次数
     * 
     * @param wechatOpenId
     * @param activityId
     * @param activityTemplateId
     * @param count
     */
    @Override
    public void setVisitCount(String wechatOpenId, Long orgId, final Integer count) {
        final String fieldStr = getAcitivityVisitCountKey(wechatOpenId, orgId);
        redisTemplate.execute(new RedisCallback<Integer>() {
            @Override
            public Integer doInRedis(RedisConnection connection) throws DataAccessException {
                connection.select(TianXiaoConstant.TX_PV_REDIS_DB);
                RedisSerializer<String> serializer = getRedisSerializer();
                byte[] keyBytes = serializer.serialize(ACTIVITY_VISITY_COUNT_KEY);
                byte[] field = serializer.serialize(fieldStr);
                RedisSerializer redisSerializer = redisTemplate.getDefaultSerializer();
                byte[] value = redisSerializer.serialize(count);
                connection.hSet(keyBytes, field, value);
                log.info("success to update visit count for for field : {} with count : {}  ", fieldStr, count);
                return null;
            }
        });
    }

    @Override
    public void setDrawActivityBase(final long activityId, final DrawCacheDto cacheDto) {
        redisTemplate.execute(new RedisCallback<Object>() {
            @Override
            public Object doInRedis(RedisConnection connection) throws DataAccessException {
                connection.select(TianXiaoConstant.TX_PV_REDIS_DB);
                RedisSerializer<String> serializer = getRedisSerializer();
                byte[] key = serializer.serialize(DRAW_ACTIVITY_KEY);
                byte[] field = serializer.serialize(getDrawActivityKey(activityId));
                RedisSerializer redisSerializer = redisTemplate.getDefaultSerializer();
                byte[] value = redisSerializer.serialize(cacheDto);
                connection.hSet(key, field, value);
                logger.info("[Redis] Update success. ActivityId=" + activityId);
                return null;
            }
        });

    }

    @Override
    public DrawCacheDto getDrawActivityBase(final long activityId) {
        logger.info("[Redis] activityId=" + activityId);
        return redisTemplate.execute(new RedisCallback<DrawCacheDto>() {
            @Override
            public DrawCacheDto doInRedis(RedisConnection connection) throws DataAccessException {
                logger.info("[Redis] Query start.");
                connection.select(TianXiaoConstant.TX_PV_REDIS_DB);
                RedisSerializer<String> serializer = getRedisSerializer();
                byte[] key = serializer.serialize(DRAW_ACTIVITY_KEY);
                byte[] field = serializer.serialize(getDrawActivityKey(activityId));

                RedisSerializer redisSerializer = redisTemplate.getDefaultSerializer();

                byte[] value = connection.hGet(key, field);
                logger.info("[Redis] Query success.");
                return (DrawCacheDto) redisSerializer.deserialize(value);
            }
        });
    }

    /**
     * 获取到投票活动
     */
    @Override
    public VoteCacheDto getVoteActivityBase(final Long activityId) {
        logger.info("[redis] activityId: ", activityId);
        return redisTemplate.execute(new RedisCallback<VoteCacheDto>() {

            @Override
            public VoteCacheDto doInRedis(RedisConnection connection) throws DataAccessException {
                logger.info("[Redis Query Start.]");
                connection.select(TianXiaoConstant.TX_PV_REDIS_DB);
                RedisSerializer<String> serializer = getRedisSerializer();
                byte[] key = serializer.serialize(VOTE_ACTIVITY_KEY);
                byte[] field = serializer.serialize(getVoteActivityKey(activityId));

                RedisSerializer<?> redisSerializer = redisTemplate.getDefaultSerializer();
                byte[] value = connection.hGet(key, field);
                VoteCacheDto vcd = (VoteCacheDto) redisSerializer.deserialize(value);
                logger.info("[Find VoteCacheDto : {} ]", vcd);
                return vcd;
            }
        });
    }

    /**
     * 设置一个投票活动
     */
    @Override
    public void setVoteActivityBase(final long activityId, final VoteCacheDto cacheDto) {
        redisTemplate.execute(new RedisCallback<Object>() {
            @Override
            public Object doInRedis(RedisConnection connection) throws DataAccessException {
                connection.select(TianXiaoConstant.TX_PV_REDIS_DB);
                RedisSerializer<String> serializer = getRedisSerializer();
                byte[] key = serializer.serialize(VOTE_ACTIVITY_KEY);
                byte[] field = serializer.serialize(getVoteActivityKey(activityId));
                RedisSerializer redisSerializer = redisTemplate.getDefaultSerializer();
                byte[] value = redisSerializer.serialize(cacheDto);
                connection.hSet(key, field, value);
                logger.info("[Redis] Update success. ActivityId=" + activityId);
                return null;
            }
        });

    }

    private String getVoteActivityKey(Long activityId) {
        StringBuilder sb = new StringBuilder();
        sb.append("VOTE").append("#").append(activityId);
        return sb.toString();
    }

    @Override
    public boolean addOrgEmailCount(final long orgId, final long activityId, final int templateTypeId,
        final EmailType emailType) {
        return redisTemplate.execute(new RedisCallback<Boolean>() {
            public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                connection.select(TianXiaoConstant.TX_PV_REDIS_DB);
                RedisSerializer<String> serializer = getRedisSerializer();
                byte[] key = serializer.serialize(emailCountKey);
                String fieldStr = getOrgKeyByDate(orgId, activityId, templateTypeId, emailType);
                byte[] field = serializer.serialize(fieldStr);
                byte[] value = connection.hGet(key, field);
                if (value == null) {
                    connection.hSet(key, field, "1".getBytes());
                    logger.info("[Email] Init,key=" + fieldStr);
                    return true;
                } else {
                    int count = Integer.parseInt(serializer.deserialize(value));
                    if (count < EMAIL_COUNT_LIMIT) {
                        count = count + 1;
                        connection.hSet(key, field, String.valueOf(count).getBytes());
                        logger.info("[Email] key=" + fieldStr + ";count=" + count);
                        return true;
                    } else {
                        return false;
                    }
                }
            }
        });
    }

    @Override
    public boolean decreaseEmailCountWhileError(final long orgId, final long activityId, final int templateTypeId,
        final EmailType emailType) {
        return redisTemplate.execute(new RedisCallback<Boolean>() {
            public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                connection.select(TianXiaoConstant.TX_PV_REDIS_DB);
                RedisSerializer<String> serializer = getRedisSerializer();
                byte[] key = serializer.serialize(emailCountKey);
                String fieldStr = getOrgKeyByDate(orgId, activityId, templateTypeId, emailType);
                byte[] field = serializer.serialize(fieldStr);
                byte[] value = connection.hGet(key, field);
                if (value != null) {
                    int count = Integer.parseInt(serializer.deserialize(value));
                    count = (count != 0) ? count - 1 : 0;
                    connection.hSet(key, field, String.valueOf(count).getBytes());
                    logger.info("[decreaseEmailCountWhileError] key=" + fieldStr + ";count=" + count);
                }
                return true;
            }
        });
    }

    private String getOrgKeyByDate(long orgId, long activityId, int typeId, EmailType emailType) {
        StringBuilder sb = new StringBuilder();
        sb.append(orgId).append("#").append(activityId).append("#").append(typeId).append("#")
            .append(emailType.getCode()).append("#").append(DateFormatUtils.format(new Date(), "yyyyMMdd"));
        return sb.toString();
    }

    private String getDrawActivityKey(long activityId) {
        StringBuilder sb = new StringBuilder();
        sb.append("DRAW").append("#").append(activityId);
        return sb.toString();
    }

    @Override
    public void clearField(final String key, final String field) {
        redisTemplate.execute(new RedisCallback<Boolean>() {
            public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                connection.select(TianXiaoConstant.TX_PV_REDIS_DB);
                RedisSerializer<String> serializer = getRedisSerializer();
                byte[] keyBytes = serializer.serialize(key);
                byte[] fieldBytes = serializer.serialize(field);
                Long ret = connection.hDel(keyBytes, fieldBytes);
                log.info(" try to clear field : {} with key  : {} ,and return is : {} ", field, key, ret);
                return true;
            }
        });
    }

    @Override
    public void setchangeActivityStatus(final Long activityId, final Integer activityType, final Integer status) {
        redisTemplate.execute(new RedisCallback<Boolean>() {
            public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                connection.select(TianXiaoConstant.TX_PV_REDIS_DB);
                RedisSerializer<String> serializer = getRedisSerializer();
                byte[] keyBytes = serializer.serialize(ACTIVITY_VISITY_STATUS_KEY);
                String format = "%s_%s";
                final String fieldStr = String.format(format, activityType, activityId);
                byte[] fieldBytes = serializer.serialize(fieldStr);
                RedisSerializer redisSerializer = redisTemplate.getDefaultSerializer();
                byte[] value = redisSerializer.serialize(status);
                connection.hSet(keyBytes, fieldBytes, value);
                log.info("success to changeActivityStatus to : {} ", status);
                return true;
            }
        });
    }

    @Override
    public Integer getActivityStatus(final Long activityId, final Integer activityType) {
        return redisTemplate.execute(new RedisCallback<Integer>() {
            public Integer doInRedis(RedisConnection connection) throws DataAccessException {
                connection.select(TianXiaoConstant.TX_PV_REDIS_DB);
                RedisSerializer<String> serializer = getRedisSerializer();
                byte[] keyBytes = serializer.serialize(ACTIVITY_VISITY_STATUS_KEY);
                String format = "%s_%s";
                final String fieldStr = String.format(format, activityType, activityId);
                byte[] fieldBytes = serializer.serialize(fieldStr);
                byte[] hGet = connection.hGet(keyBytes, fieldBytes);
                RedisSerializer redisSerializer = redisTemplate.getDefaultSerializer();
                Integer status = (Integer) redisSerializer.deserialize(hGet);
                log.info("success to changeActivityStatus to : {} ", status);
                return status;
            }
        });
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.baijia.tianxiao.sal.marketing.commons.service.RedisService#getRedisTemplate()
     */
    @Override
    public RedisTemplate<String, Object> getRedisTemplate() {
        return this.redisTemplate;
    }

    public Boolean setNXKey(final String key, final Object valueObj) {
        return this.redisTemplate.execute(new RedisCallback<Boolean>() {
            @Override
            public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                RedisSerializer redisSerializer = redisTemplate.getDefaultSerializer();
                byte[] value = redisSerializer.serialize(valueObj);
                Boolean setNX = connection.setNX(key.getBytes(), value);
                return setNX;
            }
        });
    }

    @Override
    public void setKey(final String key, final Object valueObj) {
        this.redisTemplate.execute(new RedisCallback<Boolean>() {
            @Override
            public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                RedisSerializer redisSerializer = redisTemplate.getDefaultSerializer();
                byte[] value = redisSerializer.serialize(valueObj);
                connection.set(key.getBytes(), value);
                return false;
            }
        });
    }

    public <T> T getKeyValue(final String key) {
        return this.redisTemplate.execute(new RedisCallback<T>() {
            @Override
            public T doInRedis(RedisConnection connection) throws DataAccessException {
                byte[] bs = connection.get(key.getBytes());
                return (T) redisTemplate.getDefaultSerializer().deserialize(bs);
            }
        });
    }

}
