/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */
package com.baijia.tianxiao.dal.wechat.dao.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections4.map.HashedMap;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.stereotype.Repository;

import com.baijia.tianxiao.dal.wechat.constant.WechatDataSyncStatus;
import com.baijia.tianxiao.dal.wechat.dao.AuthorizerInfoDao;
import com.baijia.tianxiao.dal.wechat.dao.UnifiedWechatAccountDao;
import com.baijia.tianxiao.dal.wechat.po.AuthorizerInfo;
import com.baijia.tianxiao.dal.wechat.po.UnifiedWechatAccount;
import com.baijia.tianxiao.sqlbuilder.SingleSqlBuilder;
import com.baijia.tianxiao.sqlbuilder.support.JdbcTemplateDaoSupport;
import com.baijia.tianxiao.util.GenericsUtils;
import com.google.common.collect.Maps;

import lombok.extern.slf4j.Slf4j;

/**
 * @author : zhenyujian
 * @title : AuthorizationInfoDaoImpl
 * @description :
 * @date : 2015年12月5日 下午4:21:41
 */
@Repository("tianxiao_dal_yunying_authorizerInfoDao")
@Slf4j
public class AuthorizerInfoDaoImpl extends JdbcTemplateDaoSupport<AuthorizerInfo> implements AuthorizerInfoDao {

    public AuthorizerInfoDaoImpl() {
        super(AuthorizerInfo.class);
    }

    /**
     * 不应该直接在dao层相互依赖.....but...
     */
    @Autowired
    private UnifiedWechatAccountDao unifiedWechatAccountDao;

    @Override
    public void deleteByOrgId(Integer orgId) {
        Map<String, Object> condition = new HashMap<String, Object>();
        condition.put("orgId", orgId);
        delByCondition(condition);
    }

    @Override
    public void resetOngoingStatus() {
        StringBuilder sb = new StringBuilder();
        sb.append(
            "update yunying.org_wechat_authorizer_info set fans_sync_status=:NOSTART,media_news_sync_status=:NOSTART ");
        sb.append("where fans_sync_status=:ONGOING or media_news_sync_status = :ONGOING");

        Map<String, Integer> params = new HashedMap<String, Integer>();
        params.put("NOSTART", WechatDataSyncStatus.NOTSTART.getValue());
        params.put("ONGOING", WechatDataSyncStatus.ONGOING.getValue());

        log.debug("wechat - AuthorizerInfoServiceImpl - resetOngoing sql={},params={}", sb.toString(), params);
        getNamedJdbcTemplate().update(sb.toString(), params);
        log.debug("wechat - AuthorizerInfoServiceImpl - resetOngoing success");
    }

    @Override
    public void deleteByAuthorizerAppId(String authorizerAppId) {
        Map<String, Object> condition = new HashMap<String, Object>();
        condition.put("authorizerAppId", authorizerAppId);
        delByCondition(condition);
    }

    /**
     * 需要加上统一公众号的查询逻辑
     */
    @Override
    public AuthorizerInfo getByOrgId(Integer orgId) {
        return this.getByOrgId(orgId, true);
    }

    @Override
    public AuthorizerInfo getByUserName(String userName) {
        SingleSqlBuilder<AuthorizerInfo> builder = createSqlBuilder();
        builder.eq("userName", userName);
        return uniqueResult(builder);
    }

    @Override
    public AuthorizerInfo getByAuthorizerAppId(String authorizerAppId) {
        SingleSqlBuilder<AuthorizerInfo> builder = createSqlBuilder();
        builder.eq("authorizerAppId", authorizerAppId);
        return uniqueResult(builder);
    }

    /**
     * 处理根据orgId查询微信appId,在某个机构没有查到的时候,继续尝试统一公众号的查询逻辑
     */
    @Override
    public Map<Long, String> getExistOrgIds(List<Long> tianxiaoOrgIds) {
        Map<String, Object> param = new HashMap<>();
        param.put("orgIds", tianxiaoOrgIds);
        String sql =
            "select org_id, authorizer_app_id from yunying.org_wechat_authorizer_info where org_id in (:orgIds)";
        Map<Long, String> query =
            this.getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<Map<Long, String>>() {
                @Override
                public Map<Long, String> extractData(ResultSet rs) throws SQLException, DataAccessException {
                    Map<Long, String> map = new HashMap<>();
                    while (rs.next()) {
                        Long orgId = rs.getLong("org_id");
                        String authorizer_app_id = rs.getString("authorizer_app_id");
                        map.put(orgId, authorizer_app_id);
                    }
                    return map;
                }
            });
        query = (query == null ? Maps.<Long, String> newHashMap() : query);
        for (Long orgId : tianxiaoOrgIds) {
            String appId = query.get(orgId);
            if (appId == null) {
                AuthorizerInfo byOrgId = this.getByOrgId(orgId.intValue(), true);
                if (byOrgId != null) {
                    String masterAppId = byOrgId.getAuthorizerAppId();
                    query.put(orgId, masterAppId);
                }
            }
        }
        return query;
    }

    @Override
    public List<AuthorizerInfo> getAuthorizerInfosByOrgIds(Collection<Long> orgIds) {
        SingleSqlBuilder<AuthorizerInfo> builder = createSqlBuilder();
        builder.in("org_id", orgIds);
        List<AuthorizerInfo> queryList = queryList(builder);
        Map<Integer, AuthorizerInfo> fieldMap = GenericsUtils.toFieldMap(queryList, "orgId");
        for (Long orgId : orgIds) {
            AuthorizerInfo ai = fieldMap.get(orgId.intValue());
            if (ai == null) {
                ai = this.getByOrgId(orgId.intValue(), true);
                if (ai != null) {
                    queryList.add(ai);
                }
            }
        }
        return queryList;
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.baijia.tianxiao.dal.wechat.dao.AuthorizerInfoDao#getByOrgIdWithoutMaster(java.lang.Integer)
     */

    @Override
    public AuthorizerInfo getByOrgId(Integer orgId, boolean withMasterSearch) {
        SingleSqlBuilder<AuthorizerInfo> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        AuthorizerInfo uniqueResult = uniqueResult(builder);
        if (uniqueResult == null && withMasterSearch) {
            UnifiedWechatAccount orgIdOfMaster =
                this.unifiedWechatAccountDao.findMasterAccountWithAnyCampusOrgId(orgId, true);
            if (orgIdOfMaster != null) {
                uniqueResult = this.getByOrgId(orgIdOfMaster.getOrgId().intValue(), false);
                if (uniqueResult != null) {
                    uniqueResult.setOrgId(orgId);
                    uniqueResult.setOrgWechatType(1);
                }
            }
        }
        return uniqueResult;
    }

}
