/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2016 All Rights Reserved.
 */

package com.baijia.tianxiao.dal.todo.dao.impl;

import com.baijia.tianxiao.constant.OrderBy;
import com.baijia.tianxiao.constants.org.BizConf;
import com.baijia.tianxiao.dal.org.constant.DeleteStatus;
import com.baijia.tianxiao.dal.todo.dao.TxBacklogDao;
import com.baijia.tianxiao.dal.todo.po.TxBacklog;
import com.baijia.tianxiao.dal.wechat.po.OrgWechatFansSummary;
import com.baijia.tianxiao.sqlbuilder.SingleSqlBuilder;
import com.baijia.tianxiao.sqlbuilder.bean.Expression;
import com.baijia.tianxiao.sqlbuilder.bean.Order;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.sqlbuilder.support.JdbcTemplateDaoSupport;
import com.baijia.tianxiao.util.date.DateUtil;
import com.google.common.base.Preconditions;

import org.apache.commons.collections4.CollectionUtils;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.stereotype.Repository;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import lombok.extern.slf4j.Slf4j;

/**
 * @title TxBacklogDaoImpl
 * @desc TODO
 * @author zhangbing
 * @date 2016年2月25日
 * @version 1.0
 */
@Repository
@Slf4j
public class TxBacklogDaoImpl extends JdbcTemplateDaoSupport<TxBacklog> implements TxBacklogDao {

    public TxBacklogDaoImpl() {
        super(TxBacklog.class);
    }

    @Override
    public List<TxBacklog> getBacklogListByRemindTime(Date startTime, Date endTime) {
        Preconditions.checkArgument(endTime != null, "endTime can't be null");

        SingleSqlBuilder<TxBacklog> builder = createSqlBuilder();

        builder.isNotNull("remindTime");
        if (startTime != null) {
            builder.ge("remindTime", startTime);
        }
        builder.lt("remindTime", endTime);
        builder.eq("delStatus", DeleteStatus.NORMAL.getValue());
        builder.eq("finish", BizConf.FALSE);
        builder.asc("id");
        return queryList(builder);
    }

    @Override
    public List<TxBacklog> getBacklogListByGroup(final Long orgId, Integer txCascadeId, final Integer groupType,
        final PageDto page, final String...queryProps) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(groupType != null && groupType.intValue() >= 0 && groupType.intValue() <= 3,
            "groupType is illegal!");

        Date currentDate = DateUtil.getDayDiff(0);
        Date tomorrowDate = DateUtil.getDayDiff(1);
        Date furtherDate = DateUtil.getDayDiff(8);

        SingleSqlBuilder<TxBacklog> builder = createSqlBuilder(queryProps);
        switch (groupType) {
            case 0: { // 今天
                builder.ge("endTime", currentDate);
                builder.lt("endTime", tomorrowDate);
                break;
            }
            case 1: { // 未来7天
                builder.ge("endTime", tomorrowDate);
                builder.lt("endTime", furtherDate);
                break;
            }
            case 2: { // 更远
                builder.ge("endTime", furtherDate);
                break;
            }
            case 3: { // 过去
                builder.lt("endTime", currentDate);
                break;
            }
            default:
                break;
        }

        builder.eq("orgId", orgId);
        // 添加子账号逻辑，当前员工账号,主管及以上等同主账号
        if (txCascadeId != null) {
            builder.eq("cascadeId", txCascadeId);
        }
        builder.eq("delStatus", DeleteStatus.NORMAL.getValue());
        builder.setPage(page);
        builder.asc("endTime");
        return queryList(builder);
    }

    @Override
    public TxBacklog getBacklogByIdAndOrgId(Long id, Long orgId, String...queryProps) {
        Preconditions.checkArgument(null != id && id > 0, "id can't be null");
        Preconditions.checkArgument(null != orgId && orgId > 0, "orgId can't be null");

        SingleSqlBuilder<TxBacklog> builder = createSqlBuilder(queryProps);
        builder.eq("id", id);
        builder.eq("orgId", orgId);
        builder.eq("delStatus", DeleteStatus.NORMAL.getValue());
        return uniqueResult(builder);
    }

    @Override
    public List<TxBacklog> getBacklogByConsulterIdAndOrgId(Long consulterId, Long orgId, Boolean expired,
        String...queryProps) {
        Preconditions.checkArgument(null != consulterId && consulterId > 0, "consulterId can't be null");
        Preconditions.checkArgument(null != orgId && orgId > 0, "orgId can't be null");

        SingleSqlBuilder<TxBacklog> builder = createSqlBuilder(queryProps);
        if (null != expired) {
            if (expired) {
                builder.lt("endTime", new Date());
            } else {
                builder.ge("endTime", new Date());
            }
        }
        builder.eq("consultUserId", consulterId);
        builder.eq("orgId", orgId);
        builder.eq("isSys", BizConf.TRUE);
        builder.eq("delStatus", DeleteStatus.NORMAL.getValue());
        return queryList(builder);
    }

    @Override
    public List<TxBacklog> getBacklogByStudentIdAndOrgId(Long studentId, Long orgId, Boolean expired,
        String...queryProps) {
        Preconditions.checkArgument(null != studentId && studentId > 0, "studentId can't be null");
        Preconditions.checkArgument(null != orgId && orgId > 0, "orgId can't be null");

        SingleSqlBuilder<TxBacklog> builder = createSqlBuilder(queryProps);
        if (null != expired) {
            if (expired) {
                builder.lt("endTime", new Date());
            } else {
                builder.ge("endTime", new Date());
            }
        }
        builder.eq("studentId", studentId);
        builder.eq("orgId", orgId);
        builder.eq("isSys", BizConf.TRUE);
        builder.eq("delStatus", DeleteStatus.NORMAL.getValue());
        return queryList(builder);
    }

    @Override
    public Map<Long, Integer> getBacklogFinish(List<Long> TianxiaoOrgIds) {
        Map<String, Object> param = new HashMap<>();
        param.put("orgIds", TianxiaoOrgIds);
        String sql = "select org_id, count(id) count from yunying.tx_backlog_list where finish = 1 group by org_id";

        return this.getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<Map<Long, Integer>>() {

            @Override
            public Map<Long, Integer> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<Long, Integer> countMap = new HashMap<>();
                while (rs.next()) {
                    Long orgId = rs.getLong("org_id");
                    Integer count = rs.getInt("count");
                    countMap.put(orgId, count);
                }
                return countMap;
            }
        });
    }

    @Override
    public Map<Long, Integer> getBacklogTotal(Date startTime, Date endTime, List<Long> TianxiaoOrgIds) {
        Map<String, Object> param = new HashMap<>();
        param.put("orgIds", TianxiaoOrgIds);
        String sql = "select org_id, count(id) count from yunying.tx_backlog_list where org_id in (:orgIds) ";
        if (startTime != null && endTime != null) {
            param.put("startTime", startTime);
            param.put("endTime", endTime);
            sql += " and create_time between :startTime and :endTime";
        }

        sql += " group by org_id";
        return this.getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<Map<Long, Integer>>() {

            @Override
            public Map<Long, Integer> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<Long, Integer> countMap = new HashMap<>();
                while (rs.next()) {
                    Long orgId = rs.getLong("org_id");
                    Integer count = rs.getInt("count");
                    countMap.put(orgId, count);
                }
                return countMap;
            }
        });
    }

    @Override
    public Map<Long, Integer> getBacklogExpired(Date startDate, Date endDate, List<Long> TianxiaoOrgIds) {
        Map<String, Object> param = new HashMap<>();
        param.put("endDate", endDate);
        param.put("orgIds", TianxiaoOrgIds);
        String sql =
            "select org_id, count(id) count from yunying.tx_backlog_list where org_id in (:orgIds) and end_time < :endDate";
        if (startDate != null) {
            param.put("startDate", startDate);
            sql += " and end_time > :startDate";
        }
        sql += " group by org_id";
        return this.getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<Map<Long, Integer>>() {

            @Override
            public Map<Long, Integer> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<Long, Integer> countMap = new HashMap<>();
                while (rs.next()) {
                    Long orgId = rs.getLong("org_id");
                    Integer count = rs.getInt("count");
                    countMap.put(orgId, count);
                }
                return countMap;
            }
        });

    }

    @Override
    public List<Long> getExistOrgIds() {
        String sql = "select org_id from yunying.tx_backlog_list group by org_id";
        return this.getNamedJdbcTemplate().query(sql, new ResultSetExtractor<List<Long>>() {

            @Override
            public List<Long> extractData(ResultSet rs) throws SQLException, DataAccessException {
                List<Long> orgIds = new ArrayList<>();
                while (rs.next()) {
                    Long orgId = rs.getLong("org_id");
                    orgIds.add(orgId);
                }
                return orgIds;
            }

        });
    }


	@Override
	public List<TxBacklog> list(List<Long> idList, Integer pageNum, Integer pageSize, Date minEndTime, Date maxEndTime, OrderBy orderByEndTime) {
		final List<TxBacklog> list = new ArrayList<TxBacklog>();
		if(CollectionUtils.isEmpty(idList)){
			return list;
		}
		
		StringBuilder sql = new StringBuilder();
		
		sql.append(" SELECT * FROM yunying.tx_backlog_list WHERE del_status=0");
		
		if(CollectionUtils.isNotEmpty(idList)){
			sql.append(" AND id IN (:idList)");
		}
		
		if(minEndTime!=null){
			sql.append(" AND end_time>=:minEndTime");
		}
		
		if(maxEndTime!=null){
			sql.append(" AND end_time<:maxEndTime");
		}
		
		if(orderByEndTime != null){
			if(orderByEndTime.equals(OrderBy.ASC)){
				sql.append(" ORDER BY end_time ASC");
			}
			if(orderByEndTime.equals(OrderBy.DESC)){
				sql.append(" ORDER BY end_time DESC");
			}
		}
		
		Integer pageStart = 0;
		if(pageNum!=null && pageSize!=null && pageNum>0 && pageSize>0){
			pageStart = (pageNum - 1) * pageSize;
			sql.append(" LIMIT :pageStart,:pageSize");
		}
		
		Map<String,Object> params = new HashMap<String,Object>();
		params.put("idList", idList);
		params.put("pageStart", pageStart);
		params.put("pageSize", pageSize);
		params.put("minEndTime", minEndTime);
		params.put("maxEndTime", maxEndTime);
		
		this.getNamedJdbcTemplate().query(sql.toString(), params, new RowMapper<Integer>() {
			TxBacklog txBacklog = null;
        	
            @Override
            public Integer mapRow(ResultSet rs, int rowNum) throws SQLException {
            	txBacklog = new TxBacklog();
            	txBacklog.setCascadeId( rs.getInt("cascade_id") );
            	txBacklog.setConsultUserId( rs.getLong("consult_user_id") );
            	txBacklog.setContent( rs.getString("content") );
            	txBacklog.setCreateTime( rs.getTimestamp("create_time") );
            	txBacklog.setDelStatus( rs.getInt("del_status") );
            	txBacklog.setEndTime( rs.getTimestamp("end_time") );
            	txBacklog.setFinish( rs.getInt("finish") );
            	txBacklog.setId( rs.getLong("id") );
            	txBacklog.setIsSys( rs.getInt("is_sys") );
            	txBacklog.setOrgId( rs.getLong("org_id") );
            	txBacklog.setRemindTime( rs.getTimestamp("remind_time") );
            	txBacklog.setStudentId( rs.getLong("student_id") );
            	txBacklog.setType( rs.getInt("type") );
            	txBacklog.setUpdateTime( rs.getTimestamp("update_time") );
            	list.add(txBacklog);
                return 1;
            }
        });
		
		return list;
	}
	
	
	@Override
	public void consultConvertToStu(Long consultUserId, Long studentId) {
		NamedParameterJdbcTemplate template = getNamedJdbcTemplate();
		
		String sql = " UPDATE yunying.tx_backlog_list SET student_id=:studentId "
				   + " WHERE student_id=0 AND consult_user_id=:consultUserId";
				
		Map<String, Object> params = new HashMap<String,Object>();
        params.put("consultUserId", consultUserId);
        params.put("studentId", studentId);
        
        //执行
        template.update(sql, params);
	}

	@Override
	public List<Long> listExpiredBacklogId(Long orgId, Date expiredTime) {
		final List<Long> list = new ArrayList<Long>();
		
		String sql = " SELECT id FROM yunying.tx_backlog_list "
				   + " WHERE org_id=:orgId AND end_time<=:expiredTime";
				
		Map<String, Object> params = new HashMap<String,Object>();
        params.put("orgId", orgId);
        params.put("expiredTime", expiredTime);
        
        this.getNamedJdbcTemplate().query(sql.toString(), params, new RowMapper<Long>() {
            @Override
            public Long mapRow(ResultSet rs, int rowNum) throws SQLException {
            	list.add( rs.getLong("id") );
                return null;
            }
        });
        return list;
	}
	
	
}