package com.baijia.tianxiao.dal.signup.dao.impl;

import com.baijia.commons.lang.utils.collection.CollectionUtils;
import com.baijia.tianxiao.dal.signup.dao.OrgSignupCourseLessonDao;
import com.baijia.tianxiao.dal.signup.po.OrgSignupCourseLesson;
import com.baijia.tianxiao.sqlbuilder.SingleSqlBuilder;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.sqlbuilder.support.JdbcTemplateDaoSupport;
import lombok.extern.slf4j.Slf4j;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.stereotype.Repository;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Slf4j
@Repository
public class OrgSignupCourseLessonDaoImpl extends JdbcTemplateDaoSupport<OrgSignupCourseLesson> implements
        OrgSignupCourseLessonDao {

    @Override
    public Map<Long, OrgSignupCourseLesson> selectSignUpLessons(Collection<Long> userIds, Long classId, Long orgId) {
        String sql =
                "select org_id,user_id,signup_course_id,sum(lesson_duration) as lesson_duration,count(*) as lesson_count,sum(amount) as amount " +
                        "from yunying.tx_signup_course_lesson where org_id=:orgId and class_id=:classId and lesson_type=:type " +
                        "and user_id in (:userIds) group by org_id,user_id,signup_course_id";
        Map<String, Object> param = new HashMap<>();
        param.put("orgId", orgId);
        param.put("classId", classId);
        param.put("userIds", userIds);
        param.put("type", 1);
        List<OrgSignupCourseLesson> list = getNamedJdbcTemplate().query(sql, param, new RowMapper<OrgSignupCourseLesson>() {
            @Override
            public OrgSignupCourseLesson mapRow(ResultSet rs, int rowNum) throws SQLException {
                OrgSignupCourseLesson stat = toOrgSignupCourseLesson(rs);
                stat.setSignupCourseId(rs.getLong("signup_course_id"));
                return stat;
            }
        });
        Map<Long, OrgSignupCourseLesson> signupLessonsMap =
                CollectionUtils.extractMap(list, new CollectionUtils.Extracter<Long, OrgSignupCourseLesson>() {
                    @Override
                    public Long extract(OrgSignupCourseLesson lesson) {
                        return lesson.getSignupCourseId();
                    }
                });
        return signupLessonsMap;
    }

    @Override
    public Map<String, OrgSignupCourseLesson> selectLessonInfoMap(Long orgId, Collection<Long> lessonIds, Collection<Long> userIds) {
        String sql = "select org_id,lesson_id,user_id,class_id,course_id,charge_unit,sum(lesson_duration) as lesson_duration,count(*) as lesson_count,sum(amount) as amount " +
                "  from yunying.tx_signup_course_lesson where org_id=:orgId and lesson_id in (:lessonIds) and user_id in (:userIds)" +
                " group by org_id,lesson_id,user_id";
        Map<String, Object> param = new HashMap<>();
        param.put("lessonIds", lessonIds);
        param.put("userIds", userIds);
        param.put("orgId", orgId);
        List<OrgSignupCourseLesson> list = getNamedJdbcTemplate().query(sql, param, new RowMapper<OrgSignupCourseLesson>() {
            @Override
            public OrgSignupCourseLesson mapRow(ResultSet rs, int rowNum) throws SQLException {
                OrgSignupCourseLesson lesson = toOrgSignupCourseLesson(rs);
                lesson.setLessonId(rs.getLong("lesson_id"));
                lesson.setUserId(rs.getLong("user_id"));
                lesson.setClassId(rs.getLong("class_id"));
                lesson.setCourseId(rs.getLong("course_id"));
                lesson.setChargeUnit(rs.getInt("charge_unit"));
                return lesson;
            }
        });
        Map<String, OrgSignupCourseLesson> signupLessonsMap =
                CollectionUtils.extractMap(list, new CollectionUtils.Extracter<String, OrgSignupCourseLesson>() {
                    @Override
                    public String extract(OrgSignupCourseLesson lesson) {
                        return lesson.getUserId() + "_" + lesson.getLessonId();
                    }
                });
        return signupLessonsMap;
    }

    @Override
    public Map<Long, OrgSignupCourseLesson> selectLessonInfoByClassId(Long orgId, Long classId, Collection<Long> userIds, Integer lessonType) {
        String sql = "select org_id,user_id,class_id,course_id,charge_unit,sum(lesson_duration) as lesson_duration,count(*) as lesson_count,sum(amount) as amount" +
                " from yunying.tx_signup_course_lesson where org_id=:orgId and class_id=:classId and user_id in (:userIds) ";
        Map<String, Object> param = new HashMap<>();
        if (lessonType != null) {
            sql += " and lesson_type=:lessonType";
            param.put("lessonType", lessonType);
        }
        sql += " group by org_id,class_id,user_id";
        param.put("classId", classId);
        param.put("userIds", userIds);
        param.put("orgId", orgId);
        List<OrgSignupCourseLesson> list = getNamedJdbcTemplate().query(sql, param, new RowMapper<OrgSignupCourseLesson>() {
            @Override
            public OrgSignupCourseLesson mapRow(ResultSet rs, int rowNum) throws SQLException {
                OrgSignupCourseLesson lesson = toOrgSignupCourseLesson(rs);
                lesson.setUserId(rs.getLong("user_id"));
                lesson.setClassId(rs.getLong("class_id"));
                lesson.setCourseId(rs.getLong("course_id"));
                lesson.setChargeUnit(rs.getInt("charge_unit"));
                return lesson;
            }
        });
        Map<Long, OrgSignupCourseLesson> signupLessonsMap =
                CollectionUtils.extractMap(list, new CollectionUtils.Extracter<Long, OrgSignupCourseLesson>() {
                    @Override
                    public Long extract(OrgSignupCourseLesson lesson) {
                        return lesson.getUserId();
                    }
                });
        return signupLessonsMap;
    }

    private OrgSignupCourseLesson toOrgSignupCourseLesson(ResultSet rs) throws SQLException {
        OrgSignupCourseLesson stat = new OrgSignupCourseLesson();
        stat.setOrgId(rs.getLong("org_id"));
        stat.setLessonCount(rs.getInt("lesson_count"));
        stat.setLessonDuration(rs.getInt("lesson_duration"));
        stat.setAmount(rs.getLong("amount"));
        return stat;
    }

    @Override
    public void batchDelLesson(long orgId, Collection<Long> lessonIds) {
        Map<String, Object> param = new HashMap<>();
        param.put("lessonId", lessonIds);
        param.put("orgId", orgId);
        delByCondition(param);
    }

    @Override
    public void batchDelStuLesson(long orgId, Collection<Long> userIds, Collection<Long> lessonIds) {
        Map<String, Object> param = new HashMap<>();
        if (userIds != null && !userIds.isEmpty()) {
            param.put("userId", userIds);
        }
        param.put("orgId", orgId);
        param.put("lessonId", lessonIds);
        delByCondition(param);
    }

    @Override
    public void batchCancelLesson(Long orgId, Long lessonId, Collection<Long> userIds) {
        if (org.apache.commons.collections4.CollectionUtils.isEmpty(userIds)) {
            return;
        }
        String sql = "update yunying.tx_signup_course_lesson set lesson_type=3,signup_course_id=-1 where user_id in (:userIds) and lesson_id = :lessonId and org_id=:orgId";
        Map<String, Object> param = new HashMap<>();
        param.put("orgId", orgId);
        param.put("lessonId", lessonId);
        param.put("userIds", userIds);
        getNamedJdbcTemplate().update(sql, param);
    }

    @Override
    public void batchUpdateLessonDuration(Long orgId, Long lessonId, Collection<Long> userIds, long lessonDuration) {
        Map<String, Object> param = new HashMap<>();
        param.put("orgId", orgId);
        param.put("lessonId", lessonId);
        param.put("userId", userIds);
        param.put("lessonDuration", lessonDuration);
        param.put("lessonType", 2);
        update(param, "lessonDuration");
    }

    @Override
    public void batchUpdateKexiaoStatus(Collection<Long> lessonIds, Collection<Long> userIds, int kexiaoStatus) {
        Map<String, Object> param = new HashMap<>();
        param.put("lessonId", lessonIds);
        param.put("userId", userIds);
        param.put("kexiaoStatus", kexiaoStatus);
        update(param, "kexiaoStatus");
    }

    @Override
    public List<OrgSignupCourseLesson> getLessonList(Long orgId, Collection<Long> lessonIds, Collection<Long> userIds) {
        SingleSqlBuilder<OrgSignupCourseLesson> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.in("lessonId", lessonIds);
        builder.in("userId", userIds);
        return queryList(builder);
    }

    @Override
    public List<OrgSignupCourseLesson> listByOrg(Long orgId, PageDto page, String... props) {
        SingleSqlBuilder<OrgSignupCourseLesson> builder = createSqlBuilder(props);
        builder.setPage(page);
        builder.eq("orgId", orgId);
        return queryList(builder);
    }

    @Override
    public List<OrgSignupCourseLesson> listBySignupCourseIds(Collection<Long> signupCourseIds, Integer lessonType, Integer kexiaoStatus) {
        String sql = "select org_id,signup_course_id,charge_unit,sum(lesson_duration) as lesson_duration,count(*) as lesson_count,sum(amount) as amount" +
                " from yunying.tx_signup_course_lesson where signup_course_id in (:signupCourseIds) ";

        Map<String, Object> param = new HashMap<>();
        if (kexiaoStatus != null) {
            sql += " and kexiao_status = :kexiaoStatus";
            param.put("kexiaoStatus", kexiaoStatus);
        }

        if (kexiaoStatus != null) {
            sql += " and lesson_type = :lessonType";
            param.put("lessonType", lessonType);
        }

        sql += " group by signup_course_id,charge_unit";
        param.put("signupCourseIds", signupCourseIds);

        List<OrgSignupCourseLesson> list = getNamedJdbcTemplate().query(sql, param, new RowMapper<OrgSignupCourseLesson>() {
            @Override
            public OrgSignupCourseLesson mapRow(ResultSet rs, int rowNum) throws SQLException {
                OrgSignupCourseLesson lesson = toOrgSignupCourseLesson(rs);
                lesson.setSignupCourseId(rs.getLong("signup_course_id"));
                lesson.setLessonDuration(rs.getInt("lesson_duration"));
                lesson.setLessonCount(rs.getInt("lesson_count"));
                lesson.setAmount(rs.getLong("amount"));
                lesson.setChargeUnit(rs.getInt("charge_unit"));
                return lesson;
            }
        });
        return list;
    }

    @Override
    public Map<String, OrgSignupCourseLesson> listBySignupCourseIdAndLessonId(Collection<Long> signupCourseIds, Collection<Long> lessonIds) {
        Map<String, OrgSignupCourseLesson> ret = new HashMap<>();
        SingleSqlBuilder<OrgSignupCourseLesson> builder = createSqlBuilder();
        builder.in("signupCourseId", signupCourseIds);
        builder.in("lessonId", lessonIds);
        List<OrgSignupCourseLesson> list = queryList(builder);
        if (list != null && list.size() > 0) {
            for (OrgSignupCourseLesson courseLesson : list) {
                ret.put(courseLesson.getSignupCourseId() + "_" + courseLesson.getLessonId(), courseLesson);
            }
        }
        return ret;
    }

    @Override
    public List<OrgSignupCourseLesson> listByUserIdAndLessonId(Collection<Long> userIds, Collection<Long> lessonIds, Integer kexiaoStatus) {
        SingleSqlBuilder<OrgSignupCourseLesson> builder = createSqlBuilder();
        builder.in("userId", userIds);
        if (kexiaoStatus != null) {
            builder.eq("kexiaoStatus", kexiaoStatus);
        }
        if (lessonIds != null && lessonIds.size() > 0) {
            builder.in("lessonId", lessonIds);
        }
        List<OrgSignupCourseLesson> list = queryList(builder);
        return list;
    }

    @Override
    public List<OrgSignupCourseLesson> listLessonsByUserIdAndClassId(Long orgId, Long userId, Long classId, Integer lessonType, Integer kexiaoStatus) {
        SingleSqlBuilder<OrgSignupCourseLesson> builder = createSqlBuilder();
        builder.eq("userId", userId);
        builder.eq("classId", classId);
        if (kexiaoStatus != null) {
            builder.eq("kexiaoStatus", kexiaoStatus);
        }
        if (kexiaoStatus != lessonType) {
            builder.eq("lessonType", lessonType);
        }
        List<OrgSignupCourseLesson> list = queryList(builder);
        return list;
    }

    @Override
    public List<OrgSignupCourseLesson> listByUserIdsAndClassId(Collection<Long> userIds, long classId, Integer kexiaoStatus) {
        SingleSqlBuilder<OrgSignupCourseLesson> builder = createSqlBuilder();
        builder.in("userId", userIds);
        builder.eq("classId", classId);
        if (kexiaoStatus != null) {
            builder.eq("kexiaoStatus", kexiaoStatus);
        }
        List<OrgSignupCourseLesson> list = queryList(builder);
        return list;
    }
}