
package com.baijia.tianxiao.dal.signup.dao.impl;

import static com.baijia.tianxiao.dal.signup.constant.SignupCourseStatus.inClassStatus;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.map.HashedMap;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.stereotype.Repository;

import com.baijia.tianxiao.consants.DataStatus;
import com.baijia.tianxiao.constant.Flag;
import com.baijia.tianxiao.dal.org.constant.DeleteStatus;
import com.baijia.tianxiao.dal.signup.constant.SignupCourseStatus;
import com.baijia.tianxiao.dal.signup.constant.SignupSupplementType;
import com.baijia.tianxiao.dal.signup.dao.OrgSignupCourseDao;
import com.baijia.tianxiao.dal.signup.dto.OrgStudentSignupStatistics;
import com.baijia.tianxiao.dal.signup.po.OrgSignupCourse;
import com.baijia.tianxiao.sqlbuilder.SingleSqlBuilder;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.sqlbuilder.support.JdbcTemplateDaoSupport;
import com.baijia.tianxiao.sqlbuilder.util.Expressions;
import com.baijia.tianxiao.util.GenericsUtils;
import com.baijia.tianxiao.util.collection.CollectorUtil;
import com.baijia.tianxiao.util.date.DateUtil;
import com.baijia.tianxiao.util.query.BatchQueryCallback;
import com.baijia.tianxiao.util.query.BatchQueryTemplate;
import com.baijia.tianxiao.util.query.ListBatchQueryTemplate;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@Repository
public class OrgSignupCourseDaoImpl extends JdbcTemplateDaoSupport<OrgSignupCourse> implements OrgSignupCourseDao {

    public OrgSignupCourseDaoImpl() {
        super(OrgSignupCourse.class);
    }

    @Override
    public List<Long> searchPurchaseIdByCourseId(@NonNull Collection<Long> courseIds, Date startTime, Date endTime) {
        if (CollectionUtils.isEmpty(courseIds)) {
            return Collections.emptyList();
        }
        SingleSqlBuilder<OrgSignupCourse> builder = createSqlBuilder();
        builder.select("signupPurchaseId");
        builder.in("orgCourseId", courseIds);
        if (startTime != null) {
            builder.ge("createTime", startTime);
        }
        if (endTime != null) {
            builder.le("createTime", endTime);
        }

        return queryForList(builder, Long.class);
    }

    @Override
    public List<OrgSignupCourse> loadByPurchaseIds(@NonNull Collection<Long> purchaseIds, final String...queryProps) {
        if (CollectionUtils.isEmpty(purchaseIds)) {
            return Collections.emptyList();
        }
        BatchQueryTemplate<Long, List<OrgSignupCourse>> queryTemplate = new ListBatchQueryTemplate<>();
        return queryTemplate.batchQuery(purchaseIds, new BatchQueryCallback<Long, List<OrgSignupCourse>>() {
            @Override
            public List<OrgSignupCourse> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<OrgSignupCourse> builder = createSqlBuilder();
                builder.select(queryProps);
                builder.in("signupPurchaseId", querySet);
                return queryList(builder);
            }
        });
    }

    @Override
    public List<OrgSignupCourse> loadByPurchaseId(@NonNull Long purchaseId, String...queryProps) {
        SingleSqlBuilder<OrgSignupCourse> builder = createSqlBuilder();
        builder.select(queryProps);
        builder.eq("signupPurchaseId", purchaseId);
        return queryList(builder);

    }

    @Override
    public List<OrgSignupCourse> listByPurchaseIds(Collection<Long> purchaseIds, Long classId, String... queryProps) {
        SingleSqlBuilder<OrgSignupCourse> builder = createSqlBuilder();
        builder.select(queryProps);
        builder.in("signupPurchaseId", purchaseIds);
        builder.eq("classId", classId);
        return queryList(builder);
    }

    @Override
    public void saveOrUpdateSignupCourse(@NonNull OrgSignupCourse signupCourse) {
        if (signupCourse.getId() > 0) {
            update(signupCourse);
        } else {
            save(signupCourse);
        }
    }

    public void updateByCourseId(@NonNull OrgSignupCourse signupCourse) {
        Map<String, Object> updateCondition = Maps.newHashMap();
        updateCondition.put("orgCourseId", signupCourse.getOrgCourseId());
        update(updateCondition, signupCourse);
    }

    @Override
    public void saveSignupCourses(List<OrgSignupCourse> courses) {
        saveAll(courses);
    }

    @Override
    public OrgSignupCourse getByCourseId(Long orgId, Long purchaseId, Long courseId) {
        SingleSqlBuilder<OrgSignupCourse> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.eq("orgCourseId", courseId);
        builder.eq("signupPurchaseId", purchaseId);
        return uniqueResult(builder);
    }

    @Override
    public List<OrgSignupCourse> searchByUserIdsAndClassId(Collection<Long> userIds, Long classId, Long orgId,
                                                           Collection<Integer> status) {
        SingleSqlBuilder<OrgSignupCourse> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.in("status", status);
        if(classId!=null){
            builder.eq("classId", classId);
        }
        builder.in("userId", userIds);
        log.debug("searchByUserIdsAndClassId sql={},params={}", builder.toSql(), builder.collectConditionValue());
        return queryList(builder);
    }

    @Override
    public List<OrgSignupCourse> getByCourseIdAndStudentId(Long orgId, Long userId, Long courseId,
        Collection<Integer> status) {
        SingleSqlBuilder<OrgSignupCourse> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.eq("orgCourseId", courseId);
        if (CollectionUtils.isNotEmpty(status)) {
            builder.in("status", status);
        }
        builder.eq("userId", userId);
        builder.eq("isDel", DataStatus.NORMAL.getValue());
        log.debug("getByCourseIdsAndStudentId sql={},params={}", builder.toSql(), builder.collectConditionValue());
        return queryList(builder);
    }

    @Override
    public List<OrgSignupCourse> getByCourseIdsAndStudentId(Long orgId, Long userId, Collection<Long> courseIds,
        Collection<Integer> status) {
        SingleSqlBuilder<OrgSignupCourse> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        if (CollectionUtils.isNotEmpty(courseIds)) {
            builder.in("orgCourseId", courseIds);
        }
        if (CollectionUtils.isNotEmpty(status)) {
            builder.in("status", status);
        }
        builder.eq("userId", userId);
        builder.eq("isDel", DataStatus.NORMAL.getValue());
        log.debug("getByCourseIdsAndStudentId sql={},params={}", builder.toSql(), builder.collectConditionValue());
        return queryList(builder);
    }

    @Override
    public int updateStatusByPurchaseId(Long signupPurchaseId, Long orgCourseId, Long userId, Integer status) {
        Preconditions.checkArgument(orgCourseId != null && orgCourseId.longValue() > 0, "orgCourseId is null!");
        Preconditions.checkArgument(userId != null && userId > 0, "userId is null!");

        StringBuilder sb = new StringBuilder(
            "update yunying.org_signup_course set status = :status,sync_lesson_count=0,update_time=now() where 1=1 ");
        Map params = new HashedMap<String, Object>();
        if (signupPurchaseId != null && signupPurchaseId.longValue() != 0) {
            sb.append("and signup_purchase_id = :signupPurchaseId");
            params.put("signupPurchaseId", signupPurchaseId);
        }
        sb.append(" and org_course_id = :orgCourseId and user_id = :userId and status != :status and status !=0");
        params.put("orgCourseId", orgCourseId);
        params.put("userId", userId);
        params.put("status", status);

        int result = getNamedJdbcTemplate().update(sb.toString(), params);
        log.debug("updateStatusByPurchaseId sql={},params={}，result={}", sb.toString(), params, result);
        return result;
    }

    @Override
    public int resyncLesson(Long signupPurchaseId, Long orgCourseId, Long userId) {
        Preconditions.checkArgument(orgCourseId != null && orgCourseId.longValue() > 0, "orgCourseId is null!");
        Preconditions.checkArgument(userId != null && userId > 0, "userId is null!");

        StringBuilder sb = new StringBuilder(
                "update yunying.org_signup_course set sync_lesson_count=0,update_time=now() where 1=1 ");
        Map params = new HashedMap<String, Object>();
        if (signupPurchaseId != null && signupPurchaseId.longValue() != 0) {
            sb.append("and signup_purchase_id = :signupPurchaseId");
            params.put("signupPurchaseId", signupPurchaseId);
        }
        sb.append(" and org_course_id = :orgCourseId and user_id = :userId and status !=0");
        params.put("orgCourseId", orgCourseId);
        params.put("userId", userId);

        int result = getNamedJdbcTemplate().update(sb.toString(), params);
        log.debug("updateStatusByPurchaseId sql={},params={}，result={}", sb.toString(), params, result);
        return result;
    }

    @Override
    public int updateStatusByPurchaseId(Long signupPurchaseId, Integer status) {
        Preconditions.checkArgument(signupPurchaseId != null && signupPurchaseId > 0, "signupPurchaseId is null!");
        StringBuilder sb = new StringBuilder(
            "update yunying.org_signup_course set status = :status,update_time=now() where signup_purchase_id = :signupPurchaseId and status !=:status");
        Map params = new HashedMap<String, Object>();
        params.put("signupPurchaseId", signupPurchaseId);
        params.put("status", status);

        int result = getNamedJdbcTemplate().update(sb.toString(), params);

        log.debug("updateStatusByPurchaseId sql={},params={}，result={}", sb.toString(), params, result);
        return result;
    }

    @Override
    public int updateLessonCountByPurchaseId(Long signupPurchaseId, Long orgCourseId) {
        Preconditions.checkArgument(signupPurchaseId != null && signupPurchaseId > 0, "signupPurchaseId is null!");
        Preconditions.checkArgument(orgCourseId != null && orgCourseId > 0, "orgCourseId is null!");
        StringBuilder sb = new StringBuilder(
            "update yunying.org_signup_course set lesson_count = count,update_time=now() where signup_purchase_id = :signupPurchaseId and org_course_id =:orgCourseId and lesson_count =0 ");
        Map params = new HashedMap<String, Object>();
        params.put("signupPurchaseId", signupPurchaseId);
        params.put("orgCourseId", orgCourseId);

        int result = getNamedJdbcTemplate().update(sb.toString(), params);

        log.info("updateStatusByPurchaseId sql={},params={}，result={}", sb.toString(), params, result);
        return result;
    }

    @Override
    public void delSignupCourseByPurchaseId(Long signupPurchaseId) {
        Preconditions.checkArgument(signupPurchaseId != null && signupPurchaseId > 0, "signupPurchaseId is null!");
        StringBuilder sb = new StringBuilder(
            "update yunying.org_signup_course set status = 1,update_time=now() where signup_purchase_id = :signupPurchaseId ");
        Map params = new HashedMap<String, Object>();
        params.put("signupPurchaseId", signupPurchaseId);

        getNamedJdbcTemplate().update(sb.toString(), params);
    }

    @Override
    public List<OrgSignupCourse> getByCourseIdsAndStudentIds(Collection<Long> delUserIds, Long orgId,
        Collection<Long> classIds, Collection<Long> userIds, Integer supplement, PageDto pageDto, String...props) {
        SingleSqlBuilder<OrgSignupCourse> builder = createSqlBuilder(props);
        builder.eq("orgId", orgId);
        if (CollectionUtils.isNotEmpty(classIds)) {
            builder.in("classId", classIds);
        }
        if (CollectionUtils.isNotEmpty(userIds)) {
            if (CollectionUtils.isNotEmpty(delUserIds)) {
                userIds.removeAll(delUserIds);
            }
            builder.in("userId", userIds);
        } else if (CollectionUtils.isNotEmpty(delUserIds)) {
            builder.notin("userId", delUserIds);
        }
        setSupplement(supplement, builder);
        builder.in("status", Lists.newArrayList(SignupCourseStatus.inClassStatus));
        builder.eq("isDel", DeleteStatus.NORMAL.getValue());
        if (null != pageDto) {
            builder.setPage(pageDto);
        }
        log.debug("sql:{}, params:{}", builder.toSql(), builder.collectConditionValue());
        List<OrgSignupCourse> result = queryList(builder);
        for (OrgSignupCourse signupCourse : result) {
            signupCourse.setSupplement(supplement);
        }
        return result;
    }

    @Override
    public List<OrgSignupCourse> getByCourseIdsAndStudentIds(Collection<Long> orgIds, Collection<Long> courseIds,
        Collection<Long> userIds, String...props) {
        if (CollectionUtils.isEmpty(orgIds) || CollectionUtils.isEmpty(courseIds) || CollectionUtils.isEmpty(userIds)) {
            return Lists.newArrayList();
        }
        SingleSqlBuilder<OrgSignupCourse> builder = createSqlBuilder(props);
        builder.in("orgId", orgIds);
        builder.in("orgCourseId", courseIds);
        builder.in("userId", userIds);
        return queryList(builder);
    }

    @Override
    public List<OrgSignupCourse> getLeftQuitClassSignupCourse(Long orgId, Long orgCourseId, Long userId,
        Long signPurchaseId, String...props) {

        SingleSqlBuilder<OrgSignupCourse> builder = createSqlBuilder(props);
        builder.eq("orgId", orgId);
        builder.eq("orgCourseId", orgCourseId);
        builder.eq("userId", userId);
        builder.ne("signupPurchaseId", signPurchaseId);
        builder.in("status", inClassStatus);
        return queryList(builder);
    }

    @Override
    public List<OrgSignupCourse> listSignupCourseByCourse(Long orgId, Long signupPurchaseId, Long orgCourseId,
        Long userId, String...props) {
        SingleSqlBuilder<OrgSignupCourse> builder = createSqlBuilder(props);
        builder.eq("orgId", orgId);
        builder.eq("orgCourseId", orgCourseId);
        builder.eq("userId", userId);
        if (signupPurchaseId != null && signupPurchaseId.intValue() != 0) {
            builder.eq("signupPurchaseId", signupPurchaseId);
        }
        builder.in("status", inClassStatus);
        return queryList(builder);
    }

    @Override
    public Map<Long, Object> getSignupedCourseIdsByStudentId(Long orgId, Long userId) {
        Object object = new Object();
        Map<Long, Object> result = Maps.newHashMap();
        SingleSqlBuilder<OrgSignupCourse> builder = createSqlBuilder("orgCourseId");
        builder.eq("orgId", orgId);
        builder.eq("userId", userId);
        builder.eq("isDel", DataStatus.NORMAL.getValue());
        // builder.eq("status", SignupCourseStatus.HAS_PAY.getCode());
        List<Long> list = queryForList(builder, Long.class);
        if (CollectionUtils.isEmpty(list)) {
            return Collections.emptyMap();
        }
        for (Long l : list) {
            result.put(l, object);
        }
        return result;
    }

    @Override
    public Map<Long, Object> getSignupedUserIdsByCourseId(Long orgId, Long courseId) {
        Object object = new Object();
        Map<Long, Object> result = Maps.newHashMap();
        SingleSqlBuilder<OrgSignupCourse> builder = createSqlBuilder("userId");
        builder.eq("orgId", orgId);
        builder.eq("orgCourseId", courseId);
        builder.eq("isDel", DataStatus.NORMAL.getValue());
        builder.in("status", inClassStatus);
        List<Long> list = queryForList(builder, Long.class);
        if (CollectionUtils.isEmpty(list)) {
            return Collections.emptyMap();
        }
        for (Long l : list) {
            result.put(l, object);
        }
        return result;
    }

    @Override
    public List<OrgSignupCourse> getByCourseIdsAndStudentIds(Long orgId, Collection<Integer> ids,
        Collection<Long> courseIds, Collection<Long> userIds, String...props) {
        SingleSqlBuilder<OrgSignupCourse> builder = createSqlBuilder(props);
        if (CollectionUtils.isNotEmpty(ids)) {
            builder.in("id", ids);
        }
        builder.eq("orgId", orgId);
        if (CollectionUtils.isNotEmpty(courseIds)) {
            builder.in("orgCourseId", courseIds);
        }
        if (CollectionUtils.isNotEmpty(userIds)) {
            builder.in("userId", userIds);
        }
        return queryList(builder);
    }

    @Override
    public Map<Long, Integer> getCourseStudentCountMap(Collection<Long> delUserIds, Long orgId, List<Long> classId,
        Integer supplement) {
        SingleSqlBuilder<OrgSignupCourse> builder = createSqlBuilder("classId");
        builder.eq("orgId", orgId);
        if (CollectionUtils.isEmpty(classId)) {
            return Maps.newHashMap();
        }
        builder.in("classId", classId);
        if (CollectionUtils.isNotEmpty(delUserIds)) {
            builder.notin("userId", delUserIds);
        }
        setSupplement(supplement, builder);
        builder.eq("isDel", DeleteStatus.NORMAL.getValue());
        builder.in("status", Lists.newArrayList(SignupCourseStatus.inClassStatus));
        builder.count("id", "count");
        builder.groupByNames("classId");
        return CollectorUtil.collectMap(queryList(builder), new Function<OrgSignupCourse, Long>() {
            @Override
            public Long apply(OrgSignupCourse input) {
                return input.getClassId();
            }
        }, new Function<OrgSignupCourse, Integer>() {
            @Override
            public Integer apply(OrgSignupCourse input) {
                return input.getCount();
            }
        });
    }

    @Override
    public Map<Long, Boolean> canShowTotalCount(Long orgId, Collection<Long> userIds, Collection<Long> courseIds) {
        if (GenericsUtils.isNullOrEmpty(userIds)) {
            return GenericsUtils.emptyMap();
        }
        SingleSqlBuilder<OrgSignupCourse> builder = createSqlBuilder("id");
        builder.eq("orgId", orgId);
        builder.in("userId", userIds);
        if (GenericsUtils.notNullAndEmpty(courseIds)) {
            builder.in("orgCourseId", courseIds);
        }
        builder.eq("isDel", DeleteStatus.NORMAL.getValue());
        builder.gt("chargeUnit", -1); // -1 对应 ChargeUnit.BY_OTHER ,修改请注意
        builder.count("id", "sqlCount");
        builder.groupByNames("userId", "orgCourseId");
        List<OrgSignupCourse> queryList = queryList(builder);
        queryList = GenericsUtils.isNullOrEmpty(queryList) ? GenericsUtils.<OrgSignupCourse> emptyList() : queryList;
        Map<Long, OrgSignupCourse> orgSignupCourseMap = GenericsUtils.toFieldMap(queryList, "userId");
        Map<Long, Boolean> retMap = Maps.newHashMap();
        for (Long userId : userIds) {
            OrgSignupCourse list = orgSignupCourseMap.get(userId);
            retMap.put(userId, (list != null && list.getSqlCount() != null && list.getSqlCount() > 0) ? true : false);
        }
        return retMap;
    }

    @Override
    public List<OrgStudentSignupStatistics> findAllOrgStudentSignupStatistics(Long orgId, Collection<Long> userIds,
        Collection<Long> courseIds) {
        log.info("userIds are :{} ", userIds);
        if (GenericsUtils.isNullOrEmpty(userIds)) {
            return GenericsUtils.emptyList();
        }
        String courseIdsCondition = "";
        Map<String, Object> paramMap = Maps.newHashMap();
        if (GenericsUtils.notNullAndEmpty(courseIds)) {
            courseIdsCondition = " and org_course_id in(:orgCourseIds) ";
            paramMap.put("orgCourseIds", courseIds);
        }
        String sql =
            "select user_id as userId,org_course_id as courseId,  sum(if(status in(2,3,4,5,6),lesson_count,0)) as buyCount, sum(if(status in(2,3,4,5,6),pay_price + student_pay_price,0)) as payMoney,count(if(status in(2,3,4,5) and lesson_count=0,1,null)) as chargeUnitStatus "
                + " from yunying.org_signup_course where status in(:status) and org_Id=:orgId and user_id in(:userIds) "
                + courseIdsCondition + " and is_del=:isDel  group by user_id,org_course_id  ";
        paramMap.put("userIds", userIds);
        paramMap.put("orgId", orgId);
        paramMap.put("isDel", DeleteStatus.NORMAL.getValue());
        paramMap.put("status",
            Arrays.asList(SignupCourseStatus.HAS_PAY.getCode(), SignupCourseStatus.INIT.getCode(),
                SignupCourseStatus.IMPORT.getCode(), SignupCourseStatus.HAS_ADD.getCode(),
                SignupCourseStatus.QUIT_PURCHASE.getCode(), SignupCourseStatus.QUIT_CLASS.getCode()));
        log.info("sql is :{} and paramMap is :{} ", sql, paramMap);

        final List<OrgStudentSignupStatistics> queryForList = Lists.newArrayList();
        try {
            this.getNamedJdbcTemplate().query(sql, paramMap, new ResultSetExtractor<OrgStudentSignupStatistics>() {
                @Override
                public OrgStudentSignupStatistics extractData(ResultSet rs) throws SQLException, DataAccessException {
                    while (rs.next()) {
                        OrgStudentSignupStatistics os = createOrgStudentSignupStatistics(rs);
                        if (os != null) {
                            queryForList.add(os);
                        }
                    }
                    return null;
                }

                private OrgStudentSignupStatistics createOrgStudentSignupStatistics(ResultSet rs) {
                    OrgStudentSignupStatistics os = null;
                    try {
                        os = new OrgStudentSignupStatistics();
                        // os.setBuyCount(rs.getLong("buyCount"));
                        os.setPayMoney(rs.getLong("payMoney"));
                        os.setCourseId(rs.getLong("courseId"));
                        os.setUserId(rs.getLong("userId"));
                        os.setChargeUnitStatus(rs.getLong("chargeUnitStatus"));
                    } catch (SQLException e) {
                        log.error("exception is :{} ", e);
                    }
                    return os;
                }
            });
        } catch (Exception e) {
            log.error("exception :{} ", e);
        }
        // log.info("queryForList are :{} ", queryForList);
        return queryForList;
    }

    @Override
    public Map<Long, OrgSignupCourse> mapKeyClassId(Long orgId, Long userId) {
        Map<Long, OrgSignupCourse> result = new HashMap<Long, OrgSignupCourse>();

        SingleSqlBuilder<OrgSignupCourse> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.eq("userId", userId);
        builder.gt("classId", -1);
        List<OrgSignupCourse> list = queryList(builder);

        if (CollectionUtils.isNotEmpty(list)) {
            for (OrgSignupCourse obj : list) {
                result.put(obj.getClassId(), obj);
            }
        }
        return result;
    }

    @Override
    public Map<Long, OrgSignupCourse> mapKeyUserId(Long orgId, Long classId) {
        Map<Long, OrgSignupCourse> result = new HashMap<Long, OrgSignupCourse>();

        SingleSqlBuilder<OrgSignupCourse> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.eq("classId", classId);
        List<OrgSignupCourse> list = queryList(builder);

        if (CollectionUtils.isNotEmpty(list)) {
            for (OrgSignupCourse obj : list) {
                result.put(obj.getUserId(), obj);
            }
        }
        return result;
    }

    @Override
    public List<OrgSignupCourse> listByUserId(Long orgId, Collection<Long> classIds, Long userId) {
        SingleSqlBuilder<OrgSignupCourse> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.gt("classId", -1);
        if (classIds != null) {
            builder.in("classId", classIds);
        }
        if (userId != null) {
            builder.eq("userId", userId);
        }

        return queryList(builder);
    }

    @Override
    public List<OrgSignupCourse> listByClassId(Long orgId, Long classId, Collection<Long> userIds,
        List<Integer> status) {
        SingleSqlBuilder<OrgSignupCourse> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.gt("classId", -1);
        if (classId != null) {
            builder.eq("classId", classId);
        }
        if (userIds != null) {
            builder.in("userId", userIds);
        }
        if (CollectionUtils.isNotEmpty(status)) {
            builder.in("status", status);
        }
        return queryList(builder);
    }

    @Override
    public List<Long> getClassIds(Collection<Long> delUserIds, Collection<Long> notInCourseIds, Long orgId,
        Collection<Long> courseNameClassIds, Integer supplement, PageDto pageDto) {
        SingleSqlBuilder<OrgSignupCourse> builder = createSqlBuilder("classId");
        builder.eq("orgId", orgId);
        if (CollectionUtils.isNotEmpty(courseNameClassIds)) {
            if(CollectionUtils.isNotEmpty(notInCourseIds)){
                courseNameClassIds.removeAll(notInCourseIds);
            }
            builder.in("classId", courseNameClassIds);
        } else {
            if (CollectionUtils.isNotEmpty(notInCourseIds)) {
                builder.notin("classId", notInCourseIds);
            }
        }
        if (CollectionUtils.isNotEmpty(delUserIds)) {
            builder.notin("userId", delUserIds);
        }
        setSupplement(supplement, builder);
        builder.eq("isDel", DeleteStatus.NORMAL.getValue());
        builder.in("status", Lists.newArrayList(SignupCourseStatus.inClassStatus));
        builder.group("classId");
        builder.desc("classId");

        Integer count = getNamedJdbcTemplate()// group 的count 不准
            .queryForObject("select count(1) from (" + builder.toSql() + ") as a", builder.collectConditionValue(),
                Integer.class);

        builder.setPage(pageDto);

        List<Long> result = Lists.transform(queryList(builder), new Function<OrgSignupCourse, Long>() {
            @Override
            public Long apply(OrgSignupCourse input) {
                return input.getClassId();
            }
        });

        pageDto.setCount(count);

        return result;
    }

    private void setSupplement(Integer supplement, SingleSqlBuilder<OrgSignupCourse> builder) {
        if (null != supplement && SignupSupplementType.ALL.getCode() != supplement) {// 跟进报名是否需要补充,做条件筛选.
            if (supplement == SignupSupplementType.OLD.getCode()) {
                builder.eq("lessonCount", Flag.FALSE.getInt());
                // builder.eq("chargeUnit", Flag.NULL.getInt());
                builder.eq("kexiaoCascadeId", Flag.NULL.getInt());
                builder.desc("createTime");
            } else if (supplement == SignupSupplementType.OLD_FIXED.getCode()) {
                builder.gt("lessonCount", Flag.FALSE.getInt());
                // builder.gt("chargeUnit", Flag.NULL.getInt());
                builder.gt("kexiaoCascadeId", Flag.NULL.getInt());
                builder.desc("kexiaoTime");
            } else if (supplement == SignupSupplementType.NEW.getCode()) {
                builder.gt("lessonCount", Flag.FALSE.getInt());
                // builder.gt("chargeUnit", Flag.NULL.getInt());
                builder.eq("kexiaoCascadeId", Flag.NULL.getInt());
                builder.desc("createTime");
            } else if (supplement == SignupSupplementType.NOT_NEW.getCode()) {
                builder.or(Expressions.eq("lessonCount", Flag.FALSE.getInt()),
                    Expressions.gt("kexiaoCascadeId", Flag.NULL.getInt()));
                // builder.or(Expressions.eq("chargeUnit", Flag.NULL.getInt()),
                // Expressions.gt("kexiaoCascadeId", Flag.NULL.getInt()));
                builder.desc("createTime");
            }
        }
    }

    private static final String syncUpdateSql =
        "update yunying.org_signup_course set class_id = :classId where class_id = -1 and org_id = :orgId and user_id =:userId and org_course_id = :courseId";

    @Override
    public void syncRealCourseId(Collection<String> syncDatas) {
        if (GenericsUtils.isNullOrEmpty(syncDatas)) {
            return;
        }
        Map<String, Object> paramMap = Maps.newHashMap();
        for (String syncDataStr : syncDatas) {
            log.info("syncData is :{} ", syncDataStr);
            List<Long> stringToNumber = GenericsUtils.stringToNumberList(syncDataStr, "_", Long.class);
            Long userId = stringToNumber.get(0);
            Long courseId = stringToNumber.get(1);
            Long orgId = stringToNumber.get(2);
            Long classId = stringToNumber.get(3);
            paramMap.put("userId", userId);
            paramMap.put("courseId", courseId);
            paramMap.put("orgId", orgId);
            paramMap.put("classId", classId);
            try {
                this.getNamedJdbcTemplate().update(syncUpdateSql, paramMap);
            } catch (Exception e) {
                log.info("can not sync :{} cause by :{} ", syncDataStr, e);
            }
            log.info("update org_signup_refund with sql:{} and paramMap: {}", syncUpdateSql, paramMap);
        }
    }

    @Override
    public List<OrgSignupCourse> getCousePurchaseListByMinId(Long minId, Date lastDate, int maxSize,
        List<Integer> notStatus, String...queryProps) {
        Preconditions.checkArgument(maxSize > 0, "maxSize is illegal");
        SingleSqlBuilder<OrgSignupCourse> builder = createSqlBuilder(queryProps);
        if (minId == null || minId == 0) {
            if (lastDate == null) {
                throw new IllegalArgumentException("must have minId or lastDate");
            }
            builder.gt("updateTime", lastDate);
        } else {
            builder.gt("id", minId);
        }

        builder.in("status", notStatus);
        builder.le("updateTime", DateUtil.getDiffDateTime(new Date(), -5, 12));// 5分钟前的数据
        builder.desc("updateTime");
        builder.setMaxSize(maxSize);
        return queryList(builder);
    }

    @Override
    public int sumAllLessonCount(Long orgId, Long orgCourseId, Long userId, List<Integer> status, String...queryProps) {
        Preconditions.checkArgument(orgCourseId != null && userId != 0, "orgCourseId userId is null!");
        SingleSqlBuilder<OrgSignupCourse> builder = createSqlBuilder(queryProps);

        builder.sum("lessonCount");
        builder.eq("orgId", orgId);
        builder.eq("orgCourseId", orgCourseId);
        builder.eq("userId", userId);
        builder.in("status", status);
        builder.eq("isDel", DataStatus.NORMAL.getValue());

        Integer count = queryForObject(builder, Integer.class);
        return count == null ? 0 : count;
    }

    @Override
    public int getSignupCourseCount(Long orgId, Long courseId, Collection<Integer> statusArr) {
        SingleSqlBuilder<OrgSignupCourse> builder = createSqlBuilder();
        builder.count("id");
        builder.eq("orgId", orgId);
        builder.eq("orgCourseId", courseId);
        builder.eq("isDel", DeleteStatus.NORMAL.getValue());
        if (CollectionUtils.isNotEmpty(statusArr)) {
            builder.in("status", statusArr);
        }
        Integer count = queryForObject(builder, Integer.class);
        return count == null ? 0 : count;
    }
    // @Override
    // public List<OrgSignupCourse> getByClassIdsAndStudentIds(Long orgId, Collection<Long> classIds,
    // Collection<Long> userIds, String...props) {
    // if (CollectionUtils.isEmpty(classIds) || CollectionUtils.isEmpty(userIds)) {
    // return Lists.newArrayList();
    // }
    // SingleSqlBuilder<OrgSignupCourse> builder = createSqlBuilder(props);
    // builder.eq("orgId", orgId);
    // builder.in("classId", classIds);
    // builder.in("userId", userIds);
    // return queryList(builder);
    // }

    /*
     * (non-Javadoc)
     * 
     * @see com.baijia.tianxiao.dal.signup.dao.OrgSignupCourseDao#listAllNeedSyncRecordWithRange(int, int)
     */

    @Override
    public List<OrgSignupCourse> listAllNeedSyncRecordWithRange(int firstId, int lastId) {
        SingleSqlBuilder<OrgSignupCourse> builder = createSqlBuilder("id", "orgId", "orgCourseId", "userId");
        builder.gt("id", firstId);
        builder.le("id", lastId);
        builder.eq("classId", -1);
        builder.asc("id");
        builder.setMaxSize(5000);
        log.info("sql is:{} ,lastRecordId {} and firstId :{} and lastId:{} ", builder.toSql(), firstId, lastId);
        List<OrgSignupCourse> retList = queryList(builder);
        log.info("retList's size is{}", retList.size());
        return retList;
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.baijia.tianxiao.dal.signup.dao.OrgSignupCourseDao#selectMaxId()
     */
    @Override
    public int selectMaxId() {

        SingleSqlBuilder<OrgSignupCourse> builder = createSqlBuilder();
        builder.max("id");
        return queryForObject(builder, Integer.class);
    }

    @Override
    public int reSyncLessonCount(List<Long> purchaseIds) {
        Preconditions.checkArgument(purchaseIds != null && purchaseIds.size() > 0, "purchaseIds is null!");
        StringBuilder sb = new StringBuilder(
                "update yunying.org_signup_course set sync_lesson_count = 0 where signup_purchase_id in (:purchaseIds)");
        Map params = new HashedMap<String, Object>();
        params.put("purchaseIds", purchaseIds);

        int result = getNamedJdbcTemplate().update(sb.toString(), params);

        log.info("updateStatusByPurchaseId sql={},params={}，result={}", sb.toString(), params, result);
        return result;
    }
}
