/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */

package com.baijia.tianxiao.dal.roster.dao.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.stereotype.Repository;

import com.baijia.tianxiao.dal.org.constant.StudentType;
import com.baijia.tianxiao.dal.roster.dao.TxStudentTagDao;
import com.baijia.tianxiao.dal.roster.po.TxStudentTag;
import com.baijia.tianxiao.sqlbuilder.SingleSqlBuilder;
import com.baijia.tianxiao.sqlbuilder.bean.impl.MatchMode;
import com.baijia.tianxiao.sqlbuilder.support.JdbcTemplateDaoSupport;
import com.baijia.tianxiao.util.GenericsUtils;
import com.google.common.collect.Maps;

import lombok.extern.slf4j.Slf4j;

/**
 * @author shanyu
 * @version 1.0
 * @title TxRosterTagDaoImpl
 * @desc TODO
 * @date 2015年12月3日
 */
@Repository
@Slf4j
public class TxStudentTagDaoImpl extends JdbcTemplateDaoSupport<TxStudentTag> implements TxStudentTagDao {

    public TxStudentTagDaoImpl() {
        super(TxStudentTag.class);
    }

    @Override
    public List<TxStudentTag> getTags(Long studentId, Long orgId, Integer type, String...queryProps) {
        SingleSqlBuilder<TxStudentTag> builder = createSqlBuilder(queryProps);
        if (type.intValue() == StudentType.CONSULT_USER.getCode()) {
            builder.eq("consultUserId", studentId);
        } else {
            builder.eq("userId", studentId);
            builder.eq("orgId", orgId);
        }
        builder.desc("createTime");
        return queryList(builder);
    }

    @Override
    public List<TxStudentTag> getTags(Collection<Long> studentIds, Long orgId, Integer type, String...queryProps) {
        SingleSqlBuilder<TxStudentTag> builder = createSqlBuilder(queryProps);
        if (type.intValue() == StudentType.CONSULT_USER.getCode()) {
            builder.in("consultUserId", studentIds);
        } else {
            builder.in("userId", studentIds);
        }
        builder.eq("orgId", orgId);
        return queryList(builder);
    }

    @Override
    public List<Long> searchByTagValue(Long orgId, String tagValue, String...queryProps) {
        if (GenericsUtils.isNullOrEmpty(tagValue)) {
            return GenericsUtils.emptyList();
        }
        SingleSqlBuilder<TxStudentTag> builder = createSqlBuilder(queryProps);
        builder.select("consultUserId");
        builder.eq("orgId", orgId);
        builder.like("content", tagValue, MatchMode.ANYWHERE);
        log.debug("searchByTagValue==sql={},params={}", builder.toSql(), builder.collectConditionValue());
        return queryForList(builder, Long.class);
    }

    @Override
    public void updateTag(Long consultUserId, Long userId) {
        SingleSqlBuilder<TxStudentTag> builder = createSqlBuilder();
        builder.eq("consultUserId", consultUserId);
        Map<String, Object> params = Maps.newHashMap();
        params.put("userId", userId);
        params.putAll(builder.collectConditionValue());
        getNamedJdbcTemplate().update(builder.toUpdateSql("userId"), params);
    }

    @Override
    public void delTags(Long userId, Long orgId) {
        SingleSqlBuilder<TxStudentTag> builder = createSqlBuilder();
        builder.eq("userId", userId);
        builder.eq("orgId", orgId);
        this.getNamedJdbcTemplate().update(builder.toDeleteSqlByCondition(), builder.collectConditionValue());
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.baijia.tianxiao.dal.roster.dao.TxStudentTagDao#getAndCachedTags(java.lang.Long, java.lang.Long,
     * java.lang.Integer)
     */

    @Override
    public Map<String, TxStudentTag> getAndCachedTags(Long studentId, Long orgId, Integer type) {
        SingleSqlBuilder<TxStudentTag> builder = createSqlBuilder();
        if (type.intValue() == StudentType.CONSULT_USER.getCode()) {
            builder.eq("consultUserId", studentId);
        } else {
            builder.eq("userId", studentId);
            builder.eq("orgId", orgId);
        }
        builder.desc("createTime");
        List<TxStudentTag> result = queryList(builder);
        if (result != null && !result.isEmpty()) {
            Map<String, TxStudentTag> ret = Maps.newHashMap();
            for (TxStudentTag tag : result) {
                ret.put(tag.getContent(), tag);
            }
            return ret;
        } else {
            return Collections.emptyMap();
        }
    }

    @Override
    public TxStudentTag getUniqueTag(Long id, Long orgId, Integer type, Long studentId, String...queryProps) {
        SingleSqlBuilder<TxStudentTag> builder = createSqlBuilder(queryProps);
        builder.eq("id", id);
        builder.eq("orgId", orgId);
        if (type.intValue() == StudentType.CONSULT_USER.getCode()) {
            builder.eq("consultUserId", studentId);
        } else {
            builder.eq("userId", studentId);
        }
        return uniqueResult(builder);
    }

    @Override
    public List<TxStudentTag> getHotTags(Long orgId, Integer listSize, Integer minTagTimes) {

        StringBuilder builder = new StringBuilder();
        builder.append(
            "SELECT content, count(id) as num FROM yunying.tx_student_tag WHERE org_id=:orgId GROUP BY content ");
        if (minTagTimes != null && minTagTimes > 0) {
            builder.append(" HAVING num>:minTagTimes ");
        }
        builder.append(" ORDER BY num DESC ");
        if (listSize != null && listSize > 0) {
            builder.append(" LIMIT :listSize");
        }

        NamedParameterJdbcTemplate template = getNamedJdbcTemplate();
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orgId", orgId);
        params.put("listSize", listSize);
        params.put("minTagTimes", minTagTimes);

        List<TxStudentTag> list = template.query(builder.toString(), params, new RowMapper<TxStudentTag>() {
            TxStudentTag tag = null;

            @Override
            public TxStudentTag mapRow(ResultSet rs, int rowNum) throws SQLException {
                tag = new TxStudentTag();
                tag.setContent(rs.getString("content"));
                return tag;
            }
        });

        return list;
    }

    @Override
    public void delTagsByConsulterId(Long consulterId, Long orgId) {
        SingleSqlBuilder<TxStudentTag> builder = createSqlBuilder();
        builder.eq("consultUserId", consulterId);
        builder.eq("orgId", orgId);
        this.getNamedJdbcTemplate().update(builder.toDeleteSqlByCondition(), builder.collectConditionValue());
    }

    @Override
    public List<TxStudentTag> getTagsByConsultId(Long consultId, Long orgId) {
        SingleSqlBuilder<TxStudentTag> builder = createSqlBuilder();
        builder.eq("consultUserId", consultId);
        builder.eq("orgId", orgId);
        return queryList(builder);
    }
}
