
/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2016 All Rights Reserved.
 */
package com.baijia.tianxiao.dal.org.dao.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.core.RowCallbackHandler;
import org.springframework.stereotype.Repository;

import com.baijia.tianxiao.constant.OrgTeacherStatus;
import com.baijia.tianxiao.dal.org.dao.OrgTeacherDao;
import com.baijia.tianxiao.dal.org.po.OrgTeacher;
import com.baijia.tianxiao.sqlbuilder.SingleSqlBuilder;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.sqlbuilder.support.JdbcTemplateDaoSupport;
import com.baijia.tianxiao.util.GenericsUtils;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;

import lombok.extern.slf4j.Slf4j;

/**
 * @author cxm
 * @version 1.0
 * @title OrgTeacherDaoImpl
 * @desc TODO
 * @date 2016年1月8日
 */
@Repository
@Slf4j
public class OrgTeacherDaoImpl extends JdbcTemplateDaoSupport<OrgTeacher> implements OrgTeacherDao {

    /**
     */
    public OrgTeacherDaoImpl() {
        super(OrgTeacher.class);
    }

    @Override
    public List<Long> getTeacherIds(Long orgId, Integer status) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        SingleSqlBuilder<OrgTeacher> builder = createSqlBuilder("userId");
        builder.eq("orgId", orgId);
        if (status != null) {
            builder.eq("status", status);
        }
        builder.eq("useStatus", 0);
        return queryForList(builder, Long.class);
    }

    @Override
    public List<OrgTeacher> listByLastId(Long orgId, Long lastId, Integer pageSize, Boolean isInUse, Integer status,
        String...queryProps) {
        SingleSqlBuilder<OrgTeacher> builder = createSqlBuilder(queryProps);
        if (lastId != null && lastId != 0) {
            builder.lt("id", lastId);
        }
        builder.eq("orgId", orgId);
        if (status != null) {
            builder.eq("status", status);
        }
        if (isInUse != null && isInUse) {
            builder.eq("useStatus", 0);
        }
        builder.setMaxSize(pageSize);
        builder.desc("id");
        log.debug("list by last id sql = {},params = {}", builder.toSql(), builder);
        return queryList(builder);
    }

    @Override
    public List<OrgTeacher> listByPageDto(Long orgId, Integer useStatus, Integer status, PageDto pageDto,
        String...queryProps) {
        SingleSqlBuilder<OrgTeacher> builder = createSqlBuilder(queryProps);
        builder.eq("orgId", orgId);
        if (status != null) {
            builder.eq("status", status);
        }
        if (useStatus == null) {
            builder.eq("useStatus", 0);
        } else {
            builder.eq("useStatus", useStatus);
        }
        builder.setPage(pageDto);
        builder.desc("createTime");
        log.debug("list by last id sql = {},params = {}", builder.toSql(), builder);
        return queryList(builder);
    }

    @Override
    public List<OrgTeacher> getTeacherByOpenIdAndOrgId(Long orgId, String weixinOpenId, Integer status,
        String...queryProps) {
        SingleSqlBuilder<OrgTeacher> builder = createSqlBuilder(queryProps);
        builder.eq("weixin", weixinOpenId);
        builder.eq("orgId", orgId);
        if (status != null) {
            builder.eq("status", status);
        }
        builder.eq("useStatus", 0);
        builder.desc("id");

        return queryList(builder);
    }

    @Override
    public List<OrgTeacher> getTeacherByMobileAndOrgId(Long orgId, String mobile, Integer maxSize,
        String...queryProps) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(StringUtils.isNoneBlank(mobile), "mobile can not be empty!");
        SingleSqlBuilder<OrgTeacher> builder = createSqlBuilder(queryProps);
        builder.eq("mobile", mobile);
        builder.eq("orgId", orgId);
        builder.eq("useStatus", 0);
        builder.eq("status", OrgTeacherStatus.SIGNED.getCode());
        builder.desc("id");

        if (maxSize != null && maxSize > 0) {
            builder.setMaxSize(maxSize);
        }
        log.debug("sql={},param={}", builder.toSql(), builder.collectConditionValue());
        return queryList(builder);
    }

    @Override
    public List<OrgTeacher> getTeacherByUserIdAndOrgId(Long orgId, Long userId, Integer status, String...queryProps) {
        SingleSqlBuilder<OrgTeacher> builder = createSqlBuilder(queryProps);
        builder.eq("userId", userId);
        builder.eq("orgId", orgId);
        builder.eq("useStatus", 0);
        if (status != null) {
            builder.eq("status", status);
        }
        builder.desc("id");

        return queryList(builder);
    }

    @Override
    public List<OrgTeacher> getTeacherByUserIdsAndOrgId(Long orgId, Collection<Long> userIds, Integer status,
        Boolean isInUse, String...queryProps) {
        SingleSqlBuilder<OrgTeacher> builder = createSqlBuilder(queryProps);
        builder.in("userId", userIds);
        builder.eq("orgId", orgId);
        if (isInUse != null && isInUse) {
            builder.eq("useStatus", 0);
        }
        if (status != null) {
            builder.eq("status", status);
        }
        builder.desc("id");

        return queryList(builder);
    }

    @Override
    public Map<Long, OrgTeacher> getMapByIds(Long orgId, Collection<Long> userIds, String...queryProps) {

        if (userIds.isEmpty()) {
            return Collections.emptyMap();
        }

        List<OrgTeacher> orgTeachers =
            getTeacherByUserIdsAndOrgId(orgId, userIds, OrgTeacherStatus.SIGNED.getCode(), true, queryProps);

        if (CollectionUtils.isEmpty(orgTeachers)) {
            return Collections.emptyMap();
        }
        Map<Long, OrgTeacher> result = Maps.newHashMap();
        for (OrgTeacher user : orgTeachers) {
            result.put(user.getUserId().longValue(), user);
        }
        return result;
    }

    @Override
    public void refreshOrgTeacher(Long orgId, Long id) {
        StringBuilder sb = new StringBuilder(
            "update yunying.org_teachers set status =2,fire_time=now() where id > :id and  org_id = :orgId");
        Map param = Maps.newHashMap();
        param.put("orgId", orgId);
        param.put("id", id);
        this.getNamedJdbcTemplate().update(sb.toString(), param);
    }

    @Override
    public void saveCdbOrgTeacher(Long userId, Long userNumber, Long orgId) {
        StringBuilder sb = new StringBuilder(
            "insert into cdb.org_teacher(user_id,user_number,is_valid,created_at,org_id) values(:userId,:userNumber,1,now(),:orgId)");
        Map param = Maps.newHashMap();
        param.put("orgId", orgId);
        param.put("userId", userId);
        param.put("userNumber", userNumber);
        this.getNamedJdbcTemplate().update(sb.toString(), param);
    }

    @Override
    public List<OrgTeacher> getTeachersByOrgId(Long orgId, PageDto page, String...queryProp) {
        SingleSqlBuilder<OrgTeacher> builder = createSqlBuilder();
        builder.select(queryProp);
        builder.eq("orgId", orgId);
        builder.eq("useStatus", 0);
        builder.eq("status", OrgTeacherStatus.SIGNED.getCode());
        builder.desc("id");
        builder.setPage(page);
        return queryList(builder);
    }

    @Override
    public Map<Long, Integer> getTeacherTotalMapByOrgIds(Date startDate, Date endDate, List<Long> TianxiaoOrgIds) {
        Map<String, Object> param = new HashMap<>();
        param.put("orgIds", TianxiaoOrgIds);
        String sql =
            "select count(1) as count, org_id from yunying.org_teachers where org_id in (:orgIds) and status = 1 and use_status=0";

        if (startDate != null && endDate != null) {
            sql += " and create_time between :startDate and :endDate";
            param.put("startDate", startDate);
            param.put("endDate", endDate);
        }

        sql += " GROUP BY org_id";
        return this.getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<Map<Long, Integer>>() {

            @Override
            public Map<Long, Integer> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<Long, Integer> map = new HashMap<>();
                while (rs.next()) {
                    Long orgId = rs.getLong("org_id");
                    Integer count = rs.getInt("count");
                    map.put(orgId, count);
                }
                return map;
            }
        });
    }

    @Override
    public List<OrgTeacher> getOrgTeacherListByOrgIds(List<Long> orgIds) {
        SingleSqlBuilder<OrgTeacher> builder = createSqlBuilder();
        builder.in("orgId", orgIds);
        builder.eq("useStatus", 0);
        builder.eq("status", 1);
        return queryList(builder);
    }

    /**
     * 获取所有停用的老师
     * 
     * @param orgIds
     * @return
     */
    @Override
    public List<OrgTeacher> getPausedOrgTeacherListByOrgIds(List<Long> orgIds) {
        SingleSqlBuilder<OrgTeacher> builder = createSqlBuilder();
        builder.in("orgId", orgIds);
        builder.eq("useStatus", 1);
        builder.eq("status", 1);
        return queryList(builder);
    }

    @Override
    public Integer getSignedTeacherCount(List<Long> orgIds) {
        SingleSqlBuilder<OrgTeacher> builder = createSqlBuilder();
        builder.count("id");
        builder.in("orgId", orgIds);
        builder.eq("useStatus", 0);
        builder.eq("status", OrgTeacherStatus.SIGNED.getCode());
        return queryForObject(builder, Integer.class);
    }

    @Override
    public Map<Integer, Integer> getSignedTeacherCountMap(List<Integer> orgIds) {
        final Map<Integer, Integer> data = Maps.newHashMap();
        SingleSqlBuilder<OrgTeacher> builder = createSqlBuilder();
        builder.count("id", "num");
        builder.in("orgId", orgIds);
        builder.group("orgId");
        builder.eq("useStatus", 0);
        builder.eq("status", OrgTeacherStatus.SIGNED.getCode());
        getNamedJdbcTemplate().query(builder.toSql(), builder.collectConditionValue(), new RowCallbackHandler() {
            @Override
            public void processRow(ResultSet rs) throws SQLException {
                data.put(rs.getInt("orgId"), rs.getInt("num"));
            }
        });
        return data;

    }

    @Override
    public OrgTeacher getTeachersByOrgIdAndId(Long orgId, Long teacherId, PageDto pageDto) {
        SingleSqlBuilder<OrgTeacher> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.eq("status", OrgTeacherStatus.SIGNED.getCode());
        builder.eq("user_id", teacherId);
        builder.eq("useStatus", 0);
        builder.desc("id");
        builder.setPage(pageDto);
        return queryForObject(builder, OrgTeacher.class);
    }

    @Override
    public OrgTeacher getTeachersByUserId(Long userId) {
        SingleSqlBuilder<OrgTeacher> builder = createSqlBuilder();
        builder.eq("status", OrgTeacherStatus.SIGNED.getCode());
        builder.eq("user_id", userId);
        builder.eq("useStatus", 0);
        log.debug("---------------sql = {},param={}", builder.toSql(), builder.collectConditionValue());
        return uniqueResult(builder, OrgTeacher.class);
    }

    @Override
    public List<OrgTeacher> listOpenIdNotNull(Long minId) {
        SingleSqlBuilder<OrgTeacher> builder = createSqlBuilder();
        builder.ne("weixin", "");
        builder.eq("useStatus", 0);
        builder.gt("id", minId);
        return queryList(builder);
    }

    @Override
    public List<OrgTeacher> findStudentWithOrgAndOpenId(List<Integer> orgIds, String openId, String...queryProps) {
        if (GenericsUtils.isNullOrEmpty(orgIds)) {
            return GenericsUtils.emptyList();
        }
        SingleSqlBuilder<OrgTeacher> builder = createSqlBuilder(queryProps);
        builder.in("orgId", orgIds);
        builder.eq("weixin", openId);
        builder.eq("useStatus", 0);
        builder.eq("status", OrgTeacherStatus.SIGNED.getCode());
        return this.queryList(builder);
    }

    @Override
    public List<OrgTeacher> getTeachersByMobileAndOrgIds(List<Long> orgIds, String mobile, String...queryProps) {
        if (GenericsUtils.isNullOrEmpty(orgIds)) {
            return GenericsUtils.emptyList();
        }
        SingleSqlBuilder<OrgTeacher> builder = createSqlBuilder(queryProps);
        builder.in("orgId", orgIds);
        builder.eq("mobile", mobile);
        builder.ne("weixin", "");
        builder.eq("useStatus", 0);
        builder.eq("status", OrgTeacherStatus.SIGNED.getCode());
        builder.desc("createTime");
        return queryList(builder);
    }

    @Override
    public List<OrgTeacher> searchHasMobileAndNeedBindWithOpenId(long orgId, PageDto pageDto) {
        SingleSqlBuilder<OrgTeacher> builder = createSqlBuilder();
        builder.select("mobile", "id");
        builder.ne("mobile", "");
        builder.eq("orgId", orgId);
        builder.eq("weixin", "");
        builder.eq("useStatus", 0);
        builder.eq("status", OrgTeacherStatus.SIGNED.getCode());
        builder.setPage(pageDto);
        builder.desc("createTime");
        return queryList(builder);
    }

    @Override
    public List<Long> listTeacherUserIds(List<Long> orgIds) {
        SingleSqlBuilder<OrgTeacher> builder = createSqlBuilder("userId");
        builder.in("orgId", orgIds);
        return queryForList(builder, Long.class);
    }

    @Override
    public Map<Long, OrgTeacher> mapKeyUserIdVauleTeacher(List<Long> orgIds) {
        Map<Long, OrgTeacher> result = new HashMap<Long, OrgTeacher>();

        SingleSqlBuilder<OrgTeacher> builder = createSqlBuilder();
        builder.in("orgId", orgIds);
        List<OrgTeacher> list = queryList(builder);

        if (CollectionUtils.isNotEmpty(list)) {
            for (OrgTeacher teacher : list) {
                result.put(teacher.getUserId(), teacher);
            }
        }
        return result;
    }

    @Override
    public Integer countTeacher(List<Long> orgIds) {
        SingleSqlBuilder<OrgTeacher> builder = createSqlBuilder();
        builder.count("id");
        builder.in("orgId", orgIds);
        Integer count = queryForObject(builder, Integer.class);
        return count != null ? count : 0;
    }

    @Override
    public OrgTeacher getByUserId(Long userId) {
        SingleSqlBuilder<OrgTeacher> builder = createSqlBuilder();
        builder.eq("user_id", userId);
        return uniqueResult(builder, OrgTeacher.class);
    }

    @Override
    public Collection<OrgTeacher> getTeachersByOrgIdAndIds(Integer orgId, Collection<Long> teacherIds) {
        SingleSqlBuilder<OrgTeacher> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.eq("status", OrgTeacherStatus.SIGNED.getCode());
        builder.in("id", teacherIds);
        // builder.eq("useStatus", 0);
        builder.desc("id");
        return queryList(builder);
    }

}