/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */

package com.baijia.tianxiao.dal.org.dao.impl;

import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Repository;

import com.baijia.tianxiao.dal.org.constant.DeleteStatus;
import com.baijia.tianxiao.dal.org.constant.OrgSubAccountStatus;
import com.baijia.tianxiao.dal.org.dao.OrgSubAccountDao;
import com.baijia.tianxiao.dal.org.po.OrgSubAccount;
import com.baijia.tianxiao.sqlbuilder.SingleSqlBuilder;
import com.baijia.tianxiao.sqlbuilder.bean.Order;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.sqlbuilder.support.JdbcTemplateDaoSupport;
import com.baijia.tianxiao.sqlbuilder.util.Expressions;
import com.baijia.tianxiao.util.GenericsUtils;

import com.google.common.collect.Lists;

import lombok.extern.slf4j.Slf4j;

/**
 * @author caoliang
 * @version 1.0
 * @title TXCampusAccountDaoImpl
 * @desc TODO
 * @date 2016年5月10日
 */

@Repository
@Slf4j
public class OrgSubAccountDaoImpl extends JdbcTemplateDaoSupport<OrgSubAccount> implements OrgSubAccountDao {

    /**
     * @param entityClass
     */
    public OrgSubAccountDaoImpl() {
        super(OrgSubAccount.class);
    }

    @Override
    public OrgSubAccount getMasterByOrgId(Integer orgId) {
        SingleSqlBuilder<OrgSubAccount> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        // 0表示主帐号
        builder.eq("pid", OrgSubAccount.MASTER_PID);
        builder.eq("status", OrgSubAccountStatus.NORMAL.getCode());
        builder.eq("isdel", DeleteStatus.NORMAL.getValue());
        return uniqueResult(builder);

    }

    @Override
    public List<OrgSubAccount> getSlavesByMasterOrgId(Integer orgId, PageDto pageDto) {
        SingleSqlBuilder<OrgSubAccount> builder = createSqlBuilder();
        builder.eq("pid", orgId);
        builder.eq("isdel", DeleteStatus.NORMAL.getValue());
        // builder.eq("status", OrgSubAccountStatus.NORMAL.getCode());
        if (pageDto != null) {
            builder.setPage(pageDto);
        }
        return queryList(builder);

    }

    @Override
    public List<OrgSubAccount> getMasterAndSlaves(Integer orgId, PageDto pageDto) {
        SingleSqlBuilder<OrgSubAccount> builder = createSqlBuilder();
        builder.or(Expressions.eq("orgId", orgId), Expressions.eq("pid", orgId));
        builder.eq("isdel", DeleteStatus.NORMAL.getValue());
        builder.setOrders(Lists.newArrayList(Order.asc("pid"), Order.desc("onlineType"), Order.asc("id")));
        if (pageDto != null) {
            builder.setPage(pageDto);
        }
        return queryList(builder);
    }

    @Override
    public List<OrgSubAccount> getSlavesIgnoreStatus(Integer orgId, PageDto pageDto) {
        SingleSqlBuilder<OrgSubAccount> builder = createSqlBuilder();
        builder.eq("pid", orgId);
        builder.eq("isdel", DeleteStatus.NORMAL.getValue());
        if (pageDto != null) {
            builder.setPage(pageDto);
        }
        return queryList(builder);

    }

    /*
     * (non-Javadoc)
     * 
     * @see com.baijia.tianxiao.dal.org.dao.OrgSubAccountDao#getSlavesCountByMasterOrgId(java.lang.Integer)
     */

    @Override
    public Integer getSlavesCountByMasterOrgId(Integer orgId) {
        SingleSqlBuilder<OrgSubAccount> builder = createSqlBuilder();
        builder.count("id");
        builder.eq("pid", orgId);
        builder.eq("isdel", DeleteStatus.NORMAL.getValue());

        // builder.eq("status", OrgSubAccountStatus.NORMAL.getCode());
        return queryForObject(builder, Integer.class);
    }

    @Override
    public OrgSubAccount getByOrgId(Integer orgId, String...props) {
        SingleSqlBuilder<OrgSubAccount> builder = createSqlBuilder(props);
        builder.eq("orgId", orgId);
        // builder.gt("pid", 0);
        builder.eq("isdel", DeleteStatus.NORMAL.getValue());
        // builder.eq("status", OrgSubAccountStatus.NORMAL.getCode()); //这里绝对不应该排除status禁用的情况 否则 orgId非唯一!! by hongyan
        return uniqueResult(builder);
    }

    @Override
    public List<OrgSubAccount> getByOrgIds(Collection<Integer> orgIds, Integer status, String...props) {
        if (CollectionUtils.isEmpty(orgIds)) {
            return Lists.newArrayList();
        }
        SingleSqlBuilder<OrgSubAccount> builder = createSqlBuilder(props);
        builder.in("orgId", orgIds);
        // builder.gt("pid", 0);
        builder.eq("isdel", DeleteStatus.NORMAL.getValue());
        if (null != status) {
            builder.eq("status", status);
        }
        return queryList(builder);
    }

    @Override
    public OrgSubAccount getByOrgIdIgnoreDelStatus(Integer orgId) {
        SingleSqlBuilder<OrgSubAccount> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        return uniqueResult(builder);
    }

    @Override
    public OrgSubAccount getSlaveIgnoreStatus(Integer orgId, Integer masterId) {
        SingleSqlBuilder<OrgSubAccount> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.eq("pid", masterId);
        builder.eq("isdel", DeleteStatus.NORMAL.getValue());
        return uniqueResult(builder);
    }

    @Override
    public List<OrgSubAccount> getOrgSubAccountByOrgIds(Collection<Long> orgIds, Integer status) {
        SingleSqlBuilder<OrgSubAccount> builder = createSqlBuilder();
        builder.in("orgId", orgIds);
        if (GenericsUtils.notNullAndEmpty(status)) {
            builder.eq("status", status);
        }
        builder.eq("isdel", DeleteStatus.NORMAL.getValue());
        return queryList(builder);
    }

    @Override
    public List<OrgSubAccount> getAllByPageDto(PageDto pageDto) {
        SingleSqlBuilder<OrgSubAccount> builder = createSqlBuilder("orgId", "isdel", "account_version");
        // builder.eq("isdel", DeleteStatus.NORMAL.getValue()); //不能排除删掉的帐号,否则无法踢出
        builder.setPage(pageDto);
        return queryList(builder);
    }

    @Override
    public void incrVersionByOrgId(int orgId) {
        String sql =
            "update yunying.org_sub_account set account_version=account_version+1, update_time=:updateTime where org_id=:orgId";
        Map<String, Object> param = new HashMap<>();
        param.put("orgId", orgId);
        param.put("updateTime", new Date());
        getNamedJdbcTemplate().update(sql, param);
    }

    @Override
    public void deleteByOrgid(int orgId) {
        String sql =
            "update yunying.org_sub_account set isdel=1, account_version=account_version+1, update_time=:updateTime where org_id=:orgId";
        Map<String, Object> param = new HashMap<>();
        param.put("orgId", orgId);
        param.put("updateTime", new Date());
        getNamedJdbcTemplate().update(sql, param);
    }

    @Override
    public OrgSubAccount getByOrgIdIgnoreIsdel(Integer orgId) {
        SingleSqlBuilder<OrgSubAccount> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        return uniqueResult(builder);
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.baijia.tianxiao.dal.org.dao.OrgSubAccountDao#resetOrgWechatType(java.lang.Integer)
     */

    /**
     * 在主账号微信公众号解绑之后,可能不需要对分校区的设置进行归零处理
     */
    @Override
    public void resetOrgWechatType(Integer orgId) {
        if (orgId == null) {
            return;
        }
        String sql =
            "update yunying.org_sub_account set org_wechat_type = 1,update_time=:updateTime where pid = :orgId";
        Map<String, Object> param = new HashMap<>();
        param.put("orgId", orgId);
        param.put("updateTime", new Date());
        getNamedJdbcTemplate().update(sql, param);
        log.info("orgId :{} need to reset orgWechatType ", orgId);
    }

    @Override
    public List<Long> findAllTxOrgIds() {
        List<Long> orgIds = Lists.newArrayList();
        List<OrgSubAccount> orgSubAccounts = null;
        PageDto pageDto = new PageDto();
        pageDto.setPageSize(100);
        pageDto.setPageNum(1);
        int total = 0;
        do {
            orgSubAccounts = this.getAllByPageDto(pageDto);
            if (CollectionUtils.isNotEmpty(orgSubAccounts)) {
                for (OrgSubAccount orgSubAccount : orgSubAccounts) {
                    if (orgSubAccount.getIsdel() != null
                        && orgSubAccount.getIsdel() == DeleteStatus.NORMAL.getValue()) {
                        total += 1;
                        orgIds.add(orgSubAccount.getOrgId().longValue());
                    }

                }
            }
            pageDto.setPageNum(pageDto.getPageNum() + 1);
        } while (orgSubAccounts.size() > 0);
        log.info("find total orgIds which not delete  is:{} and orgIds.size is:{} ", total, orgIds.size());
        return orgIds;
    }

}
