package com.baijia.tianxiao.dal.finance.dao.impl;

import com.baijia.tianxiao.consants.DataStatus;
import com.baijia.tianxiao.constant.TransferClassStatus;
import com.baijia.tianxiao.dal.finance.dao.TxTransferClassRecordDao;
import com.baijia.tianxiao.dal.finance.po.TxTransferClassRecord;
import com.baijia.tianxiao.sqlbuilder.SingleSqlBuilder;
import com.baijia.tianxiao.sqlbuilder.support.JdbcTemplateDaoSupport;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.jdbc.core.RowMapper;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Repository;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.*;

/**
 * Created by wengshengli on 2017/4/21.
 */
@Repository
@Slf4j
public class TxTransferClassRecordDaoImpl extends JdbcTemplateDaoSupport<TxTransferClassRecord>
        implements TxTransferClassRecordDao {

    @Override
    public List<TxTransferClassRecord> getOrgTransferRefundListByMinId(Long minId, Date lastDate, int maxSize,
                                                                       String... queryProps) {
        Preconditions.checkArgument(maxSize > 0, "maxSize is illegal");
        SingleSqlBuilder<TxTransferClassRecord> builder = createSqlBuilder(queryProps);
        if (minId == null || minId == 0) {
            if (lastDate == null) {
                throw new IllegalArgumentException("must have minId or lastDate");
            }
            builder.gt("updateTime", lastDate);
            builder.asc("updateTime");
        } else {
            builder.gt("id", minId);
            builder.asc("id");
        }
        builder.ne("refundType", 0);
        builder.eq("status", TransferClassStatus.SUCCESS.getCode());
        builder.eq("syncRefund", DataStatus.NORMAL.getValue());
        builder.setMaxSize(maxSize);
        return queryList(builder);
    }

    @Override
    public List<TxTransferClassRecord> groupByTransferNumber(Long outPurchaseId, Long transferOutClassId) {

        SingleSqlBuilder<TxTransferClassRecord> builder =
                this.createSqlBuilder();
        builder.select("transferNumber");
        builder.select("chargeUnit");
        builder.sum("realLessonCount","realLessonCount");
        builder.sum("freeLessonCount","freeLessonCount");
        builder.sum("lessonMoney","lessonMoney");
        builder.sum("refundMoney","refundMoney");
        builder.eq("outPurchaseId", outPurchaseId);
        if (transferOutClassId != null) {
            builder.eq("transferOutClassId", transferOutClassId);
        }
        builder.ne("status", TransferClassStatus.CANCEL.getCode());
        builder.group("transferNumber");
        builder.desc("createTime");

        log.debug("tranfer groupByTransferNumber={},{}", builder.toSql(), builder.collectConditionValue());
        return queryList(builder);
    }

    @Override
    public TxTransferClassRecord sumByPurchases(List<Long> outPurchaseIds, Long transferOutClassId) {

        SingleSqlBuilder<TxTransferClassRecord> builder = this.createSqlBuilder();
        builder.select("inPurchaseId");
        builder.select("transferNumber");
        builder.select("transferOutClassId");
        builder.select("transferOutUserId");
        builder.select("chargeUnit");
        builder.select("refundType");
        builder.select("syncRefund");
        builder.select("status");
        builder.select("createTime");
        builder.sum("realLessonCount","realLessonCount");
        builder.sum("freeLessonCount","freeLessonCount");
        builder.sum("lessonMoney","lessonMoney");
        builder.sum("refundMoney","refundMoney");
        builder.in("outPurchaseId", outPurchaseIds);
        if (transferOutClassId != null) {
            builder.eq("transferOutClassId", transferOutClassId);
        }
        builder.ne("status", TransferClassStatus.CANCEL.getCode());
        builder.setMaxSize(1);

        return uniqueResult(builder);
    }

    @Override
    public List<TxTransferClassRecord> listByUserClass(Long userId, Long classId, Long outPurchaseId, Integer status) {
        SingleSqlBuilder<TxTransferClassRecord> builder = this.createSqlBuilder();
        builder.eq("transferOutUserId", userId);
        builder.eq("transferOutClassId", classId);
        if (outPurchaseId != null) {
            builder.eq("outPurchaseId", outPurchaseId);
        }
        builder.eq("status", status);

        log.info("List listByUserClass sql={},params={}", builder.toSql(), builder.collectConditionValue());
        return queryList(builder);
    }

    @Override
    public List<TxTransferClassRecord> listByInPurchaseId(Long inPurchaseId) {

        SingleSqlBuilder<TxTransferClassRecord> builder = this.createSqlBuilder();
        builder.eq("inPurchaseId", inPurchaseId);
        builder.ne("status", TransferClassStatus.CANCEL.getCode());

        return queryList(builder);
    }
    
    @Override
    public List<TxTransferClassRecord> listByInPurchaseIds(Collection<Long> inPurchaseIds) {

        SingleSqlBuilder<TxTransferClassRecord> builder = this.createSqlBuilder();
        builder.in("inPurchaseId", inPurchaseIds);
        builder.ne("status", TransferClassStatus.CANCEL.getCode());

        return queryList(builder);
    }

    @Override
    public List<TxTransferClassRecord> listByOutPurchaseId(Long transferNumber, Long outPurchaseId) {

        SingleSqlBuilder<TxTransferClassRecord> builder = this.createSqlBuilder();
        builder.eq("outPurchaseId", outPurchaseId);
        builder.eq("transferNumber", transferNumber);
        builder.ne("status", TransferClassStatus.CANCEL.getCode());

        return queryList(builder);
    }

    public List<TxTransferClassRecord> listByOutPurchaseIds(Collection<Long> outPurchaseIds, String... props) {
        if (CollectionUtils.isEmpty(outPurchaseIds)) {
            return Lists.newArrayList();
        }
        SingleSqlBuilder<TxTransferClassRecord> builder = this.createSqlBuilder();
        builder.in("outPurchaseId", outPurchaseIds);
        builder.ne("status", TransferClassStatus.CANCEL.getCode());
        return queryList(builder);
    }

    @Override
    public List<TxTransferClassRecord> listByTransferNumber(Long transferNumber) {

        SingleSqlBuilder<TxTransferClassRecord> builder = this.createSqlBuilder();
        builder.eq("transferNumber", transferNumber);
        builder.ne("status", TransferClassStatus.CANCEL.getCode());

        return queryList(builder);
    }

    @Override
    public List<TxTransferClassRecord> listByTransferNumbers(Collection<Long> transferNumbers) {
        SingleSqlBuilder<TxTransferClassRecord> builder = this.createSqlBuilder();
        builder.in("transferNumber", transferNumbers);
        builder.ne("status", TransferClassStatus.CANCEL.getCode());
        return queryList(builder);
    }

    @Override
    public int updateRefundType(Long transferNumber, Integer refundType) {
        StringBuilder sb = new StringBuilder(
                "update yunying.tx_transfer_class_record set refund_type = :refundType,update_time = now() where transfer_number = :transferNumber");
        Map<String, Object> params = Maps.newHashMap();
        params.put("refundType", refundType);
        params.put("transferNumber", transferNumber);
        return this.getNamedJdbcTemplate().update(sb.toString(), params);
    }

    @Override
    public int updateStatus(Long transferNumber, Integer status) {
        StringBuilder sb = new StringBuilder(
                "update yunying.tx_transfer_class_record set status = :status,update_time = now() where transfer_number = :transferNumber");
        Map<String, Object> params = Maps.newHashMap();
        params.put("status", status);
        params.put("transferNumber", transferNumber);
        return this.getNamedJdbcTemplate().update(sb.toString(), params);
    }

    @Override
    public Map<String, TxTransferClassRecord> groupByPurchaseIds(Collection<Long> purchaseIds,
                                                                 Collection<Integer> status) {
        Map<String, TxTransferClassRecord> ret = new HashMap<>();
        Map<String, Object> param = new HashMap<>();
        param.put("outPurchaseId", purchaseIds);
        param.put("status", status);
        String sql =
                "select sum(real_lesson_count) as realLessonCount,sum(free_lesson_count) as freeLessonCount,sum(lesson_money) as lessonMoney,sum(refund_money) as refundMoney,"
                        + " out_purchase_id,transfer_out_class_id from yunying.tx_transfer_class_record where out_purchase_id in (:outPurchaseId) and status in (:status) group by"
                        + " out_purchase_id,transfer_out_class_id";
        List<TxTransferClassRecord> recordList =
                getNamedJdbcTemplate().query(sql, param, new RowMapper<TxTransferClassRecord>() {
                    @Override
                    public TxTransferClassRecord mapRow(ResultSet rs, int i) throws SQLException {
                        TxTransferClassRecord record = new TxTransferClassRecord();
                        record.setRealLessonCount(rs.getInt("realLessonCount"));
                        record.setFreeLessonCount(rs.getInt("freeLessonCount"));
                        record.setLessonMoney(rs.getInt("lessonMoney"));
                        record.setRefundMoney(rs.getInt("refundMoney"));
                        record.setOutPurchaseId(rs.getLong("out_purchase_id"));
                        record.setTransferOutClassId(rs.getLong("transfer_out_class_id"));
                        return record;
                    }
                });
        for (TxTransferClassRecord record : recordList) {
            ret.put(record.getOutPurchaseId() + "_" + record.getTransferOutClassId(), record);
        }
        return ret;
    }

}
