package com.baijia.tianxiao.dal.course.dao.impl;

import com.baijia.tianxiao.constants.CourseType;
import com.baijia.tianxiao.dal.course.dao.OrgCourseGroupRelateDao;
import com.baijia.tianxiao.dal.course.po.OrgCourseGroupRelate;
import com.baijia.tianxiao.dal.org.constant.DeleteStatus;
import com.baijia.tianxiao.sqlbuilder.SingleSqlBuilder;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.sqlbuilder.support.JdbcTemplateDaoSupport;
import com.baijia.tianxiao.util.collection.CollectorUtil;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.BeanPropertyRowMapper;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.core.RowCallbackHandler;
import org.springframework.stereotype.Repository;
import org.springframework.util.CollectionUtils;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.*;

/**
 * @author zhangbing
 * @version 1.0
 * @title OrgCourseGroupRelateDaoImpl
 * @desc TODO
 * @date 2015年12月13日
 */
@Repository("orgCourseGroupRelateDao")
public class OrgCourseGroupRelateDaoImpl extends JdbcTemplateDaoSupport<OrgCourseGroupRelate> implements
        OrgCourseGroupRelateDao {

    private static final Logger logger = LoggerFactory.getLogger(OrgCourseGroupRelateDaoImpl.class);

    /**
     */
    public OrgCourseGroupRelateDaoImpl() {
        super(OrgCourseGroupRelate.class);
    }

    /**
     * 查询机构自定义分类课程
     */
    @Override
    public List<OrgCourseGroupRelate> getOrgCourseGroup(Integer orgId, PageDto pageDto) {
        String sql =
                "select oc.* from org_course_group og,org_course_group_relate oc where og.org_id=?"
                        + " and og.id=oc.course_group_id and oc.is_deleted=1 and course_type!=0 order by oc.display_order desc";
        return this.getNamedJdbcTemplate().getJdbcOperations()
                .query(sql, new Object[]{orgId}, BeanPropertyRowMapper.newInstance(OrgCourseGroupRelate.class));
    }

    @Override
    public List<Long> getCourseIdsByGroup(Integer courseGroupId, Integer courseType) {
        try {
            SingleSqlBuilder<OrgCourseGroupRelate> builder = createSqlBuilder();
            builder.select("courseId");
            builder.eq("courseGroupId", courseGroupId);
            builder.eq("courseType", courseType);
            builder.eq("isDeleted", DeleteStatus.NORMAL.getValue());
            builder.desc("displayOrder");
            return queryForList(builder, Long.class);
        } catch (Exception e) {
            logger.info("courseGroupId:{}, courseType:{}, e:{}", courseGroupId, courseType, e);
        }
        return Collections.emptyList();
    }

    @Override
    public Collection<Long> getCourseIdsByGroups(Collection<Integer> courseGroupIds, Integer courseType) {
        if (courseGroupIds == null) {
            return null;
        }
        SingleSqlBuilder<OrgCourseGroupRelate> builder = createSqlBuilder("courseGroupId");
        builder.select("courseId");
        builder.in("courseGroupId", courseGroupIds);
        builder.eq("courseType", courseType);
        builder.eq("isDeleted", DeleteStatus.NORMAL.getValue());
        builder.desc("displayOrder");
        List<OrgCourseGroupRelate> list = this.queryList(builder);
        return CollectorUtil.collect(list, new Function<OrgCourseGroupRelate, Long>() {

            @Override
            public Long apply(OrgCourseGroupRelate input) {

                return input.getCourseId();

            }
        });
    }

    @Override
    public OrgCourseGroupRelate getCourseGroupByCourseId(Integer courseGroupId, Integer courseType, Long courseId) {
        SingleSqlBuilder<OrgCourseGroupRelate> builder = createSqlBuilder();
        builder.eq("courseId", courseId);
        builder.eq("courseGroupId", courseGroupId);
        builder.eq("courseType", courseType);
        builder.eq("isDeleted", DeleteStatus.NORMAL.getValue());
        return uniqueResult(builder);
    }

    @Override
    public void deleteGroupOfCourse(Long courseId, Collection<Integer> groupIds, Integer courseType) {
        if (CollectionUtils.isEmpty(groupIds)) {
            return;
        }
        Map<String, Object> updateCondtion = Maps.newHashMap();
        updateCondtion.put("courseId", courseId);
        updateCondtion.put("courseGroupId", groupIds);
        updateCondtion.put("courseType", courseType);
        OrgCourseGroupRelate obj = new OrgCourseGroupRelate();
        obj.setIsDeleted(DeleteStatus.DELETED.getValue());
        this.update(updateCondtion, false, obj, "isDeleted");
    }

    @Override
    public List<Integer> getGroupIdByCourseId(Long courseId, Integer courseType) {
        SingleSqlBuilder<OrgCourseGroupRelate> builder = createSqlBuilder();
        builder.select("courseGroupId");
        builder.eq("courseId", courseId);
        builder.eq("courseType", courseType);
        builder.eq("isDeleted", DeleteStatus.NORMAL.getValue());
        return this.queryForList(builder, Integer.class);
    }

    @Override
    public List<OrgCourseGroupRelate> getGroupByCourseId(Long courseId, Integer courseType) {
        SingleSqlBuilder<OrgCourseGroupRelate> builder = createSqlBuilder();
        builder.eq("courseId", courseId);
        builder.eq("courseType", courseType);
        builder.eq("isDeleted", courseType);
        return this.queryList(builder);
    }

    @Override
    public int getMaxDisplayOrderByExcludeIds(Integer courseGroupId, Collection<Long> excludeIds) {
        SingleSqlBuilder<OrgCourseGroupRelate> builder = createSqlBuilder();
        builder.max("displayOrder");
        builder.eq("courseGroupId", courseGroupId);
        builder.eq("isDeleted", DeleteStatus.NORMAL.getValue());
        builder.notin("courseId", excludeIds);
        Integer count = queryForObject(builder, Integer.class);

        return count == null ? 0 : count;
    }

    @Override
    public void batchAddWeike(Long groupId, Collection<Long> courseIds) {
        Preconditions.checkArgument(groupId != null, "groupid == null");
        logger.info("batchAddWeike:{}  {}", groupId, courseIds);
        if (CollectionUtils.isEmpty(courseIds)) {
            return;
        }

        for (Long courseId : courseIds) {

            OrgCourseGroupRelate relate =
                    this.getCourseGroupByCourseId(groupId.intValue(), Integer.valueOf(CourseType.ORG_COURSE.getCode()),
                            courseId);
            if (relate == null) {
                OrgCourseGroupRelate po = new OrgCourseGroupRelate();
                po.setCourseGroupId(groupId.intValue());
                po.setCourseId(courseId);
                po.setCourseType(CourseType.ORG_COURSE.getCode());
                po.setCreateTime(new Date());
                po.setIsDeleted(DeleteStatus.NORMAL.getValue());
                po.setDisplayOrder(0);
                this.save(po);
            }
        }

    }

    @Override
    public boolean isRecommendTeacher(long teacherId) {
        String sql = "select count(*) from yunying.org_recommend_teacher where teacher_id = :teacherId and is_deleted = :isDel";
        Map map = Maps.newHashMap();
        map.put("teacherId", teacherId);
        map.put("isDel", DeleteStatus.NORMAL.getValue());
        int num =
                getNamedJdbcTemplate().queryForObject(sql, map, Integer.class);

        return num > 0;
    }

    @Override
    public Map<Long, Integer> isRecommendTeacher(Collection<Long> teacherIds) {
        if (teacherIds == null || teacherIds.size() == 0) {
            return Maps.newHashMap();
        }
        String sql = "select count(id) as cc,teacher_id as tid from yunying.org_recommend_teacher where teacher_id in (:teacherIds) and is_deleted = :isDel";
        Map map = Maps.newHashMap();
        map.put("teacherIds", teacherIds);
        map.put("isDel", DeleteStatus.NORMAL.getValue());
        return this.getNamedJdbcTemplate().query(sql, map, new ResultSetExtractor<Map<Long, Integer>>() {

            @Override
            public Map<Long, Integer> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<Long, Integer> map = new HashMap<>();
                while (rs.next()) {
                    Long tid = rs.getLong("tid");
                    Integer cc = rs.getInt("cc");
                    map.put(tid, cc);
                }
                return map;
            }
        });
    }

    @Override
    public List<OrgCourseGroupRelate> getCourseGroupRelateByGroupIds(List<Integer> ids, int code) {
        try{
            if (ids == null) {
                return null;
            }
            SingleSqlBuilder<OrgCourseGroupRelate> builder = createSqlBuilder();
            builder.in("courseGroupId", ids);
            builder.eq("courseType", code);
            builder.eq("isDeleted", DeleteStatus.NORMAL.getValue());
            builder.desc("displayOrder");
            List<OrgCourseGroupRelate> list = this.queryList(builder);
            return list;
        }catch (Exception e){
            logger.debug("000",e);
        }
        return Collections.emptyList();
    }

    @Override
    public List<OrgCourseGroupRelate> getCourseGroupRelateByCourseIds(List<Integer> ids, int courseType) {
        try{
            if (ids == null) {
                return null;
            }
            SingleSqlBuilder<OrgCourseGroupRelate> builder = createSqlBuilder();
            builder.in("course_id", ids);
            builder.eq("courseType", courseType);
            builder.eq("isDeleted", DeleteStatus.NORMAL.getValue());
            builder.desc("displayOrder");
            List<OrgCourseGroupRelate> list = this.queryList(builder);
            return list;
        }catch (Exception e){
            logger.debug("000",e);
        }
        return Collections.emptyList();
    }

    @Override
    public Map<Long, Set<Integer>> getCourseGroupMap(Collection<Long> courseIds) {
        final Map<Long, Set<Integer>> map = new HashMap<>();
        if (!CollectionUtils.isEmpty(courseIds)) {
            SingleSqlBuilder<OrgCourseGroupRelate> builder = createSqlBuilder();
            builder.in("courseId", courseIds);
            builder.eq("isDeleted", DeleteStatus.NORMAL.getValue());

            this.getNamedJdbcTemplate().query(builder.toSql(), builder.collectConditionValue(), new RowCallbackHandler() {
                @Override
                public void processRow(ResultSet rs) throws SQLException {
                    Long courseId = rs.getLong("courseId");
                    Integer groupId = rs.getInt("courseGroupId");
                    Set<Integer> set = map.get(courseId);
                    if (CollectionUtils.isEmpty(set)) {
                        set = new HashSet<Integer>();
                        map.put(courseId, set);
                    }
                    set.add(groupId);
                }
            });
        }
        return map;
    }
}
