package com.baijia.tianxiao.dal.activity.dao.draw.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.stereotype.Repository;

import com.baijia.tianxiao.dal.activity.dao.draw.ActivityUserDao;
import com.baijia.tianxiao.dal.activity.po.DairyCountStatistics;
import com.baijia.tianxiao.dal.activity.po.draw.ActivityUser;
import com.baijia.tianxiao.sqlbuilder.SingleSqlBuilder;
import com.baijia.tianxiao.sqlbuilder.support.JdbcTemplateDaoSupport;
import com.baijia.tianxiao.util.GenericsUtils;

import lombok.extern.slf4j.Slf4j;

/**
 * Created by liuxp on 16/1/26.
 */
@Repository
@Slf4j
public class ActivityUserDaoImpl extends JdbcTemplateDaoSupport<ActivityUser> implements ActivityUserDao {
    public ActivityUserDaoImpl() {
        super(ActivityUser.class);
    }

    @Override
    public Long insertActivityUser(ActivityUser activityUser) {
        save(activityUser);
        return activityUser.getId();
    }

    /**
     * 统计总参与人数
     *
     * @param activityIds
     * @return
     */
    @Override
    @Deprecated
    public Map<Long, Integer> selectTotalByActivityId(List<Long> activityIds) {
        String sql =
            "select activity_id,count(id) as count from yunying.tx_activity_user where activity_id in (:activityIds) group by activity_id";
        Map<String, Object> param = new HashMap<>();
        param.put("activityIds", activityIds);
        return getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<Map<Long, Integer>>() {

            @Override
            public Map<Long, Integer> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<Long, Integer> map = new HashMap<Long, Integer>();
                while (rs.next()) {
                    map.put(rs.getLong("activity_id"), rs.getInt("count"));
                }
                return map;
            }
        });
    }

    /**
     * 根据不同的活动类型统计总参与人数
     *
     * @param activityIds
     * @return
     */
    @Override
    public Map<Long, Integer> selectTotalByActivityIdAndCategory(List<Long> activityIds, Integer category) {
        String sql =
            "select activity_id,count(id) as count from yunying.tx_activity_user where activity_id in (:activityIds) and category =:category  group by activity_id";
        Map<String, Object> param = new HashMap<>();
        param.put("activityIds", activityIds);
        param.put("category", category);
        return getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<Map<Long, Integer>>() {

            @Override
            public Map<Long, Integer> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<Long, Integer> map = new HashMap<Long, Integer>();
                while (rs.next()) {
                    map.put(rs.getLong("activity_id"), rs.getInt("count"));
                }
                return map;
            }
        });
    }

    /**
     * 统计总参与人数
     *
     * @param activityIds
     * @return
     */
    @Override
    public Map<Long, Map<Long, Integer>> statisticalVotes(List<Long> activityIds, Integer category) {
        String sql =
            "select activity_id,options_id,count(*) as count from yunying.tx_activity_user where activity_id in (:activityIds) and category =:category  group by activity_id , options_id";
        Map<String, Object> param = new HashMap<>();
        param.put("activityIds", activityIds);
        param.put("category", category);
        return getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<Map<Long, Map<Long, Integer>>>() {

            @Override
            public Map<Long, Map<Long, Integer>> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<Long, Map<Long, Integer>> map = new HashMap<Long, Map<Long, Integer>>();
                while (rs.next()) {
                    Long activityId = rs.getLong("activity_id");
                    Map<Long, Integer> voteResults = map.get(activityId);
                    if (voteResults == null) {
                        voteResults = new HashMap<>();
                        map.put(activityId, voteResults);
                    }
                    voteResults.put(rs.getLong("options_id"), rs.getInt("count"));
                }
                return map;
            }
        });
    }

    /**
     * 根据微信号以及参与到的活动类型来查找
     */
    @Override
    public ActivityUser selectActivityUser(String wechatOpenId, String date, long activityId, int category) {
        SingleSqlBuilder<ActivityUser> sb = createSqlBuilder();
        sb.eq("wechatOpenId", wechatOpenId);
        if (GenericsUtils.notNullAndEmpty(date)) {
            sb.eq("done_date", date);
        }
        sb.eq("activityId", activityId);
        sb.eq("category", category);

        List<ActivityUser> list = queryList(sb);
        if (list != null && list.size() > 0) {
            return list.get(0);
        }
        return null;
    }

    /**
     * 获取某个微信号对某个活动所有的投票记录
     * 
     * @param wechateOpenId
     * @param date
     * @param activityId
     * @param category
     * @return
     */
    @Override
    public List<ActivityUser> selectActivityUsers(String wechateOpenId, String date, Long activityId,
        Integer category) {
        SingleSqlBuilder<ActivityUser> builder = createSqlBuilder();
        builder.eq("wechatOpenId", wechateOpenId);
        builder.eq("activityId", activityId);
        if (GenericsUtils.notNullAndEmpty(date)) {
            builder.eq("done_date", date);
        }
        builder.eq("category", category);

        return queryList(builder);
    }

    @Override
    public void updateActivityUserDoneCount(ActivityUser user) {
        update(user, false);
    }

    /**
     * 原始的数据统计方法
     */
    @Override
    public List<DairyCountStatistics> selectDairyCountStatistics(long activityId, int category) {
        String sql =
            "select done_date,count(id) as count from yunying.tx_activity_user where activity_id=:activityId and category=:category group by done_date order by done_date desc";
        Map<String, Object> param = new HashMap<>();
        param.put("activityId", activityId);
        param.put("category", category);
        return getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<List<DairyCountStatistics>>() {

            @Override
            public List<DairyCountStatistics> extractData(ResultSet rs) throws SQLException, DataAccessException {
                List<DairyCountStatistics> list = new ArrayList<>();
                while (rs.next()) {
                    DairyCountStatistics statistics = new DairyCountStatistics();
                    statistics.setDate(rs.getString("done_date"));
                    statistics.setCount(rs.getInt("count"));
                    list.add(statistics);
                }
                return list;
            }
        });
    }

    /**
     * 原始的数据统计方法
     */
    @Override
    public List<DairyCountStatistics> selectDairyCountStatisticsWithSingle(long activityId, int category) {
        String sql =
            "select done_date ,count(*) as count from (select done_date,wechat_open_id  from yunying.tx_activity_user where activity_id=:activityId and category=:category group by done_date,wechat_open_id ) as a group by done_date order by done_date desc";
        Map<String, Object> param = new HashMap<>();
        param.put("activityId", activityId);
        param.put("category", category);
        return getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<List<DairyCountStatistics>>() {

            @Override
            public List<DairyCountStatistics> extractData(ResultSet rs) throws SQLException, DataAccessException {
                List<DairyCountStatistics> list = new ArrayList<>();
                while (rs.next()) {
                    DairyCountStatistics statistics = new DairyCountStatistics();
                    statistics.setDate(rs.getString("done_date"));
                    statistics.setCount(rs.getInt("count"));
                    list.add(statistics);
                }
                return list;
            }
        });
    }

    @Override
    public Integer getUserTotal(List<Long> activityIds, Integer category, Date startDate, Date endDate) {
        SingleSqlBuilder<ActivityUser> builder = createSqlBuilder();
        builder.in("activityId", activityIds);
        builder.eq("category", category);
        if (startDate != null) {
            builder.between("createTime", startDate, endDate);
        }
        builder.count("id");
        return queryForObject(builder, Integer.class);
    }

    @Override
    public Map<String, Integer> getUserTotalMap(List<Long> activityIds, Integer category, Date startTime,
        Date endTime) {
        String sql =
            "select count(id) as count, DATE_FORMAT(create_time,'%y-%m-%d') createTime from yunying.tx_activity_user where "
                + "activity_id in (:activityIds) and category=:category ";
        Map<String, Object> param = new HashMap<>();
        param.put("activityIds", activityIds);
        param.put("category", category);

        if (startTime != null && endTime != null) {
            sql += " and create_time between :startTime and :endTime ";
            param.put("endTime", endTime);
            param.put("startTime", startTime);
        }
        sql += "group by MONTH(create_time), DAY(create_time)";
        return getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<Map<String, Integer>>() {

            @Override
            public Map<String, Integer> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<String, Integer> map = new HashMap<String, Integer>();
                while (rs.next()) {
                    String accessDate = rs.getString("createTime");
                    Integer count = rs.getInt("count");
                    map.put(accessDate, count);
                }
                return map;
            }
        });

    }

    @Override
    public List<ActivityUser> selectAllUsers(long activityId, int category) {
        SingleSqlBuilder<ActivityUser> builder = createSqlBuilder();
        builder.eq("activityId", activityId);
        builder.eq("category", category);
        return queryList(builder);
    }

    @Override
    public String searchFillUserInfo(String wechatOpenId, long activityId, int category) {
        SingleSqlBuilder<ActivityUser> builder = createSqlBuilder();
        builder.eq("activityId", activityId);
        builder.eq("category", category);
        builder.eq("wechatOpenId", wechatOpenId);
        builder.ne("userInfo", "");
        List<ActivityUser> queryList = this.queryList(builder);
        log.info("sql is : {} and result is : {} ", builder.toSql(), queryList.size());
        if (GenericsUtils.notNullAndEmpty(queryList)) {
            return queryList.get(0).getUserInfo();
        }
        return null;
    }

}
