package com.baijia.tianxiao.dal.org.dao.impl;


import com.baijia.tianxiao.dal.org.constant.DeleteStatus;
import com.baijia.tianxiao.dal.org.constant.TXCascadeAccountStatus;
import com.baijia.tianxiao.dal.org.dao.TXCascadeAccountDao;
import com.baijia.tianxiao.dal.org.po.TXCascadeAccount;
import com.baijia.tianxiao.sqlbuilder.SingleSqlBuilder;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.sqlbuilder.support.JdbcTemplateDaoSupport;
import com.baijia.tianxiao.util.GenericsUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.stereotype.Repository;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.*;

/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */

/**
 * @author caoliang
 * @version 1.0
 * @title TXCascadeAccountDaoImpl
 * @desc TODO
 * @date 2016年5月10日
 */
@Slf4j
@Repository
public class TxCascadeAccountDaoImpl extends JdbcTemplateDaoSupport<TXCascadeAccount> implements TXCascadeAccountDao {

    /**
     * @param
     */
    public TxCascadeAccountDaoImpl() {
        super(TXCascadeAccount.class);
    }

    @Override
    public Integer getCountByOrgId(Integer orgId) {
        SingleSqlBuilder<TXCascadeAccount> builder = createSqlBuilder();
        builder.count("id");
        builder.eq("orgId", orgId);
        builder.eq("isdel", DeleteStatus.NORMAL.getValue());
        // builder.eq("status", TXCascadeAccountStatus.VALID.getCode());
        return queryForObject(builder, Integer.class);

    }

    @Override
    public TXCascadeAccount getByIdAndOrgId(Integer id, Integer orgId) {
        SingleSqlBuilder<TXCascadeAccount> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.eq("id", id);
        builder.eq("isdel", DeleteStatus.NORMAL.getValue());
        builder.eq("status", TXCascadeAccountStatus.VALID.getCode());
        return uniqueResult(builder);

    }

    @Override
    public List<TXCascadeAccount> getByOrgIds(List<Integer> orgIds, PageDto pageDto) {
        String sql = "select id,credential_id,account_type,org_id,title,status,isdel,update_time,create_time from yunying.tx_cascade_account "
                + "where org_id in ( :orgIds ) and isdel=0 order by create_time desc ";
        Map<String, Object> paramMap = Maps.newHashMap();
        if (CollectionUtils.isNotEmpty(orgIds)) {
            paramMap.put("orgIds", orgIds);
        } else {
            return Lists.newArrayList();
        }
        if (null != pageDto) {
            Integer pageSize = pageDto.getPageSize();
            Integer curPageCount = pageDto.getCurPageCount();
            Integer pageNum = pageDto.getPageNum();
            if (null != pageSize) {
                sql = sql + " limit :pageSize ";
                paramMap.put("pageSize", pageSize);
            }
            if (null != curPageCount) {
                sql = sql + " offset :curPageCount ";
                paramMap.put("curPageCount", curPageCount);
            } else if (null != pageNum) {
                sql = sql + " offset :offSet ";
                paramMap.put("offSet", (pageNum - 1) * pageSize);
            }
        }
        log.info("getByOrgIds input param is orgIds:{},pageDto:{},sql:{}", orgIds, pageDto, sql);
        List<TXCascadeAccount> accounts = this.getNamedJdbcTemplate().query(sql, paramMap, new RowMapper<TXCascadeAccount>() {
            @Override
            public TXCascadeAccount mapRow(ResultSet rs, int rowNum) throws SQLException {
                TXCascadeAccount txa = new TXCascadeAccount();
                txa.setId(rs.getInt("id"));
                txa.setCredentialId(rs.getInt("credential_id"));
                txa.setAccountType(rs.getInt("account_type"));
                txa.setOrgId(rs.getInt("org_id"));
                txa.setTitle(rs.getString("title"));
                txa.setStatus(rs.getInt("status"));
                txa.setIsdel(rs.getInt("isdel"));
                txa.setUpdateTime(rs.getDate("update_time"));
                txa.setCreateTime(rs.getDate("create_time"));
                return txa;
            }
        });
        if (null != pageDto) {
            pageDto.setCurPageCount(accounts.size());
        }
        log.debug("the length of txCascadeAccount result is:{}", accounts.size());
        return accounts;
    }

    @Override
    public Integer getCountByOrgIds(List<Integer> orgIds) {
        SingleSqlBuilder<TXCascadeAccount> builder = createSqlBuilder();
        builder.count("id");
        builder.in("orgId", orgIds);
        builder.eq("isdel", DeleteStatus.NORMAL.getValue());
        builder.eq("status", TXCascadeAccountStatus.VALID.getCode());
        return queryForObject(builder, Integer.class);
    }

    @Override
    public Integer getCountByOrgIds(List<Integer> orgIds, Integer status) {
        SingleSqlBuilder<TXCascadeAccount> builder = createSqlBuilder();
        builder.count("id");
        builder.in("orgId", orgIds);
        builder.eq("isdel", DeleteStatus.NORMAL.getValue());
        if (status != null) {
            builder.eq("status", status);
        }
        return queryForObject(builder, Integer.class);
    }

    @Override
    public List<TXCascadeAccount> getByCIdAndOrgId(Integer orgId, Integer credentialId) {
        SingleSqlBuilder<TXCascadeAccount> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.eq("credentialId", credentialId);
        builder.eq("isdel", DeleteStatus.NORMAL.getValue());
        return queryList(builder);

    }

    @Override
    public List<TXCascadeAccount> getTXCascadeAccountListByOrgId(Long orgId) {
        SingleSqlBuilder<TXCascadeAccount> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.eq("status", TXCascadeAccountStatus.VALID.getCode());
        builder.eq("isdel", DeleteStatus.NORMAL.getValue());
        return queryList(builder);
    }

    @Override
    public List<Integer> listIdByAccountType(Integer orgId, Integer accountType) {
        StringBuilder builder = new StringBuilder();
        builder.append("SELECT id FROM yunying.tx_cascade_account WHERE isdel=0");
        if (orgId != null) {
            builder.append(" AND org_id=").append(orgId);
        }
        if (accountType != null) {
            builder.append(" AND account_type=").append(accountType);
        }
        // String sql = "SELECT id FROM yunying.tx_cascade_account WHERE org_id=" + orgId;
        return this.getNamedJdbcTemplate().query(builder.toString(), new ResultSetExtractor<List<Integer>>() {

            @Override
            public List<Integer> extractData(ResultSet rs) throws SQLException, DataAccessException {
                List<Integer> appIds = new ArrayList<>();
                while (rs.next()) {
                    Integer appId = rs.getInt("id");
                    appIds.add(appId);
                }
                return appIds;
            }
        });
    }

    @Override
    public TXCascadeAccount getByCredentialIdAndOrgId(Integer orgId, Integer credentialId) {
        SingleSqlBuilder<TXCascadeAccount> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.eq("credentialId", credentialId);
        builder.eq("isdel", DeleteStatus.NORMAL.getValue());
        builder.eq("status", TXCascadeAccountStatus.VALID.getCode());
        List<TXCascadeAccount> list = queryList(builder);
        if (list != null && list.size() > 0) {
            return list.get(0);
        } else {
            return null;
        }
    }

    @Override
    public List<TXCascadeAccount> getTXCascadeAccountListByOrgId(Long orgId, Integer status) {
        SingleSqlBuilder<TXCascadeAccount> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.eq("isdel", DeleteStatus.NORMAL.getValue());
        if (status != null) {
            builder.eq("status", status);
        }
        return queryList(builder);
    }

    @Override
    public List<TXCascadeAccount> getTXCascadeAccountListByStatusAndOrgId(Integer cascadeId, int status, Integer orgId) {
        SingleSqlBuilder<TXCascadeAccount> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        if (GenericsUtils.notNullAndEmpty(cascadeId)) {
            builder.ne("id", cascadeId);
        }
        builder.eq("status", status);
        builder.eq("isdel", DeleteStatus.NORMAL.getValue());
        return queryList(builder);
    }

    @Override
    public List<TXCascadeAccount> getTXCascadeAccountListByOrgIds(Collection<Integer> orgIds) {
        SingleSqlBuilder<TXCascadeAccount> builder = createSqlBuilder();
        builder.in("orgId", orgIds);
        builder.eq("isdel", DeleteStatus.NORMAL.getValue());
        return queryList(builder);
    }

    @Override
    public List<TXCascadeAccount> getByCredentialId(Integer credentialId) {
        SingleSqlBuilder<TXCascadeAccount> builder = createSqlBuilder();
        builder.eq("credentialId", credentialId);
        builder.eq("isdel", DeleteStatus.NORMAL.getValue());
        return queryList(builder);

    }

    @Override
    public List<TXCascadeAccount> listIdByOrgId(Long orgId, Integer cascadeId) {
        SingleSqlBuilder<TXCascadeAccount> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        if (cascadeId != null) {
            builder.ne("credentialId", cascadeId);
        }
        builder.eq("isdel", DeleteStatus.NORMAL.getValue());//未删除
        builder.eq("status", TXCascadeAccountStatus.VALID.getCode());//正常状态
//		builder.eq("accountType", 4);//普通员工
        return queryList(builder);
    }

    @Override
    public List<TXCascadeAccount> getValidAccountsByCredentialId(Integer credentialId) {
        SingleSqlBuilder<TXCascadeAccount> builder = createSqlBuilder();
        builder.eq("credentialId", credentialId);
        builder.eq("isdel", DeleteStatus.NORMAL.getValue());
        builder.eq("status", TXCascadeAccountStatus.VALID.getCode());
        return queryList(builder);
    }

    @Override
    public List<TXCascadeAccount> getAllByPageDto(PageDto pageDto) {
        SingleSqlBuilder<TXCascadeAccount> builder = createSqlBuilder();
//        builder.eq("isdel", DeleteStatus.NORMAL.getValue()); //不能排除删掉的帐号,否则无法踢出
        builder.setPage(pageDto);
        return queryList(builder);

    }

    @Override
    public void incrVersionByOrgId(int orgId) {
        String sql = "update yunying.tx_cascade_account set account_version=account_version+1, update_time=:updateTime where org_id=:orgId";
        Map<String, Object> param = new HashMap<>();
        param.put("orgId", orgId);
        param.put("updateTime", new Date());
        getNamedJdbcTemplate().update(sql, param);
    }

    @Override
    public void incrVersionByCredentialId(int credentialId) {
        String sql = "update yunying.tx_cascade_account set account_version=account_version+1, update_time=:updateTime where credential_id=:credentialId";
        Map<String, Object> param = new HashMap<>();
        param.put("credentialId", credentialId);
        param.put("updateTime", new Date());
        getNamedJdbcTemplate().update(sql, param);
    }

    @Override
    public void deleteById(int cascadeId) {
        String sql = "update yunying.tx_cascade_account set isdel=1, account_version=account_version+1, update_time=:updateTime where id=:cascadeId";
        Map<String, Object> param = new HashMap<>();
        param.put("cascadeId", cascadeId);
        param.put("updateTime", new Date());
        getNamedJdbcTemplate().update(sql, param);
    }

    @Override
    public void deleteByOrgId(int orgId) {
        String sql = "update yunying.tx_cascade_account set isdel=1, account_version=account_version+1, update_time=:updateTime where org_id=:orgId";
        Map<String, Object> param = new HashMap<>();
        param.put("orgId", orgId);
        param.put("updateTime", new Date());
        getNamedJdbcTemplate().update(sql, param);
    }
}