/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */
package com.baijia.tianxiao.dal.org.dao.impl;

import com.baijia.tianxiao.dal.org.constant.DeleteStatus;
import com.baijia.tianxiao.dal.org.dao.OrgStudentDao;
import com.baijia.tianxiao.dal.org.po.OrgStudent;
import com.baijia.tianxiao.dto.query.AutoMatchQueryRequest;
import com.baijia.tianxiao.sqlbuilder.SingleSqlBuilder;
import com.baijia.tianxiao.sqlbuilder.bean.Expression;
import com.baijia.tianxiao.sqlbuilder.bean.impl.MatchMode;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.sqlbuilder.support.JdbcTemplateDaoSupport;
import com.baijia.tianxiao.sqlbuilder.util.Expressions;
import com.baijia.tianxiao.util.AutoMatchUtils;
import com.baijia.tianxiao.util.GenericsUtils;
import com.baijia.tianxiao.util.ListUtil;
import com.baijia.tianxiao.util.collection.CollectorUtil;
import com.baijia.tianxiao.util.mobile.MaskUtil;
import com.baijia.tianxiao.util.query.BatchQueryCallback;
import com.baijia.tianxiao.util.query.BatchQueryTemplate;
import com.baijia.tianxiao.util.query.ListBatchQueryTemplate;
import com.baijia.tianxiao.util.query.MapBatchQueryTemplate;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.core.RowCallbackHandler;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.stereotype.Repository;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.*;

import lombok.extern.slf4j.Slf4j;


@Slf4j
@Repository
public class OrgStudentDaoImpl extends JdbcTemplateDaoSupport<OrgStudent> implements OrgStudentDao {

    public OrgStudentDaoImpl() {
        super(OrgStudent.class);
    }

    @Override
    public OrgStudent getStudent(Long orgId, Long userId, Integer delStatus, String...queryProps) {
        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder(queryProps);
        builder.eq("orgId", orgId);
        builder.eq("userId", userId);
        if (delStatus != null && delStatus >= 0) {
            builder.eq("delStatus", delStatus);
        } else {
            builder.eq("delStatus", DeleteStatus.NORMAL.getValue());
        }
        return uniqueResult(builder);
    }

    @Override
    public OrgStudent getStudentByUserId(Long orgId, Long userId, String...queryProps) {
        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder(queryProps);
        builder.eq("orgId", orgId);
        builder.eq("userId", userId);
        List<OrgStudent> students = queryList(builder);
        if (students != null && students.size() > 0) {
            return students.get(0);
        }
        return null;
    }

    @Override
    public List<OrgStudent> getStudentsByPage(PageDto pageDto) {
        String sql = "select id,org_id,mobile,student_name from yunying.org_students limit :start,:size";
        Map<String, Object> map = new HashMap<>();
        map.put("start", (pageDto.getPageNum() - 1) * pageDto.getPageSize());
        map.put("size", pageDto.getPageSize());
        return getNamedJdbcTemplate().query(sql, map, new RowMapper<OrgStudent>() {
            @Override
            public OrgStudent mapRow(ResultSet rs, int rowNum) throws SQLException {
                OrgStudent student = new OrgStudent();
                student.setId(rs.getLong("id"));
                student.setMobile(rs.getString("mobile"));
                student.setOrgId(rs.getLong("org_id"));
                student.setName(rs.getString("student_name"));
                return student;
            }
        });
    }

    @Override
    public List<OrgStudent> getStudentId(Long orgId, String mobile, Integer delStatus, String name, String...propes) {
        Preconditions.checkArgument(StringUtils.isNoneBlank(name), "学生姓名不能为空");
        Preconditions.checkArgument(StringUtils.isNoneBlank(mobile), "学生手机号码不能为空");
        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder(propes);
        builder.eq("orgId", orgId);
        if (StringUtils.isNotEmpty(name)) {
            builder.eq("name", name);
        }
        if (StringUtils.isNotEmpty(mobile)) {
            Expression exp = Expressions.eq("mobile", mobile);
            exp = Expressions.or(exp, Expressions.eq("showMobile", mobile));
            builder.add(exp);
        }
        if (delStatus != null) {
            builder.eq("delStatus", delStatus);
        }
        builder.desc("id");
        return queryList(builder);
    }

    @Override
    public List<OrgStudent> getStudents(Long orgId, String searchKey, Integer delStatus, PageDto page,
        String...queryProps) {
        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder(queryProps);
        builder.eq("orgId", orgId);
        if (StringUtils.isNoneBlank(searchKey)) {
            builder.add(Expressions.or(Expressions.like("name", searchKey, MatchMode.ANYWHERE),
                Expressions.like("nickName", searchKey, MatchMode.ANYWHERE),
                Expressions.like("pinyin", searchKey, MatchMode.ANYWHERE),
                // Expressions.like("showMobile", searchKey, MatchMode.ANYWHERE),
                Expressions.like("mobile", searchKey, MatchMode.ANYWHERE)));
        }
        if (delStatus != null) {
            builder.eq("delStatus", delStatus);
        }
        builder.setPage(page);
        builder.asc("name", true);
        // builder.desc("createTime");
        return queryList(builder);
    }

    @Override
    public List<OrgStudent> getStudents(Long orgId, List<Long> userIds, String searchKey, Integer delStatus,
        PageDto page, String...queryProps) {
        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder(queryProps);
        builder.eq("orgId", orgId);
        if (StringUtils.isNoneBlank(searchKey)) {
            builder.add(Expressions.or(Expressions.like("name", searchKey, MatchMode.ANYWHERE),
                Expressions.like("nickName", searchKey, MatchMode.ANYWHERE),
                Expressions.like("pinyin", searchKey, MatchMode.ANYWHERE),
                // Expressions.like("showMobile", searchKey, MatchMode.ANYWHERE),
                Expressions.like("mobile", searchKey, MatchMode.ANYWHERE)));
        }
        if (delStatus != null) {
            builder.eq("delStatus", delStatus);
        }
        if (CollectionUtils.isNotEmpty(userIds)) {
            builder.in("userId", userIds);
        }
        builder.setPage(page);
        builder.asc("name", true);
        // builder.desc("createTime");
        return queryList(builder);
    }

    @Override
    public List<OrgStudent> getStudentByOrgIdAndCourseName(Long orgId, String searchKey) {
        SingleSqlBuilder<OrgStudent> builder = this.createSqlBuilder();
        builder.eq("orgId", orgId);
        if (StringUtils.isNotBlank(searchKey)) {
            builder.like("name", searchKey, MatchMode.ANYWHERE);
        }
        // builder.desc("createTime");
        List<OrgStudent> list = this.queryList(builder);
        log.debug("list = {}", list);
        return queryList(builder);

    }

    @Override
    public List<OrgStudent> getStudentList(Long orgId, String searchKey, Integer delStatus, PageDto page,
        String...queryProps) {
        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder(queryProps);
        builder.eq("orgId", orgId);
        if (StringUtils.isNoneBlank(searchKey)) {
            builder.add(Expressions.or(Expressions.like("name", searchKey, MatchMode.ANYWHERE),
                Expressions.like("nickName", searchKey, MatchMode.ANYWHERE),
                Expressions.like("pinyin", searchKey, MatchMode.ANYWHERE),
                Expressions.like("mobile", searchKey, MatchMode.ANYWHERE)));
        }
        if (delStatus != null) {
            builder.eq("delStatus", delStatus);
        }
        builder.setPage(page);
        builder.asc("pinyin");
        return queryList(builder);
    }

    /**
     * @Warning 如果后期这个学员的课程状态进行了添加或者变更，这里需要进行同步修改
     */
    public static final List<Integer> emptyLessonStatus = Arrays.asList(1, 2, 3);

    /**
     * 
     * 根据课程的状态(在读,往期,待续费) 以及机构信息,学员id来查找某些学员
     * 
     */
    @Override
    public List<OrgStudent> getStudentList(int orgId, List<Integer> lessonStatus, Collection<Long> addCascadeIds,
        Collection<Long> studentIds, boolean asc, final PageDto dto) {

        Map<String, Object> params = Maps.newHashMap();
        params.put("orgId", orgId);
        if (GenericsUtils.isNullOrEmpty(lessonStatus)) {
            lessonStatus = emptyLessonStatus;
        }
        String inStudentIdsSql = "";
        if (GenericsUtils.notNullAndEmpty(studentIds)) {
            inStudentIdsSql = " and user_id in (:userIds) ";
            params.put("userIds", studentIds);
        }
        String inAddCascadeIdsSql = "";
        if (GenericsUtils.notNullAndEmpty(addCascadeIds)) {
            inAddCascadeIdsSql = " and add_cascade_id in (:addCascadeIds) ";
            params.put("add_cascade_id", addCascadeIds);
        }

        params.put("lessonStatus", lessonStatus);
        int pageIndex = (dto.getPageNum() - 1) * dto.getPageSize();
        params.put("pageIndex", pageIndex);
        params.put("pageSize", dto.getPageSize());
        String sort = asc ? "asc" : "desc";

        String fromTable =
            "(select id,student_name,lesson_status, mobile ,weixin,user_id,pinyin,avatar  from yunying.org_students "
                + "where org_id=:orgId AND del_status =  0  AND lesson_status IN (:lessonStatus) " + inStudentIdsSql
                + inAddCascadeIdsSql + " and  pinyin >= 'A' and pinyin < '{')  " + " union all "
                + " (select id,student_name, lesson_status,mobile ,weixin,user_id, CONCAT('~',pinyin) as pinyin,avatar from yunying.org_students "
                + "where org_id=:orgId AND del_status =  0  AND lesson_status IN (:lessonStatus) " + inStudentIdsSql
                + inAddCascadeIdsSql + " and  ( pinyin < 'A' or pinyin >= '{'))  order by pinyin " + sort
                + " limit :pageIndex,:pageSize";

        String countSql = "select count(*) as count from yunying.org_students where org_id=:orgId AND del_status =  0 "
            + inStudentIdsSql + inAddCascadeIdsSql + " AND lesson_status IN (:lessonStatus) ";

        final List<OrgStudent> orgStudents = Lists.newArrayList();
        this.getNamedJdbcTemplate().query(fromTable, params, new ResultSetExtractor<OrgStudent>() {
            @Override
            public OrgStudent extractData(ResultSet rs) throws SQLException, DataAccessException {
                while (rs.next()) {
                    OrgStudent os = new OrgStudent();
                    os.setId(rs.getLong("id"));
                    os.setName(rs.getString("student_name"));
                    os.setMobile(rs.getString("mobile"));
                    os.setWeixin(rs.getString("weixin"));
                    os.setLessonStatus(rs.getInt("lesson_status"));
                    os.setUserId(rs.getLong("user_id"));
                    os.setPinyin(rs.getString("pinyin"));
                    os.setAvatar(rs.getLong("avatar"));
                    orgStudents.add(os);
                }
                return null;
            }
        });

        Integer totalCount = this.getNamedJdbcTemplate().query(countSql, params, new ResultSetExtractor<Integer>() {
            @Override
            public Integer extractData(ResultSet rs) throws SQLException, DataAccessException {
                if (rs.next()) {
                    return rs.getInt("count");
                }
                return 0;
            }
        });
        dto.setCount(totalCount);

        return orgStudents;
    }

    @Override
    public List<OrgStudent> getStudentsNotInUserIds(Long orgId, Collection<Long> usrIds, String searchKey,
        Integer delStatus, PageDto page, String...queryProps) {

        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder(queryProps);
        builder.eq("orgId", orgId);
        if (StringUtils.isNotEmpty(searchKey)) {
            builder.add(Expressions.or(Expressions.like("name", searchKey, MatchMode.ANYWHERE),
                Expressions.like("nickName", searchKey, MatchMode.ANYWHERE),
                Expressions.like("pinyin", searchKey, MatchMode.ANYWHERE),
                // Expressions.like("showMobile", searchKey, MatchMode.ANYWHERE),
                Expressions.like("mobile", searchKey, MatchMode.ANYWHERE)));
        }

        if (CollectionUtils.isNotEmpty(usrIds)) {
            builder.notin("userId", usrIds);
        }

        if (delStatus != null) {
            builder.eq("delStatus", delStatus);
        }
        builder.setPage(page);
        builder.asc("pinyin", true);
        // builder.desc("createTime");
        return queryList(builder);

    }

    @Override
    public List<OrgStudent> getStudentsUserIdsSortedByPinyin(Long orgId, String searchKey, Integer delStatus,
        PageDto page) {
        Map<String, Object> params = Maps.newHashMap();
        StringBuilder sql = new StringBuilder(
            " select *,if(pinyin < 'A' || pinyin >= '{','~',pinyin) as py from yunying.org_students where 1=1 ");
        sql.append(" and org_id=:orgId");
        params.put("orgId", orgId);
        if (delStatus != null) {
            sql.append(" and del_status=:delStatus");
            params.put("delStatus", delStatus);
        }
        if (StringUtils.isNotEmpty(searchKey)) {
            sql.append(" and ( ");
            sql.append(" student_name like :searchKey");
            sql.append(" or nick_name like :searchKey");
            sql.append(" or pinyin like :searchKey");
            sql.append(" or mobile like :searchKey)");
            params.put("searchKey", "%" + searchKey + "%");
        }
        int pageIndex = (page.getPageNum() - 1) * page.getPageSize();
        sql.append(" order by py ");
        String countSql = sql.toString();
        sql.append("limit :pageIndex,:pageSize ;");
        params.put("pageIndex", pageIndex);
        params.put("pageSize", page.getPageSize());
        log.info("getStudent SortedByPinyin sql:{}", sql);
        List<OrgStudent> result = this.getNamedJdbcTemplate().query(sql.toString(), params, new RowMapper<OrgStudent>() {
            @Override
            public OrgStudent mapRow(ResultSet rs, int rowNum) throws SQLException {
                OrgStudent os = new OrgStudent();
                os.setId(rs.getLong("id"));// 卧槽 太他妈长了 累死了
                os.setName(rs.getString("student_name"));
                os.setNickName(rs.getString("nick_name"));
                os.setParentName(rs.getString("parent_name"));
                os.setParentMobile(rs.getString("parent_mobile"));
                os.setSchool(rs.getString("school"));
                os.setRollNumber(rs.getString("roll_number"));
                os.setMobile(rs.getString("mobile"));
                os.setShowMobile(rs.getString("show_mobile"));
                os.setBranchId(rs.getLong("branch_id"));
                os.setCreateTime(rs.getDate("create_time"));
                os.setUpdateTime(rs.getDate("update_time"));
                os.setDelStatus(rs.getInt("del_status"));
                os.setOrgId(rs.getLong("org_id"));
                os.setAvatar(rs.getLong("avatar"));
                os.setUserId(rs.getLong("user_id"));
                os.setSource(rs.getInt("source"));
                os.setBirthday(rs.getDate("birthday"));
                os.setRemark(rs.getString("remark"));
                os.setWeixin(rs.getString("weixin"));
                os.setNextRemindTime(rs.getDate("next_remind_time"));
                os.setQq(rs.getString("qq"));
                os.setMail(rs.getString("mail"));
                os.setDegreeClass(rs.getString("degree_class"));
                os.setAddress(rs.getString("address"));
                os.setFatherOccupation(rs.getString("father_occupation"));
                os.setMatherOccupation(rs.getString("mather_occupation"));
                os.setAdvisoryStatus(rs.getInt("advisory_status"));
                os.setOrigin(rs.getInt("origin"));
                os.setPinyin(rs.getString("pinyin"));
                os.setLatitude(rs.getDouble("latitude"));
                os.setLongitude(rs.getDouble("longitude"));
                os.setAreaId(rs.getLong("area_id"));
                os.setGender(rs.getInt("gender"));
                os.setRelationship(rs.getInt("relationship"));
                os.setLessonStatus(rs.getInt("lesson_status"));
                os.setAddCascadeId(rs.getInt("add_cascade_id"));
                return os;
            }
        });
        page.setCount(this.getNamedJdbcTemplate().query(countSql.toString().replace("*", "count(*) as count"), params,
            new ResultSetExtractor<Integer>() {
                @Override
                public Integer extractData(ResultSet rs) throws SQLException, DataAccessException {
                    rs.next();
                    return rs.getInt("count");
                }
            }));
        page.setCurPageCount(result.size());
        return result;
    }

    public static void main(String[] args) {
        OrgStudentDaoImpl impl = new OrgStudentDaoImpl();
        PageDto page = new PageDto();
        page.setPageNum(2);
        page.setPageSize(20);
        impl.getStudentsUserIdsSortedByPinyin(1234L, "waba", 1, page);
    }

    @Override
    public List<OrgStudent> getStudents(final Long orgId, Collection<Long> userIds, final Integer delStatus,
        final PageDto page, final String...queryProps) {
        if (CollectionUtils.isEmpty(userIds)) {
            return Lists.newArrayList();
        }
        BatchQueryTemplate<Long, List<OrgStudent>> batchQueryTemplate = new ListBatchQueryTemplate<>();
        return batchQueryTemplate.batchQuery(userIds, new BatchQueryCallback<Long, List<OrgStudent>>() {

            @Override
            public List<OrgStudent> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<OrgStudent> builder = createSqlBuilder(queryProps);
                builder.eq("orgId", orgId);
                if (page != null) {
                    builder.setPage(page);
                }
                if (delStatus != null) {
                    builder.eq("delStatus", delStatus);
                }
                builder.in("userId", querySet);
                builder.asc("name", true);
                return queryList(builder);
            }
        });
    }

    @Override
    public List<OrgStudent> getStudents(Date createTime, Integer delStatus, PageDto page, String...queryProps) {
        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder(queryProps);
        builder.gt("createTime", createTime);
        builder.setPage(page);
        if (delStatus != null) {
            builder.eq("delStatus", delStatus);
        }
        builder.desc("createTime");
        return queryList(builder);
    }

    @Override
    public List<OrgStudent> getStudents(final Long orgId, Collection<String> mobiles, final Integer delStatus,
        final String...queryProps) {
        if (CollectionUtils.isEmpty(mobiles)) {
            return Lists.newArrayList();
        }
        BatchQueryTemplate<String, List<OrgStudent>> batchQueryTemplate = new ListBatchQueryTemplate<>();
        return batchQueryTemplate.batchQuery(mobiles, new BatchQueryCallback<String, List<OrgStudent>>() {

            @Override
            public List<OrgStudent> doQuery(Collection<String> querySet) {
                SingleSqlBuilder<OrgStudent> builder = createSqlBuilder(queryProps);
                builder.eq("orgId", orgId);
                if (delStatus != null) {
                    builder.eq("delStatus", delStatus);
                }
                builder.in("mobile", querySet);
                return queryList(builder);
            }
        });
    }

    @Override
    public OrgStudent getStudentByMobileAndOrgId(Long orgId, String mobile, String...queryProps) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(StringUtils.isNoneBlank(mobile), "mobile can not be empty!");
        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder(queryProps);
        builder.eq("mobile", mobile);
        builder.eq("orgId", orgId);
        builder.desc("createTime");

        builder.eq("delStatus", DeleteStatus.NORMAL.getValue());
        builder.setMaxSize(1);
        return uniqueResult(builder);

    }

    @Override
    public List<OrgStudent> getStudentByMobileAndOrgId(Long orgId, String mobile, Integer maxSize,
        String...queryProps) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(StringUtils.isNoneBlank(mobile), "mobile can not be empty!");
        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder(queryProps);
        builder.eq("mobile", mobile);
        builder.eq("orgId", orgId);
        builder.desc("createTime");

        builder.eq("delStatus", DeleteStatus.NORMAL.getValue());
        if (maxSize != null && maxSize > 0) {
            builder.setMaxSize(maxSize);
        }
        return queryList(builder);

    }

    @Override
    public List<OrgStudent> getStudentByOpenIdAndOrgId(Long orgId, String weixinOpenId, String...queryProps) {
        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder(queryProps);
        builder.eq("weixin", weixinOpenId);
        builder.eq("orgId", orgId);
        builder.desc("createTime");

        builder.eq("delStatus", DeleteStatus.NORMAL.getValue());
        return queryList(builder);
    }

    @Override
    public OrgStudent getLastStudentByMobileOrParentMobile(Long orgId, String mobile, String...queryProps) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(StringUtils.isNoneBlank(mobile), "mobile can not be empty!");
        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder(queryProps);

        Expression expression = Expressions.or(Expressions.eq("mobile", mobile),
            Expressions.and(Expressions.ne("mobile", mobile), Expressions.eq("parentMobile", mobile)));

        builder.add(expression);
        builder.eq("orgId", orgId);
        builder.desc("createTime");
        builder.setMaxSize(1);

        return uniqueResult(builder);
    }

    @Override
    public List<OrgStudent> getStudentByIds(final Long orgId, Collection<Long> studentIds, final String...propes) {

        if (studentIds.isEmpty()) {
            log.warn("student ids is empty.");
            return Lists.newArrayList();
        }

        BatchQueryTemplate<Long, List<OrgStudent>> queryTemplate = new ListBatchQueryTemplate<>();
        List<OrgStudent> students =
            queryTemplate.batchQuery(studentIds, new BatchQueryCallback<Long, List<OrgStudent>>() {
                @Override
                public List<OrgStudent> doQuery(Collection<Long> querySet) {
                    SingleSqlBuilder<OrgStudent> builder = createSqlBuilder(propes);
                    builder.in("id", querySet);
                    builder.eq("orgId", orgId);
                    return queryList(builder);
                }
            });

        return students;

    }

    @Override
    public List<OrgStudent> getStudentByUserIds(final Long orgId, Collection<Long> userIds, final String...propes) {
        if (userIds.isEmpty()) {
            log.warn("user ids is empty.");
            return Lists.newArrayList();
        }
        BatchQueryTemplate<Long, List<OrgStudent>> queryTemplate = new ListBatchQueryTemplate<>();
        List<OrgStudent> students = queryTemplate.batchQuery(userIds, new BatchQueryCallback<Long, List<OrgStudent>>() {
            @Override
            public List<OrgStudent> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<OrgStudent> builder = createSqlBuilder(propes);
                builder.in("userId", querySet);
                builder.eq("orgId", orgId);
                builder.eq("delStatus", DeleteStatus.NORMAL.getValue());
                // builder.asc("name");
                return queryList(builder);
            }
        });
        return students;
    }
    
    @Override
    public List<OrgStudent> getStudentByUserIdsAndDelStatus(final Long orgId, Collection<Long> userIds, 
        final Integer delStatus, final String...propes) {
        if (userIds.isEmpty()) {
            log.warn("user ids is empty.");
            return Lists.newArrayList();
        }
        BatchQueryTemplate<Long, List<OrgStudent>> queryTemplate = new ListBatchQueryTemplate<>();
        List<OrgStudent> students = queryTemplate.batchQuery(userIds, new BatchQueryCallback<Long, List<OrgStudent>>() {
            @Override
            public List<OrgStudent> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<OrgStudent> builder = createSqlBuilder(propes);
                builder.in("userId", querySet);
                builder.eq("orgId", orgId);
                if(null != delStatus){
                    builder.eq("delStatus", delStatus);
                }
                // builder.asc("name");
                return queryList(builder);
            }
        });
        return students;
    }

    @Override
    public List<OrgStudent> getPageStudentByUserIds(final Long orgId, Collection<Long> userIds,
        final Collection<Long> inStudentIds, final PageDto pageDto) {

        if (userIds.isEmpty()) {
            log.warn("user ids is empty.");
            return Lists.newArrayList();
        }

        BatchQueryTemplate<Long, List<OrgStudent>> queryTemplate = new ListBatchQueryTemplate<>();
        List<OrgStudent> students = queryTemplate.batchQuery(userIds, new BatchQueryCallback<Long, List<OrgStudent>>() {
            @Override
            public List<OrgStudent> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<OrgStudent> builder = createSqlBuilder();
                builder.in("userId", querySet);
                if (!inStudentIds.isEmpty()) {
                    builder.in("id", inStudentIds);
                }
                builder.eq("orgId", orgId);
                builder.setPage(pageDto);
                return queryList(builder);
            }
        });

        return students;

    }

    @Override
    public Long getUserId(Long studentId) {
        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder("userId");
        builder.eq("id", studentId);
        return this.queryForObject(builder, Long.class);

    }

    @Override
    public Long getStudentId(Long userId) {
        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder("id");
        builder.eq("userId", userId);
        return this.queryForObject(builder, Long.class);

    }

    @Override
    public Map<Long, Long> getStudentIdUserIdMap(Collection<Long> studentIds) {

        if (CollectionUtils.isEmpty(studentIds)) {
            return Collections.emptyMap();
        }
        return new MapBatchQueryTemplate<Long, Long, Long>().batchQuery(studentIds,
            new BatchQueryCallback<Long, Map<Long, Long>>() {
                @Override
                public Map<Long, Long> doQuery(Collection<Long> studentIds) {
                    SingleSqlBuilder<OrgStudent> builder = createSqlBuilder();
                    builder.in("id", studentIds);
                    final Map<Long, Long> result = Maps.newHashMap();
                    getNamedJdbcTemplate().query(builder.toSql(), builder.collectConditionValue(),
                        new RowCallbackHandler() {
                            @Override
                            public void processRow(ResultSet rs) throws SQLException {
                                result.put(rs.getLong("id"), rs.getLong("userId"));
                            }
                        });
                    return result;
                }
            });
    }

    @Override
    public Map<Long, Long> getStudentUserIdAndIdMap(Collection<Long> studentIds) {
        if (CollectionUtils.isEmpty(studentIds)) {
            return Collections.emptyMap();
        }
        return new MapBatchQueryTemplate<Long, Long, Long>().batchQuery(studentIds,
            new BatchQueryCallback<Long, Map<Long, Long>>() {
                @Override
                public Map<Long, Long> doQuery(Collection<Long> studentIds) {
                    SingleSqlBuilder<OrgStudent> builder = createSqlBuilder();
                    builder.in("id", studentIds);
                    builder.eq("delStatus", DeleteStatus.NORMAL.getValue());
                    final Map<Long, Long> result = Maps.newHashMap();
                    getNamedJdbcTemplate().query(builder.toSql(), builder.collectConditionValue(),
                        new RowCallbackHandler() {
                            @Override
                            public void processRow(ResultSet rs) throws SQLException {
                                result.put(rs.getLong("userId"), rs.getLong("id"));
                            }
                        });
                    return result;
                }
            });
    }

    @Override
    public Map<Long, Long> getUserIdStudentIdMap(Collection<Long> userIds, final Long orgId) {
        return this.getUserIdStudentIdMapWithFilter(userIds, orgId, false);
    }

    @Override
    public Map<Long, Long> getUserIdStudentIdMap(Collection<Long> userIds, final Long orgId, boolean needFilterDel) {
        return this.getUserIdStudentIdMapWithFilter(userIds, orgId, true);
    }

    private Map<Long, Long> getUserIdStudentIdMapWithFilter(Collection<Long> userIds, final Long orgId,
        final Boolean needFilterDel) {
        if (CollectionUtils.isEmpty(userIds)) {
            return Collections.emptyMap();
        }

        return new MapBatchQueryTemplate<Long, Long, Long>().batchQuery(userIds,
            new BatchQueryCallback<Long, Map<Long, Long>>() {
                @Override
                public Map<Long, Long> doQuery(Collection<Long> userIds) {
                    SingleSqlBuilder<OrgStudent> builder = createSqlBuilder();
                    builder.in("userId", userIds);
                    builder.eq("orgId", orgId);
                    if (needFilterDel) {
                        builder.eq("delStatus", 0);
                    }
                    final Map<Long, Long> result = Maps.newHashMap();
                    getNamedJdbcTemplate().query(builder.toSql(), builder.collectConditionValue(),
                        new RowCallbackHandler() {
                            @Override
                            public void processRow(ResultSet rs) throws SQLException {
                                result.put(rs.getLong("userId"), rs.getLong("id"));
                            }
                        });
                    return result;
                }
            });
    }

    @Override
    public Map<Long, String> getStudentNameMap(Collection<Long> studentIds) {
        if (CollectionUtils.isEmpty(studentIds)) {
            return Collections.emptyMap();
        }
        return new MapBatchQueryTemplate<Long, Long, String>().batchQuery(studentIds,
            new BatchQueryCallback<Long, Map<Long, String>>() {
                @Override
                public Map<Long, String> doQuery(Collection<Long> studentIds) {
                    long current = System.currentTimeMillis();
                    SingleSqlBuilder<OrgStudent> builder = createSqlBuilder("id", "name", "nickName", "mobile");
                    builder.in("id", studentIds);
                    final Map<Long, String> result = Maps.newHashMap();
                    getNamedJdbcTemplate().query(builder.toSql(), builder.collectConditionValue(),
                        new RowCallbackHandler() {
                            @Override
                            public void processRow(ResultSet rs) throws SQLException {
                                if (StringUtils.isBlank(rs.getString("name"))) {
                                    if (StringUtils.isNotBlank(rs.getString("nickName"))) {
                                        result.put(rs.getLong("id"), rs.getString("nickName"));
                                    } else {
                                        result.put(rs.getLong("id"), MaskUtil.maskMobile(rs.getString("mobile")));
                                    }
                                } else {
                                    result.put(rs.getLong("id"), rs.getString("name"));
                                }
                            }
                        });
                    log.debug("cost:{}ms", System.currentTimeMillis() - current);
                    return result;
                }
            });
    }

    @Override
    public Map<Long, OrgStudent> getStudentMap(Collection<Long> userIds, final Long orgId) {
        if (userIds.isEmpty()) {
            log.warn("user ids is empty.");
            return Maps.newHashMap();
        }

        BatchQueryTemplate<Long, List<OrgStudent>> queryTemplate = new ListBatchQueryTemplate<>();
        List<OrgStudent> students = queryTemplate.batchQuery(userIds, new BatchQueryCallback<Long, List<OrgStudent>>() {
            @Override
            public List<OrgStudent> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<OrgStudent> builder = createSqlBuilder();
                builder.eq("orgId", orgId);
                builder.in("userId", querySet);
                return queryList(builder);
            }
        });
        return CollectorUtil.collectMap(students, new Function<OrgStudent, Long>() {
            @Override
            public Long apply(OrgStudent arg0) {
                return arg0.getUserId();
            }
        });
    }

    @Override
    public List<OrgStudent> getUserByRemindTime(Date startTime) {
        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder("id", "orgId", "userId", "name", "nickName");
        builder.lt("nextRemindTime", new Date());
        builder.gt("nextRemindTime", startTime);
        return queryList(builder);
    }

    @Override
    public List<OrgStudent> getUserByRemindTime(Date startTime, Date endTime) {
        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder("id", "orgId", "userId", "name", "nickName");
        builder.lt("nextRemindTime", endTime);
        builder.ge("nextRemindTime", startTime);
        return queryList(builder);
    }

    /*
     * (non-Javadoc)
     *
     * @see com.baijia.tianxiao.dal.org.dao.OrgStudentDao#searchHasMobileConsulter(com.baijia.tianxiao.sqlbuilder.dto.
     * PageDto, java.lang.String, java.lang.String)
     */
    @Override
    public List<OrgStudent> searchHasMobileConsulter(PageDto pageDto, String format, String value) {
        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder();
        builder.select("name", "nickName", "orgId");
        builder.ne("mobile", "");
        builder.dateformat("birthday", format, value);
        builder.eq("delStatus", DeleteStatus.NORMAL.getValue());
        builder.group("mobile");

        builder.setPage(pageDto);
        return queryList(builder);
    }

    @Override
    public List<OrgStudent> searchHasMobileAndNeedBindWithOpenId(Long orgId, PageDto pageDto) {
        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder();
        builder.select("mobile", "id");
        builder.ne("mobile", "");
        builder.eq("orgId", orgId);
        builder.eq("weixin", "");
        builder.eq("delStatus", DeleteStatus.NORMAL.getValue());
        // builder.group("mobile");
        builder.setPage(pageDto);
        return queryList(builder);
    }

    @Override
    public List<OrgStudent> getStudentsByUserIdsAndOrgIds(final Collection<Long> orgIds, Collection<Long> userIds) {
        if (CollectionUtils.isEmpty(userIds) || CollectionUtils.isEmpty(orgIds)) {
            return Lists.newArrayList();
        }
        BatchQueryTemplate<Long, List<OrgStudent>> queryTemplate = new ListBatchQueryTemplate<>();
        return queryTemplate.batchQuery(userIds, new BatchQueryCallback<Long, List<OrgStudent>>() {
            @Override
            public List<OrgStudent> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<OrgStudent> builder = createSqlBuilder();
                builder.in("orgId", orgIds);
                builder.in("userId", querySet);
                return queryList(builder);
            }
        });
    }

    @Override
    public OrgStudent getStudentByMobileAndName(Long orgId, String mobile, String studentName) {
        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.eq("name", studentName);
        builder.eq("mobile", mobile);
        builder.eq("delStatus", DeleteStatus.NORMAL.getValue());
        builder.desc("createTime");

        builder.setMaxSize(1);
        return uniqueResult(builder);
    }

    @Override
    public List<OrgStudent> getStudentsByMobileAndName(Long orgId, String mobile, String studentName) {
        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.eq("name", studentName);
        builder.eq("mobile", mobile);
        builder.eq("delStatus", DeleteStatus.NORMAL.getValue());
        builder.desc("createTime");
        return queryList(builder);
    }

    @Override
    public List<OrgStudent> getStudentsByMobileAndOrgIds(Collection<Long> orgIds, String mobile, String...queryProps) {
        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder(queryProps);
        builder.in("orgId", orgIds);
        builder.eq("mobile", mobile);
        builder.ne("weixin", "");
        builder.eq("delStatus", DeleteStatus.NORMAL.getValue());
        builder.desc("createTime");
        return queryList(builder);
    }

    @Override
    public List<OrgStudent> searchStudentByCustomParam(long orgId, AutoMatchQueryRequest params,
        Collection<Long> inStudentIds, PageDto pageDto) {
        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder();

        builder.eq("orgId", orgId);
        builder.eq("delStatus", DeleteStatus.NORMAL.getValue());
        log.debug("AutoMatchQueryRequest student={}", params);
        if (params != null) {
            // 自动匹配查询条件以及查询方式
            AutoMatchUtils.wrap(builder, params);
        } else {
            builder.desc("id");
        }

        if (!inStudentIds.isEmpty()) {
            builder.in("id", inStudentIds);
        }

        if (pageDto != null) {
            builder.setPage(pageDto);
        }
        log.debug("builder sql={},params={}", builder.toSql(), builder.collectConditionValue());
        return queryList(builder);
    }

    @Override
    public Map<Long, Integer> getStudentTotal(Date startDate, Date endDate, List<Long> TianxiaoOrgIds) {
        Map<String, Object> param = new HashMap<>();
        param.put("orgIds", TianxiaoOrgIds);
        String sql =
            "select org_id, count(id) count from yunying.org_students where org_id in (:orgIds) and del_status = 0";
        if (startDate != null && endDate != null) {
            sql += " and create_time between :startDate and :endDate";
            param.put("endDate", endDate);
            param.put("startDate", startDate);
        }
        sql += " group by org_id";
        return this.getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<Map<Long, Integer>>() {
            @Override
            public Map<Long, Integer> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<Long, Integer> map = new HashMap<>();
                while (rs.next()) {
                    Long orgId = rs.getLong("org_id");
                    Integer count = rs.getInt("count");
                    map.put(orgId, count);
                }
                return map;
            }
        });
    }

    @Override
    public Map<Long, Integer> getStudentTotalByStatus(List<Long> TianxiaoOrgIds, Integer lessonStatus) {
        Map<String, Object> param = new HashMap<>();
        param.put("orgIds", TianxiaoOrgIds);
        String sql =
            "select org_id, count(id) count from yunying.org_students where org_id in (:orgIds) and del_status = 0 ";
        if (lessonStatus != null) {
            param.put("lessonStatus", lessonStatus);
            sql += " and lesson_status = :lessonStatus ";
        }
        sql += " group by org_id";
        return this.getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<Map<Long, Integer>>() {
            @Override
            public Map<Long, Integer> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<Long, Integer> map = new HashMap<>();
                while (rs.next()) {
                    Long orgId = rs.getLong("org_id");
                    Integer count = rs.getInt("count");
                    map.put(orgId, count);
                }
                return map;
            }
        });
    }

    @Override
    public Map<Long, Integer> getOrgWechatAuthorizerOfStudentTotal(List<Long> TianxiaoOrgIds) {
        Map<String, Object> param = new HashMap<>();
        param.put("orgIds", TianxiaoOrgIds);
        String sql =
            "select org_id, count(id) count from yunying.org_students where org_id in (:orgIds) and weixin != '' and del_status = 0 group by org_id";

        return this.getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<Map<Long, Integer>>() {
            @Override
            public Map<Long, Integer> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<Long, Integer> map = new HashMap<>();
                while (rs.next()) {
                    Long orgId = rs.getLong("org_id");
                    Integer count = rs.getInt("count");
                    map.put(orgId, count);
                }
                return map;
            }
        });
    }

    @Override
    public void refreshOrgStudent(Long orgId, Long id) {
        StringBuilder sb =
            new StringBuilder("update yunying.org_students set del_status =1 where id > :id and  org_id = :orgId");
        Map<String, Object> param = Maps.newHashMap();
        param.put("orgId", orgId);
        param.put("id", id);
        this.getNamedJdbcTemplate().update(sb.toString(), param);
    }

    @Override
    public Map<Long, OrgStudent> getOrgStudentMapByUserIds(Collection<Long> userIds) {
        if (userIds.isEmpty()) {
            log.warn("user ids is empty.");
            return Maps.newHashMap();
        }
        BatchQueryTemplate<Long, List<OrgStudent>> queryTemplate = new ListBatchQueryTemplate<>();
        List<OrgStudent> students = queryTemplate.batchQuery(userIds, new BatchQueryCallback<Long, List<OrgStudent>>() {
            @Override
            public List<OrgStudent> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<OrgStudent> builder = createSqlBuilder();
                builder.in("userId", querySet);
                return queryList(builder);
            }
        });
        return CollectorUtil.collectMap(students, new Function<OrgStudent, Long>() {
            @Override
            public Long apply(OrgStudent arg0) {
                return arg0.getUserId();
            }
        });
    }

    @Override
    public Map<String, Integer> getStudentTotalMap(Date startDate, Date endDate, List<Long> TianxiaoOrgIds) {
        Map<String, Object> param = new HashMap<>();
        param.put("orgIds", TianxiaoOrgIds);
        String sql =
            "select count(id) count, DATE_FORMAT(create_time,'%y-%m-%d') createTime from yunying.org_students where org_id in (:orgIds) and del_status = 0";
        if (startDate != null && endDate != null) {
            sql += " and create_time between :startDate and :endDate";
            param.put("endDate", endDate);
            param.put("startDate", startDate);
        }
        sql += " group by MONTH(create_time), DAY(create_time)";
        return this.getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<Map<String, Integer>>() {
            @Override
            public Map<String, Integer> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<String, Integer> map = new HashMap<>();
                while (rs.next()) {
                    String createTime = rs.getString("createTime");
                    Integer count = rs.getInt("count");
                    map.put(createTime, count);
                    log.info("createTime===========" + rs.getString("createTime") + "count==========" + count);
                }
                return map;
            }
        });
    }
    /*
     * (non-Javadoc)
     * 
     * @see com.baijia.tianxiao.dal.org.dao.OrgStudentDao#getStudentSum(java.lang.Integer)
     */

    @Override
    public Integer getStudentSum(Integer orgId) {
        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.eq("delStatus", DeleteStatus.NORMAL.getValue());
        builder.count("id");
        return queryForObject(builder, Integer.class);
    }

    @Override
    public Integer getStudentSum(List<Long> orgIds) {
        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder();
        builder.in("orgId", orgIds);
        builder.eq("delStatus", DeleteStatus.NORMAL.getValue());
        builder.count("id");
        return queryForObject(builder, Integer.class);
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.baijia.tianxiao.dal.org.dao.OrgStudentDao#getStudentByUserIds(java.util.List)
     */
    @Override
    public List<OrgStudent> getStudentByUserId(Long userId, Integer delStatus) {
        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder();
        builder.eq("userId", userId);
        if (delStatus != null) {
            builder.eq("delStatus", delStatus);
        }
        return queryList(builder);

    }

    @Override
    public Integer getStudentCountByTime(Date startTime, Date endTime, Long orgId) {
        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder();
        builder.count("id");
        builder.eq("orgId", orgId);
        builder.eq("delStatus", DeleteStatus.NORMAL.getValue());
        builder.le("createTime", endTime);
        builder.ge("createTime", startTime);
        return queryForObject(builder, Integer.class);
    }

    @Override
    public Map<Long, List<Long>> getUserIdMapByOrgIds(List<Long> TianxiaoOrgIds) {
        Map<String, Object> param = new HashMap<>();
        param.put("orgIds", TianxiaoOrgIds);
        String sql = "select * from yunying.org_students where org_id in (:orgIds) and del_status = 0 ";

        return this.getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<Map<Long, List<Long>>>() {

            @Override
            public Map<Long, List<Long>> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<Long, List<Long>> map = new HashMap<>();
                while (rs.next()) {
                    Long userId = rs.getLong("user_id");
                    Long orgId = rs.getLong("org_id");
                    List<Long> userIds = map.get(orgId);
                    if (GenericsUtils.isNullOrEmpty(userIds)) {
                        userIds = new ArrayList<>();
                        map.put(orgId, userIds);
                    }
                    userIds.add(userId);

                }
                return map;
            }
        });
    }

    @Override
    public List<OrgStudent> searchStudent(Long orgId, Collection<? extends Long> studentIds, boolean isAll,
        PageDto pageDto) {
        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder("id", "name", "nickName", "mobile", "showMobile");
        builder.eq("orgId", orgId);
        builder.eq("delStatus", 0);
        if (isAll) {
            if (GenericsUtils.notNullAndEmpty(studentIds)) {
                builder.notin("id", studentIds);
            }
        } else {
            if (GenericsUtils.isNullOrEmpty(studentIds)) {
                return Collections.emptyList();
            }
            builder.in("id", studentIds);
        }
        if (pageDto != null) {
            builder.setPage(pageDto);
        }

        String sql = builder.toSql();
        log.info("sql is : {} ", sql);

        return queryList(builder);
    }

    public void batchUpdateStudentStatus(long orgId, Collection<Long> userIds, int status) {
        String sql =
            "update yunying.org_students set lesson_status=:status,update_time=:updateTime where org_id=:orgId AND user_id in (:ids) and lesson_status!=:status";
        Map<String, Object> map = new HashMap<>();
        map.put("status", status);
        map.put("orgId", orgId);
        map.put("updateTime", new Date());
        map.put("ids", userIds);
        log.info("sql={},param={}", sql, map);
        getNamedJdbcTemplate().update(sql, map);
    }

    public void batchUpdateStudentToStudying(long orgId, Collection<Long> userIds) {
        String sql =
            "update yunying.org_students set lesson_status=1,update_time=:updateTime where org_id=:orgId AND user_id in (:ids) and lesson_status=-1";
        Map<String, Object> map = new HashMap<>();
        map.put("orgId", orgId);
        map.put("updateTime", new Date());
        map.put("ids", userIds);
        log.info("sql={},param={}", sql, map);
        getNamedJdbcTemplate().update(sql, map);
    }

    @Override
    public Map<Integer, Integer> getStudentCountMap(List<Integer> orgIds) {
        final Map<Integer, Integer> data = Maps.newHashMap();
        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder();
        builder.count("id", "num");
        builder.in("orgId", orgIds);
        builder.group("orgId");
        builder.eq("delStatus", DeleteStatus.NORMAL.getValue());
        getNamedJdbcTemplate().query(builder.toSql(), builder.collectConditionValue(), new RowCallbackHandler() {
            @Override
            public void processRow(ResultSet rs) throws SQLException {
                data.put(rs.getInt("orgId"), rs.getInt("num"));
            }
        });
        return data;
    }

    @Override
    public Map<Integer, Integer> getStatisticsByLessonStatus(int orgId, Collection<Long> studentIds) {
        String sql =
            "select count(1) as c,lesson_status from yunying.org_students where org_id=:orgId and del_status=0";
        Map<String, Object> param = new HashMap<>();
        param.put("orgId", orgId);
        if (studentIds != null && studentIds.size() > 0) {
            sql += " and id in (:studentIds)";
            param.put("studentIds", studentIds);
        }
        sql += " group by lesson_status";
        final Map<Integer, Integer> ret = new HashMap<>();
        log.info("[Statistics] sql={},param={}", sql, param);
        getNamedJdbcTemplate().query(sql, param, new RowMapper<Object>() {
            @Override
            public Object mapRow(ResultSet rs, int rowNum) throws SQLException {
                ret.put(rs.getInt("lesson_status"), rs.getInt("c"));
                return null;
            }
        });
        return ret;
    }

    @Override
    public void updateStudentStatusByOrgId(Long orgId, int status) {
        String sql =
            "update yunying.org_students set lesson_status=:status,update_time=:updateTime where org_id=:orgId and lesson_status!=:status";
        Map<String, Object> map = new HashMap<>();
        map.put("status", status);
        map.put("updateTime", new Date());
        map.put("orgId", orgId);
        int num = getNamedJdbcTemplate().update(sql, map);
        log.info("[UpdateStudentStatus] param={},number={}", map, num);
    }

    @Override
    public List<OrgStudent> getOrgStudents(Long orgId, List<String> mobiles) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(CollectionUtils.isNotEmpty(mobiles), "mobiles can not be empty!");
        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder();
        builder.in("mobile", mobiles);
        builder.eq("orgId", orgId);
        builder.desc("createTime");
        builder.eq("delStatus", DeleteStatus.NORMAL.getValue());
        return queryList(builder);
    }

    @Override
    public OrgStudent getStudent(Long orgId, String mobile, Long userId) {
        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.eq("userId", userId);
        builder.eq("mobile", mobile);
        builder.eq("delStatus", DeleteStatus.NORMAL.getValue());
        return uniqueResult(builder);
    }

    @Override
    public List<Long> getStudentIdsByAdder(int cascadeId, long orgId) {
        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder("userId");
        builder.eq("orgId", orgId);
        builder.eq("addCascadeId", cascadeId);
        builder.eq("delStatus", 0);
        List<OrgStudent> students = queryList(builder);
        if (students != null) {
            return ListUtil.toKeyList(students, "userId", OrgStudent.class);
        } else {
            return Collections.emptyList();
        }
    }

    @Override
    public List<OrgStudent> listPinYinNull(Long maxId, Integer size) {
        final List<OrgStudent> list = new ArrayList<OrgStudent>();
        String sql = "select * from yunying.org_students where pinyin='' and id>:maxId limit :size";
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("maxId", maxId);
        params.put("size", size);

        this.getNamedJdbcTemplate().query(sql, params, new ResultSetExtractor<Object>() {

            @Override
            public Object extractData(ResultSet rs) throws SQLException, DataAccessException {
                OrgStudent student = null;
                while (rs.next()) {
                    student = new OrgStudent();
                    student.setId(rs.getLong("id"));
                    student.setName(rs.getString("student_name"));
                    list.add(student);
                }
                return null;
            }
        });
        return list;
    }

    @Override
    public void replacePortrait(String oldUrl, String newUrl) {
        StringBuilder sb = new StringBuilder("UPDATE yunying.org_students SET avatar =:newUrl WHERE avatar =:oldUrl");
        Map<String, String> param = new HashMap<String, String>();
        param.put("oldUrl", oldUrl);
        param.put("newUrl", newUrl);
        this.getNamedJdbcTemplate().update(sb.toString(), param);
    }

    @Override
    public Map<Integer, Integer> getStatisticsDayByLessonStatus(int lessonStatus, List<Integer> orgIds) {
        String sql =
            "select count(1) as c,org_id from yunying.org_students where lesson_status=:lessonStatus and del_status=0 and org_id in (:orgIds)";
        Map<String, Object> param = new HashMap<>();
        param.put("lessonStatus", lessonStatus);
        param.put("orgIds", orgIds);
        sql += " group by org_id";
        final Map<Integer, Integer> ret = new HashMap<>();
        getNamedJdbcTemplate().query(sql, param, new RowMapper<Object>() {
            @Override
            public Object mapRow(ResultSet rs, int rowNum) throws SQLException {
                ret.put(rs.getInt("org_id"), rs.getInt("c"));
                return null;
            }
        });
        return ret;
    }

    @Override
    public List<OrgStudent> listOpenIdNotNull(Long minId) {
        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder();
        builder.ne("weixin", "");
        builder.gt("id", minId);
        return queryList(builder);
    }

    @Override
    public List<OrgStudent> fuzzySearchStudents(Long orgId, String param, PageDto page, String...queryProps) {
        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder(queryProps);
        builder.eq("orgId", orgId);
        if (StringUtils.isNotBlank(param)) {
            builder.add(Expressions.or(Expressions.like("name", param, MatchMode.ANYWHERE),
                Expressions.like("mobile", param, MatchMode.ANYWHERE)));
        }
        if (page != null) {
            builder.setPage(page);
        }
        builder.eq("delStatus", DeleteStatus.NORMAL.getValue());
        builder.asc("name", true);
        return queryList(builder);
    }

    @Override
    public void updateNextRemindTime(Long studentId) {
        Map<String, Object> param = new HashMap<>();
        param.put("studentId", studentId);
        String sql = "update yunying.org_students set next_remind_time = 0 where id = :studentId";
        getNamedJdbcTemplate().update(sql, param);
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.baijia.tianxiao.dal.org.dao.OrgStudentDao#queryStudentClassHours(java.util.Set, java.lang.Long)
     */
    @Override
    public Integer countStudentsByTime(Long orgId, Date startTime, Date endTime) {
        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.eq("delStatus", DeleteStatus.NORMAL.getValue());
        builder.count("id");
        if (startTime != null) {
            builder.ge("createTime", startTime);
        }
        if (endTime != null) {
            builder.le("createTime", endTime);
        }
        return queryForObject(builder, Integer.class);
    }

    @Override
    public Map<Integer, Integer> getStatisticsByLessonStatus(int orgId, boolean needDataAuthority,
        List<Long> studentIds, Integer cascadeId) {
        String sql =
            "select count(1) as c,lesson_status from yunying.org_students where org_id=:orgId and del_status = 0 ";
        Map<String, Object> param = new HashMap<>();
        param.put("orgId", orgId);

        if (needDataAuthority) {
            if (GenericsUtils.notNullAndEmpty(studentIds)) {
                sql += " and user_id in(:userIds) ";
                param.put("userIds", studentIds);
            }
            sql += " and add_cascade_id = :addCascadeId ";
            param.put("addCascadeId", cascadeId);
        }

        sql += " group by lesson_status ";
        final Map<Integer, Integer> ret = new HashMap<>();
        log.info("[Statistics] sql={},param={}", sql, param);
        getNamedJdbcTemplate().query(sql, param, new RowMapper<Object>() {
            @Override
            public Object mapRow(ResultSet rs, int rowNum) throws SQLException {
                ret.put(rs.getInt("lesson_status"), rs.getInt("c"));
                return null;
            }
        });
        return ret;
    }

    @Override
    public List<OrgStudent> findStudentWithOrgAndOpenId(List<Integer> orgIds, String openId, String...queryProps) {
        if (GenericsUtils.isNullOrEmpty(orgIds)) {
            return GenericsUtils.emptyList();
        }
        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder(queryProps);
        builder.in("orgId", orgIds);
        builder.eq("weixin", openId);
        builder.eq("delStatus", DeleteStatus.NORMAL.getValue());
        return this.queryList(builder);
    }

    public int countBySource(Long orgId, Long sourceId) {
        SingleSqlBuilder<OrgStudent> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.eq("source", sourceId);
        builder.eq("delStatus", DeleteStatus.NORMAL.getValue());
        builder.count("id");
        Integer count = queryForObject(builder, Integer.class);
        return count == null ? 0 : count;
    }


    @Override
    public Map<Long, String> mapIdVsName(Collection<Long> ids) {
        final Map<Long, String> map = new HashMap<Long, String>();
        
        NamedParameterJdbcTemplate template = getNamedJdbcTemplate();
        
        String sql = "SELECT id,student_name FROM yunying.org_students WHERE id IN (:ids)";
        
        Map<String, Object> params = new HashMap<String,Object>();
        params.put("ids", ids);
        
        template.query(sql, params, new RowMapper<Long>() {
            @Override
            public Long mapRow(ResultSet rs, int rowNum) throws SQLException {
                map.put(rs.getLong("id"), rs.getString("student_name"));
                return  0L;
            }
        });
        
        return map;
    }

}