/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */

package com.baijia.tianxiao.dal.org.dao.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections4.map.HashedMap;
import org.apache.commons.lang3.StringUtils;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.core.RowCallbackHandler;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.stereotype.Repository;

import com.baijia.tianxiao.dal.org.dao.OrgInfoDao;
import com.baijia.tianxiao.dal.org.po.OrgBaseInfoDto;
import com.baijia.tianxiao.dal.org.po.OrgInfo;
import com.baijia.tianxiao.sqlbuilder.SingleSqlBuilder;
import com.baijia.tianxiao.sqlbuilder.bean.impl.MatchMode;
import com.baijia.tianxiao.sqlbuilder.support.JdbcTemplateDaoSupport;
import com.baijia.tianxiao.util.query.BatchQueryCallback;
import com.baijia.tianxiao.util.query.ListBatchQueryTemplate;
import com.baijia.tianxiao.util.query.MapBatchQueryTemplate;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

/**
 * @author shizuwei
 * @version 1.0
 * @title OrgInfoDaoImpl
 * @desc TODO
 * @date 2015年12月4日
 */
@Repository("tianxiao_dal_yunying_orgInfoDao")
@Slf4j
public class OrgInfoDaoImpl extends JdbcTemplateDaoSupport<OrgInfo> implements OrgInfoDao {

    public OrgInfoDaoImpl() {
        super(OrgInfo.class);
    }

    @Override
    public OrgInfo getOrgInfo(@NonNull Integer orgId, String... queryProps) {
        SingleSqlBuilder<OrgInfo> builder = createSqlBuilder();
        builder.select(queryProps);
        builder.eq("orgId", orgId);
        return uniqueResult(builder);
    }

    @Override
    public String getOrgNameByOrgId(Integer orgId) {
        SingleSqlBuilder<OrgInfo> builder = createSqlBuilder();
        builder.select("name");
        builder.eq("orgId", orgId);
        return queryForObject(builder, String.class);
    }

    @Override
    public String getOrgShortNameByOrgId(Integer orgId) {
        if (orgId == null) {
            return "";
        }
        SingleSqlBuilder<OrgInfo> builder = createSqlBuilder();
        builder.select("shortName");
        builder.eq("orgId", orgId);
        return queryForObject(builder, String.class);
    }

    @Override
    public List<OrgInfo> getOrgInfos(Collection<Integer> orgIds, final String... properties) {
        if (CollectionUtils.isEmpty(orgIds)) {
            return Collections.emptyList();
        }

        ListBatchQueryTemplate<Integer, OrgInfo> queryTemplate = new ListBatchQueryTemplate<>();

        return queryTemplate.batchQuery(orgIds, new BatchQueryCallback<Integer, List<OrgInfo>>() {
            @Override
            public List<OrgInfo> doQuery(Collection<Integer> querySet) {
                SingleSqlBuilder<OrgInfo> builder = createSqlBuilder(properties);
                builder.in("orgId", querySet);
                return queryList(builder);
            }
        });
    }

    @Override
    public List<OrgInfo> getOrgInfos(Collection<Integer> orgIds, final Long areaId, final String... properties) {
        if (CollectionUtils.isEmpty(orgIds)) {
            return Collections.emptyList();
        }

        ListBatchQueryTemplate<Integer, OrgInfo> queryTemplate = new ListBatchQueryTemplate<>();

        return queryTemplate.batchQuery(orgIds, new BatchQueryCallback<Integer, List<OrgInfo>>() {
            @Override
            public List<OrgInfo> doQuery(Collection<Integer> querySet) {
                SingleSqlBuilder<OrgInfo> builder = createSqlBuilder(properties);
                if (areaId != null) {
                    builder.eq("areaId", areaId);
                }
                builder.in("orgId", querySet);
                return queryList(builder);
            }
        });
    }

    @Override
    public List<Integer> getOrgIdByOrgName(String name) {
        SingleSqlBuilder<OrgInfo> builder = createSqlBuilder();
        builder.select("orgId");
        builder.like("shortName", name, MatchMode.ANYWHERE);
        return queryForList(builder, Integer.class);
    }

    @Override
    public List<OrgInfo> getOrgInfosByExtentions(Collection<String> extensions, final String... properties) {
        if (CollectionUtils.isEmpty(extensions)) {
            return Collections.emptyList();
        }
        ListBatchQueryTemplate<String, OrgInfo> queryTemplate = new ListBatchQueryTemplate<>();

        return queryTemplate.batchQuery(extensions, new BatchQueryCallback<String, List<OrgInfo>>() {
            @Override
            public List<OrgInfo> doQuery(Collection<String> querySet) {
                SingleSqlBuilder<OrgInfo> builder = createSqlBuilder(properties);
                builder.in("extension", querySet);
                return queryList(builder);
            }
        });
    }

    @Override
    public List<OrgInfo> getOrgInfoByNameAndAreaId(String name, Long areaId, String... properties) {
        if (StringUtils.isBlank(name) && areaId == null) {
            return Lists.newArrayList();
        }
        SingleSqlBuilder<OrgInfo> builder = createSqlBuilder(properties);
        if (StringUtils.isNotEmpty(name)) {
            builder.like("shortName", name, MatchMode.ANYWHERE);
        }
        if (areaId != null) {
            builder.eq("areaId", areaId);
        }
        return queryList(builder);
    }

    @Override
    public OrgInfo getByExtension(String extension, String... properties) {
        if (StringUtils.isBlank(extension)) {
            return null;
        }
        SingleSqlBuilder<OrgInfo> builder = createSqlBuilder(properties);
        builder.eq("extension", extension);
        return uniqueResult(builder);

    }

    @Override
    public boolean updateMarkingStatusByOrgId(Integer orgId, Integer markingStatus) {
        StringBuilder sb = new StringBuilder();
        sb.append("update yunying.org_info set marketing_status =:markingStatus where org_id = :orgId");

        Map map = new HashedMap();
        map.put("markingStatus", markingStatus);
        map.put("orgId", orgId);

        return getNamedJdbcTemplate().update(sb.toString(), map) > 0;
    }


    @Override
    public boolean updateTeacherSigninByOrgId(Long orgId, Integer signStatus) {
        StringBuilder sb = new StringBuilder();
        sb.append("update yunying.org_info set teacher_signin =:signStatus where org_id = :orgId");

        Map map = new HashedMap();
        map.put("signStatus", signStatus);
        map.put("orgId", orgId);

        return getNamedJdbcTemplate().update(sb.toString(), map) > 0;
    }

    @Override
    public Map<Integer, Long> getAreaMap(Collection<Integer> orgIds) {
        return new MapBatchQueryTemplate<Integer, Integer, Long>().batchQuery(orgIds,
                new BatchQueryCallback<Integer, Map<Integer, Long>>() {
                    @Override
                    public Map<Integer, Long> doQuery(Collection<Integer> querySet) {
                        SingleSqlBuilder<OrgInfo> builder = createSqlBuilder("orgId", "areaId");
                        builder.in("orgId", querySet);
                        final Map<Integer, Long> result = Maps.newHashMap();
                        getNamedJdbcTemplate().query(builder.toSql(), builder.collectConditionValue(),
                                new RowCallbackHandler() {
                                    @Override
                                    public void processRow(ResultSet rs) throws SQLException {
                                        result.put(rs.getInt("orgId"), rs.getLong("areaId"));
                                    }
                                });
                        return result;
                    }
                });
    }

    @Override
    public Map<Long, String> getExtentionsByOrgIds(List<Long> orgIds) {
        Map<String, Object> param = new HashMap<>();
        param.put("orgIds", orgIds);
        String sql = "select org_id, extension from yunying.org_info where org_id in (:orgIds) and (extension is not null)";

        return this.getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<Map<Long, String>>() {

            @Override
            public Map<Long, String> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<Long, String> map = new HashMap<>();

                while (rs.next()) {
                    Long orgId = rs.getLong("org_id");
                    String extension = rs.getString("extension");
                    map.put(orgId, extension);
                }
                return map;
            }
        });
    }

    @Override
    public OrgBaseInfoDto getBaseInfo(int orgId) {
        String sql =
                "select oa.id, oa.number, oa.mobile, oa.country_code, oi.shortname from yunying.org_account oa, yunying.org_info oi where oa.id=:orgId and oa.id=oi.org_id";

        OrgBaseInfoDto dto = null;
        Map<String, Object> param = new HashMap<>();
        param.put("orgId", orgId);

        try {
            dto = this.getNamedJdbcTemplate().queryForObject(sql, param, new RowMapper<OrgBaseInfoDto>() {
                @Override
                public OrgBaseInfoDto mapRow(ResultSet rs, int rowNum) throws SQLException {
                    OrgBaseInfoDto dto = new OrgBaseInfoDto();
                    dto.setMobile(rs.getString("mobile"));
                    dto.setNumber(rs.getInt("number"));
                    dto.setOrgId(rs.getInt("id"));
                    dto.setShortName(rs.getString("shortname"));
                    dto.setCountryCode(rs.getString("country_code"));
                    return dto;
                }
            });
        } catch (Exception e) {
            log.info("get orginfo failed, orgId:{}, e:{}", orgId, e);
        }
        return dto;
    }

	@Override
	public OrgInfo queryOrgInfo(int orgId, String key) {
		SingleSqlBuilder<OrgInfo> builder = createSqlBuilder();
		builder.eq("orgId", orgId);
		if(StringUtils.isNotEmpty(key)) {
			builder.like("contacts", key, MatchMode.ANYWHERE);
		}
		return uniqueResult(builder);
	}

	@Override
	public Map<Integer, OrgInfo> getOrgInfoMap(Collection<Integer> orgIds, final String... properties) {
		if (CollectionUtils.isEmpty(orgIds)) {
            return Maps.newHashMap();
        }

        ListBatchQueryTemplate<Integer, OrgInfo> queryTemplate = new ListBatchQueryTemplate<>();

        List<OrgInfo> orgInfoList = queryTemplate.batchQuery(orgIds, new BatchQueryCallback<Integer, List<OrgInfo>>() {
            @Override
            public List<OrgInfo> doQuery(Collection<Integer> querySet) {
                SingleSqlBuilder<OrgInfo> builder = createSqlBuilder(properties);
                builder.in("orgId", querySet);
                return queryList(builder);
            }
        });
        Map<Integer, OrgInfo> data = Maps.newHashMap();
        if(CollectionUtils.isNotEmpty(orgInfoList)){
        	for(OrgInfo orgInfo : orgInfoList){
        		data.put(orgInfo.getOrgId(), orgInfo);
        	}
        }
        return data;
	}
}
