/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */
package com.baijia.tianxiao.dal.callservice.dao.impl;

import com.baijia.tianxiao.dal.advisory.dao.dto.CallStatisticDto;
import com.baijia.tianxiao.dal.callservice.constant.PartyCallType;
import com.baijia.tianxiao.dal.callservice.dao.CallServiceInfoDao;
import com.baijia.tianxiao.dal.callservice.po.CallServiceInfo;
import com.baijia.tianxiao.sqlbuilder.SingleSqlBuilder;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.sqlbuilder.support.JdbcTemplateDaoSupport;
import com.baijia.tianxiao.util.GenericsUtils;
import com.baijia.tianxiao.util.date.DateUtil;

import com.google.common.base.Preconditions;

import org.apache.commons.lang3.StringUtils;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.stereotype.Repository;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @title CallServiceInfoDaoImpl
 * @desc TODO
 * @author cxm
 * @date 2015年12月5日
 * @version 1.0
 */
@Repository
public class CallServiceInfoDaoImpl extends JdbcTemplateDaoSupport<CallServiceInfo> implements CallServiceInfoDao {

    /**
     * @param entityClass
     */
    public CallServiceInfoDaoImpl() {
        super(CallServiceInfo.class);
    }

    @Override
    public Map<String, Object> queryCallCountAndTime(String mobile, Long callParty, PartyCallType callType) {
        Preconditions.checkArgument(StringUtils.isNoneBlank(mobile), "mobile can not be null");
        Preconditions.checkArgument(callParty != null && callParty > 0, "call party is illegal");
        SingleSqlBuilder<CallServiceInfo> builder = createSqlBuilder();
        builder.count("id");
        builder.max("createTime", "MAX_TIME");
        builder.eq("callSubscriberNum", mobile);
        builder.eq("calledParty", callParty);
        if (callType != null) {
            builder.eq("callType", callType.getCode());
        }
        return getNamedJdbcTemplate().queryForMap(builder.toSql(), builder.collectConditionValue());
    }

    /* (non-Javadoc)
     * @see com.baijia.tianxiao.dal.callservice.dao.CallServiceInfoDao#getTotalDurationByOrgId(java.lang.Integer)
     */

    @Override
    public Integer getTotalDurationByOrgId(Integer orgId) {
        SingleSqlBuilder<CallServiceInfo> builder = createSqlBuilder();
        builder.sum("duration");
        builder.eq("callSubscriber", Long.valueOf(orgId));
        builder.eq("month", DateUtil.getCurrentYM());
        Integer sum = this.queryForObject(builder, Integer.class);
        if(sum == null){
            return 0;
        }
        return sum;

    }

    @Override
    public List<CallServiceInfo> listCallServiceInfoDesc(Integer orgId, String mobile, PageDto pageDto) {
        final List<CallServiceInfo> list = new ArrayList<CallServiceInfo>();

        StringBuilder builder = new StringBuilder();
        builder.append(" SELECT * FROM yunying.callservice_info ")
                .append(" WHERE (call_subscriber=:orgId AND call_subscriber_num=:mobile) ")
                .append(" OR (called_party=:orgId AND called_party_num=:mobile)");

        builder.append(" ORDER BY id DESC");
        if( pageDto!=null ){
            builder.append(" LIMIT :start,:size");
        }


        NamedParameterJdbcTemplate template = getNamedJdbcTemplate();
        Map<String, Object> params = new HashMap<String,Object>();
        params.put("orgId", orgId);
        params.put("mobile", mobile);
        if( pageDto!=null ){
            params.put("start", pageDto.firstNum());
            params.put("size", pageDto.getPageSize());
        }

        template.query(builder.toString(), params, new RowMapper<Integer>() {
            CallServiceInfo info;
            @Override
            public Integer mapRow(ResultSet rs, int rowNum) throws SQLException {
                info = new CallServiceInfo();
                info.setCallback( rs.getString("callback") );
                info.setCalledParty( rs.getLong("called_party") );
                info.setCalledPartyNum( rs.getString("called_party_num") );
                info.setCallEnd( rs.getTimestamp("call_end") );
                info.setCallStart( rs.getTimestamp("call_start") );
                info.setCallSubscriber( rs.getLong("call_subscriber") );
                info.setCallSubscriberNum( rs.getString("call_subscriber_num") );
                info.setCallType( rs.getInt("call_type") );
                info.setCdrStatus( rs.getInt("cdr_status") );
                info.setCreateTime( rs.getTimestamp("create_time") );
                info.setDuration( rs.getInt("duration") );
                info.setId( rs.getLong("id") );
                info.setMonth( rs.getString("month") );
                info.setResult( rs.getInt("result") );
                info.setStorageId( rs.getLong("storage_id") );
                info.setUniqueId( rs.getString("unique_id") );
                info.setUpdateTime( rs.getTimestamp("update_time") );

                list.add(info);
                return 1;
            }
        });
        return list;
    }

    @Override
    public List<CallServiceInfo> page(Long minId, PageDto pageDto) {
        final List<CallServiceInfo> list = new ArrayList<CallServiceInfo>();

        StringBuilder builder = new StringBuilder();
        builder.append("SELECT * FROM yunying.callservice_info WHERE (call_type=2 OR call_type=4)");
        if (minId!=null && minId!=0) {
            builder.append(" AND id>:minId");
        }

        if (pageDto != null) {
            builder.append(" LIMIT :start,:size");
        }

        NamedParameterJdbcTemplate template = getNamedJdbcTemplate();
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("minId", minId);
        params.put("start", pageDto.firstNum());
        params.put("size", pageDto.getPageSize());

        template.query(builder.toString(), params, new RowMapper<Integer>() {
            CallServiceInfo info;

            @Override
            public Integer mapRow(ResultSet rs, int rowNum) throws SQLException {
                info = new CallServiceInfo();
                info.setCallback( rs.getString("callback") );
                info.setCalledParty(rs.getLong("called_party"));
                info.setCalledPartyNum(rs.getString("called_party_num"));
                info.setCallEnd(rs.getTimestamp("call_end"));
                info.setCallStart(rs.getTimestamp("call_start"));
                info.setCallSubscriber(rs.getLong("call_subscriber"));
                info.setCallSubscriberNum(rs.getString("call_subscriber_num"));
                info.setCallType(rs.getInt("call_type"));
                info.setCdrStatus(rs.getInt("cdr_status"));
                info.setCreateTime(rs.getTimestamp("create_time"));
                info.setDuration(rs.getInt("duration"));
                info.setId(rs.getLong("id"));
                info.setMonth(rs.getString("month"));
                info.setResult(rs.getInt("result"));
                info.setStatus(rs.getInt("status"));
                info.setStorageId( rs.getLong("storage_id") );
                info.setUniqueId( rs.getString("unique_id") );
                info.setUpdateTime( rs.getTimestamp("update_time") );


                list.add(info);
                return 1;
            }
        });

        return list;
    }

    @Override
    public Map<Long, CallStatisticDto> getTotalDurationAndCount(Date startTime, Date endTime, List<Long> orgIds) {
        Map<String , Object> param = new HashMap<>();
        param.put("orgIds", orgIds);
        String sql = "select count(1) count, sum(duration) duration, call_subscriber from yunying.callservice_info where "
                + "call_subscriber in (:orgIds)";
        if(GenericsUtils.notNullAndEmpty(startTime)){
            sql += " and create_time between :startTime and :endTime";
            param.put("endTime", endTime);
            param.put("startTime", startTime);
        }

        sql += " group by call_subscriber";

        return this.getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<Map<Long, CallStatisticDto>>() {

            @Override
            public Map<Long, CallStatisticDto> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<Long, CallStatisticDto> map = new HashMap<>();

                while(rs.next()){
                    Long orgId = rs.getLong("call_subscriber");
                    Integer count = rs.getInt("count");
                    Integer duration = rs.getInt("duration");
                    CallStatisticDto dto = new CallStatisticDto();
                    dto.setCount(count);
                    dto.setDuration(duration);
                    map.put(orgId, dto);
                }

                return map;
            }
        });
    }

}