/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */

package com.baijia.tianxiao.dal.advisory.dao.impl;

import com.baijia.tianxiao.dal.advisory.dao.OrgCallRecorderDao;
import com.baijia.tianxiao.dal.advisory.dao.dto.CallStatisticDto;
import com.baijia.tianxiao.dal.advisory.po.OrgCallRecordCount;
import com.baijia.tianxiao.dal.advisory.po.OrgCallRecorder;
import com.baijia.tianxiao.sqlbuilder.SingleSqlBuilder;
import com.baijia.tianxiao.sqlbuilder.bean.Expression;
import com.baijia.tianxiao.sqlbuilder.bean.impl.MatchMode;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.sqlbuilder.support.JdbcTemplateDaoSupport;
import com.baijia.tianxiao.sqlbuilder.util.Expressions;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.BeanPropertyRowMapper;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.stereotype.Repository;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.*;

/**
 * @title OrgCallRecorderDaoImpl
 * @desc TODO
 * @author shanyu
 * @date 2015年11月12日
 * @version 1.0
 */
@Slf4j
@Repository
public class OrgCallRecorderDaoImpl extends JdbcTemplateDaoSupport<OrgCallRecorder> implements OrgCallRecorderDao {

    public OrgCallRecorderDaoImpl() {
        super(OrgCallRecorder.class);
    }

    @Override
    public List<OrgCallRecorder> query(Date startTime, Date endTime, String extension, String key,
        Collection<Integer> statuses, PageDto page) {
        if (StringUtils.isBlank(extension)) {
            return Collections.emptyList();
        }
        SingleSqlBuilder<OrgCallRecorder> builder = createSqlBuilder();
        if (StringUtils.isNoneBlank(key)) {
            builder.or(Expressions.like("customerNumber", key, MatchMode.ANYWHERE),
                Expressions.like("connectedNumber", key, MatchMode.ANYWHERE));
        }
        if (CollectionUtils.isNotEmpty(statuses)) {
            Expression inExpression = Expressions.in("processStatus", statuses);
            if (statuses.contains(50) && statuses.contains(51)) {
                builder.or(Expressions.eq("processStatus", 0), inExpression);
            } else if (statuses.contains(50)) {
                builder.or(Expressions.and(Expressions.eq("processStatus", 0), Expressions.eq("status", 1)),
                    inExpression);
            } else if (statuses.contains(51)) {
                builder.or(Expressions.and(Expressions.eq("processStatus", 0), Expressions.ne("status", 1)),
                    inExpression);
            } else {
                builder.add(inExpression);
            }
        }
        if (startTime != null && endTime != null) {
            builder.between("startTime", startTime, endTime);
        }
        builder.eq("extention", extension);
        builder.desc("startTime");
        builder.setPage(page);
        return queryList(builder);

    }

    @Override
    public List<OrgCallRecorder> query(Date startTime, Date endTime, Collection<String> extensions, PageDto page) {
        SingleSqlBuilder<OrgCallRecorder> builder = createSqlBuilder();
        if (startTime != null && endTime != null) {
            builder.between("startTime", startTime, endTime);
        }
        if (CollectionUtils.isNotEmpty(extensions)) {
            builder.in("extention", extensions);
        }
        builder.desc("startTime");
        builder.setPage(page);
        return queryList(builder);
    }

    @Override
    public Integer queryCount(Date startTime, Date endTime, Collection<String> extensions, Integer status,
        Integer processStatus) {
        SingleSqlBuilder<OrgCallRecorder> builder = createSqlBuilder();
        if (startTime != null && endTime != null) {
            builder.between("startTime", startTime, endTime);
        }
        if (CollectionUtils.isNotEmpty(extensions)) {
            builder.in("extention", extensions);
        }
        if (status != null) {
            builder.eq("status", status);
        }
        if (processStatus != null) {
            builder.eq("processStatus", processStatus);
        }
        builder.desc("startTime");
        builder.count("id");
        return queryForObject(builder, Integer.class);
    }

    @Override
    public OrgCallRecordCount queryOrgCallRecordCount(Date startTime, Date endTime, Collection<String> extensions) {
        String sql =
            "select sum(case when process_status=50 then 1 when process_status=0 and status = 1 then 1 else 0 end) answered,"
                + "sum(case when process_status=51 then 1 when process_status=0 and status in (2,3) then 1 else 0 end) notAnswered,"
                + " sum(case when process_status=20 then 1 else 0 end) needCallBack,"
                + "sum(case when process_status=52 then 1 else 0 end) complete "
                + "from yunying.callservice_org_call_record "
                + "where extention in (:extention) and start_time between :startTime_l and :startTime_r order by start_time desc";
        Map<String, Object> nameValueMap = Maps.newHashMap();
        nameValueMap.put("extention", extensions);
        nameValueMap.put("startTime_l", startTime);
        nameValueMap.put("startTime_r", endTime);
        return this.getNamedJdbcTemplate().queryForObject(sql, nameValueMap,
            new BeanPropertyRowMapper<OrgCallRecordCount>(OrgCallRecordCount.class));
    }
    
    @Override
    public Map<String, Object> queryCallCountAndTime(String mobile, String extension) {
        Preconditions.checkArgument(StringUtils.isNoneBlank(mobile), "mobile can not be null");
        Preconditions.checkArgument(StringUtils.isNoneBlank(extension), "org extension is null");
        SingleSqlBuilder<OrgCallRecorder> builder = createSqlBuilder();
        builder.count("id");
        builder.max("startTime", "MAX_TIME");
        builder.eq("customerNumber", mobile);
        builder.eq("extention", extension);
        String sql = builder.toSql();
        Map<String, Object> paramMap = builder.collectConditionValue();
        log.debug("query sql:{},param:{}", sql, paramMap);
        return getNamedJdbcTemplate().queryForMap(sql, paramMap);
    }
    
    @Override
    public Map<String, CallStatisticDto> getCallStatisticDtoMapByExtensions(Date startTime, Date endTime,
        Collection<String> extensions) {
        Map<String, Object> param = new HashMap<>();
        param.put("extensions", extensions);
        
        String sql = "select extention, count(1) count, sum(during_time) duration from yunying.callservice_org_call_record where extention in (:extensions) ";
        
        if(startTime!= null){
            sql += " and start_time between :startTime and :endTime ";
            param.put("startTime", startTime);
            param.put("endTime", endTime);
        }
        
        sql += " group by extention";
        
        return this.getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<Map<String, CallStatisticDto>>() {

            @Override
            public Map<String, CallStatisticDto> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<String, CallStatisticDto> map = new HashMap<>();
                while(rs.next()){
                    String extention = rs.getString("extention");
                    Integer count = rs.getInt("count");
                    Integer duration = rs.getInt("duration");
                    CallStatisticDto dto = new CallStatisticDto();
                    dto.setCount(count);
                    dto.setDuration(duration);
                    map.put(extention, dto);
                }
                return map;
            }
        });
    }
    
    
    @Override
    public Integer getOrgCallRecordCountByExtexsion(Date startTime, Date endTime, String extention){
        Map<String, Object> param = new HashMap<>();
        param.put("extention", extention);
        String sql = "select count(1) count "
            + " from yunying.callservice_org_call_record where extention = :extention";
        if(startTime!= null){
            sql += " and start_time between :startTime and :endTime";
            param.put("startTime", startTime);
            param.put("endTime", endTime);
        }
        return this.getNamedJdbcTemplate().queryForObject(sql, param, Integer.class);
    }
    
    @Override
    public Integer getOrgCallRecordTimeByExtension(Date startTime, Date endTime, String extension){
        Map<String, Object> param = new HashMap<>();
        param.put("extention", extension);
        String sql = "select sum(during_time)  from yunying.callservice_org_call_record where extention = :extention";
        if(startTime!= null){
            sql += " and start_time between :startTime and :endTime";
            param.put("startTime", startTime);
            param.put("endTime", endTime);
        }
        return this.getNamedJdbcTemplate().queryForObject(sql, param, Integer.class);
    }
    
}
