/**
 * s * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */

package com.baijia.tianxiao.dal.org.dao.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.core.RowCallbackHandler;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.stereotype.Repository;

import com.baijia.tianxiao.consants.UserRole;
import com.baijia.tianxiao.dal.org.dao.OrgStudentLessonDao;
import com.baijia.tianxiao.dal.org.dto.StudentClasHourDocument;
import com.baijia.tianxiao.dal.org.po.OrgLessonComment;
import com.baijia.tianxiao.dal.org.po.OrgLessonSign;
import com.baijia.tianxiao.dal.org.po.OrgStudentLesson;
import com.baijia.tianxiao.dal.org.po.OrgStudentLessonCounter;
import com.baijia.tianxiao.sqlbuilder.SingleSqlBuilder;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.sqlbuilder.support.JdbcTemplateDaoSupport;
import com.baijia.tianxiao.util.CollectorUtil;
import com.baijia.tianxiao.util.GenericsUtils;
import com.baijia.tianxiao.util.date.DateUtil;
import com.baijia.tianxiao.util.query.BatchQueryCallback;
import com.baijia.tianxiao.util.query.BatchQueryTemplate;
import com.baijia.tianxiao.util.query.ListBatchQueryTemplate;
import com.baijia.tianxiao.util.query.MapBatchQueryTemplate;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import lombok.extern.slf4j.Slf4j;

@Slf4j
@Repository
public class OrgStudentLessonDaoImpl extends JdbcTemplateDaoSupport<OrgStudentLesson> implements OrgStudentLessonDao {

    public OrgStudentLessonDaoImpl() {
        super(OrgStudentLesson.class);
    }

    @Override
    public void delStudentFromLesson(Long orgId, Collection<Long> lessonIds, Long userId) {

        if (CollectionUtils.isEmpty(lessonIds)) {
            log.info("lessonIds = {}", lessonIds);
            return;
        }
        String sql = "update tts.org_student_lesson set del_status=1 where org_id=:orgId " +
                "and lesson_id in (:lessonId) and student_id = :userId";
        Map<String, Object> condition = Maps.newHashMap();
        condition.put("orgId", orgId);
        condition.put("lessonId", lessonIds);
        condition.put("userId", userId);
        getNamedJdbcTemplate().update(sql,condition);
    }
    
    @Override
    public int delStudentFromLesson(Long orgId, Collection<Long> lessonIds, Collection<Long> userIds) {

        if (CollectionUtils.isEmpty(lessonIds)) {
            log.info("lessonIds = {}", lessonIds);
            return 0;
        }
        String sql = "update tts.org_student_lesson set del_status=1 where org_id=:orgId " +
                "and lesson_id in (:lessonId) and student_id in (:userId)";
        Map<String, Object> condition = Maps.newHashMap();
        condition.put("orgId", orgId);
        condition.put("lessonId", lessonIds);
        condition.put("userId", userIds);
        return getNamedJdbcTemplate().update(sql,condition);
    }

    @Override
    public List<Long> getLessIdsOfStudent(final Long orgId, Collection<Long> lessonIds, final Long userId) {
        if (CollectionUtils.isEmpty(lessonIds)) {
            log.info("lessonIds = {}", lessonIds);
            return Lists.newArrayList();
        }
        BatchQueryTemplate<Long, List<Long>> batch = new ListBatchQueryTemplate<>();
        return batch.batchQuery(lessonIds, new BatchQueryCallback<Long, List<Long>>() {
            @Override
            public List<Long> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<OrgStudentLesson> builder = createSqlBuilder("lessonId");
                builder.in("lessonId", querySet);
                builder.eq("userId", userId);
                builder.eq("orgId", orgId);
                builder.eq("delStatus", 0);
                return queryForList(builder, Long.class);
            }
        });
    }

    @Override
    public List<Long> getUserIds(Long lessonId, Long orgId) {
        SingleSqlBuilder<OrgStudentLesson> builder = createSqlBuilder("userId");
        builder.eq("lessonId", lessonId);
        builder.eq("orgId", orgId);
        builder.eq("delStatus", 0);
        return queryForList(builder, Long.class);
    }

    @Override
    public Map<Long, List<Long>> getStudentCount(Collection<Long> lessonIds, final Long orgId) {
        if (CollectionUtils.isEmpty(lessonIds)) {
            log.info("lessonIds = {}", lessonIds);
            return Maps.newHashMap();
        }
        BatchQueryTemplate<Long, List<OrgStudentLesson>> batch = new ListBatchQueryTemplate<>();
        List<OrgStudentLesson> list =
                batch.batchQuery(lessonIds, new BatchQueryCallback<Long, List<OrgStudentLesson>>() {
                    @Override
                    public List<OrgStudentLesson> doQuery(Collection<Long> querySet) {
                        SingleSqlBuilder<OrgStudentLesson> builder = createSqlBuilder("lessonId");
                        builder.in("lessonId", querySet);
                        builder.eq("orgId", orgId);
                        builder.eq("delStatus", 0);
                        return queryList(builder);
                    }
                });
        return CollectorUtil.group(list, new Function<OrgStudentLesson, Long>() {
            @Override
            public Long apply(OrgStudentLesson arg0) {
                return arg0.getLessonId();
            }
        }, new Function<OrgStudentLesson, Long>() {
            @Override
            public Long apply(OrgStudentLesson arg0) {
                return arg0.getUserId();
            }
        });
    }

    @Override
    public Map<Long, List<Long>> getLessonIdsOfStudents(final Long orgId, Collection<Long> userIds,
                                                        final Collection<Long> lessonIds) {
        if (CollectionUtils.isEmpty(userIds) || CollectionUtils.isEmpty(lessonIds)) {
            log.info("userIds = {} lessonIds={}", userIds, lessonIds);
            return Maps.newHashMap();
        }

        BatchQueryTemplate<Long, List<OrgStudentLesson>> batch = new ListBatchQueryTemplate<>();
        List<OrgStudentLesson> list = batch.batchQuery(userIds, new BatchQueryCallback<Long, List<OrgStudentLesson>>() {
            @Override
            public List<OrgStudentLesson> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<OrgStudentLesson> builder = createSqlBuilder("userId", "lessonId");
                builder.in("userId", querySet);
                if (lessonIds != null) {
                    builder.in("lessonId", lessonIds);
                }
                builder.eq("orgId", orgId);
                builder.eq("delStatus", 0);
                return queryList(builder);
            }
        });

        return CollectorUtil.group(list, new Function<OrgStudentLesson, Long>() {
            @Override
            public Long apply(OrgStudentLesson arg0) {
                return arg0.getUserId();
            }
        }, new Function<OrgStudentLesson, Long>() {
            @Override
            public Long apply(OrgStudentLesson arg0) {
                return arg0.getLessonId();
            }
        });
    }

    /**
     * 不管删除没删除，都查出来
     */
    @Override
    public List<OrgStudentLesson> getLessonsOfStudent(final Long orgId,final Long userId,
                                                        final Collection<Long> lessonIds) {
        if (CollectionUtils.isEmpty(lessonIds) || CollectionUtils.isEmpty(lessonIds)) {
        	log.info("lessonIds = {}", lessonIds);
        	return Lists.newArrayList();
        }
        SingleSqlBuilder<OrgStudentLesson> builder = createSqlBuilder();
        builder.in("lessonId", lessonIds);
        builder.eq("userId", userId);
        builder.eq("orgId", orgId);
        return queryList(builder);
    }

    @Override
    public Map<Long, Integer> queryLessonStudentCountMap(final Long orgId, Collection<Long> lessonIds) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        if (CollectionUtils.isEmpty(lessonIds)) {
            return Collections.emptyMap();
        }
        return new MapBatchQueryTemplate<Long, Long, Integer>().batchQuery(lessonIds,
                new BatchQueryCallback<Long, Map<Long, Integer>>() {
                    @Override
                    public Map<Long, Integer> doQuery(Collection<Long> lessonIds) {
                        SingleSqlBuilder<OrgStudentLesson> builder = createSqlBuilder();
                        builder.distinctCount("userId", "studentCount");
                        builder.eq("orgId", orgId);
                        builder.in("lessonId", lessonIds);
                        builder.eq("delStatus", 0);
                        builder.group("lessonId");
                        final Map<Long, Integer> result = Maps.newHashMap();
                        log.debug("query lesson student count sql:{} params:{}", builder.toSql(),
                                builder.collectConditionValue());
                        getNamedJdbcTemplate().query(builder.toSql(), builder.collectConditionValue(),
                                new RowCallbackHandler() {
                                    @Override
                                    public void processRow(ResultSet rs) throws SQLException {
                                        result.put(rs.getLong("lessonId"), rs.getInt("studentCount"));
                                    }
                                });
                        return result;
                    }
                });
    }

    @Override
    public Integer queryLessonStudentCount(Long orgId, Long lessonId) {
        SingleSqlBuilder<OrgStudentLesson> builder = createSqlBuilder();
        builder.distinctCount("userId", "studentCount");
        builder.eq("orgId", orgId);
        builder.eq("lessonId", lessonId);
        builder.eq("delStatus", 0);
        return queryForObject(builder, Integer.class);
    }

    @Override
    public Integer queryLessonStudentCount(Long orgId, Collection<Long> lessonIds) {
        if (CollectionUtils.isEmpty(lessonIds)) {
            return 0;
        }
        SingleSqlBuilder<OrgStudentLesson> builder = createSqlBuilder();
        builder.count("id");
        builder.eq("orgId", orgId);
        builder.in("lessonId", lessonIds);
        builder.eq("delStatus", 0);
        return queryForObject(builder, Integer.class);
    }

    @Override
    public List<Long> getLessonIdsOfStudent(Long orgId, Long userId) {
        SingleSqlBuilder<OrgStudentLesson> builder = createSqlBuilder("lessonId");
        builder.eq("orgId", orgId);
        builder.eq("userId", userId);
        builder.eq("delStatus", 0);
        return queryForList(builder, Long.class);
    }

    @Override
    public Integer getStudentLessonCountToday(Long orgId, Long userId) {
        Map<String, Object> paramMap = Maps.newHashMap();
        paramMap.put("orgId", orgId);
        paramMap.put("userId", userId);
        paramMap.put("startTime", DateUtil.getCurrentDate());
        paramMap.put("endTime", DateUtil.getDayDiff(1));
        String sql = "select count(ocl.id) from tts.org_student_lesson osl join tts.org_class_lesson ocl "
                + "on osl.lesson_id = ocl.id where ocl.org_id = :orgId " + "AND ocl.start_time > :startTime "
                + "AND ocl.end_time <= :endTime AND osl.student_id = :userId AND ocl.del_status=0 and osl.del_status = 0 ";
        return this.getNamedJdbcTemplate().queryForObject(sql, paramMap, Integer.class);
    }
    
    @Override
    public Integer getStudentLessonCountTodayByCourseId(Long orgId, Long userId,List<Long> courseIds) {
        if(CollectionUtils.isEmpty(courseIds)){
            return 0;
        }
        Map<String, Object> paramMap = Maps.newHashMap();
        paramMap.put("orgId", orgId);
        paramMap.put("userId", userId);
        paramMap.put("courseIds", courseIds);
        paramMap.put("startTime", DateUtil.getCurrentDate());
        paramMap.put("endTime", DateUtil.getDayDiff(1));
        String sql = "select count(ocl.id) from tts.org_student_lesson osl join tts.org_class_lesson ocl "
                + "on osl.lesson_id = ocl.id where ocl.org_id = :orgId " + "AND ocl.start_time > :startTime "
                + "AND ocl.end_time <= :endTime AND osl.student_id = :userId AND ocl.del_status=0 and osl.del_status = 0 and ocl.course_id IN (:courseIds )";
        return this.getNamedJdbcTemplate().queryForObject(sql, paramMap, Integer.class);
    }

    @Override
    public boolean isStudentInlesson(Long lessonId, Long userId) {
        SingleSqlBuilder<OrgStudentLesson> builder = createSqlBuilder("lessonId");
        builder.eq("lessonId", lessonId);
        builder.eq("userId", userId);
        builder.eq("delStatus", 0);
        List<Long> list = queryForList(builder, Long.class);
        if (CollectionUtils.isNotEmpty(list)) {
            return true;
        }
        return false;
    }

    @Override
    public Map<Long, Integer> getStudentLessonCountOfCourses(final Long orgId, final Long userId,
                                                             Collection<Long> courseIds) {

        if (CollectionUtils.isEmpty(courseIds)) {
            return Collections.emptyMap();
        }
        final Map<Long, Integer> map = Maps.newHashMap();
        String sql = "select count(ocl.id) as cnt,ocl.course_id from tts.org_student_lesson osl "
                + "join tts.org_class_lesson ocl ON osl.lesson_id = ocl.id "
                + "where  ocl.org_id = :orgId AND osl.student_id = :userId "
                + "AND ocl.course_id in(:courseIds) and osl.del_status = 0 group by(ocl.course_id)";
        Map<String, Object> paramMap = Maps.newHashMap();
        paramMap.put("orgId", orgId);
        paramMap.put("userId", userId);
        paramMap.put("courseIds", courseIds);
        
        this.getNamedJdbcTemplate().query(sql, paramMap, new RowCallbackHandler() {
            @Override
            public void processRow(ResultSet rs) throws SQLException {
                map.put(rs.getLong("course_id"), rs.getInt("cnt"));
            }
        });
        return map;
    }

    @Override
    public Map<Long, List<Long>> getLessonStudentList(Collection<Long> lessonIds) {
        if (CollectionUtils.isEmpty(lessonIds)) {
            return Maps.newHashMap();
        }
        BatchQueryTemplate<Long, List<OrgStudentLesson>> batch = new ListBatchQueryTemplate<>();
        List<OrgStudentLesson> list =
                batch.batchQuery(lessonIds, new BatchQueryCallback<Long, List<OrgStudentLesson>>() {
                    @Override
                    public List<OrgStudentLesson> doQuery(Collection<Long> querySet) {
                        SingleSqlBuilder<OrgStudentLesson> builder = createSqlBuilder("lessonId", "userId");
                        builder.in("lessonId", querySet);
                        builder.eq("delStatus", 0);
                        return queryList(builder);
                    }
                });
        return CollectorUtil.group(list, new Function<OrgStudentLesson, Long>() {
            @Override
            public Long apply(OrgStudentLesson arg0) {
                return arg0.getLessonId();

            }
        }, new Function<OrgStudentLesson, Long>() {
            @Override
            public Long apply(OrgStudentLesson arg0) {
                return arg0.getUserId();
            }
        });
    }
    
    @Override
    public List<OrgStudentLesson> getLessonStudentListDelStatus(Collection<Long> lessonIds) {
        SingleSqlBuilder<OrgStudentLesson> builder = createSqlBuilder();
        builder.in("lessonId", lessonIds);
        builder.eq("delStatus", 1);
        return queryList(builder);
    }

    @Override
    public Map<Long, List<Long>> getStudentLessonList(Collection<Long> studentIds) {
        if (CollectionUtils.isEmpty(studentIds)) {
            return Maps.newHashMap();
        }
        BatchQueryTemplate<Long, List<OrgStudentLesson>> batch = new ListBatchQueryTemplate<>();
        List<OrgStudentLesson> list =
                batch.batchQuery(studentIds, new BatchQueryCallback<Long, List<OrgStudentLesson>>() {
                    @Override
                    public List<OrgStudentLesson> doQuery(Collection<Long> querySet) {
                        SingleSqlBuilder<OrgStudentLesson> builder = createSqlBuilder("lessonId", "userId");
                        builder.in("userId", querySet);
                        builder.eq("delStatus", 0);
                        return queryList(builder);
                    }
                });
        return CollectorUtil.group(list, new Function<OrgStudentLesson, Long>() {
            @Override
            public Long apply(OrgStudentLesson arg0) {
                return arg0.getUserId();

            }
        }, new Function<OrgStudentLesson, Long>() {
            @Override
            public Long apply(OrgStudentLesson arg0) {
                return arg0.getLessonId();
            }
        });
    }

    @Override
    public List<Long> getUserIdsByLessonIds(Long orgId, Collection<Long> lessonIds) {
        if (CollectionUtils.isEmpty(lessonIds)) {
            return Lists.newArrayList();
        }
        BatchQueryTemplate<Long, List<Long>> batch = new ListBatchQueryTemplate<>();
        return batch.batchQuery(lessonIds, new BatchQueryCallback<Long, List<Long>>() {
            @Override
            public List<Long> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<OrgStudentLesson> builder = createSqlBuilder("userId");
                builder.in("lessonId", querySet);
                builder.eq("delStatus", 0);
                return queryForList(builder, Long.class);
            }
        });
    }
    

    @Override
    public Map<Long, List<Long>> getStudentLessonByStartTime(Date startTime, Date endTime) {
        StringBuilder sb = new StringBuilder();
        sb.append(
                "select student.lesson_id as lessonId, student.student_id as userId from tts.org_student_lesson student ");
        sb.append("join tts.org_class_lesson lesson on student.lesson_id = lesson.id and lesson.del_status=0 and student.del_status=0 ");
        sb.append(" and lesson.start_time >=:startTime and lesson.start_time <:endTime");
        Map<String, Object> param = Maps.newHashMap();
        param.put("startTime", startTime);
        param.put("endTime", endTime);
        final Map<Long, List<Long>> result = Maps.newHashMap();
        this.getNamedJdbcTemplate().query(sb.toString(), param, new RowCallbackHandler() {
            @Override
            public void processRow(ResultSet rs) throws SQLException {
                Long userId = rs.getLong("userId");
                Long lessonId = rs.getLong("lessonId");
                if (!result.containsKey(lessonId)) {
                    result.put(lessonId, Lists.<Long>newArrayList());
                }
                result.get(lessonId).add(userId);
            }
        });
        return result;
    }

    @Override
    public Map<Long, List<Long>> getStudentLessonByEndTime(Date startTime, Date endTime) {
        StringBuilder sb = new StringBuilder();
        sb.append(
                "select student.lesson_id as lessonId, student.student_id as userId from tts.org_student_lesson student ");
        sb.append("join tts.org_class_lesson lesson on student.lesson_id = lesson.id and lesson.del_status=0 and student.del_status = 0");
        sb.append(" and lesson.end_time >=:startTime and lesson.end_time <:endTime");
        Map<String, Object> param = Maps.newHashMap();
        param.put("startTime", startTime);
        param.put("endTime", endTime);
        final Map<Long, List<Long>> result = Maps.newHashMap();
        this.getNamedJdbcTemplate().query(sb.toString(), param, new RowCallbackHandler() {
            @Override
            public void processRow(ResultSet rs) throws SQLException {
                Long userId = rs.getLong("userId");
                Long lessonId = rs.getLong("lessonId");
                if (!result.containsKey(lessonId)) {
                    result.put(lessonId, Lists.<Long>newArrayList());
                }
                result.get(lessonId).add(userId);
            }
        });
        return result;
    }

    @Override
    public Map<Long, Integer> getStudentCountOfTeacher(Long orgId, Collection<Long> teacherIds, Date endTime) {
        Preconditions.checkArgument(orgId != null, "orgId is null!");
        if (CollectionUtils.isEmpty(teacherIds)) {
            return Maps.newHashMap();
        }
        Map<String, Object> params = Maps.newHashMap();
        StringBuilder sb = new StringBuilder();
        sb.append(
                "select count(student.student_id) as num, teacher.teacher_id as teacherId from tts.org_student_lesson student ");
        sb.append("join tts.org_teacher_lesson teacher on student.lesson_id = teacher.lesson_id ");
        sb.append("join tts.org_class_lesson lesson on student.lesson_id = lesson.id ");
        sb.append("where student.org_id =:orgId ");
        sb.append("and teacher.teacher_id in (:teacherIds) ");
        params.put("orgId", orgId);
        params.put("teacherIds", teacherIds);
        sb.append("and lesson.end_time <=:startTime and student.del_status = 0 ");
        if (endTime != null) {
            sb.append("and lesson.end_time >:endTime ");
            params.put("endTime", endTime);
        }
        params.put("startTime", new Date());

        sb.append(" group by (teacher.teacher_id)");
        final Map<Long, Integer> result = Maps.newHashMap();
        getNamedJdbcTemplate().query(sb.toString(), params, new RowCallbackHandler() {
            @Override
            public void processRow(ResultSet rs) throws SQLException {
                result.put(rs.getLong("teacherId"), rs.getInt("num"));
            }
        });
        return result;
    }

    @Override
    public Integer getFinishLessonStudentCount(Long orgId) {
        StringBuilder sb = new StringBuilder();
        sb.append("select count(student.id) as num from tts.org_student_lesson student ");
        sb.append("join tts.org_class_lesson lesson on student.lesson_id = lesson.id ");
        sb.append("where student.org_id =:orgId and lesson.end_time<=:endTime and student.del_status = 0 ");
        Map<String, Object> params = Maps.newHashMap();
        params.put("orgId", orgId);
        params.put("endTime", new Date());
        return getNamedJdbcTemplate().query(sb.toString(), params, new ResultSetExtractor<Integer>() {
            @Override
            public Integer extractData(ResultSet rs) throws SQLException, DataAccessException {
                if (rs.next()) {
                    return rs.getInt("num");
                }
                return 0;
            }
        });
    }

    @Override
    public List<OrgStudentLesson> queryStudentsByLessonId(Long orgId, Long lessonId) {
        SingleSqlBuilder<OrgStudentLesson> builder = createSqlBuilder();
        if (orgId != null) {
            builder.eq("orgId", orgId);
        }
        builder.eq("lessonId", lessonId);
        builder.eq("delStatus", 0);
        return queryList(builder);
    }
    
    @Override
    public List<OrgStudentLesson> getOrgStudentLessonsByLessonIds(Collection<Long> lessonIds){
        
        SingleSqlBuilder<OrgStudentLesson> builder = createSqlBuilder();
        builder.in("lessonId", lessonIds);
        builder.eq("delStatus", 0);
        List<OrgStudentLesson> orgStudentLessons = queryList(builder);
        return orgStudentLessons;
        
    }

    @Override
    public List<OrgStudentLesson> getStudentLessonsByPage(Date updateTime, PageDto pageDto) {
        SingleSqlBuilder<OrgStudentLesson> builder = createSqlBuilder();
        if(updateTime!=null) {
            builder.ge("updateTime", updateTime);
        }
        builder.eq("delStatus", 0);
        builder.setPage(pageDto);
        return queryList(builder);
    }

    @Override
    public void batchUpdateStatus(Collection<Long> lessonIds, int status) {
        String sql = "update tts.org_student_lesson set start_status=:status,update_time=:updateTime " +
                "where lesson_id in (:ids) AND start_status != :status";
        Map<String,Object> map = new HashMap<>();
        map.put("status",status);
        map.put("updateTime",new Date());
        map.put("ids",lessonIds);
        getNamedJdbcTemplate().update(sql, map);
    }

    @Override
    public void batchUpdateKexiaoStatus(Collection<Long> lessonIds, int status) {
        String sql = "update tts.org_student_lesson set kexiao_status=:status,update_time=:updateTime " +
                "where lesson_id in (:ids) AND kexiao_status != :status";
        Map<String,Object> map = new HashMap<>();
        map.put("status",status);
        map.put("updateTime", new Date());
        map.put("ids", lessonIds);
        getNamedJdbcTemplate().update(sql, map);
    }


    @Override
    public void batchUpdateKexiaoStatusBySign(List<OrgLessonSign> signs, int status) {
        if(signs==null || signs.size()<1){
            return;
        }
        log.info("[Param] sign={},status={}",signs,status);
        String sql = "update tts.org_student_lesson set kexiao_status=:status,update_time=:updateTime " +
                "where lesson_id = :lessonId AND student_id=:userId AND kexiao_status != :status";
        Map<String,Object>[] list = new HashMap[signs.size()];
        for (int i=0;i<list.length;i++) {
            Map<String, Object> map = new HashMap<>();
            map.put("status", status);
            map.put("updateTime", new Date());
            map.put("lessonId", signs.get(i).getLessonId());
            map.put("userId", signs.get(i).getUserId());
            list[i] = map;
        }
        getNamedJdbcTemplate().batchUpdate(sql, list);

    }

    @Override
    public OrgStudentLessonCounter getStudentLessonCounter(final long orgId,long userId,long courseId) {
        String sql = "select sl.student_id,cl.course_id,count(1) as total,sum(kexiao_status) as finished from " +
                " tts.org_student_lesson sl left join tts.org_class_lesson cl on sl.lesson_id = cl.id where sl.org_id=:orgId AND sl.student_id=:userId and cl.course_id=:courseId and sl.del_status = 0 " +
                " group by sl.student_id,cl.course_id";
        Map<String,Object> param = new HashMap<>();
        param.put("userId",userId);
        param.put("orgId",orgId);
        param.put("courseId", courseId);
        List<OrgStudentLessonCounter> counters = getNamedJdbcTemplate().query(sql, param, new RowMapper<OrgStudentLessonCounter>() {
            @Override
            public OrgStudentLessonCounter mapRow(ResultSet rs, int rowNum) throws SQLException {
                OrgStudentLessonCounter counter = new OrgStudentLessonCounter();
                counter.setCourseId(rs.getLong("course_id"));
                counter.setUserId(rs.getLong("student_id"));
                counter.setTotalCount(rs.getLong("total"));
                counter.setFinishedCount(rs.getLong("finished"));
                counter.setOrgId(orgId);
                return counter;
            }
        });
        if(counters!=null && counters.size()>0) {
            return counters.get(0);
        }else {
            return null;
        }
    }

    @Override
    public List<OrgStudentLessonCounter> getStudentLessonCounterList(final long orgId,PageDto pageDto) {
        int start = (pageDto.getPageNum()-1)*pageDto.getPageSize();
        String sql = "select sl.student_id,cl.course_id,count(1) as total,sum(kexiao_status) as finished from " +
                " tts.org_student_lesson sl left join tts.org_class_lesson cl on sl.lesson_id = cl.id where sl.org_id=:orgId and sl.del_status = 0 " +
                " group by sl.student_id,cl.course_id limit :start,:limit";
        Map<String,Object> param = new HashMap<>();
        param.put("orgId",orgId);
        param.put("start",start);
        param.put("limit",pageDto.getPageSize());
        List<OrgStudentLessonCounter> counters = getNamedJdbcTemplate().query(sql, param, new RowMapper<OrgStudentLessonCounter>() {
            @Override
            public OrgStudentLessonCounter mapRow(ResultSet rs, int rowNum) throws SQLException {
                OrgStudentLessonCounter counter = new OrgStudentLessonCounter();
                counter.setCourseId(rs.getLong("course_id"));
                counter.setUserId(rs.getLong("student_id"));
                counter.setTotalCount(rs.getLong("total"));
                counter.setFinishedCount(rs.getLong("finished"));
                counter.setOrgId(orgId);
                return counter;
            }
        });
        return counters;
    }
    
    @Override
    public Map<Long, Integer> getStudentsLeftLessonMap(Long orgId, Long courseId, Collection<Long> userIds) {
        final Map<Long, Integer> map = Maps.newHashMap();
        
        String sql = "select sl.student_id, count(sl.id) as leftLessons, cl.course_id from tts.org_student_lesson sl left join tts.org_class_lesson cl on sl.lesson_id = cl.id "
                + " where sl.org_id=:orgId and sl.student_id in (:userIds) and cl.course_id=:courseId and sl.del_status = 0 and cl.del_status = 0 and cl.start_time > now() "
                + " group by sl.student_id, cl.course_id"; 
        Map<String,Object> param = new HashMap<>();
        param.put("userIds",userIds);
        param.put("orgId",orgId);
        param.put("courseId", courseId);
        
        getNamedJdbcTemplate().query(sql, param, new RowCallbackHandler() {
            
            @Override
            public void processRow(ResultSet rs) throws SQLException {
                map.put(rs.getLong("student_id"), rs.getInt("leftLessons"));
            }
        });
        
        return map;
    }
    
    @Override
    public Map<Long, List<Long>> getStudentCourseLessonMap(Long orgId, Long courseId, Collection<Long> userIds) {
        final Map<Long, List<Long>> map = Maps.newHashMap();
        String sql = "select sl.student_id, sl.lesson_id from tts.org_student_lesson sl left join tts.org_class_lesson cl on sl.lesson_id = cl.id " +
                " where sl.org_id=:orgId and sl.student_id in (:userIds) and cl.course_id=:courseId and sl.del_status = 0 and cl.del_status = 0"; 
        Map<String,Object> param = new HashMap<>();
        param.put("userIds",userIds);
        param.put("orgId",orgId);
        param.put("courseId", courseId);
        
        getNamedJdbcTemplate().query(sql, param, new RowCallbackHandler() {
            @Override
            public void processRow(ResultSet rs) throws SQLException {
                Long userId = rs.getLong("student_id");
                Long lessonId = rs.getLong("lesson_id");
                List<Long> lessons = map.get(userId);
                if (lessons == null) {
                    lessons = Lists.newArrayList();
                    map.put(userId, lessons);
                }
                lessons.add(lessonId);
            }
        });
        return map;
    }

    @Override
    public List<OrgStudentLesson> getOrgStudentLessonsByCourseIds(Collection<Long> courseIds) {
        String sql = "select sl.* from tts.org_student_lesson sl left join tts.org_class_lesson cl on sl.lesson_id = cl.id where cl.course_id in (:courseIds) AND sl.del_status=0";
        Map<String,Object> param = new HashMap<>();
        param.put("courseIds",courseIds);
        List<OrgStudentLesson> counters = getNamedJdbcTemplate().query(sql, param, new RowMapper<OrgStudentLesson>() {
            @Override
            public OrgStudentLesson mapRow(ResultSet rs, int rowNum) throws SQLException {
                OrgStudentLesson lesson = new OrgStudentLesson();
                lesson.setLessonId(rs.getLong("lesson_id"));
                lesson.setKexiaoStatus(rs.getInt("kexiao_status"));
                lesson.setId(rs.getLong("id"));
                lesson.setUserId(rs.getLong("student_id"));
                return lesson;
            }
        });
        return counters;
    }
    
    @Override
    public List<StudentClasHourDocument> queryStudentClassHours(Set<Long> studentIds, Long orgId) {
        if (GenericsUtils.isNullOrEmpty(studentIds)) {
            return GenericsUtils.emptyList();
        }
        String sql =
            "select sl.student_id as userId,cl.course_id as courseId,count(1) as total,sum(kexiao_status) as finished,sum(if(kexiao_status=0,1,0)) as leftCount "
                + "from tts.org_student_lesson sl inner join  tts.org_class_lesson cl on sl.lesson_id = cl.id "
                + "where sl.del_status= 0 and sl.student_id in (:studentIds)  group by sl.student_id,cl.course_id";

        Map<String, Object> param = Maps.newHashMap();
        param.put("studentIds", studentIds);
        final List<StudentClasHourDocument> ret = Lists.newArrayList();
        this.getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<StudentClasHourDocument>() {
            @Override
            public StudentClasHourDocument extractData(ResultSet rs) throws SQLException, DataAccessException {
                while (rs.next()) {
                    Long userId = rs.getLong("userid");
                    Long courseId = rs.getLong("courseId");
                    Long total = rs.getLong("total");
                    Long finished = rs.getLong("finished");
                    Long leftCount = rs.getLong("leftCount");
                    StudentClasHourDocument scd = new StudentClasHourDocument();
                    scd.setUserId(userId);
                    scd.setCourseId(courseId);
                    scd.setTotal(total);
                    scd.setFinished(finished);
                    scd.setLeftCount(leftCount);
                    ret.add(scd);
                }
                return null;
            }
        });
        return ret;
    }
    

    
}