package com.baijia.tianxiao.dal.org.dao.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.core.RowCallbackHandler;
import org.springframework.stereotype.Repository;

import com.baijia.tianxiao.dal.org.dao.OrgStudentCourseDao;
import com.baijia.tianxiao.dal.org.dto.StudentClassHourStatusDocument;
import com.baijia.tianxiao.dal.org.po.OrgStudentCourse;
import com.baijia.tianxiao.sqlbuilder.SingleSqlBuilder;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.sqlbuilder.support.JdbcTemplateDaoSupport;
import com.baijia.tianxiao.sqlbuilder.util.Expressions;
import com.baijia.tianxiao.util.CollectorUtil;
import com.baijia.tianxiao.util.GenericsUtils;
import com.baijia.tianxiao.util.query.BatchQueryCallback;
import com.baijia.tianxiao.util.query.BatchQueryTemplate;
import com.baijia.tianxiao.util.query.ListBatchQueryTemplate;
import com.baijia.tianxiao.util.query.MapBatchQueryTemplate;
import com.google.common.base.Function;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

@Repository
@Slf4j
public class OrgStudentCourseDaoImpl extends JdbcTemplateDaoSupport<OrgStudentCourse> implements OrgStudentCourseDao {

    public OrgStudentCourseDaoImpl() {
        super(OrgStudentCourse.class);
    }

    @Override
    public List<Long> getStudents(Long orgId, Long courseId) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder("userId");
        builder.eq("courseId", courseId);
        builder.eq("orgId", orgId);
        builder.eq("delStatus", 0);
        return queryForList(builder, Long.class);
    }

    @Override
    public OrgStudentCourse getStudentCourse(@NonNull Long orgId, @NonNull Long courseId, @NonNull Long userId) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
        builder.eq("courseId", courseId);
        builder.eq("orgId", orgId);
        // builder.eq("delStatus", 0);
        if (userId != null) {
            builder.eq("userId", userId);
        }

        return this.uniqueResult(builder);
    }

    @Override
    public List<OrgStudentCourse> getStudentListByCourseId(Long orgId, Long courseId, String...queryProps) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
        builder.eq("courseId", courseId);
        builder.add(Expressions.or(Expressions.eq("orgId", orgId), Expressions.eq("orgId", 0)));
        builder.eq("delStatus", 0);
        builder.select(queryProps);
        builder.desc("createTime");
        return queryList(builder);

    }

    @Override
    public List<OrgStudentCourse> getStudentListByCourseId(Long orgId, Long courseId, PageDto pageDto) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
        builder.eq("courseId", courseId);
        builder.add(Expressions.or(Expressions.eq("orgId", orgId), Expressions.eq("orgId", 0)));
        builder.eq("delStatus", 0);
        // 学生列表不包含转班和退班的学生
        builder.eq("status", 0);
        builder.desc("createTime");

        builder.setPage(pageDto);
        return queryList(builder);
    }

    @Override
    public void deleteOrgCourseStudent(Long orgId, Long courseId, Long userId, Integer status) {
        Map<String, Object> condition = Maps.newHashMap();
        condition.put("orgId", orgId);
        condition.put("courseId", courseId);
        condition.put("userId", userId);
        OrgStudentCourse obj = new OrgStudentCourse();
        obj.setStatus(status);
        this.update(condition, obj, "status");
    }

    @Override
    public void batchDeleteOrgCourseStudent(Long orgId, Long courseId, Collection<Long> userIds, Integer status) {
        Map<String, Object> condition = Maps.newHashMap();
        condition.put("orgId", orgId);
        condition.put("courseId", courseId);
        condition.put("userId", userIds);
        OrgStudentCourse obj = new OrgStudentCourse();
        obj.setStatus(status);
        this.update(condition, obj, "status");
    }

    @Override
    public List<OrgStudentCourse> getStudentListByStudentId(Long userId, String...queryProps) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
        builder.eq("userId", userId);
        builder.select(queryProps);
        builder.desc("createTime");
        return queryList(builder);
    }

    @Override
    public Map<Long, List<OrgStudentCourse>> getStudentMapBycourseIds(Collection<Long> courseIds,
        final String...queryProps) {
        BatchQueryTemplate<Long, List<OrgStudentCourse>> batchQueryTemplate = new ListBatchQueryTemplate<>();
        List<OrgStudentCourse> courses =
            batchQueryTemplate.batchQuery(courseIds, new BatchQueryCallback<Long, List<OrgStudentCourse>>() {

                @Override
                public List<OrgStudentCourse> doQuery(Collection<Long> querySet) {
                    SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
                    builder.in("courseId", querySet);
                    builder.eq("delStatus", 0);
                    builder.select(queryProps);
                    return queryList(builder);
                }
            });
        if (CollectionUtils.isEmpty(courses)) {
            return Collections.emptyMap();
        }
        Map<Long, List<OrgStudentCourse>> map = Maps.newHashMap();
        for (OrgStudentCourse orgStudentCourse : courses) {
            List<OrgStudentCourse> orgStudentCourses = map.get(orgStudentCourse.getCourseId());
            if (orgStudentCourses == null) {
                orgStudentCourses = Lists.newArrayList();
                map.put(orgStudentCourse.getCourseId(), orgStudentCourses);
            }
            orgStudentCourses.add(orgStudentCourse);
        }
        return map;
    }

    @Override
    public Map<Long, List<OrgStudentCourse>> getStudentMapBycourseIds(Collection<Long> courseIds, final int status,
        final String...queryProps) {
        BatchQueryTemplate<Long, List<OrgStudentCourse>> batchQueryTemplate = new ListBatchQueryTemplate<>();
        List<OrgStudentCourse> courses =
            batchQueryTemplate.batchQuery(courseIds, new BatchQueryCallback<Long, List<OrgStudentCourse>>() {

                @Override
                public List<OrgStudentCourse> doQuery(Collection<Long> querySet) {
                    SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
                    builder.in("courseId", querySet);
                    builder.eq("status", status);
                    builder.eq("delStatus", 0);
                    builder.select(queryProps);
                    return queryList(builder);
                }
            });
        if (CollectionUtils.isEmpty(courses)) {
            return Collections.emptyMap();
        }
        Map<Long, List<OrgStudentCourse>> map = Maps.newHashMap();
        for (OrgStudentCourse orgStudentCourse : courses) {
            List<OrgStudentCourse> orgStudentCourses = map.get(orgStudentCourse.getCourseId());
            if (orgStudentCourses == null) {
                orgStudentCourses = Lists.newArrayList();
                map.put(orgStudentCourse.getCourseId(), orgStudentCourses);
            }
            orgStudentCourses.add(orgStudentCourse);
        }
        return map;
    }

    @Override
    public Map<Long, List<Long>> getStudentIdMapBycourseIds(Collection<Long> courseIds, final Long orgId) {
        BatchQueryTemplate<Long, List<OrgStudentCourse>> batchQueryTemplate = new ListBatchQueryTemplate<>();
        List<OrgStudentCourse> courses =
            batchQueryTemplate.batchQuery(courseIds, new BatchQueryCallback<Long, List<OrgStudentCourse>>() {

                @Override
                public List<OrgStudentCourse> doQuery(Collection<Long> querySet) {
                    SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
                    builder.in("courseId", querySet);
                    builder.eq("delStatus", 0);
                    builder.eq("status", 0);
                    builder.eq("orgId", orgId);
                    builder.select("courseId", "userId");
                    return queryList(builder);
                }
            });
        return CollectorUtil.group(courses, new Function<OrgStudentCourse, Long>() {
            @Override
            public Long apply(OrgStudentCourse arg0) {
                return arg0.getCourseId();
            }
        }, new Function<OrgStudentCourse, Long>() {
            @Override
            public Long apply(OrgStudentCourse arg0) {
                return arg0.getUserId();
            }
        });

    }

    @Override
    public Map<Long, List<Long>> getStudentIdMapBycourseIdsNoStatus(Collection<Long> courseIds, final Long orgId) {
        BatchQueryTemplate<Long, List<OrgStudentCourse>> batchQueryTemplate = new ListBatchQueryTemplate<>();
        List<OrgStudentCourse> courses =
            batchQueryTemplate.batchQuery(courseIds, new BatchQueryCallback<Long, List<OrgStudentCourse>>() {

                @Override
                public List<OrgStudentCourse> doQuery(Collection<Long> querySet) {
                    SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
                    builder.in("courseId", querySet);
                    builder.eq("delStatus", 0);

                    builder.eq("orgId", orgId);
                    builder.select("courseId", "userId");
                    return queryList(builder);
                }
            });
        return CollectorUtil.group(courses, new Function<OrgStudentCourse, Long>() {
            @Override
            public Long apply(OrgStudentCourse arg0) {
                return arg0.getCourseId();
            }
        }, new Function<OrgStudentCourse, Long>() {
            @Override
            public Long apply(OrgStudentCourse arg0) {
                return arg0.getUserId();
            }
        });

    }

    @Override
    public Map<Long, List<Long>> getCourseIdMapByStuIds(Collection<Long> stuIds, final Long orgId) {
        BatchQueryTemplate<Long, List<OrgStudentCourse>> batchQueryTemplate = new ListBatchQueryTemplate<>();
        List<OrgStudentCourse> courses =
            batchQueryTemplate.batchQuery(stuIds, new BatchQueryCallback<Long, List<OrgStudentCourse>>() {

                @Override
                public List<OrgStudentCourse> doQuery(Collection<Long> querySet) {
                    SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
                    builder.in("userId", querySet);
                    builder.eq("orgId", orgId);
                    builder.eq("status", 0);
                    builder.select("courseId", "userId");
                    return queryList(builder);
                }
            });
        return CollectorUtil.group(courses, new Function<OrgStudentCourse, Long>() {
            @Override
            public Long apply(OrgStudentCourse arg0) {
                return arg0.getUserId();
            }
        }, new Function<OrgStudentCourse, Long>() {
            @Override
            public Long apply(OrgStudentCourse arg0) {
                return arg0.getCourseId();
            }
        });
    }

    @Override
    public Map<Long, Integer> getCurStudentCntByCourseIds(final Long orgId, Collection<Long> courseIds,
        final String...queryProps) {

        MapBatchQueryTemplate<Long, Long, Integer> batchTemplate = new MapBatchQueryTemplate<>();
        return batchTemplate.batchQuery(courseIds, new BatchQueryCallback<Long, Map<Long, Integer>>() {
            @Override
            public Map<Long, Integer> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
                builder.count("id");
                builder.add(Expressions.or(Expressions.eq("orgId", orgId), Expressions.eq("orgId", 0)));
                builder.eq("delStatus", 0);
                builder.group("courseId");
                builder.in("courseId", querySet);
                List<Map<String, Object>> queryList = query2ListMap(builder);

                if (CollectionUtils.isNotEmpty(queryList)) {
                    Map<Long, Integer> result = Maps.newHashMap();
                    for (Map<String, Object> objMap : queryList) {
                        result.put(((Number) objMap.get("courseId")).longValue(),
                            ((Number) objMap.get("cnt_id")).intValue());
                    }
                    return result;
                } else {
                    return Collections.emptyMap();
                }
            }
        });

    }

    @Override
    public void updateStudentName(Long userId, String studentName) {

        Map<String, Object> params = Maps.newHashMap();
        params.put("userId", userId);
        params.put("studentName", studentName);
        this.getNamedJdbcTemplate()
            .update("update tts.org_student_course set student_name = :studentName where student_id = :userId", params);
    }

    @Override
    public List<Long> getOrgCourseIds(Long orgId, List<Long> courseIds, boolean ascByStuCnt, PageDto pageDto) {
        if (CollectionUtils.isEmpty(courseIds)) {
            return Collections.emptyList();
        }
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder("courseId");
        builder.count("id", "cnt");
        builder.in("courseId", courseIds);
        builder.eq("orgId", orgId);
        builder.eq("delStatus", 0);
        builder.eq("status", 0);
        builder.eq("tradeComplete", 1);
        builder.group("courseId");

        if (ascByStuCnt) {
            builder.asc("cnt");
        } else {
            builder.desc("cnt");
        }
        if (pageDto != null) {
            builder.setPage(pageDto);
        }
        return this.queryForList(builder, Long.class);
    }

    @Override
    public List<Long> getStudents(Long orgId, Long courseId, Integer status) {

        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder("userId");
        builder.eq("courseId", courseId);
        builder.eq("orgId", orgId);
        builder.eq("status", status);
        builder.eq("delStatus", 0);

        return this.queryForList(builder, Long.class);

    }

    @Override
    public Map<Long, Integer> getStudentCount(final Long orgId, Collection<Long> courseIds) {
        return getStudentCount(orgId, courseIds, null);
    }

    @Override
    public Map<Long, Integer> getStudentCount(final Long orgId, Collection<Long> courseIds,
        final Boolean tradeComplete) {
        if (CollectionUtils.isEmpty(courseIds)) {
            return Maps.newHashMap();
        }
        return new MapBatchQueryTemplate<Long, Long, Integer>().batchQuery(courseIds,
            new BatchQueryCallback<Long, Map<Long, Integer>>() {
                @Override
                public Map<Long, Integer> doQuery(Collection<Long> querySet) {
                    SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder("courseId");
                    builder.count("id", "num");
                    builder.in("courseId", querySet);
                    builder.eq("orgId", orgId);
                    builder.eq("delStatus", 0);
                    builder.eq("status", 0);
                    if (tradeComplete != null) {
                        builder.eq("tradeComplete", tradeComplete ? 1 : 0);
                    }
                    builder.group("courseId");
                    final Map<Long, Integer> result = Maps.newHashMap();
                    getNamedJdbcTemplate().query(builder.toSql(), builder.collectConditionValue(),
                        new RowCallbackHandler() {
                            @Override
                            public void processRow(ResultSet rs) throws SQLException {
                                result.put(rs.getLong("courseId"), rs.getInt("num"));
                            }
                        });
                    return result;
                }
            });
    }
    
    @Override
    public Map<Long, Integer> getStudentCountNoStatus(final Long orgId, Collection<Long> courseIds,
        final Boolean tradeComplete) {
        if (CollectionUtils.isEmpty(courseIds)) {
            return Maps.newHashMap();
        }
        return new MapBatchQueryTemplate<Long, Long, Integer>().batchQuery(courseIds,
            new BatchQueryCallback<Long, Map<Long, Integer>>() {
                @Override
                public Map<Long, Integer> doQuery(Collection<Long> querySet) {
                    SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder("courseId");
                    builder.count("id", "num");
                    builder.in("courseId", querySet);
                    builder.eq("orgId", orgId);
                    builder.eq("delStatus", 0);
                    
                    if (tradeComplete != null) {
                        builder.eq("tradeComplete", tradeComplete ? 1 : 0);
                    }
                    builder.group("courseId");
                    final Map<Long, Integer> result = Maps.newHashMap();
                    getNamedJdbcTemplate().query(builder.toSql(), builder.collectConditionValue(),
                        new RowCallbackHandler() {
                            @Override
                            public void processRow(ResultSet rs) throws SQLException {
                                result.put(rs.getLong("courseId"), rs.getInt("num"));
                            }
                        });
                    return result;
                }
            });
    }

    @Override
    public List<OrgStudentCourse> getOrgCourseIds(Long orgId, Long userId, Integer status, PageDto pageDto) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
        if (userId != null) {
            builder.eq("userId", userId);
        }
        builder.eq("orgId", orgId);
        if (status != null) {
            builder.eq("status", status);
        }

        if (pageDto != null) {
            builder.setPage(pageDto);
        }
        builder.eq("delStatus", 0);
        builder.asc("createTime");
        return this.queryList(builder);
    }

    @Override
    public List<OrgStudentCourse> getOrgCourseIds(Long orgId, Long userId, PageDto pageDto) {
        return getOrgCourseIds(orgId, userId, null, pageDto);
    }

    @Override
    public List<OrgStudentCourse> getOrgCourseByOrgId(Long orgId, Collection<Long> courseIds, PageDto pageDto) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder("courseId");
        builder.eq("orgId", orgId);
        builder.eq("delStatus", 0);
        if (courseIds != null && !courseIds.isEmpty()) {
            builder.in("courseId", courseIds);
        }
        builder.group("courseId");
        builder.desc("createTime");
        if (pageDto != null) {
            builder.setPage(pageDto);
        }
        return this.queryList(builder);
    }

    @Override
    public List<OrgStudentCourse> getOrgCourseByOrgIdAndIds(Long orgId, List<Long> courseIds) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.eq("delStatus", 0);
        builder.in("courseId", courseIds);
        builder.eq("status", 0); // 找不到常量 0不包括转班、退班
        return this.queryList(builder);
    }

    @Override
    public Integer getStudentCourseCount(Long orgId, Long userId, Integer status) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
        builder.count("id");
        builder.eq("userId", userId);
        builder.eq("orgId", orgId);
        if (status != null) {
            builder.eq("status", status);
        }
        return this.queryForObject(builder, Integer.class);
    }

    @Override
    public Integer getStudentSignupCourseCount(Long orgId, Long userId, Integer status) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
        builder.distinctCount("courseId");
        builder.eq("userId", userId);
        builder.eq("orgId", orgId);
        builder.eq("delStatus", 0);
        if (status != null) {
            builder.eq("status", status);
        }
        return this.queryForObject(builder, Integer.class);
    }

    @Override
    public List<Long> getStudentCourseIds(Long orgId, Long userId, Integer status) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder("courseId");
        builder.eq("userId", userId);
        builder.eq("orgId", orgId);
        builder.eq("delStatus", 0);
        if (status != null) {
            builder.eq("status", status);
        }
        return this.queryForList(builder, Long.class);
    }

    @Override
    public List<Long> getStudentCourseIdsByStatus(Long orgId, Long userId, List<Integer> status) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder("courseId");
        builder.eq("userId", userId);
        builder.eq("orgId", orgId);
        if (status != null) {
            builder.in("status", status);
        }
        return this.queryForList(builder, Long.class);
    }

    @Override
    public List<Long> getStudentIdsByCourseIds(final Long orgId, Collection<Long> courseIds) {
        if (CollectionUtils.isEmpty(courseIds)) {
            return Lists.newArrayList();
        }
        BatchQueryTemplate<Long, List<Long>> batch = new ListBatchQueryTemplate<>();
        return batch.batchQuery(courseIds, new BatchQueryCallback<Long, List<Long>>() {
            @Override
            public List<Long> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder("userId");
                builder.eq("orgId", orgId);
                builder.in("courseId", querySet);
                return queryForList(builder, Long.class);
            }
        });
    }

    @Override
    public void refreshOrgStudentCourse(Long orgId, Long id) {
        StringBuilder sb =
            new StringBuilder("update tts.org_student_course set del_status =1 where id > :id and  org_id = :orgId");
        Map param = Maps.newHashMap();
        param.put("orgId", orgId);
        param.put("id", id);
        this.getNamedJdbcTemplate().update(sb.toString(), param);
    }

    @Override
    public void deleteByOrgCourseId(Integer orgId, Integer courseId) {
        StringBuilder sb = new StringBuilder(
            "update tts.org_student_course set del_status =1 where course_id = :courseId and  org_id = :orgId");
        Map param = Maps.newHashMap();
        param.put("orgId", orgId);
        param.put("courseId", courseId);
        this.getNamedJdbcTemplate().update(sb.toString(), param);
    }

    @Override
    public List<Long> getEnrollClassUserIdByUserIds(final Long orgId, Collection<Long> userIds) {

        BatchQueryTemplate<Long, List<Long>> batch = new ListBatchQueryTemplate<>();
        return batch.batchQuery(userIds, new BatchQueryCallback<Long, List<Long>>() {
            @Override
            public List<Long> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder("userId");
                builder.eq("orgId", orgId);
                builder.in("userId", querySet);
                builder.eq("delStatus", 0);
                builder.group("userId");
                return queryForList(builder, Long.class);
            }
        });
    }

    @Override
    public Map<Integer, Integer> allCourseOrgIdMap() {

        Map<String, Object> param = new HashMap<>();
        String sql =
            "select org_id,course_id from tts.org_student_course where course_id <= 46763 and course_id>0 and org_id>0 and del_status=0  group by org_id,course_id";

        return this.getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<Map<Integer, Integer>>() {
            @Override
            public Map<Integer, Integer> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<Integer, Integer> map = new HashMap<>();
                while (rs.next()) {
                    Integer courseId = rs.getInt("course_id");
                    Integer orgId = rs.getInt("org_id");
                    map.put(courseId, orgId);
                }
                return map;
            }
        });
    }

    @Override
    public List<OrgStudentCourse> getOrgStudentCourseByUserIds(Long orgId, Collection<Long> userIds, String...props) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder(props);
        builder.eq("orgId", orgId);
        builder.in("userId", userIds);
        builder.eq("delStatus", 0);
        return queryList(builder);
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.baijia.tianxiao.dal.org.dao.OrgStudentCourseDao#searchStudentWithCourseIds(java.lang.Long,
     * java.util.Set, boolean)
     */
    @Override
    public List<OrgStudentCourse> searchStudentWithCourseIds(Long orgId, Set<Long> filterIds, List<Long> noDelCourseIds,
        boolean allCourse) {
        if (orgId == null) {
            return Collections.emptyList();
        }
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.eq("delStatus", 0);
        builder.eq("status", 0); // 不包括 退班/转班的情况
        log.info("orgId:{} , filterIds:{} , noDelCourseIds:{} ,allCourse:{}", orgId, filterIds, noDelCourseIds,
            allCourse);
        if (allCourse) {
            noDelCourseIds.removeAll(filterIds);
            if (GenericsUtils.notNullAndEmpty(noDelCourseIds)) {
                builder.in("courseId", noDelCourseIds);
            }
        } else {
            filterIds.retainAll(noDelCourseIds);
            if (GenericsUtils.isNullOrEmpty(filterIds)) {
                return Collections.emptyList();
            }
            builder.in("courseId", filterIds);
        }
        log.info("sql is : {} ", builder.toSql());
        return queryList(builder);
    }

    @Override
    public List<OrgStudentCourse> getOrgStudentCourseByOrgIds(Collection<Long> orgIds, PageDto pageDto) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
        builder.in("orgId", orgIds);
        builder.setPage(pageDto);
        return queryList(builder);
    }

    /*
     * 获取学员所有的报班记录
     */
    @Override
    public Map<Long, List<Long>> getOrgCourseIdMap(long orgId, Set<Long> userIds) {
        List<OrgStudentCourse> orgStudentCourseByUserIds =
            this.getOrgStudentCourseByUserIds(orgId, userIds, "userId", "courseId");
        Map<Long, List<Long>> emptyMap = GenericsUtils.emptyMap();
        if (GenericsUtils.isNullOrEmpty(orgStudentCourseByUserIds)) {
            return emptyMap;
        }
        for (OrgStudentCourse osc : orgStudentCourseByUserIds) {
            Long studentId = osc.getUserId();
            GenericsUtils.addListIfNotExists(emptyMap, studentId, osc.getCourseId());
        }
        return emptyMap;
    }

    @Override
    public int sumOneToOneClassHours(Collection<Long> classIds) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
        builder.in("courseId", classIds);
        builder.sum("lessonCount");
        return queryForObject(builder, Integer.class);
    }
    
    /*
     * (non-Javadoc)
     * 
     * @see com.baijia.tianxiao.dal.org.dao.OrgStudentCourseDao#searchStudentClassHourStatus(java.util.Set,
     * java.lang.Long)
     */
    @Override
    public List<StudentClassHourStatusDocument> searchStudentClassHourStatus(Set<Long> studentIds, Long orgId) {
        SingleSqlBuilder<OrgStudentCourse> builder =
            createSqlBuilder("courseId", "userId", "status", "lessonCount");
        builder.eq("orgId", orgId);
        builder.in("userId", studentIds);
        List<OrgStudentCourse> queryList = queryList(builder);
        List<StudentClassHourStatusDocument> ret = Lists.newArrayList();
        if (GenericsUtils.notNullAndEmpty(queryList)) {
            for (OrgStudentCourse osc : queryList) {
                StudentClassHourStatusDocument scsd = new StudentClassHourStatusDocument();
                scsd.setCourseId(osc.getCourseId());
                scsd.setLessonCount(osc.getLessonCount());
                scsd.setStatus(osc.getStatus());
                scsd.setUserId(osc.getUserId());
                scsd.setOrgId(orgId);
            }
        }
        return ret;
    }

//	@Override
//	public Double getVipClassTotalPrice(Long orgId, Long courseId) {
//		SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder("payAmount");
//		builder.eq("orgId", orgId);
//		builder.eq("courseId", courseId);
//		OrgStudentCourse orgStudentCourse = queryForObject(builder, OrgStudentCourse.class);
//		return (double) (orgStudentCourse == null ? 0 : orgStudentCourse.getPayAmount());
//	}
}