/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */

package com.baijia.tianxiao.dal.comment.dao.impl;

import com.baijia.tianxiao.consants.UserRole;
import com.baijia.tianxiao.dal.comment.dao.OrgLessonCommentDao;
import com.baijia.tianxiao.dal.comment.po.CommentAudit;
import com.baijia.tianxiao.dal.comment.po.CommentStatic;
import com.baijia.tianxiao.dal.comment.po.OrgCommentAudit;
import com.baijia.tianxiao.dal.comment.po.OrgLessonComment;
import com.baijia.tianxiao.sqlbuilder.SingleSqlBuilder;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.sqlbuilder.support.JdbcTemplateDaoSupport;
import com.baijia.tianxiao.sqlbuilder.util.Expressions;
import com.baijia.tianxiao.util.CollectorUtil;
import com.baijia.tianxiao.util.GenericsUtils;
import com.baijia.tianxiao.util.query.BatchQueryCallback;
import com.baijia.tianxiao.util.query.BatchQueryTemplate;
import com.baijia.tianxiao.util.query.ListBatchQueryTemplate;
import com.baijia.tianxiao.util.query.MapBatchQueryTemplate;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.core.RowCallbackHandler;
import org.springframework.stereotype.Repository;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.*;

import lombok.NonNull;

/**
 * @author shanyu
 * @version 1.0
 * @title OrgLessonCommentDaoImpl
 * @desc TODO
 * @date 2015年12月30日
 */
@Repository
@Slf4j
public class OrgLessonCommentDaoImpl extends JdbcTemplateDaoSupport<OrgLessonComment> implements OrgLessonCommentDao {

    public OrgLessonCommentDaoImpl() {
        super(OrgLessonComment.class);
    }

    @Override
    public List<OrgLessonComment> getLessonStudentComments(Long lessonId, Boolean isSystem, String... queryProps) {
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder(queryProps);
        builder.eq("lessonId", lessonId);
        builder.eq("userRole", UserRole.STUDENT.getRole());
        if (isSystem != null) {
            builder.eq("isSystem", isSystem);
        }
        return queryList(builder);
    }

    @Override
    public List<OrgLessonComment> getLessonTeacherComments(Long lessonId, Boolean isSystem, String... queryProps) {
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder(queryProps);
        builder.eq("lessonId", lessonId);
        builder.eq("userRole", UserRole.TEACHER.getRole());
        if (isSystem != null) {
            builder.eq("isSystem", isSystem);
        }
        return queryList(builder);
    }

    @Override
    public OrgLessonComment getLessonCommentDetail(Long lessonId, Long userId, Integer userRole, Boolean isSystem,
                                                   String... queryProps) {
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder(queryProps);
        builder.eq("lessonId", lessonId);
        builder.eq("userRole", userRole);
        builder.eq("fromId", userId);
        if (isSystem != null) {
            builder.eq("isSystem", isSystem);
        }
        return uniqueResult(builder);
    }

    @Override
    public OrgLessonComment getLessonCommentByFromAndTo(Long lessonId, Long fromId, Long toId, Integer userRole, Boolean isSystem,
                                                        String... queryProps) {
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder(queryProps);
        builder.eq("lessonId", lessonId);
        builder.eq("userRole", userRole);
        builder.eq("fromId", fromId);
        builder.eq("toId", toId);
        if (isSystem != null) {
            builder.eq("isSystem", isSystem);
        }
        return uniqueResult(builder);
    }

    @Override
    public List<OrgLessonComment> getStudentComments(Long orgId, Long userId, Boolean isSystem, PageDto pageDto,
                                                     String... queryProps) {
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder(queryProps);
        builder.eq("orgId", orgId);
        builder.eq("fromId", userId);
        builder.eq("userRole", UserRole.STUDENT.getRole());
        if (isSystem != null) {
            builder.eq("isSystem", isSystem);
        }
        if (null != pageDto) {
            builder.setPage(pageDto);
        }
        return queryList(builder);
    }

    @Override
    public List<OrgLessonComment> getTeacherCommentsOfStudent(Long orgId, Long userId, Boolean isSystem,
                                                              PageDto pageDto, String... queryProps) {
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder(queryProps);
        builder.eq("orgId", orgId);
        builder.eq("toId", userId);
        builder.eq("userRole", UserRole.TEACHER.getRole());
        if (isSystem != null) {
            builder.eq("isSystem", isSystem);
        }
        if (null != pageDto) {
            builder.setPage(pageDto);
        }
        return queryList(builder);
    }

    @Override
    public CommentAudit getLessonCommentAudit(Long lessonId, Long orgId, Boolean isSystem) {
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder("lessonId");
        builder.count("id", "num");
        builder.sum("score", "totalScore");
        builder.eq("lessonId", lessonId);
        builder.eq("orgId", orgId);
        builder.eq("userRole", UserRole.STUDENT.getRole());
        if (isSystem != null) {
            builder.eq("isSystem", isSystem);
        }
        List<CommentAudit> result = queryList(builder, CommentAudit.class);
        if (CollectionUtils.isNotEmpty(result)) {
            return result.get(0);
        }
        return null;
    }

    @Override
    public CommentAudit starScoreLessonCommentAudit(Long lessonId, Long orgId, Boolean isSystem) {
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder("lessonId");
        builder.count("id", "num");
        builder.sum("starScore", "totalScore");
        builder.eq("lessonId", lessonId);
        builder.eq("orgId", orgId);
        builder.eq("userRole", UserRole.STUDENT.getRole());
        if (isSystem != null) {
            builder.eq("isSystem", isSystem);
        }
        List<CommentAudit> result = queryList(builder, CommentAudit.class);
        if (CollectionUtils.isNotEmpty(result)) {
            return result.get(0);
        }
        return null;
    }

    @Override
    public List<CommentAudit> getLessonsCommentAudit(Collection<Long> lessonIds, Long orgId, Integer userRole,
                                                     Boolean isSystem) {
        if (CollectionUtils.isEmpty(lessonIds)) {
            return Lists.newArrayList();
        }
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder("lessonId");
        builder.count("id", "num");
        builder.group("lessonId");
        builder.sum("score", "totalScore");
        builder.in("lessonId", lessonIds);
        builder.eq("orgId", orgId);
        builder.eq("userRole", userRole);
        if (isSystem != null) {
            builder.eq("isSystem", isSystem);
        }
        return queryList(builder, CommentAudit.class);
    }

    @Override
    public CommentAudit getCommentAudit(Collection<Long> lessonIds, Long orgId, Boolean isSystem) {
        if (CollectionUtils.isEmpty(lessonIds)) {
            return new CommentAudit();
        }
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder();
        builder.count("id", "num");
        builder.sum("score", "totalScore");
        builder.in("lessonId", lessonIds);
        builder.eq("orgId", orgId);
        builder.eq("userRole", UserRole.STUDENT.getRole());
        if (isSystem != null) {
            builder.eq("isSystem", isSystem);
        }
        List<CommentAudit> result = queryList(builder, CommentAudit.class);
        if (CollectionUtils.isNotEmpty(result)) {
            return result.get(0);
        }
        return new CommentAudit();
    }

    @Override
    public Map<Long, Integer> getLessonCommentCountMap(Collection<Long> lessonIds, final Long orgId, final Long fromId,
                                                       final Long toId, final Integer userRole, Boolean isSystem) {
        if (CollectionUtils.isEmpty(lessonIds)) {
            return Maps.newHashMap();
        }
        return new MapBatchQueryTemplate<Long, Long, Integer>().batchQuery(lessonIds,
                new BatchQueryCallback<Long, Map<Long, Integer>>() {
                    @Override
                    public Map<Long, Integer> doQuery(Collection<Long> querySet) {
                        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder("lessonId");
                        builder.count("id", "num");
                        builder.group("lessonId");
                        builder.in("lessonId", querySet);
                        builder.eq("userRole", userRole);
                        builder.eq("orgId", orgId);
                        if (fromId != null) {
                            builder.eq("fromId", fromId);
                        }
                        if (toId != null) {
                            builder.eq("toId", toId);
                        }
                        final Map<Long, Integer> result = Maps.newHashMap();
                        getNamedJdbcTemplate().query(builder.toSql(), builder.collectConditionValue(),
                                new RowCallbackHandler() {
                                    @Override
                                    public void processRow(ResultSet rs) throws SQLException {
                                        result.put(rs.getLong("lessonId"), rs.getInt("num"));
                                    }
                                });
                        return result;
                    }
                });
    }

    @Override
    public List<OrgLessonComment> getStudentComments(Long orgId, Long userId, Integer userRole, Boolean isFrom,
                                                     Boolean isSystem, Collection<Long> lessonIds, PageDto pageDto, String... queryPorps) {
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder(queryPorps);
        if (GenericsUtils.isNullOrEmpty(lessonIds)) {
            return GenericsUtils.emptyList();
        }
        builder.eq("orgId", orgId);
        if (isFrom) {
            builder.eq("fromId", userId);
        } else {
            builder.eq("toId", userId);
        }
        if (isSystem != null) {
            builder.eq("isSystem", isSystem);
        }
        builder.in("lessonId", lessonIds);
        builder.eq("userRole", userRole);
        builder.desc("createTime");
        builder.setPage(pageDto);
        return queryList(builder);
    }

    @Override
    public Integer getStudentLessonCommentCount(Long orgId, Long userId, Boolean isSystem) {
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder();
        builder.count("id");
        builder.eq("fromId", userId);
        builder.eq("orgId", orgId);
        if (isSystem != null) {
            builder.eq("isSystem", isSystem);
        }
        builder.eq("userRole", UserRole.STUDENT.getRole());
        return queryForObject(builder, Integer.class);
    }

    @Override
    public List<Long> getCommentLessonIdsOfStudent(Long userId) {

        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder("lessonId");
        builder.eq("fromId", userId);
        // builder.eq("isSystem", 0);
        builder.eq("userRole", UserRole.STUDENT.getRole());
        return queryForList(builder, Long.class);

    }

    @Override
    public Map<Long, List<Long>> getLessonCommentStudentMap(Collection<Long> lessonIds, final Boolean isSystem,
                                                            final Integer userRole) {
        BatchQueryTemplate<Long, List<OrgLessonComment>> batch = new ListBatchQueryTemplate<>();
        List<OrgLessonComment> list =
                batch.batchQuery(lessonIds, new BatchQueryCallback<Long, List<OrgLessonComment>>() {
                    @Override
                    public List<OrgLessonComment> doQuery(Collection<Long> querySet) {
                        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder("lessonId", "fromId", "toId");
                        builder.in("lessonId", querySet);
                        builder.eq("userRole", userRole);
                        if (isSystem != null) {
                            builder.eq("isSystem", isSystem);
                        }
                        return queryList(builder);
                    }
                });
        return CollectorUtil.group(list, new Function<OrgLessonComment, Long>() {
            @Override
            public Long apply(OrgLessonComment arg0) {
                return arg0.getLessonId();

            }
        }, new Function<OrgLessonComment, Long>() {
            @Override
            public Long apply(OrgLessonComment arg0) {
                if (userRole.intValue() == UserRole.STUDENT.getRole()) {
                    return arg0.getFromId();
                }
                return arg0.getToId();
            }
        });
    }

    @Override
    public Map<Long, CommentAudit> getTeacherCommentCountMap(Long orgId, Collection<Long> teacherIds, Integer userRole,
                                                             Date endTime, Boolean isSystem) {
        Preconditions.checkArgument(orgId != null, "orgId is null!");
        Preconditions.checkArgument(userRole != null, "userRole is null!");
        if (CollectionUtils.isEmpty(teacherIds)) {
            return Maps.newHashMap();
        }
        Map<String, Object> params = Maps.newHashMap();
        StringBuilder sb = new StringBuilder();
        sb.append("select count(comment.id) as num, sum(comment.score) as score, ");
        if (userRole.intValue() == UserRole.STUDENT.getRole()) {
            sb.append("comment.to_id as teacherId ");
        } else {
            sb.append("comment.from_id as teacherId ");
        }
        sb.append(
                "from tts.org_lesson_comment comment join tts.org_class_lesson lesson on comment.lesson_id = lesson.id ");
        sb.append("where comment.org_id =:orgId and comment.user_role =:userRole ");
        if (isSystem != null) {
            sb.append("and comment.is_system =:isSystem ");
            params.put("isSystem", isSystem);
        }
        params.put("orgId", orgId);
        params.put("userRole", userRole);
        if (endTime != null) {
            sb.append("and lesson.end_time >:endTime ");
            params.put("endTime", endTime);

        }
        if (userRole.intValue() == UserRole.STUDENT.getRole()) {
            sb.append("and comment.to_id in (:teacherIds) ");
            sb.append("group by (comment.to_id) ");
        } else {
            sb.append("and comment.from_id in (:teacherIds) ");
            sb.append("group by (comment.from_id) ");
        }
        params.put("teacherIds", teacherIds);
        final Map<Long, CommentAudit> result = Maps.newHashMap();
        getNamedJdbcTemplate().query(sb.toString(), params, new RowCallbackHandler() {

            @Override
            public void processRow(ResultSet rs) throws SQLException {
                CommentAudit audit = new CommentAudit();
                audit.setNum(rs.getInt("num"));
                audit.setTotalScore(rs.getInt("score"));
                result.put(rs.getLong("teacherId"), audit);
            }
        });
        return result;
    }

    @Override
    public Map<Integer, Integer> getCommentCountOfOrg(Long orgId, Boolean isSystem, Integer lowScore) {
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder("userRole");
        builder.count("id", "num");
        builder.eq("orgId", orgId);
        if (isSystem != null) {
            builder.eq("isSystem", isSystem);
        }
        if (lowScore != null) {
            builder.lt("score", lowScore);
        }
        builder.group("userRole");
        final Map<Integer, Integer> result = Maps.newHashMap();
        getNamedJdbcTemplate().query(builder.toSql(), builder.collectConditionValue(), new RowCallbackHandler() {
            @Override
            public void processRow(ResultSet rs) throws SQLException {
                result.put(rs.getInt("userRole"), rs.getInt("num"));
            }
        });
        return result;
    }

    @Override
    public List<OrgCommentAudit> getOrgCommentAudit(Date startTime, Date endTime, final Boolean isSystem) {
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder("orgId");
        builder.count("id", "commentCount");
        builder.sum("score", "totalScore");
        builder.eq("userRole", UserRole.STUDENT.getRole());
        if (startTime != null) {
            builder.ge("createTime", startTime);
        }
        if (endTime != null) {
            builder.lt("createTime", endTime);
        }
        if (isSystem != null) {
            builder.eq("isSystem", isSystem);
        }
        builder.group("orgId");
        final List<OrgCommentAudit> result = Lists.newArrayList();
        getNamedJdbcTemplate().query(builder.toSql(), builder.collectConditionValue(), new RowCallbackHandler() {
            @Override
            public void processRow(ResultSet rs) throws SQLException {
                OrgCommentAudit audit = new OrgCommentAudit();
                if (isSystem == null) {// 包含默认评价
                    audit.setCommentCountAll(rs.getInt("commentCount"));
                    audit.setOrgId(rs.getLong("orgId"));
                    audit.setTotalScoreAll(rs.getInt("totalScore"));
                } else {// 不包含默认评价
                    audit.setCommentCount(rs.getInt("commentCount"));
                    audit.setOrgId(rs.getLong("orgId"));
                    audit.setTotalScore(rs.getInt("totalScore"));
                }
                result.add(audit);
            }
        });
        return result;
    }

    @Override
    public List<OrgLessonComment> getComments(Long orgId, Collection<Long> lessonIds, Integer userRole,
                                              Integer lowSocre, Boolean hasPic, Integer highScore, PageDto page, Boolean isSystem, String... queryProps) {
        Preconditions.checkArgument(orgId != null, "orgId is null!");
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder(queryProps);
        builder.eq("orgId", orgId);
        builder.eq("userRole", userRole);
        if (lowSocre != null) {
            builder.lt("score", lowSocre);
        }
        if (hasPic != null && hasPic) {
            builder.ne("storageIds", "");
        }
        if (highScore != null) {
            builder.gt("score", highScore);
        }
        if (isSystem != null) {
            builder.eq("isSystem", isSystem);
        }
        if (CollectionUtils.isNotEmpty(lessonIds)) {
            builder.in("lessonId", lessonIds);
        }
        builder.desc("createTime");
        builder.setPage(page);
        return queryList(builder);
    }

    @Override
    public List<OrgLessonComment> getLessonComments(Long lessonId, Boolean isSystem, Integer userRole,
                                                    String... queryProps) {
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder(queryProps);
        builder.eq("lessonId", lessonId);
        if (userRole != null) {
            builder.eq("userRole", userRole);
        }
        if (isSystem != null) {
            builder.eq("isSystem", isSystem);
        }
        return queryList(builder);
    }

    @Override
    public Map<Long, Integer> getComment(int userRole, Date startTime, Date endTime, List<Long> TianxiaoOrgIds) {
        Map<String, Object> param = new HashMap<>();
        param.put("orgIds", TianxiaoOrgIds);
        param.put("userRole", userRole);
        String sql =
                "select org_id, count(id) count from tts.org_lesson_comment where user_role = :userRole and org_id in (:orgIds) and is_system = 0";
        if (startTime != null && endTime != null) {
            sql += " AND create_time between :startTime and :endTime ";
            param.put("startTime", startTime);
            param.put("endTime", endTime);
        }
        sql += " group by org_id ";

        return this.getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<Map<Long, Integer>>() {

            @Override
            public Map<Long, Integer> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<Long, Integer> map = new HashMap<>();
                while (rs.next()) {
                    Long orgId = rs.getLong("org_id");
                    Integer count = rs.getInt("count");
                    map.put(orgId, count);
                }
                return map;
            }
        });
    }

    @Override
    public Integer getCommentCount(Long orgId, Collection<Long> lessonIds, Integer userRole, Integer lowSocre,
                                   Boolean hasPic, Integer highScore, Boolean isSystem) {
        Preconditions.checkArgument(orgId != null, "orgId is null!");
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder();
        builder.count("id");
        builder.eq("orgId", orgId);
        builder.eq("userRole", userRole);
        if (lowSocre != null) {
            builder.lt("score", lowSocre);
        }
        if (hasPic != null && hasPic) {
            builder.ne("storageIds", "");
        }
        if (highScore != null) {
            builder.gt("score", highScore);
        }
        if (isSystem != null) {
            builder.eq("isSystem", isSystem);
        }
        if (CollectionUtils.isNotEmpty(lessonIds)) {
            builder.in("lessonId", lessonIds);
        }
        return queryForObject(builder, Integer.class);

    }

    @Override
    public Integer getPeriodComment(Long orgId, Collection<Long> lessonIds, Integer userRole, Integer lowSocre,
                                    Boolean hasPic, Integer highScore, Boolean isSystem, String startDate, String endDate) {
        Preconditions.checkArgument(orgId != null, "orgId is null!");
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder();
        builder.count("id");
        builder.eq("orgId", orgId);
        builder.between("createTime", startDate, endDate);
        builder.eq("userRole", userRole);
        if (lowSocre != null) {
            builder.lt("score", lowSocre);
        }
        if (hasPic != null && hasPic) {
            builder.ne("storageIds", "");
        }
        if (highScore != null) {
            builder.gt("score", highScore);
        }
        if (isSystem != null) {
            builder.eq("isSystem", isSystem);
        }
        if (CollectionUtils.isNotEmpty(lessonIds)) {
            builder.in("lessonId", lessonIds);
        }
        return queryForObject(builder, Integer.class);
    }

    @Override
    public Long getSumCommentByTeacher(Long teacherUserId, String... queryProps) {
        Preconditions.checkArgument(teacherUserId != null, "teacherUserId is null!");
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder();
        builder.sum("score");
        builder.eq("toId", teacherUserId);
        return queryForObject(builder, Long.class);
    }

    @Override
    public Long getCountCommentByTeacher(Long teacherUserId, String... queryProps) {
        Preconditions.checkArgument(teacherUserId != null, "teacherUserId is null!");
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder();
        builder.count("id");
        builder.eq("toId", teacherUserId);
        return queryForObject(builder, Long.class);
    }

    @Override
    public Map<Long, Long> getAveCommentByTeacherIds(Collection<Long> teacherUserIds, String... queryProps) {
        if (teacherUserIds == null || teacherUserIds.size() == 0) {
            return Maps.newHashMap();
        }
        StringBuilder sb = new StringBuilder();
        sb.append(
                "select sum(score)/count(id) as score,to_id as tid from tts.org_lesson_comment where to_id in (:teacherIds) group by to_id");

        Map<String, Object> params = Maps.newHashMap();
        params.put("teacherIds", teacherUserIds);

        return this.getNamedJdbcTemplate().query(sb.toString(), params, new ResultSetExtractor<Map<Long, Long>>() {

            @Override
            public Map<Long, Long> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<Long, Long> map = new HashMap<>();
                while (rs.next()) {
                    Long tid = rs.getLong("tid");
                    Long score = rs.getLong("score");
                    map.put(tid, score);
                }
                return map;
            }
        });
    }

    @Override
    public Map<String, Integer> getOrgCommentMap(Date startTime, Date endTime, List<Long> TianxiaoOrgIds,
                                                 Integer userRole) {
        Map<String, Object> param = new HashMap<>();
        param.put("orgIds", TianxiaoOrgIds);
        param.put("userRole", userRole);
        param.put("startTime", startTime);
        param.put("endTime", endTime);

        String sql =
                "SELECT count(id) count, DATE_FORMAT(create_time,'%y-%m-%d') 'createTime' FROM tts.org_lesson_comment where "
                        + "user_role = :userRole and is_system = 0 and org_id in (:orgIds) and create_time>=:startTime and create_time<=:endTime";
        sql += " group by MONTH(create_time),DAY(create_time)";

        return this.getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<Map<String, Integer>>() {

            @Override
            public Map<String, Integer> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<String, Integer> map = new HashMap<>();
                while (rs.next()) {
                    String date = rs.getString("createTime");
                    Integer count = rs.getInt("count");
                    map.put(date, count);
                }
                return map;
            }
        });
    }

    @Override
    public List<OrgLessonComment> getComments(@NonNull Long orgId, @NonNull Long lessonId, List<Long> fromIds,
                                              List<Long> toIds, @NonNull Integer userRole) {
        if (CollectionUtils.isEmpty(fromIds) && CollectionUtils.isEmpty(toIds)) {
            return Lists.newArrayList();
        }
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.eq("lessonId", lessonId);
        if (CollectionUtils.isNotEmpty(fromIds)) {
            builder.in("fromId", fromIds);
        }
        builder.eq("userRole", userRole);
        if (CollectionUtils.isNotEmpty(toIds)) {
            builder.in("toId", toIds);
        }
        return queryList(builder);
    }

    @Override
    public OrgLessonComment getByIdAndOrgId(Long commentId, Long orgId) {
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder();
        builder.eq("id", commentId);
        builder.eq("orgId", orgId);
        return uniqueResult(builder);
    }

    @Override
    public List<OrgLessonComment> listComment(Long orgId, Collection<Long> classIds, Collection<Long> lessonIds, Long searchTeacherUserId, Long cascadeTeacherUserId, Date beginDate, Date endDate, Integer userRole, Integer startScore, Long lastId, Integer pageSize) {
        if (lessonIds != null && CollectionUtils.isEmpty(lessonIds)) {
            return Collections.EMPTY_LIST;
        }
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        if (lessonIds != null && CollectionUtils.isNotEmpty(lessonIds)) {
            builder.in("lessonId", lessonIds);
        }
        if (userRole != null && userRole != -1) {
            builder.eq("userRole", userRole);
        }
        if (startScore != null && startScore != -1) {
            buildStarScore(builder, startScore);
        }

        if (searchTeacherUserId != null && searchTeacherUserId.longValue() > 0) {
            builder.or(Expressions.eq("fromId", searchTeacherUserId), Expressions.eq("toId", searchTeacherUserId));
        }

        buildCoureseTeacher(builder, cascadeTeacherUserId, classIds);

        if (beginDate != null) {
            builder.ge("updateTime", beginDate);
        }
        if (endDate != null) {
            builder.lt("updateTime", endDate);
        }

        if (lastId != null && lastId != -1) {
            builder.lt("id", lastId);
        }

        builder.eq("isSystem", 0);

        builder.desc("id");
        builder.setMaxSize(pageSize);
        log.debug("listComment = {},{}", builder.toSql(), builder.collectConditionValue());
        return queryList(builder);
    }

    @Override
    public List<OrgLessonComment> listSearchComment(Long orgId, Long cascadeTeacherUserId, Collection<Long> classIds, Collection<Long> lessonIds, List<Long> teacherUserid, List<Long> studentUserId, String
            searchKey, Integer pageSize) {
        if (lessonIds != null && CollectionUtils.isEmpty(lessonIds)) {
            return Collections.EMPTY_LIST;
        }

        Map<String, Object> params = new HashMap<String, Object>();
        final List<OrgLessonComment> list = new ArrayList<OrgLessonComment>();

        StringBuilder sql = new StringBuilder("select * from tts.org_lesson_comment where org_id = :orgId ");
        if (lessonIds != null) {
            sql.append(" and lesson_id in (:lessonIds) ");
            params.put("lessonIds", lessonIds);
        }

        if (cascadeTeacherUserId != null && cascadeTeacherUserId.longValue() > 0) {
            sql.append(" and ((user_role = 0 and from_id = :cascadeTeacherUserId) or (user_role = 2 and to_id = :cascadeTeacherUserId))");
            params.put("cascadeTeacherUserId", cascadeTeacherUserId);
        }
        if (CollectionUtils.isEmpty(classIds)) {
            sql.append(" and ((user_role = 0 and (content like :searchKey or from_id in (:teacherUserid) or to_id in (:studentUserId))) ");
            sql.append(" or (user_role = 2 and (content like :searchKey or from_id in (:studentUserId) or to_id in (:teacherUserid))))");
        } else {
            sql.append(" and ((user_role = 0 and (content like :searchKey or from_id in (:teacherUserid) or to_id in (:studentUserId))) ");
            sql.append(" or (user_role = 2 and (content like :searchKey or from_id in (:studentUserId) or to_id in (:teacherUserid)))");
            sql.append(" or (course_id in (:classIds)))");
            params.put("classIds", classIds);
        }
        sql.append(" and is_system = 0 order by update_time desc limit :pageSize");

        params.put("orgId", orgId);
        params.put("searchKey", "%" + searchKey + "%");
        params.put("teacherUserid", teacherUserid);
        params.put("studentUserId", studentUserId);
        params.put("pageSize", pageSize);

        log.debug("listSearcgComment = {},{}", sql, params);

        this.getNamedJdbcTemplate().query(sql.toString(), params, new ResultSetExtractor<Object>() {

            @Override
            public Object extractData(ResultSet rs) throws SQLException, DataAccessException {
                OrgLessonComment orgLessonComment = null;
                while (rs.next()) {
                    orgLessonComment = new OrgLessonComment();
                    orgLessonComment.setId(rs.getLong("id"));
                    orgLessonComment.setOrgId(rs.getLong("org_id"));
                    orgLessonComment.setLessonId(rs.getLong("lesson_id"));
                    orgLessonComment.setCourseId(rs.getLong("course_id"));
                    orgLessonComment.setFromId(rs.getLong("from_id"));
                    orgLessonComment.setToId(rs.getLong("to_id"));
                    orgLessonComment.setUserRole(rs.getInt("user_role"));
                    orgLessonComment.setContent(rs.getString("content"));
                    orgLessonComment.setCustomFastComment(rs.getString("custom_fast_comment"));
                    orgLessonComment.setStorageIds(rs.getString("storage_ids"));
                    orgLessonComment.setSoundId(rs.getInt("sound_id"));
                    orgLessonComment.setSoundLength(rs.getInt("sound_length"));
                    orgLessonComment.setCreateTime(rs.getDate("create_time"));
                    orgLessonComment.setUpdateTime(rs.getDate("update_time"));
                    orgLessonComment.setIsSystem(rs.getBoolean("is_system"));
                    orgLessonComment.setScore(rs.getInt("score"));
                    orgLessonComment.setStarScore(rs.getInt("star_score"));
                    list.add(orgLessonComment);
                }
                return null;
            }
        });
        return list;

    }

    @Override
    public List<OrgLessonComment> listCommentByStudent(Long orgId, Long lessonId, Long userId) {
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.eq("lessonId", lessonId);
        builder.or(Expressions.eq("fromId", userId),
                Expressions.eq("toId", userId));
        builder.desc("updateTime");
        return queryList(builder);
    }

    @Override
    public CommentStatic listCommentStatic(Long orgId, Collection<Long> classIds, Collection<Long> lessonIds, Long searchTeacherUserId, Long cascadeTeacherUserId, Date
            beginDate, Date endDate, Integer userRole, Integer startScore) {

        if (lessonIds != null && CollectionUtils.isEmpty(lessonIds)) {
            return new CommentStatic();
        }
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder();

        builder.sum("starScore", "averageScore");
        builder.count("id", "count");

        builder.eq("orgId", orgId);
        if (lessonIds != null && CollectionUtils.isNotEmpty(lessonIds)) {
            builder.in("lessonId", lessonIds);
        }
        if (userRole != null && userRole != -1) {
            builder.eq("userRole", userRole);
        }
        if (startScore != null && startScore != -1) {
            buildStarScore(builder, startScore);
        }

        if (searchTeacherUserId != null && searchTeacherUserId.longValue() > 0) {
            builder.or(Expressions.eq("fromId", searchTeacherUserId), Expressions.eq("toId", searchTeacherUserId));
        }

        buildCoureseTeacher(builder, cascadeTeacherUserId, classIds);

        if (beginDate != null) {
            builder.ge("updateTime", beginDate);
        }
        if (endDate != null) {
            builder.lt("updateTime", endDate);
        }

        builder.eq("isSystem", 0);

        log.debug("listCommentStatic = {},{}", builder.toSql(), builder.collectConditionValue());
        return uniqueResult(builder, CommentStatic.class);

    }

    private void buildCoureseTeacher(SingleSqlBuilder<OrgLessonComment> builder, Long cascadeTeacherUserId, Collection<Long> classIds) {
        if (cascadeTeacherUserId != null && cascadeTeacherUserId.intValue() > -1) {
            if (CollectionUtils.isNotEmpty(classIds)) {
                builder.or(Expressions.in("courseId", classIds), Expressions.eq("fromId", cascadeTeacherUserId), Expressions.eq("toId", cascadeTeacherUserId));
            } else {
                builder.or(Expressions.eq("fromId", cascadeTeacherUserId), Expressions.eq("toId", cascadeTeacherUserId));
            }
        } else {
            if (CollectionUtils.isNotEmpty(classIds)) {
                builder.in("courseId", classIds);
            }
        }
    }

    private void buildStarScore(SingleSqlBuilder<OrgLessonComment> builder, Integer starScore) {
        if (starScore.intValue() == 1) {
            builder.ge("starScore", 400);
        } else if (starScore.intValue() == 2) {
            builder.ge("starScore", 200);
            builder.lt("starScore", 400);
        } else if (starScore.intValue() == 3) {
            builder.lt("starScore", 200);
        }
    }
}