/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */

package com.baijia.tianxiao.dal.org.dao.impl;

import com.baijia.tianxiao.consants.DataStatus;
import com.baijia.tianxiao.dal.org.dao.OrgClassLessonDao;
import com.baijia.tianxiao.dal.org.dto.DeleteStatus;
import com.baijia.tianxiao.dal.org.dto.TimeTableRequestDto;
import com.baijia.tianxiao.dal.org.po.ClassHour;
import com.baijia.tianxiao.dal.org.po.OrgClassLesson;
import com.baijia.tianxiao.dal.org.po.OrgLessonConflict;
import com.baijia.tianxiao.sqlbuilder.SingleSqlBuilder;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.sqlbuilder.support.JdbcTemplateDaoSupport;
import com.baijia.tianxiao.sqlbuilder.util.Expressions;
import com.baijia.tianxiao.util.GenericsUtils;
import com.baijia.tianxiao.util.collection.CollectorUtil;
import com.baijia.tianxiao.util.date.DateUtil;
import com.baijia.tianxiao.util.query.BatchQueryCallback;
import com.baijia.tianxiao.util.query.BatchQueryTemplate;
import com.baijia.tianxiao.util.query.ListBatchQueryTemplate;
import com.baijia.tianxiao.util.query.MapBatchQueryTemplate;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.BeanPropertyRowMapper;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.core.RowCallbackHandler;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.stereotype.Repository;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author shanyu
 * @version 1.0
 * @title OrgClassLessonDaoImpl
 * @desc TODO
 * @date 2015年12月8日
 */
@Repository
@Slf4j
public class OrgClassLessonDaoImpl extends JdbcTemplateDaoSupport<OrgClassLesson> implements OrgClassLessonDao {

    public OrgClassLessonDaoImpl() {
        super(OrgClassLesson.class);
    }

    @Override
    public List<OrgClassLesson> getLessonDates(@NonNull Long orgId, Collection<Long> lessonIds, Long roomId,
                                               @NonNull Date startDate, @NonNull Date endDate, String... queryProps) {
        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder(queryProps);
        builder.eq("orgId", orgId);
        if (CollectionUtils.isNotEmpty(lessonIds)) {
            builder.in("id", lessonIds);
        }
        if (roomId != null && roomId > 0) {
            builder.eq("roomId", roomId);
        }
        builder.between("start_time", startDate, endDate);
        builder.eq("delStatus", DeleteStatus.NORMAL.getValue());
        return queryList(builder);
    }

    @Override
    public List<OrgClassLesson> queryLessonsByEndTime(Long orgId, Collection<Long> courseIds, Date endTime, PageDto page) {
        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder();
        if (CollectionUtils.isNotEmpty(courseIds)) {
            builder.in("courseId", courseIds);
        }
        builder.eq("orgId", orgId);
        if (endTime != null) {
            builder.gt("endTime", endTime);
        }
        builder.asc("startTime", "id");
        builder.eq("delStatus", DataStatus.NORMAL.getValue());
        if (page != null) {
            builder.setPage(page);
        }
        return queryList(builder);
    }

    @Override
    public List<OrgClassLesson> queryLessons(Long orgId, Collection<Long> courseIds, Collection<Long> roomIds,
                                             Collection<Long> lessonIds, Date startTime, Date endTime, PageDto page, Boolean timeIncFlag,
                                             String... queryProps) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder(queryProps);
        if (CollectionUtils.isNotEmpty(courseIds)) {
            builder.in("courseId", courseIds);
        }
        builder.eq("orgId", orgId);
        if (CollectionUtils.isNotEmpty(roomIds)) {
            builder.in("roomId", roomIds);
        }

        if (CollectionUtils.isNotEmpty(lessonIds)) {
            builder.in("id", lessonIds);
        }

        if (startTime != null) {
            builder.ge("startTime", startTime);
        }
        if (endTime != null) {
            builder.lt("startTime", endTime);
        }
        if (endTime != null) {
            if (null != timeIncFlag && timeIncFlag) {
                builder.asc("startTime");
            } else {
                builder.desc("startTime");
            }
        } else {
            builder.asc("startTime", "id");
        }
        builder.eq("delStatus", DataStatus.NORMAL.getValue());
        if (page != null) {
            builder.setPage(page);
        }
        return queryList(builder);
    }

    @Override//FIXME 脏逻辑
    @Deprecated
    public List<OrgClassLesson> queryLessons(Collection<Long> notInCourseIds, Long orgId, Collection<Long> courseIds,
                                             Collection<Long> roomIds, Collection<Long> lessonIds, Date startTime, Date endTime, PageDto page,
                                             Boolean timeIncFlag, String... queryProps) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder(queryProps);
        if (CollectionUtils.isNotEmpty(courseIds)) {
            courseIds.removeAll(notInCourseIds);
            builder.in("courseId", courseIds);
        } else {
            if (CollectionUtils.isNotEmpty(notInCourseIds)) {
                builder.notin("courseId", notInCourseIds);
            }
        }
        builder.eq("orgId", orgId);
        if (CollectionUtils.isNotEmpty(roomIds)) {
            builder.in("roomId", roomIds);
        }

        if (CollectionUtils.isNotEmpty(lessonIds)) {
            builder.in("id", lessonIds);
        }

        if (startTime != null) {
            builder.gt("startTime", startTime);
        }
        if (endTime != null) {
            builder.lt("startTime", endTime);
        }
        if (endTime != null) {
            if (null != timeIncFlag && timeIncFlag) {
                builder.asc("startTime", "id");
            } else {
                builder.desc("startTime");
            }
        } else {
            builder.asc("startTime", "id");
        }
        builder.eq("delStatus", DataStatus.NORMAL.getValue());
        if (page != null) {
            builder.setPage(page);
        }
        return queryList(builder);
    }


    @Override
    @Deprecated // FIXME 脏逻辑 数据没修
    public List<OrgClassLesson> queryLessons(Collection<Long> notInCourseIds, Long orgId, Collection<Long> courseIds,
                                             Date startTime, Date endTime, PageDto page, Boolean timeIncFlag, String... queryProps) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        notInCourseIds = null == notInCourseIds ? Lists.<Long>newArrayList() : notInCourseIds;
        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder(queryProps);
        if (CollectionUtils.isNotEmpty(courseIds)) {
            courseIds.removeAll(notInCourseIds);
            builder.in("courseId", courseIds);
        } else {
            if (CollectionUtils.isNotEmpty(notInCourseIds)) {
                builder.notin("courseId", notInCourseIds);
            }
        }
        builder.eq("orgId", orgId);
        builder.asc("startTime", "id");
        if (startTime != null) {
            builder.ge("startTime", startTime);
        }
        if (endTime != null) {
            builder.lt("startTime", endTime);
        }
        if (endTime != null) {
            if (null != timeIncFlag && timeIncFlag) {
                builder.asc("startTime");
            } else {
                builder.desc("startTime");
            }
        }
        builder.eq("delStatus", DataStatus.NORMAL.getValue());
        if (page != null) {
            builder.setPage(page);
        }
        return queryList(builder);
    }

    @Override
    public List<OrgClassLesson> querySignInLessons(Long orgId, Collection<Long> courseIds,
                                                   Collection<Long> notInCourseIds, Collection<Long> roomIds, Collection<Long> lessonIds, Date startTime,
                                                   Date endTime, PageDto page, Long lessonId, String... queryProps) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        StringBuilder sql = new StringBuilder();
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("orgId", orgId);
        paramMap.put("delStatus", DataStatus.NORMAL.getValue());

        sql.append("select * from tts.org_class_lesson where org_id = :orgId and del_status = :delStatus ");

        if (CollectionUtils.isNotEmpty(courseIds)) {
            sql.append("and course_id in (:courseIds) ");
            paramMap.put("courseIds", courseIds);
        }
        if (CollectionUtils.isNotEmpty(notInCourseIds)) {
            sql.append("and course_id not in (:notInCourseIds) ");
            paramMap.put("notInCourseIds", notInCourseIds);
        }
        if (startTime != null) {
            if (lessonId != null) {
                paramMap.put("id", lessonId);
                sql.append("and (start_time > :startTime or (start_time=:startTime and id > :id)) ");
            } else {
                sql.append("and start_time > :startTime ");
            }
            sql.append(" order by start_time asc,id asc ");
            paramMap.put("startTime", startTime);
        } else if (endTime != null) {
            if (lessonId != null) {
                paramMap.put("id", lessonId);
                sql.append("and (start_time < :startTime or (start_time=:startTime and id < :id))");
            } else {
                sql.append("and start_time < :startTime ");
            }
            sql.append(" order by start_time desc,id desc ");
            paramMap.put("startTime", endTime);
        }
        if (page != null) {
            paramMap.put("offset", (page.getPageNum() - 1) * page.getPageSize());
            paramMap.put("size", page.getPageSize());
            sql.append(" limit :offset,:size");
        }
        return this.getNamedJdbcTemplate().query(sql.toString(), paramMap,
                new ResultSetExtractor<List<OrgClassLesson>>() {
                    @Override
                    public List<OrgClassLesson> extractData(ResultSet rs) throws SQLException, DataAccessException {
                        List<OrgClassLesson> orgClassLessons = new ArrayList<>();
                        while (rs.next()) {
                            orgClassLessons.add(toOrgClassLesson(rs));
                        }
                        return orgClassLessons;
                    }
                });
    }

    private OrgClassLesson toOrgClassLesson(ResultSet rs) throws SQLException {
        OrgClassLesson orgClassLesson = new OrgClassLesson();
        orgClassLesson.setCourseId(rs.getLong("course_id"));
        orgClassLesson.setCreateTime(new Date(rs.getTimestamp("create_time").getTime()));
        orgClassLesson.setDelStatus(rs.getInt("del_status"));
        orgClassLesson.setEndTime(new Date(rs.getTimestamp("end_time").getTime()));
        orgClassLesson.setId(rs.getLong("id"));
        orgClassLesson.setLayoutId(rs.getLong("layout_id"));
        orgClassLesson.setNumber(rs.getInt("number"));
        orgClassLesson.setOrgId(rs.getLong("org_id"));
        orgClassLesson.setRoomId(rs.getLong("room_id"));
        orgClassLesson.setStartTime(new Date(rs.getTimestamp("start_time").getTime()));
        orgClassLesson.setUpdateTime(new Date(rs.getTimestamp("update_time").getTime()));
        return orgClassLesson;
    }

    @Override
    public Map<Long, Integer> getLessonTimemap(Collection<Long> courseIds, Date date, Long orgId, Integer delStatus) {
        List<OrgClassLesson> lessons = getLessons(courseIds, date, orgId, delStatus, "courseId");
        if (CollectionUtils.isEmpty(lessons)) {
            return Maps.newHashMap();
        }
        Map<Long, Integer> map = Maps.newHashMap();
        for (OrgClassLesson orgClassLesson : lessons) {
            if (!map.containsKey(orgClassLesson.getCourseId())) {
                map.put(orgClassLesson.getCourseId(), 0);
            }
            map.put(orgClassLesson.getCourseId(), map.get(orgClassLesson.getCourseId()) + 1);
        }
        return map;
    }

    @Override
    public List<OrgClassLesson> getLessons(Collection<Long> courseIds, final Date date, final Long orgId,
                                           final Integer delStatus, final String... queryProps) {
        if (CollectionUtils.isEmpty(courseIds)) {
            return Lists.newArrayList();
        }
        BatchQueryTemplate<Long, List<OrgClassLesson>> batchQueryTemplate = new ListBatchQueryTemplate<>();
        return batchQueryTemplate.batchQuery(courseIds, new BatchQueryCallback<Long, List<OrgClassLesson>>() {
            @Override
            public List<OrgClassLesson> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder(queryProps);
                builder.in("courseId", querySet);
                builder.eq("orgId", orgId);
                builder.eq("delStatus", delStatus);
                if (date != null) {
                    builder.le("endTime", date);
                }
                builder.asc("startTime");
//                builder.setOrder(Order.asc("startTime"));
                return queryList(builder);
            }
        });
    }

    @Override
    public List<OrgClassLesson> getLessons(Collection<Long> courseIds, final Date startTime, final Date endTime,
                                           PageDto pageDto) {
        if (CollectionUtils.isEmpty(courseIds)) {
            return Lists.newArrayList();
        }
        BatchQueryTemplate<Long, List<OrgClassLesson>> batchQueryTemplate = new ListBatchQueryTemplate<>();
        return batchQueryTemplate.batchQuery(courseIds, new BatchQueryCallback<Long, List<OrgClassLesson>>() {
            @Override
            public List<OrgClassLesson> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder();
                builder.in("courseId", querySet);
                if (startTime != null) {
                    builder.ge("startTime", startTime);
                }
                if (endTime != null) {
                    builder.lt("startTime", endTime);
                }
                builder.eq("delStatus", DataStatus.NORMAL.getValue());
                return queryList(builder);
            }
        });
    }

    @Override
    public List<Long> getLeftLessonIds(Long orgId, Long courseId) {
        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder("id");
        builder.eq("courseId", courseId);
        builder.eq("orgId", orgId);
        builder.ge("startTime", new Date());
        log.debug("leftLessonIds ={},{}", builder.toSql(), builder.collectConditionValue());
        return this.queryForList(builder, Long.class);
    }

    @Override
    public Integer getLessonCountOfStudent(Long orgId, Long courseId, Long studentId) {

        String sql = "select count(ocl.id) from tts.org_class_lesson ocl join tts.org_student_lesson osl on "
                + "osl.lesson_id = ocl.id where osl.student_id = :studentId "
                + "and ocl.course_id = :courseId and ocl.org_id = :orgId";
        Map<String, Object> paramMap = Maps.newHashMap();
        paramMap.put("orgId", orgId);
        paramMap.put("courseId", courseId);
        paramMap.put("studentId", studentId);
        return this.getNamedJdbcTemplate().queryForObject(sql, paramMap, Integer.class);
    }

    @Override
    public int updateLessonClassRoom(Long orgId, Collection<Long> lessonIds, Long roomId) {
        if (CollectionUtils.isEmpty(lessonIds)) {
            return 0;
        }
        if (roomId == null) {
            roomId = 0l;
        }
        Map<String, Object> params = Maps.newHashMap();
        params.put("id", lessonIds);
        params.put("orgId", orgId);
        params.put("roomId", roomId);

        String sql = "update tts.org_class_lesson set room_id=:roomId where orgId=:orgId and id in :lessonIds ";
        log.debug("update lesson sql:{} params:{}", sql, params);
        return this.update(params, "roomId");
        // return getNamedJdbcTemplate().update(sql, params);
    }

    @Override
    public int updateLessonName(Long orgId, Collection<Long> lessonIds, String name) {
        if (CollectionUtils.isEmpty(lessonIds)) {
            return 0;
        }
        if (name == null) {
            name = "";
        }
        Map<String, Object> params = Maps.newHashMap();
        params.put("orgId", orgId);
        params.put("name", name);
        params.put("id", lessonIds);

        String sql = "update tts.org_class_lesson set name=:name where orgId=:orgId and id in :lessonIds ";
        log.debug("update lesson sql:{} params:{}", sql, params);
        return this.update(params, "name");
        // return getNamedJdbcTemplate().update(sql, params);
    }

    @Override
    public void batchUpdateLessonNumber(Map<String, ?>[] lessonIdNumberMaps) {
        if (ArrayUtils.isEmpty(lessonIdNumberMaps)) {
            return;
        }
        String sql = "update tts.org_class_lesson set number=:number where  id=:id ";
        log.debug("sql:{} batch update params:{}", sql, lessonIdNumberMaps);
        this.getNamedJdbcTemplate().batchUpdate(sql, lessonIdNumberMaps);
    }

    @Override
    public void batchUpdateLessonName(List<Long> ids, String name) {
        if (ids.isEmpty() || name == null) {
            return;
        }
        String sql = "update tts.org_class_lesson set name=:name where  id in ( :ids ) ";
        log.debug("sql:{} batch update params:{},{}", sql, ids, name);
        Map<String, Object> params = Maps.newHashMap();
        params.put("name", name);
        params.put("ids", ids);
        this.getNamedJdbcTemplate().update(sql, params);
    }

    @Override
    public Map<Long, Integer> getFinishLessonCount(final Long orgId, Collection<Long> courseIds,
                                                   final Integer delStatus) {
        if (CollectionUtils.isEmpty(courseIds)) {
            return Maps.newHashMap();
        }
        final Date now = new Date();
        return new MapBatchQueryTemplate<Long, Long, Integer>().batchQuery(courseIds,
                new BatchQueryCallback<Long, Map<Long, Integer>>() {
                    @Override
                    public Map<Long, Integer> doQuery(Collection<Long> querySet) {
                        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder("courseId");
                        builder.count("id", "num");
                        builder.in("courseId", querySet);
                        builder.group("courseId");
                        if (delStatus != null) {
                            builder.eq("delStatus", delStatus);
                        }
                        builder.le("endTime", now);
                        final Map<Long, Integer> result = Maps.newHashMap();
                        getNamedJdbcTemplate().query(builder.toSql(), builder.collectConditionValue(),
                                new RowCallbackHandler() {
                                    @Override
                                    public void processRow(ResultSet rs) throws SQLException {
                                        result.put(rs.getLong("courseId"), rs.getInt("num"));
                                    }
                                });
                        return result;
                    }
                });
    }

    @Override
    public Map<Long, Integer> getWxFinishLessonCount(Long orgId, Collection<Long> courseIds, final Integer delStatus) {
        if (CollectionUtils.isEmpty(courseIds)) {
            return Maps.newHashMap();
        }
        final Date now = new Date();
        return new MapBatchQueryTemplate<Long, Long, Integer>().batchQuery(courseIds,
                new BatchQueryCallback<Long, Map<Long, Integer>>() {
                    @Override
                    public Map<Long, Integer> doQuery(Collection<Long> querySet) {
                        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder("courseId");
                        builder.count("id", "num");
                        builder.in("courseId", querySet);
                        builder.group("courseId");
                        if (delStatus != null) {
                            builder.eq("delStatus", delStatus);
                        }
                        builder.le("startTime", now);
                        final Map<Long, Integer> result = Maps.newHashMap();
                        getNamedJdbcTemplate().query(builder.toSql(), builder.collectConditionValue(),
                                new RowCallbackHandler() {
                                    @Override
                                    public void processRow(ResultSet rs) throws SQLException {
                                        result.put(rs.getLong("courseId"), rs.getInt("num"));
                                    }
                                });
                        return result;
                    }
                });
    }

    @Override
    public Map<Long, Integer> getLessonCount(final Long orgId, Collection<Long> courseIds, final Integer delStatus) {
        if (CollectionUtils.isEmpty(courseIds)) {
            return Maps.newHashMap();
        }
        return new MapBatchQueryTemplate<Long, Long, Integer>().batchQuery(courseIds,
                new BatchQueryCallback<Long, Map<Long, Integer>>() {
                    @Override
                    public Map<Long, Integer> doQuery(Collection<Long> querySet) {
                        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder("courseId");
                        builder.count("id", "num");
                        builder.eq("orgId", orgId);
                        builder.in("courseId", querySet);
                        builder.group("courseId");
                        if (delStatus != null) {
                            builder.eq("delStatus", delStatus);
                        }
                        final Map<Long, Integer> result = Maps.newHashMap();
                        getNamedJdbcTemplate().query(builder.toSql(), builder.collectConditionValue(),
                                new RowCallbackHandler() {
                                    @Override
                                    public void processRow(ResultSet rs) throws SQLException {
                                        result.put(rs.getLong("courseId"), rs.getInt("num"));
                                    }
                                });
                        return result;
                    }
                });
    }

    @Override
    public Map<Long, Map<Long, Integer>> getLessonLengthMap(final Long orgId, Collection<Long> courseIds,
                                                            final Integer delStatus) {
        if (CollectionUtils.isEmpty(courseIds)) {
            return Maps.newHashMap();
        }
        return new MapBatchQueryTemplate<Long, Long, Map<Long, Integer>>().batchQuery(courseIds,
                new BatchQueryCallback<Long, Map<Long, Map<Long, Integer>>>() {
                    @Override
                    public Map<Long, Map<Long, Integer>> doQuery(Collection<Long> querySet) {
                        SingleSqlBuilder<OrgClassLesson> builder =
                                createSqlBuilder("id", "courseId", "startTime", "endTime");
                        builder.eq("orgId", orgId);
                        builder.in("courseId", querySet);
                        if (delStatus != null) {
                            builder.eq("delStatus", delStatus);
                        }
                        final Map<Long, Map<Long, Integer>> result = Maps.newHashMap();
                        getNamedJdbcTemplate().query(builder.toSql(), builder.collectConditionValue(),
                                new RowCallbackHandler() {
                                    @Override
                                    public void processRow(ResultSet rs) throws SQLException {
                                        Map<Long, Integer> innerMap = result.get(rs.getLong("courseId"));
                                        if (null == innerMap) {
                                            innerMap = Maps.newHashMap();
                                            result.put(rs.getLong("courseId"), innerMap);
                                        }
                                        innerMap.put(rs.getLong("id"),
                                                DateUtil.getMinuteDiff(rs.getTimestamp("startTime"), rs.getTimestamp("endTime")));
                                    }
                                });
                        return result;
                    }
                });
    }

    @Override
    public Map<Long, Map<Long, Integer>> getFinishLessonLengthMap(final Long orgId, Collection<Long> courseIds,
                                                                  final Integer delStatus) {
        if (CollectionUtils.isEmpty(courseIds)) {
            return Maps.newHashMap();
        }
        final Date now = new Date();
        return new MapBatchQueryTemplate<Long, Long, Map<Long, Integer>>().batchQuery(courseIds,
                new BatchQueryCallback<Long, Map<Long, Map<Long, Integer>>>() {
                    @Override
                    public Map<Long, Map<Long, Integer>> doQuery(Collection<Long> querySet) {
                        SingleSqlBuilder<OrgClassLesson> builder =
                                createSqlBuilder("id", "courseId", "startTime", "endTime");
                        builder.eq("orgId", orgId);
                        builder.in("courseId", querySet);
                        if (delStatus != null) {
                            builder.eq("delStatus", delStatus);
                        }
                        builder.le("endTime", now);
                        final Map<Long, Map<Long, Integer>> result = Maps.newHashMap();
                        getNamedJdbcTemplate().query(builder.toSql(), builder.collectConditionValue(),
                                new RowCallbackHandler() {
                                    @Override
                                    public void processRow(ResultSet rs) throws SQLException {
                                        Map<Long, Integer> innerMap = result.get(rs.getLong("courseId"));
                                        if (null == innerMap) {
                                            innerMap = Maps.newHashMap();
                                            result.put(rs.getLong("courseId"), innerMap);
                                        }
                                        innerMap.put(rs.getLong("id"),
                                                DateUtil.getMinuteDiff(rs.getTimestamp("startTime"), rs.getTimestamp("endTime")));
                                    }
                                });
                        return result;
                    }
                });
    }

    @Override
    public List<OrgClassLesson> getLessonByCourseIds(Collection<Long> courseIds, Integer delStatus, String... props) {
        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder(props);
        builder.in("courseId", courseIds);
        if (delStatus != null) {
            builder.eq("delStatus", delStatus);
        }
        return this.queryList(builder, OrgClassLesson.class);
    }

    @Override
    public Map<Long, OrgClassLesson> getClassLessonMap(final Long orgId, Collection<Long> lessonIds,
                                                       final Integer delStatus, final String... props) {
        if (CollectionUtils.isEmpty(lessonIds)) {
            return Maps.newHashMap();
        }
        BatchQueryTemplate<Long, List<OrgClassLesson>> batchQueryTemplate = new ListBatchQueryTemplate<>();
        List<OrgClassLesson> list =
                batchQueryTemplate.batchQuery(lessonIds, new BatchQueryCallback<Long, List<OrgClassLesson>>() {
                    @Override
                    public List<OrgClassLesson> doQuery(Collection<Long> querySet) {
                        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder(props);
                        builder.in("id", querySet);
                        builder.eq("orgId", orgId);
                        builder.eq("delStatus", delStatus);
                        return queryList(builder);
                    }
                });

        Map<Long, OrgClassLesson> map = Maps.newHashMap();
        for (OrgClassLesson po : list) {
            map.put(po.getId(), po);
        }
        return map;
    }

    @Override
    public Map<Long, OrgClassLesson> getClassLessonMap(Long orgId, Long courseId, Integer delStatus, String... props) {
        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder(props);
        builder.eq("courseId", courseId);
        builder.eq("orgId", orgId);
        if (delStatus != null) {
            builder.eq("delStatus", delStatus);
        }

        List<OrgClassLesson> list = this.queryList(builder);

        Map<Long, OrgClassLesson> map = Maps.newHashMap();
        for (OrgClassLesson po : list) {
            map.put(po.getId(), po);
        }
        return map;
    }

    @Override
    public Integer getFinishLessonCount(Long orgId, Long courseId, Integer delStatus) {
        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder();
        builder.count("id", "num");
        builder.eq("courseId", courseId);
        if (delStatus != null) {
            builder.eq("delStatus", delStatus);
        }
        builder.le("endTime", new Date());
        return queryForObject(builder, Integer.class);
    }

    @Override
    public Integer getLessonCount(Long orgId, Long courseId, Integer delStatus) {
        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder();
        builder.count("id", "num");
        builder.eq("courseId", courseId);
        if (delStatus != null) {
            builder.eq("delStatus", delStatus);
        }
        return queryForObject(builder, Integer.class);
    }

    @Override
    public List<OrgClassLesson> querylessonsByEndTime(Date startTime, Date endTime, Integer delStatus,
                                                      PageDto pageDto) {
        Preconditions.checkArgument(endTime != null, "endTime is null!");
        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder();
        builder.gt("courseId", 0);
        if (startTime != null) {
            builder.ge("endTime", startTime);
        }
        builder.lt("endTime", endTime);
        builder.desc("endTime");
        if (delStatus != null) {
            builder.eq("delStatus", delStatus);
        }
        builder.setPage(pageDto);
        return queryList(builder);
    }

    @Override
    public Map<Long, Integer> getCourseLessonCount(Long orgId, Collection<Long> courseIds, Long studentId,
                                                   Long teacherId, Date endTime) {
        Preconditions.checkArgument(orgId != null, "orgId is null!");
        if (CollectionUtils.isEmpty(courseIds)) {
            return Maps.newHashMap();
        }
        final StringBuilder sb = new StringBuilder();
        final Map<String, Object> params = Maps.newHashMap();
        sb.append("select lesson.course_id AS courseId, count(lesson.id) AS num from tts.org_class_lesson lesson ");
        if (studentId != null) {
            sb.append("join tts.org_student_lesson student on lesson.id = student.lesson_id ");
            params.put("studentId", studentId);
        }
        if (teacherId != null) {
            sb.append("join tts.org_teacher_lesson teacher on lesson.id = teacher.lesson_id ");
            params.put("teacherId", teacherId);
        }
        sb.append("where lesson.org_id=:orgId and lesson.course_id in (:courseIds) ");
        if (studentId != null) {
            sb.append("and student.student_id =:studentId ");
        }
        if (teacherId != null) {
            sb.append("and teacher.teacher_id =:teacherId ");
        }
        if (endTime != null) {
            sb.append("and lesson.end_time <=:endTime ");
            params.put("endTime", endTime);
        }
        params.put("orgId", orgId);
        sb.append(" and lesson.del_status=0 group by lesson.course_id ");
        return new MapBatchQueryTemplate<Long, Long, Integer>().batchQuery(courseIds,
                new BatchQueryCallback<Long, Map<Long, Integer>>() {
                    @Override
                    public Map<Long, Integer> doQuery(Collection<Long> querySet) {
                        params.put("courseIds", querySet);
                        final Map<Long, Integer> result = Maps.newHashMap();
                        getNamedJdbcTemplate().query(sb.toString(), params, new RowCallbackHandler() {
                            @Override
                            public void processRow(ResultSet rs) throws SQLException {
                                result.put(rs.getLong("courseId"), rs.getInt("num"));
                            }
                        });
                        return result;
                    }
                });
    }

    @Override
    public List<Long> filterLessonListByTime(Collection<Long> courseIds, Collection<Long> lessonIds, Date date) {
        if (CollectionUtils.isEmpty(courseIds)) {
            return Lists.newArrayList();
        }
        if (CollectionUtils.isEmpty(lessonIds)) {
            return Lists.newArrayList();
        }
        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder("id");
        builder.in("id", lessonIds);
        builder.in("courseId", courseIds);
        builder.le("endTime", date);
        return queryForList(builder, Long.class);
    }

    @Override
    public List<Long> finishLessonId(Collection<Long> lessonIds) {
        if (CollectionUtils.isEmpty(lessonIds)) {
            return Lists.newArrayList();
        }
        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder("id");
        builder.in("id", lessonIds);
        builder.le("endTime", DateUtil.getStartOfDay(new Date()));
        return queryForList(builder, Long.class);
    }

    @Override
    public List<OrgClassLesson> getLeftRoomIdLessons(Long orgId, Long roomId, Date startTime, Integer delStatus) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(roomId != null && roomId > 0, "orgId is illegal");
        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.eq("roomId", roomId);
        if (startTime != null) {
            builder.gt("startTime", startTime);
        }
        if (null != delStatus) {
            builder.eq("delStatus", delStatus);
        }
        return queryList(builder);
    }

    @Override
    public List<Long> getOrgLessonIds(Long orgId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder("id");
        builder.eq("orgId", orgId);
        builder.desc("startTime");
        return queryForList(builder, Long.class);
    }

    @Override
    public Map<Long, Integer> getCourseAttendCountMap(Long orgId, Collection<Long> courseIds) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        if (CollectionUtils.isEmpty(courseIds)) {
            return Collections.emptyMap();
        }
        String sql = "select ocl.course_id as courseId,count(osl.id) as cnt "
                + "from tts.org_class_lesson ocl join tts.org_student_lesson osl on osl.lesson_id = ocl.id "
                + "where ocl.org_id = :orgId and ocl.course_id in(:courseIds) and osl.del_status=0 and ocl.start_time < :now "
                + "group by ocl.course_id order by field(courseId,:courseIds)";
        final Map<Long, Integer> resultMap = Maps.newHashMap();
        Date now = new Date();
        Map<String, Object> paramMap = Maps.newHashMap();
        paramMap.put("orgId", orgId);
        paramMap.put("courseIds", courseIds);
        paramMap.put("now", now);
        log.debug("sql = {},param = {}", sql, paramMap);
        this.getNamedJdbcTemplate().query(sql, paramMap, new RowCallbackHandler() {

            @Override
            public void processRow(ResultSet rs) throws SQLException {
                resultMap.put(rs.getLong("courseId"), rs.getInt("cnt"));
            }

        });
        return resultMap;

    }

    @Override
    public Map<Long, Integer> getStudentAttendCountMap(Long orgId, Collection<Long> userIds) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        if (CollectionUtils.isEmpty(userIds)) {
            return Collections.emptyMap();
        }
        String sql = "select osl.student_id as userId, count(osl.id) as cnt from tts.org_class_lesson ocl "
                + " join tts.org_student_lesson osl on osl.lesson_id = ocl.id "
                + " where osl.student_id in(:userIds) and ocl.org_id = :orgId and osl.del_status=0 "
                + " and ocl.start_time < :now group by osl.student_id order by field(userId,:userIds)";
        final Map<Long, Integer> resultMap = Maps.newHashMap();
        Date now = new Date();
        Map<String, Object> paramMap = Maps.newHashMap();
        paramMap.put("orgId", orgId);
        paramMap.put("userIds", userIds);
        paramMap.put("now", now);
        log.debug("sql = {},param = {}", sql, paramMap);
        this.getNamedJdbcTemplate().query(sql, paramMap, new RowCallbackHandler() {

            @Override
            public void processRow(ResultSet rs) throws SQLException {
                resultMap.put(rs.getLong("userId"), rs.getInt("cnt"));
            }

        });
        return resultMap;
    }

    @Override
    public List<Long> filterLessonList(Long orgId, Collection<Long> courseIds, Collection<Long> teacherIds) {
        Preconditions.checkArgument(orgId != null, "orgId is null!");
        if (CollectionUtils.isEmpty(teacherIds) && CollectionUtils.isEmpty(courseIds)) {
            return Lists.newArrayList();
        }
        Map<String, Object> params = Maps.newHashMap();
        StringBuilder sb = new StringBuilder();
        sb.append("select distinct teacher.lesson_id from tts.org_teacher_lesson teacher ");
        sb.append("join tts.org_class_lesson lesson on teacher.lesson_id = lesson.id ");
        sb.append("where teacher.org_id =:orgId ");
        params.put("orgId", orgId);
        if (CollectionUtils.isNotEmpty(teacherIds)) {
            sb.append("and teacher.teacher_id in (:teacherIds) ");
            params.put("teacherIds", teacherIds);
        }
        if (CollectionUtils.isNotEmpty(courseIds)) {
            sb.append("and lesson.course_id in (:courseIds)");
            params.put("courseIds", courseIds);
        }
        return getNamedJdbcTemplate().queryForList(sb.toString(), params, Long.class);
    }

    @Override
    public List<Long> queryFinishedLessonIds(Long orgId, Long teacherId, Date startTime, Date endTime) {
        Preconditions.checkArgument(orgId != null, "orgId is null");
        Preconditions.checkArgument(startTime != null, "startTime is null");
        Preconditions.checkArgument(endTime != null, "endTime is null");
        final List<Long> result = Lists.newArrayList();
        Map<String, Object> params = Maps.newHashMap();
        params.put("startTime", startTime);
        params.put("endTime", endTime);
        params.put("orgId", orgId);
        StringBuilder sb = new StringBuilder();
        if (teacherId != null) {
            params.put("teacherId", teacherId);
            sb.append(
                    "select distinct lesson.id from tts.org_class_lesson lesson,tts.org_teacher_lesson teacher ")
                    .append("where lesson.id = teacher.lesson_id ")
                    .append("and lesson.org_id=:orgId ").append("and teacher.teacher_id =:teacherId ")
                    .append("and lesson.start_time >:startTime and lesson.start_time <:endTime");
        } else {
            sb.append("select distinct lesson.id from tts.org_class_lesson lesson ")
                    .append("where lesson.org_id=:orgId ")
                    .append("and lesson.start_time >:startTime and lesson.start_time <:endTime ");
        }

        this.getNamedJdbcTemplate().query(sb.toString(), params, new RowCallbackHandler() {
            @Override
            public void processRow(ResultSet rs) throws SQLException {
                result.add(rs.getLong("id"));
            }
        });
        return result;
    }

    @Override
    public List<ClassHour> queryFinishedLessonCountByTeacherIds(Long orgId, Collection<Long> teacherIds, Date startTime,
                                                                Date endTime) {
        Preconditions.checkArgument(orgId != null, "orgId is null");
        Preconditions.checkArgument(startTime != null, "startTime is null");
        Preconditions.checkArgument(endTime != null, "endTime is null");

        if (teacherIds == null || teacherIds.size() == 0) {
            return Collections.emptyList();
        }
        Map<String, Object> params = Maps.newHashMap();
        params.put("startTime", startTime);
        params.put("endTime", endTime);
        params.put("orgId", orgId);
        StringBuilder sb = new StringBuilder();
        params.put("teacherId", teacherIds);
        sb.append(
                "select count(distinct lesson.id) as lessonCount ,teacher.teacher_id as teacherId from tts.org_class_lesson lesson, tts.org_student_lesson stu, tts.org_teacher_lesson teacher ")
                .append("where lesson.id = stu.lesson_id ").append("and lesson.id = teacher.lesson_id ")
                .append("and lesson.org_id=:orgId ").append("and teacher.teacher_id in (:teacherId) ").append(
                "and lesson.start_time >:startTime and lesson.start_time <:endTime and lesson.del_status=0 and stu.start_status = 1 group by teacher.teacher_id");

        final List<ClassHour> result = Lists.newArrayList();
        this.getNamedJdbcTemplate().query(sb.toString(), params, new RowCallbackHandler() {

            @Override
            public void processRow(ResultSet rs) throws SQLException {
                ClassHour classHour = new ClassHour();
                classHour.setTeacherId(rs.getLong("teacherId"));
                classHour.setLessonCount(rs.getInt("lessonCount"));
                result.add(classHour);
            }

        });
        return result;
    }

    @Override
    public List<ClassHour> queryTeacherClassHourList(Collection<Long> lessonIds, Long teacherId, final PageDto page,
                                                     final boolean groupByCourse) {
        Map<String, Object> params = Maps.newHashMap();
        params.put("lessonId", lessonIds);
        if (teacherId != null) {
            params.put("teacherId", teacherId);
        }
        if (page != null) {
            StringBuilder countSb = new StringBuilder();
            if (groupByCourse) {
                countSb.append("select count(distinct lesson.course_id) num from tts.org_class_lesson lesson ")
                        .append("where lesson.id in (:lessonId) ");
            } else {
                countSb.append("select count(distinct teacher.teacher_id) num from tts.org_teacher_lesson teacher ")
                        .append("where teacher.lesson_id in (:lessonId) ");
            }

            this.getNamedJdbcTemplate().query(countSb.toString(), params, new RowCallbackHandler() {
                @Override
                public void processRow(ResultSet rs) throws SQLException {
                    page.setCount(rs.getInt("num"));
                }
            });
        }
        StringBuilder querySb = new StringBuilder();

        if (groupByCourse) {
            querySb.append(
                    "select teacher.teacher_id teacherId, l.course_id courseId, count(l.id) lessonCount, sum(TIMESTAMPDIFF(MINUTE,l.start_time,l.end_time)) minutes from tts.org_class_lesson l join tts.org_teacher_lesson teacher on l.id = teacher.lesson_id ");
            querySb.append("where teacher.lesson_id in (:lessonId) ");
            if (teacherId != null) {
                querySb.append("and teacher.teacher_id =:teacherId ");
            }
            querySb.append("group by teacher.teacher_id, l.course_id order by minutes desc ");
        } else {
            querySb.append(
                    "select teacher.teacher_id teacherId, count(l.id) lessonCount, sum(TIMESTAMPDIFF(MINUTE,l.start_time,l.end_time)) minutes from tts.org_class_lesson l join tts.org_teacher_lesson teacher on l.id = teacher.lesson_id ");
            querySb.append("where teacher.lesson_id in (:lessonId) ");
            if (teacherId != null) {
                querySb.append("and teacher.teacher_id=:teacherId ");
            }
            querySb.append("group by teacher.teacher_id order by minutes desc ");
        }
        if (page != null) {
            querySb.append("limit ").append((page.getPageNum() - 1) * page.getPageSize()).append(",")
                    .append(page.getPageNum() * page.getPageSize());
        }
        final List<ClassHour> reuslt = Lists.newArrayList();

        this.getNamedJdbcTemplate().query(querySb.toString(), params, new RowCallbackHandler() {

            @Override
            public void processRow(ResultSet rs) throws SQLException {
                ClassHour classHour = new ClassHour();
                classHour.setTeacherId(rs.getLong("teacherId"));
                if (groupByCourse) {
                    classHour.setCourseId(rs.getLong("courseId"));
                }
                classHour.setLessonCount(rs.getInt("lessonCount"));
                classHour.setMinutes(rs.getInt("minutes"));
                reuslt.add(classHour);
            }

        });
        if (page != null) {
            page.setCurPageCount(reuslt.size());
        }
        return reuslt;
    }

    @Override
    public Map<Long, List<OrgClassLesson>> getLessonMapByCourseAndLessonIds(final Long orgId,
                                                                            final Collection<Long> courseIds, Collection<Long> lessonIds) {
        if (CollectionUtils.isEmpty(lessonIds) || CollectionUtils.isEmpty(courseIds)) {
            return Maps.newHashMap();
        }
        BatchQueryTemplate<Long, List<OrgClassLesson>> batchQueryTemplate = new ListBatchQueryTemplate<>();
        List<OrgClassLesson> list =
                batchQueryTemplate.batchQuery(lessonIds, new BatchQueryCallback<Long, List<OrgClassLesson>>() {
                    @Override
                    public List<OrgClassLesson> doQuery(Collection<Long> querySet) {
                        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder();
                        builder.in("id", querySet);
                        builder.in("courseId", courseIds);
                        builder.eq("orgId", orgId);
                        return queryList(builder);
                    }
                });
        return CollectorUtil.group(list, new Function<OrgClassLesson, Long>() {
            @Override
            public Long apply(OrgClassLesson arg0) {

                return arg0.getCourseId();
            }
        });
    }

    @Override
    public ClassHour queryTeacherClassHour(Collection<Long> lessonIds, Long teacherId) {
        Map<String, Object> params = Maps.newHashMap();
        params.put("lessonId", lessonIds);
        if (teacherId != null) {
            params.put("teacherId", teacherId);
        }
        StringBuilder querySb = new StringBuilder();

        querySb.append(
                "select teacher.teacher_id teacherId, count(l.id) lessonCount, sum(TIMESTAMPDIFF(MINUTE,l.start_time,l.end_time)) minutes from tts.org_class_lesson l join tts.org_teacher_lesson teacher on l.id = teacher.lesson_id ");
        querySb.append("where teacher.lesson_id in (:lessonId) ");
        if (teacherId != null) {
            querySb.append("and teacher.teacher_id=:teacherId ");
        }
        querySb.append("group by teacher.teacher_id order by minutes desc ");
        final ClassHour classHour = new ClassHour();
        this.getNamedJdbcTemplate().query(querySb.toString(), params, new RowCallbackHandler() {

            @Override
            public void processRow(ResultSet rs) throws SQLException {
                classHour.setTeacherId(rs.getLong("teacherId"));
                classHour.setLessonCount(rs.getInt("lessonCount"));
                classHour.setMinutes(rs.getInt("minutes"));
            }
        });
        return classHour;
    }

    @Override
    public Map<Long, Integer> getLessonTotal(Date startTime, Date endTime, List<Long> TianxiaoOrgIds) {
        Map<String, Object> param = new HashMap<>();
        param.put("orgIds", TianxiaoOrgIds);
        String sql = "select org_id, count(id) count from tts.org_class_lesson where org_id in (:orgIds)";
        if (startTime != null && endTime != null) {
            sql += " and create_time between :startTime and :endTime ";
            param.put("endTime", endTime);
            param.put("startTime", startTime);
        }
        sql += " group by org_id";
        return this.getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<Map<Long, Integer>>() {

            @Override
            public Map<Long, Integer> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<Long, Integer> map = new HashMap<>();
                while (rs.next()) {
                    Long orgNamber = rs.getLong("org_id");
                    Integer count = rs.getInt("count");
                    map.put(orgNamber, count);
                }
                return map;
            }
        });

    }

    @Override
    public void refreshOrgClassLesson(Long orgId, Long id) {
        StringBuilder sb =
                new StringBuilder("update tts.org_class_lesson set del_status =1 where id > :id and  org_id = :orgId");
        Map param = Maps.newHashMap();
        param.put("orgId", orgId);
        param.put("id", id);
        this.getNamedJdbcTemplate().update(sb.toString(), param);
    }

    @Override
    public Map<Long, Integer> getArrangedClassCount(Date startTime, Date endTime, List<Long> TianxiaoOrgIds) {
        Map<String, Object> param = new HashMap<>();
        param.put("orgIds", TianxiaoOrgIds);
        String sql =
                "select org_id, count(distinct course_id) count from tts.org_class_lesson where org_id in (:orgIds)";
        if (startTime != null && endTime != null) {
            sql += " and create_time between :startTime and :endTime ";
            param.put("endTime", endTime);
            param.put("startTime", startTime);
        }
        sql += " group by org_id";

        return this.getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<Map<Long, Integer>>() {

            @Override
            public Map<Long, Integer> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<Long, Integer> map = new HashMap<>();
                while (rs.next()) {
                    Long orgNamber = rs.getLong("org_id");
                    Integer count = rs.getInt("count");
                    map.put(orgNamber, count);
                }
                return map;
            }
        });
    }

    @Override
    public Integer getLessonCount(Integer orgId, List<Long> courseIds, Date endTime) {
        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder();
        builder.count("id");
        if (CollectionUtils.isNotEmpty(courseIds)) {
            builder.in("courseId", courseIds);
        }
        builder.eq("delStatus", 0);
        builder.eq("orgId", orgId.longValue());
        builder.gt("endTime", endTime);
        builder.lt("endTime", DateUtil.getEndOfDay(endTime));
        return queryForObject(builder, Integer.class);
    }

    @Override
    public List<OrgClassLesson> querylessonsByStartTime(Date startTime, Long courseId, Long orgId) {
        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.eq("courseId", courseId);
        builder.gt("startTime", startTime);
        builder.eq("delStatus", DataStatus.NORMAL.getValue());
        return queryList(builder);
    }

    @Override
    public List<OrgClassLesson> queryEndedlessons(Long courseId, Long orgId) {
        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.eq("courseId", courseId);
        builder.lt("endTime", new Date());
        builder.eq("delStatus", DataStatus.NORMAL.getValue());
        return queryList(builder);
    }

    @Override
    public Map<Long, List<OrgClassLesson>> querylessonsByStartTime(final Date startTime,
                                                                   final Collection<Long> courseIds, final Long orgId) {
        if (CollectionUtils.isEmpty(courseIds)) {
            return Maps.newHashMap();
        }
        BatchQueryTemplate<Long, List<OrgClassLesson>> batchQueryTemplate = new ListBatchQueryTemplate<>();
        List<OrgClassLesson> list =
                batchQueryTemplate.batchQuery(courseIds, new BatchQueryCallback<Long, List<OrgClassLesson>>() {
                    @Override
                    public List<OrgClassLesson> doQuery(Collection<Long> querySet) {
                        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder();

                        builder.eq("orgId", orgId);
                        builder.in("courseId", courseIds);
                        if (startTime != null) {
                            builder.gt("startTime", startTime);
                        }
                        builder.eq("delStatus", DataStatus.NORMAL.getValue());
                        builder.asc("startTime");
                        return queryList(builder);
                    }

                });
        return CollectorUtil.group(list, new Function<OrgClassLesson, Long>() {
            @Override
            public Long apply(OrgClassLesson arg0) {
                return arg0.getCourseId();
            }

        });
    }

    @Override
    public Map<Long, List<OrgClassLesson>> queryEndedlessons(final Collection<Long> courseIds, final Long orgId) {
        if (CollectionUtils.isEmpty(courseIds)) {
            return Maps.newHashMap();
        }
        BatchQueryTemplate<Long, List<OrgClassLesson>> batchQueryTemplate = new ListBatchQueryTemplate<>();
        List<OrgClassLesson> list =
                batchQueryTemplate.batchQuery(courseIds, new BatchQueryCallback<Long, List<OrgClassLesson>>() {
                    @Override
                    public List<OrgClassLesson> doQuery(Collection<Long> querySet) {
                        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder();
                        builder.eq("orgId", orgId);
                        builder.in("courseId", courseIds);
                        builder.lt("endTime", new Date());
                        builder.eq("delStatus", DataStatus.NORMAL.getValue());
                        builder.asc("startTime");
                        return queryList(builder);
                    }

                });
        return CollectorUtil.group(list, new Function<OrgClassLesson, Long>() {
            @Override
            public Long apply(OrgClassLesson arg0) {
                return arg0.getCourseId();
            }
        });
    }

    @Override
    public List<Long> getOrgCourseIdList(Date startTime, Date endTime, Collection<Long> courseIds) {
        Map<String, Object> param = new HashMap<>();
        param.put("courseIds", courseIds);

        String sql = "select * from tts.org_class_lesson where course_id in (:courseIds) and del_status = 0 ";

        if (GenericsUtils.isNullOrEmpty(startTime)) {// 获取截至到现在未开始课节或者结束课节
            Date now = new Date();
            sql += " and start_time > :now or end_time < :now ";
            param.put("now", now);
        } else {

            sql += " and start_time > :endTime or end_time < :startTime";
            param.put("startTime", startTime);
            param.put("endTime", endTime);
        }

        return this.getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<List<Long>>() {

            @Override
            public List<Long> extractData(ResultSet rs) throws SQLException, DataAccessException {
                List<Long> courseIds = new ArrayList<>();

                while (rs.next()) {
                    Long courseId = rs.getLong("course_id");
                    courseIds.add(courseId);
                }
                return courseIds;
            }
        });
    }

    @Override
    public List<OrgClassLesson> getOrgClassLessonsByOrgIds(Collection<Long> orgIds, Date nowDate) {
        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder();
        builder.in("orgId", orgIds);
        builder.eq("delStatus", 0);
        if (GenericsUtils.notNullAndEmpty(nowDate)) {
            builder.le("startTime", nowDate); // 正在上课的
            builder.ge("endTime", nowDate);
            builder.or(Expressions.ge("startTime", nowDate)); // 已经排课但是还没有开课
        }
        List<OrgClassLesson> orgClassLessons = this.queryList(builder);
        log.info("orgClassLessons size:{}", orgClassLessons.size());
        return orgClassLessons;
    }

    @Override
    public List<OrgClassLesson> getOrgClassLessonList(Collection<Long> courseIds) {
        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder();
        builder.in("course_id", courseIds);
        builder.eq("delStatus", 0);
        return queryList(builder);
    }

    @Override
    //FIXME 不走索引
    public List<OrgClassLesson> getOrgClassLessonsByStartTime(Date startTime, Date endTime, PageDto pageDto) {
        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder();
        builder.ge("startTime", startTime);
        builder.le("startTime", endTime);
        if (pageDto != null) {
            builder.setPage(pageDto);
        }
        return queryList(builder);
    }

    @Override
    public List<OrgClassLesson> getOrgClassLessonsByUpdateTime(Date startTime, Date endTime, PageDto pageDto) {
        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder();
        builder.ge("updateTime", startTime);
        builder.le("updateTime", endTime);
        if (pageDto != null) {
            builder.setPage(pageDto);
        }
        return queryList(builder);
    }

    @Override
    @Deprecated
    public List<OrgClassLesson> getOrgClassLessons(Long orgId, Long courseId, Long studentId) {
        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        if (null != courseId) {
            builder.eq("courseId", courseId);
        }
        return queryList(builder);
    }

    @Override
    public List<OrgClassLesson> queryByParams(Long orgId, List<Long> teacherLessonIds, List<Long> confictLessonIds,
                                              List<Long> courseIds) {
        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.eq("delStatus", false);
        if (CollectionUtils.isNotEmpty(teacherLessonIds)) {
            builder.in("id", teacherLessonIds);
        }
        if (CollectionUtils.isNotEmpty(confictLessonIds)) {
            builder.in("id", confictLessonIds);
        }
        if (CollectionUtils.isNotEmpty(courseIds)) {
            builder.in("courseId", courseIds);
        }
        return queryList(builder);
    }

    @Override
    public List<OrgClassLesson> queryByClassRooms(Long orgId, List<Long> classRoomIds, List<Long> courseIds,
                                                  List<Long> confictLessonIds) {
        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        if (CollectionUtils.isNotEmpty(classRoomIds)) {
            builder.in("roomId", classRoomIds);
        }
        if (CollectionUtils.isNotEmpty(courseIds)) {
            builder.in("courseId", courseIds);
        }
        if (CollectionUtils.isNotEmpty(confictLessonIds)) {
            builder.in("id", confictLessonIds);
        }
        builder.eq("delStatus", false);
        return queryList(builder);
    }

    @Override
    public List<OrgClassLesson> getOrgClassLessonByParams(Collection<Long> notInCourseIds, Long orgId,
                                                          TimeTableRequestDto params, Collection<Long> includeIds, PageDto pageDto) {
        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder();
        if (CollectionUtils.isNotEmpty(includeIds)) {
            builder.in("id", includeIds);
        }
        builder.eq("orgId", orgId);
        if (CollectionUtils.isNotEmpty(params.getRoomIds())) {
            builder.in("roomId", params.getRoomIds());
        }
        if (CollectionUtils.isNotEmpty(params.getCourseIds())) {
            if (CollectionUtils.isNotEmpty(notInCourseIds)) {
                params.getCourseIds().removeAll(notInCourseIds);
            }
            if (CollectionUtils.isNotEmpty(params.getCourseIds())) {
                builder.in("courseId", params.getCourseIds());
            }
        } else {
            if (CollectionUtils.isNotEmpty(notInCourseIds)) {
                builder.notin("courseId", notInCourseIds);
            }
        }
        if (params.getStartTime() != null) {
            builder.ge("startTime", DateFormatUtils.format(params.getStartTime(), "yyyy-MM-dd HH:mm:ss"));
        }
        if (params.getEndTime() != null) {
            builder.le("startTime", DateFormatUtils.format(params.getEndTime(), "yyyy-MM-dd HH:mm:ss"));
        }
        builder.eq("delStatus", 0);

        builder.asc("startTime");
        if (pageDto != null) {
            builder.setPage(pageDto);
        }

        return queryList(builder);
    }

    @Override
    public List<OrgClassLesson> queryForSelectCourse(Long orgId, List<Long> classRoomIds, List<Long> confictLessonIds,
                                                     List<Long> teacherLessonIds) {
        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.eq("delStatus", false);
        if (CollectionUtils.isNotEmpty(classRoomIds)) {
            builder.in("roomId", classRoomIds);
        }
        if (CollectionUtils.isNotEmpty(confictLessonIds)) {
            builder.in("id", confictLessonIds);
        }
        if (CollectionUtils.isNotEmpty(teacherLessonIds)) {
            builder.in("id", teacherLessonIds);
        }
        return queryList(builder);
    }

    @Override
    public OrgClassLesson getByIdForUpdate(Long id) {
        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder();
        builder.eq("id", id);
        return uniqueResult(builder, true);
    }

    @Override
    public void deleteByCourseIds(Long orgId, Collection<Long> courseIds) {
        if (CollectionUtils.isNotEmpty(courseIds)) {
            Map<String, Object> param = new HashMap<>();
            param.put("orgId", orgId);
            param.put("courseIds", courseIds);
            String sql =
                    "update tts.org_class_lesson set del_status=1 where org_id = :orgId and course_id in (:courseIds)";
            this.getNamedJdbcTemplate().update(sql, param);
        }
    }

    @Override
    public void updateDelByLessonIds(Long orgId, Collection<Long> lessonIds) {
        if (CollectionUtils.isNotEmpty(lessonIds)) {
            Map<String, Object> param = new HashMap<>();
            param.put("orgId", orgId);
            param.put("id", lessonIds);
            param.put("delStatus", 1);
            this.update(param, "delStatus");
        }
    }

    @Override
    public List<OrgLessonConflict> getflushConflictData() {
        // 分页获取避免数据量大导致性能问题
        return queryFlushConflictData(0, 10000);
    }

    private List<OrgLessonConflict> queryFlushConflictData(int start, int pageSize) {
        String sql =
                "select ocl.course_id as courseId,ocl.org_id as orgId,ocl.id as lessonId,otl.teacher_id as teacherId,ocl.room_id as roomId,ocl.start_time as startTime,ocl.end_time as endTime,ocl.del_status as delStatus from tts.org_class_lesson ocl left join tts.org_teacher_lesson otl on ocl.id=otl.lesson_id limit :start,:pageSize";
        Map<String, Object> params = Maps.newHashMap();
        params.put("start", start);
        params.put("pageSize", pageSize);
        List<OrgLessonConflict> queryList = this.getNamedJdbcTemplate().query(sql, params,
                new BeanPropertyRowMapper<OrgLessonConflict>(OrgLessonConflict.class));
        if (CollectionUtils.isNotEmpty(queryList)) {
            List<OrgLessonConflict> result = queryFlushConflictData(start + pageSize, pageSize);
            if (CollectionUtils.isNotEmpty(result)) {
                queryList.addAll(result);
            }
        }
        return queryList;
    }

    @Override
    public List<OrgClassLesson> queryByCourseIds(Long orgId, Collection<Long> courseIds, Integer delStatus) {
        Map<String, Object> param = new HashMap<>();
        param.put("orgId", orgId);
        param.put("courseIds", courseIds);
        if (delStatus != null) {
            param.put("delStatus", delStatus);
        }
        return queryByCondition(param, null);

    }

    @Override
    public List<OrgClassLesson> queryByIds(Collection<Long> lessonIds) {
        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder();
        builder.in("id", lessonIds);
        return queryList(builder);
    }

    @Override
    public Long sumLessonDuration(Collection<Long> lessonIds) {
        String sql = "select SUM(UNIX_TIMESTAMP(end_time) - UNIX_TIMESTAMP(start_time)) FROM tts.org_class_lesson WHERE id IN (:lessonIds)";
        Map<String, Object> params = Maps.newHashMap();
        params.put("lessonIds", lessonIds);

        Long result = this.getNamedJdbcTemplate().queryForObject(sql, params, Long.class);
        return result != null ? result : 0L;
    }

    @Override
    public List<OrgClassLesson> queryBy(Date startTime, Date endTime, Long classId, Collection<Long> lessonIds) {
        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder();
        builder.ge("startTime", startTime);
        builder.lt("startTime", endTime);
        builder.eq("courseId", classId);
        if (CollectionUtils.isNotEmpty(lessonIds)) {
            builder.in("id", lessonIds);
        }
        return queryList(builder);
    }

    @Override
    public Map<Long, OrgClassLesson> mapOrgClassLesson(Collection<Long> lessonIds) {
        Map<Long, OrgClassLesson> result = new HashMap<Long, OrgClassLesson>();

        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder();
        builder.in("id", lessonIds);
        List<OrgClassLesson> list = queryList(builder);
        if (CollectionUtils.isNotEmpty(list)) {
            for (OrgClassLesson ocl : list) {
                result.put(ocl.getId(), ocl);
            }
        }
        return result;
    }

    @Override
    public List<OrgClassLesson> getStartLessons(Long orgId, Long roomId, Date date) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.eq("roomId", roomId);
        if (date != null) {
            builder.lt("startTime", date);
        }
        builder.eq("delStatus", DeleteStatus.NORMAL.getValue());
        return queryList(builder);
    }

    @Override
    public Map<Long, Integer> getRoomArrangedMinute(Collection<Long> courseIds, final Date startTime, final Date endTime) {
        if (CollectionUtils.isEmpty(courseIds)) {
            return Maps.newHashMap();
        }
        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder("id", "roomId", "startTime", "endTime");
        builder.in("courseId", courseIds);
        builder.ge("startTime", startTime);
        builder.le("endTime", endTime);
        builder.eq("delStatus", DeleteStatus.NORMAL.getValue());
        final Map<Long, Integer> result = Maps.newHashMap();
        getNamedJdbcTemplate().query(builder.toSql(), builder.collectConditionValue(),
                new RowCallbackHandler() {
                    @Override
                    public void processRow(ResultSet rs) throws SQLException {
                        int minute = DateUtil.getMinuteDiff(rs.getTimestamp("startTime"), rs.getTimestamp("endTime"));
                        if (result.containsKey(rs.getLong("roomId"))) {
                            result.put(rs.getLong("roomId"), result.get(rs.getLong("roomId")) + minute);
                        } else {
                            result.put(rs.getLong("roomId"), minute);
                        }
                    }
                });
        return result;
    }

    @Override
    public Map<Long, Integer> getRoomArrangedCount(Collection<Long> courseIds, Date startTime, Date endTime) {
        if (CollectionUtils.isEmpty(courseIds)) {
            return Maps.newHashMap();
        }
        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder("id", "roomId", "startTime", "endTime");
        builder.in("courseId", courseIds);
        builder.ge("startTime", startTime);
        builder.le("endTime", endTime);
        builder.eq("delStatus", DeleteStatus.NORMAL.getValue());
        final Map<Long, Integer> result = Maps.newHashMap();
        getNamedJdbcTemplate().query(builder.toSql(), builder.collectConditionValue(),
                new RowCallbackHandler() {
                    @Override
                    public void processRow(ResultSet rs) throws SQLException {
                        if (result.containsKey(rs.getLong("roomId"))) {
                            Integer count = result.get(rs.getLong("roomId")) + 1;
                            result.put(rs.getLong("roomId"), count);
                        } else {
                            result.put(rs.getLong("roomId"), 1);
                        }
                    }
                });
        return result;
    }

    @Override
    public Map<Long, Integer> getRoomClassRecordCount(Long orgId, List<Long> roomIds) {
        final Map<Long, Integer> data = Maps.newHashMap();
        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder();
        builder.count("id", "num");
        builder.eq("orgId", orgId);
        builder.in("roomId", roomIds);
        builder.group("roomId");
        builder.eq("delStatus", DeleteStatus.NORMAL.getValue());
        getNamedJdbcTemplate().query(builder.toSql(), builder.collectConditionValue(), new RowCallbackHandler() {
            @Override
            public void processRow(ResultSet rs) throws SQLException {
                data.put(rs.getLong("roomId"), rs.getInt("num"));
            }
        });
        return data;
    }

    @Override
    public List<Long> listCourseLessonTeacherId(long lessonId) {
        final StringBuilder sb = new StringBuilder();
        final Map<String, Object> params = Maps.newHashMap();
        sb.append("select teacher_id from tts.org_class_lesson lesson join tts.org_teacher_lesson teacher on lesson.id = teacher.lesson_id where lesson.id=:lessonId");
        params.put("lessonId", lessonId);
        return getNamedJdbcTemplate().queryForList(sb.toString(), params, Long.class);
    }

    @Override
    public List<OrgClassLesson> listTeacherLessons(long teacherId, Date startTime, Date endTime) {
        final StringBuilder sb = new StringBuilder();
        final Map<String, Object> params = Maps.newHashMap();
        sb.append("select lesson.* from tts.org_class_lesson lesson join tts.org_teacher_lesson teacher on lesson.id = teacher.lesson_id ");
        sb.append("  where teacher.teacher_id=:teacherId and lesson.start_time >= :startTime and lesson.start_time <= :endTime and lesson.del_status=0");
        params.put("teacherId", teacherId);
        params.put("startTime", startTime);
        params.put("endTime", endTime);
        return getNamedJdbcTemplate().query(sb.toString(), params, new RowMapper<OrgClassLesson>() {
            @Override
            public OrgClassLesson mapRow(ResultSet resultSet, int i) throws SQLException {
                return toOrgClassLesson(resultSet);
            }
        });
    }

    @Override
    public List<Long> listCourseIdByTeacherId(long teacherId) {
        final StringBuilder sb = new StringBuilder();
        final Map<String, Object> params = Maps.newHashMap();
        sb.append("select lesson.course_id from tts.org_class_lesson lesson join tts.org_teacher_lesson teacher on lesson.id = teacher.lesson_id where teacher.teacher_id=:teacherId");
        params.put("teacherId", teacherId);
        return getNamedJdbcTemplate().queryForList(sb.toString(), params, Long.class);
    }

    @Override
    public List<OrgClassLesson> list1V1ByClassId(Long courseId) {
        Preconditions.checkArgument(courseId != null && courseId > 0, "courseId is illegal");
        SingleSqlBuilder<OrgClassLesson> builder = createSqlBuilder();
        builder.eq("courseId", courseId);
        builder.eq("delStatus", DeleteStatus.NORMAL.getValue());
        builder.asc("startTime");
        return queryList(builder);
    }
}
