
package com.baijia.tianxiao.dal.org.dao.impl;

import com.baijia.tianxiao.dal.org.dao.OrgCourseRoomDao;
import com.baijia.tianxiao.dal.org.po.OrgCourseRoom;
import com.baijia.tianxiao.sqlbuilder.SingleSqlBuilder;
import com.baijia.tianxiao.sqlbuilder.support.JdbcTemplateDaoSupport;
import com.baijia.tianxiao.util.CollectorUtil;
import com.baijia.tianxiao.util.query.BatchQueryCallback;
import com.baijia.tianxiao.util.query.BatchQueryTemplate;
import com.baijia.tianxiao.util.query.ListBatchQueryTemplate;
import com.baijia.tianxiao.util.query.MapBatchQueryTemplate;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import org.apache.commons.collections4.CollectionUtils;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.core.RowCallbackHandler;
import org.springframework.stereotype.Repository;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import lombok.extern.slf4j.Slf4j;

@Repository
@Slf4j
public class OrgCourseRoomDaoImpl extends JdbcTemplateDaoSupport<OrgCourseRoom> implements OrgCourseRoomDao {

    public OrgCourseRoomDaoImpl() {
        super(OrgCourseRoom.class);
    }

    @Override
    public List<OrgCourseRoom> getCourseRoomList(Long orgId, Long courseId, String... queryProps) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(courseId != null && courseId > 0, "courseId is illegal");
        SingleSqlBuilder<OrgCourseRoom> builder = createSqlBuilder(queryProps);
        builder.eq("orgId", orgId);
        builder.eq("courseId", courseId);
        builder.desc("id");
        return queryList(builder);
    }

    @Override
    public List<Long> getCourseRoomIds(Long orgId, Long courseId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(courseId != null && courseId > 0, "courseId is illegal");
        SingleSqlBuilder<OrgCourseRoom> builder = createSqlBuilder();
        builder.select("roomId");
        builder.eq("orgId", orgId);
        builder.eq("courseId", courseId);
        builder.desc("id");
        return queryForList(builder, Long.class);
    }

    @Override
    public Map<Long, List<OrgCourseRoom>> getCourseRoomMap(final Long orgId, Collection<Long> courseIds,
                                                           final String... queryProps) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        if (CollectionUtils.isEmpty(courseIds)) {
            return Collections.emptyMap();
        }
        List<OrgCourseRoom> courseRooms = new ListBatchQueryTemplate<Long, OrgCourseRoom>().batchQuery(courseIds,
                new BatchQueryCallback<Long, List<OrgCourseRoom>>() {
                    @Override
                    public List<OrgCourseRoom> doQuery(Collection<Long> querySet) {
                        SingleSqlBuilder<OrgCourseRoom> builder = createSqlBuilder(queryProps);
                        builder.eq("orgId", orgId);
                        builder.in("courseId", querySet);
                        return queryList(builder);
                    }
                });
        return CollectorUtil.group(courseRooms, new Function<OrgCourseRoom, Long>() {
            @Override
            public Long apply(OrgCourseRoom input) {
                return input.getCourseId();
            }
        });
    }

    @Override
    public OrgCourseRoom getCourseDefaultRoom(Long orgId, Long courseId, String... queryProps) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(courseId != null && courseId > 0, "courseId is illegal");
        SingleSqlBuilder<OrgCourseRoom> builder = createSqlBuilder(queryProps);
        builder.eq("orgId", orgId);
        builder.eq("courseId", courseId);
        builder.eq("isDefault", true);
        return uniqueResult(builder);

    }

    @Override
    public List<OrgCourseRoom> getCourseDefaultRoom(final Long orgId, Collection<Long> courseIds,
                                                    final String... queryProps) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        if (CollectionUtils.isEmpty(courseIds)) {
            return Collections.emptyList();
        }
        return new ListBatchQueryTemplate<Long, OrgCourseRoom>().batchQuery(courseIds,
                new BatchQueryCallback<Long, List<OrgCourseRoom>>() {
                    @Override
                    public List<OrgCourseRoom> doQuery(Collection<Long> querySet) {
                        SingleSqlBuilder<OrgCourseRoom> builder = createSqlBuilder(queryProps);
                        builder.eq("orgId", orgId);
                        builder.in("courseId", querySet);
                        return queryList(builder);
                    }
                });
    }

    @Override
    public int delCourseRoom(Long orgId, Long courseId, Collection<Long> roomIds) {
        if (CollectionUtils.isEmpty(roomIds)) {
            log.warn("room id is empty ,skip delete course rooms");
            return 0;
        }
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        SingleSqlBuilder<OrgCourseRoom> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        if (courseId != null) {
            builder.eq("courseId", courseId);
        }
        builder.in("roomId", roomIds);
        return delete(builder);
    }

    @Override
    public void courseDefaultRoomSet(Long orgId, Long courseId, Long roomId, boolean enable) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(courseId != null && courseId > 0, "courseId is illegal");
        Preconditions.checkArgument(roomId != null && roomId > 0, "roomId is illegal");

        Map<String, Object> updateCondition = Maps.newHashMap();
        updateCondition.put("orgId", orgId);
        updateCondition.put("courseId", courseId);
        updateCondition.put("roomId", roomId);
        OrgCourseRoom po = new OrgCourseRoom();
        po.setIsDefault(enable);
        int result = update(updateCondition, po, "isDefault");
        if (enable && result > 0) {
            // 如果启用了默认教室,需要将原来的默认教室设置为false
            SingleSqlBuilder<OrgCourseRoom> builder = createSqlBuilder();
            builder.eq("orgId", orgId);
            builder.eq("courseId", courseId);
            builder.ne("roomId", roomId);
            builder.eq("isDefault", true);
            updateCondition = builder.collectConditionValue();
            updateCondition.put("isDefault", false);
            String sql = builder.toUpdateSql("isDefault");
            log.debug("update other to false sql:{},update params:{}", sql, updateCondition);
            this.getNamedJdbcTemplate().update(sql, updateCondition);
        }
    }

    @Override
    public Map<Long, Integer> getCourseRoomCount(final Long orgId, Collection<Long> courseIds) {
        if (CollectionUtils.isEmpty(courseIds)) {
            return Maps.newHashMap();
        }
        return new MapBatchQueryTemplate<Long, Long, Integer>().batchQuery(courseIds,
                new BatchQueryCallback<Long, Map<Long, Integer>>() {
                    @Override
                    public Map<Long, Integer> doQuery(Collection<Long> querySet) {
                        SingleSqlBuilder<OrgCourseRoom> builder = createSqlBuilder("courseId");
                        builder.count("id", "num");
                        builder.in("courseId", querySet);
                        builder.eq("orgId", orgId);
                        builder.group("courseId");
                        final Map<Long, Integer> result = Maps.newHashMap();
                        getNamedJdbcTemplate().query(builder.toSql(), builder.collectConditionValue(),
                                new RowCallbackHandler() {
                                    @Override
                                    public void processRow(ResultSet rs) throws SQLException {
                                        result.put(rs.getLong("courseId"), rs.getInt("num"));
                                    }
                                });
                        return result;
                    }
                });
    }

    @Override
    public Map<Long, Integer> getOrgRoomTotal() {
        String sql = "select count(1) as count,org_id from tts.org_course_room GROUP BY org_id";
        return this.getNamedJdbcTemplate().query(sql, new ResultSetExtractor<Map<Long, Integer>>() {

            @Override
            public Map<Long, Integer> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<Long, Integer> map = new HashMap<>();
                while (rs.next()) {
                    Long orgId = rs.getLong("org_id");
                    Integer count = rs.getInt("count");
                    map.put(orgId, count);
                }
                return map;
            }
        });
    }


    @Override
    public List<Long> getcoursIdsByRoomIds(Collection<Long> roomIds) {
        if (CollectionUtils.isEmpty(roomIds)) {
            return Lists.newArrayList();
        }
        BatchQueryTemplate<Long, List<Long>> queryTemplate = new ListBatchQueryTemplate<>();
        return queryTemplate.batchQuery(roomIds, new BatchQueryCallback<Long, List<Long>>() {
            @Override
            public List<Long> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<OrgCourseRoom> builder = createSqlBuilder("courseId");
                builder.in("roomId", querySet);
                return queryForList(builder, Long.class);
            }
        });
    }

}
