package com.baijia.tianxiao.dal.solr.query.impl;

import com.baijia.tianxiao.dal.solr.constant.ConsultConstant;
import com.baijia.tianxiao.dal.solr.constant.SolrConstant;
import com.baijia.tianxiao.dal.solr.dto.ConsulterListDto;
import com.baijia.tianxiao.dal.solr.dto.ConsulterListQueryParam;
import com.baijia.tianxiao.dal.solr.dto.TimeRange;
import com.baijia.tianxiao.dal.solr.enums.PCConsultOrderEnum;
import com.baijia.tianxiao.dal.solr.enums.TimeType;
import com.baijia.tianxiao.dal.solr.query.ConsultUserQuery;
import com.baijia.tianxiao.dal.solr.utils.SolrUtil;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.GenericsUtils;
import com.baijia.tianxiao.util.TwoTuple;
import com.baijia.tianxiao.util.date.DateUtil;
import com.google.common.collect.Lists;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.solr.client.solrj.SolrQuery;
import org.apache.solr.client.solrj.SolrRequest;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.apache.solr.client.solrj.util.ClientUtils;
import org.apache.solr.common.SolrDocument;
import org.apache.solr.common.SolrDocumentList;
import org.springframework.stereotype.Repository;

import java.io.IOException;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;

import lombok.extern.slf4j.Slf4j;

/**
 * Created by liuxp on 16/5/24.
 */
@Repository
@Slf4j
public class ConsultUserQueryImpl extends SolrAbstractServiceImpl implements ConsultUserQuery {
    private static final Long ONE_DAY_TIME = 86400000L;

    private static final String keys =
        "id,org_id,is_consulter,finally_hold_time,last_pull_time,last_push_time,last_invalid_time,"
            + "last_browse_time,last_remind_time,user_id,user_number,sex,student_name,student_id,"
            + "name,nick_name,consult_source,mobile,weixin,weixin_nick_name,weixin_app_id,weixin_open_id,"
            + "parent_name,parent_mobile,portrait,intension_level,consult_status,del_status,next_remind_Time,"
            + "school,qq,mail,birthday,degree_class,address,create_time,relatives,update_time,manually,area_id,longitude,"
            + "latitude,pinyin,is_invalid,campus_org_id,cascade_id,custom_search_value";

    private static final String QUERY_NO_THING = "NEED NOT QUERY ANYTHING";

    @Override
    public List<ConsulterListDto> queryConsulter(Collection<Integer> cascadeIds, ConsulterListQueryParam param,
        PageDto pageDto) {

        log.info("consulterListQueryParam is : {} ", param);
        String querySql = getconsultUserIds(cascadeIds, param);
        if (querySql.equals(QUERY_NO_THING)) {
            log.info("query for tag and find nothing , will ret empty list with queryParams:{},cascadeIds are:{}",
                param, cascadeIds);
            return GenericsUtils.emptyList();
        }

        log.info("[solr query is : {} ] ", querySql);

        ArrayList<ConsulterListDto> retList = new ArrayList<>();
        SolrQuery query = new SolrQuery();
        query.setQuery(querySql);

        int start = 0;
        int limit = Integer.MAX_VALUE;
        if (pageDto != null) {
            start = (pageDto.getPageNum() - 1) * pageDto.getPageSize();
            limit = pageDto.getPageSize();
        }
        query.setStart(start);
        query.setRows(limit);
        TwoTuple<String, Integer> sortRule = param.getSortRule();
        log.info("sortRule is -- orderName: {} and sortType:{} ", sortRule.first, sortRule.second);
        query.addSort(sortRule.first, sortRule.second == 0 ? SolrQuery.ORDER.asc : SolrQuery.ORDER.desc);

        QueryResponse response = null;
        try {
            log.info("Solr QueryParam=" + query.toString() + "=========Start=" + start + ";limit=" + limit);
            response = getSolr().query(SolrConstant.CRM_CONSULT_USER_COLLECTION, query, SolrRequest.METHOD.POST);
            SolrDocumentList documentList = response.getResults();
            for (Iterator<SolrDocument> iterator = documentList.iterator(); iterator.hasNext();) {
                retList.add(buildConsulterListDto(iterator.next(), param.getConsulterType()));
            }
            long total = documentList.getNumFound();
            if (pageDto == null) {
                pageDto = new PageDto();
            }
            pageDto.setCount((int) total);
            log.info("[Solr] Query count=" + total);
        } catch (SolrServerException e) {
            log.error("[Solr] query consult user SolrServerException", e);
        } catch (IOException e) {
            log.error("[Solr] query consult user IOException", e);
        }

        return retList;
    }

    private String getconsultUserIds(Collection<Integer> cascadeIds, ConsulterListQueryParam param) {
        StringBuilder sb = new StringBuilder();

        // 筛选是线索且cascade_id在指定集合中的线索
        sb.append("id:c_*  AND del_status:0").append(" AND is_consulter:").append(param.getIsConsulter());
        // 如果当前线索属于主账号，添加主账号筛选条件
        sb.append(createEquals("org_id", param.getOrgId()));

        if (param.getConsulterType() != ConsultConstant.INVALID && GenericsUtils.notNullAndEmpty(cascadeIds)) {
            sb.append(" AND ").append(SolrUtil.or("cascade_id", cascadeIds));
        }

        // 筛选线索的有效性
        if (param.getIsInvalid() != null) {
            sb.append(createEquals("is_invalid", param.getIsInvalid()));
        }

        // 筛选线索目标的性别
        if (param.getSex() != null) {
            sb.append(createEquals("sex", param.getSex()));
        }

        // 筛选线索的来源
        if (CollectionUtils.isNotEmpty(param.listConsultSource())) {
            int size = param.listConsultSource().size();
            if (size == 1) {
                sb.append(createEquals("consult_source", param.listConsultSource().get(0)));
            } else {
                sb.append(createIn("consult_source", param.listConsultSource()));
            }
        }

        // 筛选线索的意向程度
        if (param.getIntentLevel() != null && param.getIntentLevel() != 0) {
            sb.append(createEquals("intension_level", param.getIntentLevel()));
        }

        // 筛选线索的跟进状态
        if (param.getConsultStatus() != null) {
            sb.append(createEquals("consult_status", param.getConsultStatus()));
        }

        // 根据关键字进行筛选 姓名，电话号码，标签
        String keyWrod = param.getKeyword();
        if (GenericsUtils.notNullAndEmpty(keyWrod)) {
            keyWrod = ClientUtils.escapeQueryChars(keyWrod);
            String keyFieldName = PCConsultOrderEnum.getMatchFieldWithDefault(param.getKeyFieldName(), null, true);
            if (keyFieldName != null) {
                log.info("keyFieldName is : {} and queryValue is : {} ", keyFieldName, keyWrod);
                if (!keyFieldName.equals(PCConsultOrderEnum.TAG.matchField)) {
                    if (keyFieldName.equals("custom_search_value")) {
                        sb.append(" AND custom_search_value:").append(StringEscapeUtils.unescapeJava(param.getKeyword()));
                    }else{
                        sb.append(createEquals(keyFieldName, "*" + keyWrod + "*"));
                    }
                    
                } else {
                    Set<Long> queryConsultIdByTags = this.queryConsultIdByTags(keyWrod, param.getOrgId());
                    if (GenericsUtils.notNullAndEmpty(queryConsultIdByTags)) {
                        if (queryConsultIdByTags.size() >= 1000) {
                            Set<Long> subSets = new HashSet<>(1000);
                            Iterator<Long> iterator = queryConsultIdByTags.iterator();
                            int index = 0;
                            while (iterator.hasNext()) {
                                subSets.add(iterator.next());
                                index++;
                                if (index > 1000) {
                                    break;
                                }
                            }
                            queryConsultIdByTags = subSets;
                        }
                        String or = SolrUtil.or("c_self_id", queryConsultIdByTags);
                        sb.append(" AND ").append(or);
                    } else {
                        return QUERY_NO_THING;
                    }
                }
            } else {
                Map<String, String> valByField = new HashMap<>();
                valByField.put("name", "*" + keyWrod + "*"); // 可以在索引的时候将原有数据倒置后添加索引，加快检索
                valByField.put("pinyin", "*" + keyWrod + "*"); // 添加拼音的转换
                if(keyWrod.length()>3){
                	try{
                		new BigDecimal(keyWrod);
                		valByField.put("mobile", "*" + keyWrod + "*");
                	}catch(Exception e){
                	}
                }
                sb.append(" AND ( ").append(SolrUtil.or(valByField));

                Set<Long> queryConsultIdByTags = this.queryConsultIdByTags(keyWrod, param.getOrgId());
                if (GenericsUtils.notNullAndEmpty(queryConsultIdByTags)) {
                    if (GenericsUtils.notNullAndEmpty(queryConsultIdByTags)) {
                        if (queryConsultIdByTags.size() >= 1000) {
                            Set<Long> subSets = new HashSet<>(1000);
                            Iterator<Long> iterator = queryConsultIdByTags.iterator();
                            int index = 0;
                            while (iterator.hasNext()) {
                                subSets.add(iterator.next());
                                index++;
                                if (index > 1000) {
                                    break;
                                }
                            }
                            queryConsultIdByTags = subSets;
                        }
                        String or = SolrUtil.or("c_self_id", queryConsultIdByTags);
                        sb.append(" OR ").append(or);
                    }
                }
                sb.append(")");
            }
        }

        // 对于转为学员或者本身为学员的线索不进行处理
        // sb.append(createEquals("user_id", ConsultConstant.CONSULT_USER_ID_DEFAULT_VALUE));
        sb.append(createEquals("student_id", ConsultConstant.CONSULT_STUDENT_ID_DEFAULT_VALUE));

        // 最近几天待跟进,从tx_consult_user表中获取下次提醒时间
        Integer remindDayNum = param.getRemindDayNum();
        Date now = new Date();
        if (remindDayNum != null) {
            TimeRange timeRange = new TimeRange();
            timeRange.setStartTime(DateUtil.getStartOfDay(now));
            timeRange.setEndTime(DateUtil.getStartOfDay(new Date(now.getTime() + remindDayNum * ONE_DAY_TIME)));
            sb.append(createTimeRangeSql("next_remind_time", timeRange, true));
        }

        // 最近几天内创建
        Integer createDayNum = param.getCreateDayNum();
        if (createDayNum != null) {
            TimeRange timeRange = new TimeRange();
            timeRange.setStartTime(DateUtil.getStartOfDay(new Date(now.getTime() - (createDayNum - 1) * ONE_DAY_TIME)));
            timeRange.setEndTime(now);
            sb.append(createTimeRangeSql("create_time", timeRange));
        }

        // 处理最近几天标记为无效
        Integer setInvalidDayNum = param.getInvalidDayNum();
        if (setInvalidDayNum != null) {
            TimeRange timeRange = new TimeRange();
            timeRange
                .setStartTime(DateUtil.getStartOfDay(new Date(now.getTime() - (setInvalidDayNum - 1) * ONE_DAY_TIME)));
            timeRange.setEndTime(now);
            sb.append(createTimeRangeSql("last_invalid_time", timeRange));
        }

        // 处理剩余跟进时间
        Integer minRemainingDay = param.getMinRemainingDay();
        Integer maxRemainingDay = param.getMaxRemainingDay();
        if (minRemainingDay != null && maxRemainingDay != null) {
            if (minRemainingDay <= 1) {
                minRemainingDay = 0;
            }
            if (maxRemainingDay <= 1) {
                maxRemainingDay = 1;
            }
            if (minRemainingDay > maxRemainingDay) {
                Integer temp = minRemainingDay;
                minRemainingDay = maxRemainingDay;
                maxRemainingDay = temp;
            }
            Long minDayDept = now.getTime() + minRemainingDay * ONE_DAY_TIME;
            Long maxDayDept = now.getTime() + (maxRemainingDay + 1) * ONE_DAY_TIME;
            TimeRange timeRange = new TimeRange();
            timeRange.setStartTime(DateUtil.getStartOfDay(new Date(minDayDept)));
            timeRange.setEndTime(DateUtil.getStartOfDay(new Date(maxDayDept)));
            sb.append(createTimeRangeSql("finally_hold_time", timeRange));

        }

        // 最后跟进时间
        if (param.getLastRemindTimeState() != null) {
            TimeType tr = TimeType.getTimeTypeByCode(param.getLastRemindTimeState());
            if (tr != null) {
                TimeRange timeRange = tr.timeRange();
                sb.append(createTimeRangeSql("last_remind_time", timeRange, true));
            }
        }

        // 按时间筛选
        if (param.getDateFieldKey() != null) {
            String dateFieldKey = ClientUtils.escapeQueryChars(param.getDateFieldKey());
            String keyFieldName = PCConsultOrderEnum.getMatchFieldWithDefault(dateFieldKey, null, true);
            if (keyFieldName != null) {
                sb.append(createTimeRangeSql(keyFieldName, new TimeRange(param.getStart(),param.getEnd()), true));
            }
        }

        // 处理最近七天回归的线索
        Integer backDayNum = param.getBackDayNum();
        if (backDayNum != null) {
            TimeRange timeRange = new TimeRange();
            timeRange.setStartTime(DateUtil.getStartOfDay(new Date(now.getTime() - (backDayNum - 1) * ONE_DAY_TIME)));
            timeRange.setEndTime(now);
            sb.append(createTimeRangeSql("last_push_time", timeRange));
        }

        // 处理最近七天浏览的线索
        Integer recentBrowseDayNum = param.getRecentBrowseDayNum();
        if (recentBrowseDayNum != null) {
            TimeRange timeRange = new TimeRange();
            timeRange.setStartTime(
                DateUtil.getStartOfDay(new Date(now.getTime() - (recentBrowseDayNum - 1) * ONE_DAY_TIME)));
            timeRange.setEndTime(DateUtil.getStartOfDay(new Date(now.getTime() + ONE_DAY_TIME)));
            sb.append(createTimeRangeSql("last_browse_time", timeRange));
        }

        // 处理最近几天未跟进的线索
        Integer notFollowDayNum = param.getUnRemindDayNum();
        if (notFollowDayNum != null) {
            TimeRange timeRange = new TimeRange();
            timeRange.setStartTime(null);
            timeRange
                .setEndTime(DateUtil.getStartOfDay(new Date(now.getTime() - (notFollowDayNum - 1) * ONE_DAY_TIME)));
            sb.append(" AND ((*:* NOT last_remind_time:*) OR ");
            sb.append(createTimeRangeSql("last_remind_time", timeRange).replace(" AND ", "")).append(" )");
        }

        log.info("[ Solr Qusery is: {}  ]", sb.toString());

        return sb.toString();
    }

    private String createEquals(String fieldName, Object value) {
        return " AND " + SolrUtil.equal(fieldName, value) + " ";
    }

    private String createIn(String fieldName, Collection<?> value) {
        return " AND " + SolrUtil.or(fieldName, value) + " ";
    }

    public ConsulterListDto buildConsulterListDto(SolrDocument doc, Integer consulterType) {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        ConsulterListDto dto = new ConsulterListDto();
        dto.setName((String) doc.get("name"));
        dto.setMobile((String) doc.get("mobile"));
        dto.setPortrait((String) doc.get("portrait"));
        dto.setId((Long) doc.get("c_self_id"));
        dto.setStudentId((Long) doc.get("student_id"));
        dto.setWenxinOpenId((String)doc.get("weixin_open_id"));
        Date finallyHoldTime = (Date) doc.get("finally_hold_time");
        Date now = new Date();
        if (finallyHoldTime != null) {
            finallyHoldTime = getDateFromSolr(sdf, finallyHoldTime);
        }
        Integer lessDayNum = 0;
        if (finallyHoldTime != null && finallyHoldTime.after(now)) {
            if (DateUtil.getStartOfDayAccurateToMillSeconde(finallyHoldTime)
                .equals(DateUtil.getStartOfDayAccurateToMillSeconde(now))) {
                lessDayNum = 1;
            } else {
                lessDayNum = (int) ((DateUtil.getStartOfDayAccurateToMillSeconde(finallyHoldTime).getTime()
                    - DateUtil.getStartOfDayAccurateToMillSeconde(now).getTime()) / ONE_DAY_TIME);
            }
            if (lessDayNum == 0) {
                lessDayNum = 1;
            }
        }
        log.info("lessDayNum is : {} for finallyHoldTime : {} ", lessDayNum, finallyHoldTime);
        dto.setTimeRemaining(lessDayNum);
        String inital = (String) doc.get("pinyin");
        if (GenericsUtils.notNullAndEmpty(inital)) {
            inital = inital.substring(0, 1).toUpperCase();
            if (inital.equals("~")) {
                inital = "#";
            }
        } else {
            inital = "#";
        }
        dto.setInitial(inital);

        dto.setIsInvalid((Integer) doc.get("is_invalid"));
        dto.setCascadeId((Integer) (doc.get("cascade_id")));
        dto.setConsulterType(consulterType);

        Date birthday = getDateFromSolr(sdf, (Date) doc.get("birthday"));
        dto.setBirthday(birthday);
        dto.setCascadeId((Integer) doc.get("cascade_id"));
        dto.setConsultSource((Integer) doc.get("consult_source"));
        dto.setConsultStatus((Integer) doc.get("consult_status"));
        dto.setDegreeClass(String.valueOf(doc.get("degree_class")));
        dto.setIntensionLevel((Integer) doc.get("intension_level"));
        dto.setMail(String.valueOf(doc.get("mail")));
        dto.setQq(String.valueOf(doc.get("qq")));
        dto.setAddress(String.valueOf(doc.get("address")));
        dto.setRelatives((Integer) (doc.get("relatives")));
        dto.setSchool(String.valueOf(doc.get("school")));
        dto.setSex((Integer) (doc.get("sex")));
        dto.setParentName(String.valueOf(doc.get("parent_name")));
        dto.setParentMobile(String.valueOf(doc.get("parent_mobile")));
        dto.setNextRemindTime(this.getDateFromSolr(sdf, (Date) doc.get("next_remind_time")));
        dto.setCreateTime(this.getDateFromSolr(sdf, (Date) doc.get("create_time")));
        dto.setLastRemindTime(this.getDateFromSolr(sdf, (Date) doc.get("last_remind_time")));
        return dto;
    }

    /**
     * @param sdf
     * @param date
     * @return
     */

    private Date getDateFromSolr(SimpleDateFormat sdf, Date date) {
        if (date == null) {
            return null;
        }
        Long time = date.getTime();
        int timeDiff = TimeZone.getDefault().getRawOffset() - TimeZone.getTimeZone("GMT").getRawOffset();// 本地时区和目标时区差
        Long time1 = time - timeDiff;
        date = new Date(time1);
        log.debug("===== time is : {} ", sdf.format(date));
        return date;
    }

    public Set<Long> queryConsultIdByTags(String tag, Long orgId) {
        StringBuilder sb = new StringBuilder();
        sb.append("id:t_* AND ").append(" org_id:").append(orgId).append(" AND content:*").append(tag).append("*")
            .append(" AND user_id:0 ");
        SolrQuery query = new SolrQuery();
        log.info("[Solr] query tag param:" + sb.toString());
        query.setQuery(sb.toString());
        query.setRows(Integer.MAX_VALUE);

        StringBuilder fq1 = new StringBuilder("-t_consult_user_id:").append("0");
        query.addFilterQuery(fq1.toString());
        QueryResponse response = null;
        Set<Long> consultUserIds = new HashSet<>();
        try {
            response = getSolr().query(SolrConstant.CRM_STUDENT_COLLECTION, query, SolrRequest.METHOD.POST);
            SolrDocumentList list = response.getResults();
            log.info("[Solr] query tag result:" + list.getNumFound());
            for (Iterator<SolrDocument> iterator = list.iterator(); iterator.hasNext();) {
                SolrDocument doc = iterator.next();
                Long userId = (Long) doc.get("t_consult_user_id");
                if (userId == null) {
                    continue;
                }
                log.info("find userId is : {} ", userId);
                consultUserIds.add(userId);
            }
            if (GenericsUtils.isNullOrEmpty(consultUserIds)) {
                return Collections.emptySet();
            }
            return consultUserIds;
        } catch (SolrServerException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }
        return null;
    }

    public String queryConsultIdByTagsStr(String tag, Long orgId) {
        StringBuilder sb = new StringBuilder();
        sb.append("id:t_* AND ").append(" org_id:").append(orgId).append(" AND content:*").append(tag).append("*")
            .append(" AND user_id:0 ").append(" AND ").append(SolrUtil.notEqual("t_consult_user_id", 0));
        return sb.toString();
    }

    @Override
    public Integer countConsulter(Collection<Integer> cascadeIds, ConsulterListQueryParam param) {
        String queryStr = this.getconsultUserIds(cascadeIds, param);
        SolrQuery query = new SolrQuery();
        query.setQuery(queryStr);
        QueryResponse response = null;
        try {
            log.info("Solr QueryParam=" + query.toString());
            response = getSolr().query(SolrConstant.CRM_CONSULT_USER_COLLECTION, query, SolrRequest.METHOD.POST);
            SolrDocumentList documentList = response.getResults();
            long total = documentList.getNumFound();
            log.info("[Solr] Query count=" + total);
            return (int) total;
        } catch (SolrServerException e) {
            log.error("[Solr] query consult user SolrServerException", e);
        } catch (IOException e) {
            log.error("[Solr] query consult user IOException", e);
        }
        return 0;
    }

    @Override
    public void updateOldRow(Map<String, String> rs) throws SolrServerException, IOException {
        rs = mapKeyToUpper(rs);
        Map<String, Object> insertKeyMap = getValueMap(rs);
        log.info("Debug insertKeyMap" + insertKeyMap + "collection:" + SolrConstant.CRM_CONSULT_USER_COLLECTION);
        this.update(SolrConstant.CRM_CONSULT_USER_COLLECTION, insertKeyMap);
        //this.getSolr().commit(SolrConstant.CRM_CONSULT_USER_COLLECTION, true, true, true);
    }

    @Override
    public void deleteOldRow(Map<String, String> rs) throws SolrServerException, IOException {
        rs = mapKeyToUpper(rs);
        String key = rs.get("ID");
        String query = "id:c_" + key;
        this.delete(SolrConstant.CRM_CONSULT_USER_COLLECTION, query);
        //this.getSolr().commit(SolrConstant.CRM_CONSULT_USER_COLLECTION, true, true, true);
    }

    @Override
    public void insertNewRow(Map<String, String> rs) throws SolrServerException, IOException {
        rs = mapKeyToUpper(rs);
        Map<String, Object> insertKeyMap = getValueMap(rs);
        this.add(SolrConstant.CRM_CONSULT_USER_COLLECTION, insertKeyMap);
        //this.getSolr().commit(SolrConstant.CRM_CONSULT_USER_COLLECTION, true, true, true);
    }

    private Map<String, Object> getValueMap(Map<String, String> rs) {
        Map<String, Object> retMap = new HashMap<>();
        log.info("rs map is : {} ", rs);
        Map<String, Object> newValues = new HashMap<>();
        for (String key : new ArrayList<>(Arrays.asList(keys.split(",")))) {
            key = key.trim().toUpperCase();
            String value = rs.get(key);
            log.info("key is : {} and value is :{} ", key, value);
            newValues.put(key, value);
        }

        String id = (String) newValues.get("ID");
        if (id != null) {
            newValues.put("ID", "c_" + id);
            newValues.put("C_SELF_ID", id);
        } else {
            log.info("can not find id field ");
        }
        log.info("ID is : {} ", newValues.get("ID"));

        Object pinyin = newValues.get("PINYIN");
        if (pinyin != null && StringUtils.isNotBlank(pinyin.toString())) {
            char init = pinyin.toString().charAt(0);
            if (init < 'a' || init > 'z') {
                pinyin = "~";
            }
        } else {
            pinyin = "~";
        }
        newValues.put("PINYIN", pinyin);

        if (newValues.get("CUSTOM_SEARCH_VALUE") != null ){
            String[] array = newValues.get("CUSTOM_SEARCH_VALUE").toString().split(" ");
            List<String> customSearchValueList = Lists.newArrayList();
            if(array.length>0){
                customSearchValueList = Lists.newArrayList(array);
            }
            newValues.put("CUSTOM_SEARCH_VALUE",customSearchValueList);
        }
        
        try {
            solveFields("c_self_id,org_id,user_id,is_consulter,user_number,relatives,student_id,avatar,campus_org_id"
                .toUpperCase(), newValues, Integer.class);
            solveFields(
                "consult_source,intension_level,consult_status,del_status,gender,branch_id,status,cascade_id,is_invalid,manually,area_id"
                    .toUpperCase(),
                newValues, Long.class);
            solveFields("longitude,latitude".toUpperCase(), newValues, Double.class);
            solveFields(
                "finally_hold_time,last_pull_time,last_push_time,last_invalid_time,last_browse_time,create_time,last_remind_time,update_time,next_remind_Time,birthday"
                    .toUpperCase(),
                newValues, Date.class);

            for (Map.Entry<String, Object> entry : newValues.entrySet()) {
                retMap.put(entry.getKey().toLowerCase(), entry.getValue());
            }

        } catch (Exception e) {
            log.error("can not insert new values cause by : {} ", e);
        }
        log.info("after populate , the valueMap is : {} ", retMap);
        return retMap;
    }

}
