package com.baijia.tianxiao.dal.solr.query.impl;

import java.io.IOException;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.solr.client.solrj.SolrQuery;
import org.apache.solr.client.solrj.SolrRequest;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.response.FacetField;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.apache.solr.client.solrj.util.ClientUtils;
import org.apache.solr.common.SolrDocument;
import org.apache.solr.common.SolrDocumentList;
import org.springframework.stereotype.Repository;

import com.baijia.commons.lang.utils.collection.CollectionUtils;
import com.baijia.tianxiao.dal.solr.constant.SolrConstant;
import com.baijia.tianxiao.dal.solr.dto.StudentDto;
import com.baijia.tianxiao.dal.solr.dto.StudentQueryParam;
import com.baijia.tianxiao.dal.solr.dto.TimeRange;
import com.baijia.tianxiao.dal.solr.enums.OpType;
import com.baijia.tianxiao.dal.solr.enums.PCStudentOrderEnum;
import com.baijia.tianxiao.dal.solr.enums.StudentLessonStatus;
import com.baijia.tianxiao.dal.solr.exceptions.SolrException;
import com.baijia.tianxiao.dal.solr.po.StudentClass;
import com.baijia.tianxiao.dal.solr.po.StudentClassHour;
import com.baijia.tianxiao.dal.solr.po.StudentStatusStatistics;
import com.baijia.tianxiao.dal.solr.query.CrmStudentQuery;
import com.baijia.tianxiao.dal.solr.utils.SolrUtil;
import com.baijia.tianxiao.enums.CrmErrorCode;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.GenericsUtils;
import com.baijia.tianxiao.util.HanZiPinYinUtils;
import com.baijia.tianxiao.util.TupleUtil;
import com.baijia.tianxiao.util.TwoTuple;
import com.baijia.tianxiao.util.date.DateUtil;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import lombok.extern.slf4j.Slf4j;

/**
 * Created by liuxp on 16/4/28.
 */
@Repository
@Slf4j
public class CrmStudentQueryImpl extends SolrAbstractServiceImpl implements CrmStudentQuery {
    private static final int MAX_QUERY_LIMIT = 1000;

    private static final String SOLR_DATE_FORMATTER = "yyyy-MM-dd'T'HH:mm:ss'Z'";

    private ExecutorService executorService = Executors.newCachedThreadPool();

    @Override
    public List<StudentStatusStatistics> queryCountByStatus(long orgId, boolean needDataAuthority,
        Collection<Long> studentIds, Integer cascadeId) {
        long begin = System.currentTimeMillis();

        List<StudentStatusStatistics> result = StudentStatusStatistics.getDefaut();
        StringBuilder sb = new StringBuilder();
        sb.append("id:s_* AND del_status:0 AND org_id:").append(orgId);
        if (needDataAuthority) {
            sb.append(" AND (");
            if (studentIds != null && studentIds.size() > 0) {
                sb.append(SolrUtil.or("user_id", studentIds));
                sb.append(" OR ");
            }
            sb.append(" add_cascade_id:").append(cascadeId);
            sb.append(")");
        }
        SolrQuery query = new SolrQuery();
        query.setFacet(true);
        query.setRows(0);
        query.addFacetField("lesson_status");
        log.info("[Solr] query param:" + sb.toString());
        query.setQuery(sb.toString());

        Map<Integer, Integer> statusCount = Maps.newHashMap();
        try {
            QueryResponse response =
                getSolr().query(SolrConstant.CRM_STUDENT_COLLECTION, query, SolrRequest.METHOD.POST);
            List<FacetField> facets = response.getFacetFields();
            for (FacetField facet : facets) {
                if (facet.getName().equals("lesson_status")) {
                    List<FacetField.Count> counts = facet.getValues();
                    for (FacetField.Count count : counts) {
                        statusCount.put(Integer.parseInt(count.getName()), (int) count.getCount());
                    }
                }
            }
            log.info("[Solr] Query status cost={},orgId={}", (System.currentTimeMillis() - begin), orgId);
        } catch (SolrServerException e) {
            log.error("[Solr] SolrServerException ", e);
            throw new SolrException(CrmErrorCode.SOLR_EXCEPTION);
        } catch (IOException e) {
            log.error("[Solr] IOException ", e);
            throw new SolrException(CrmErrorCode.SOLR_EXCEPTION);
        }

        log.info("countMap :{} ", statusCount);
        for (StudentStatusStatistics sss : result) {
            List<Integer> status = StudentLessonStatus.findSubStatus(sss.getStatus());
            sss.setNumberFromMap(statusCount, status);
            log.info("sss:{} and status:{} ", sss, status);
        }

        log.info("[Solr] Query status cost={},orgId={}", (System.currentTimeMillis() - begin), orgId);
        return result;
    }

    @Override
    public List<StudentClassHour> queryStudentClassCount(Long orgId,Collection<Long> userIds, PageDto pageDto) {
        List<StudentClassHour> ret = new ArrayList<>();
        StringBuilder sb = new StringBuilder();
        sb.append("id:stu_lesson_* AND org_id:" + orgId);
        if(userIds!=null && userIds.size()>0){
            sb.append(" AND ").append(SolrUtil.or("student_id", userIds));
        }
        SolrQuery query = new SolrQuery();
        log.info("[Solr] query param:" + sb.toString());
        int start = (pageDto.getPageNum() - 1) * pageDto.getPageSize();
        int limit = pageDto.getPageSize();
        query.addSort("student_id", SolrQuery.ORDER.desc);
        query.setStart(start);
        query.setRows(limit);
        query.setQuery(sb.toString());
        QueryResponse response = null;
        try {
            response = getSolr().query(SolrConstant.CRM_STUDENT_COLLECTION, query, SolrRequest.METHOD.POST);
            SolrDocumentList list = response.getResults();
            for (Iterator<SolrDocument> iterator = list.iterator(); iterator.hasNext();) {
                SolrDocument document = iterator.next();
                StudentClassHour classHour = buildStudentClassHour(document);
                ret.add(classHour);
            }
        } catch (SolrServerException e) {
            log.error("[Solr] SolrServerException ", e);
            throw new SolrException(CrmErrorCode.SOLR_EXCEPTION);
        } catch (IOException e) {
            log.error("[Solr] IOException ", e);
            throw new SolrException(CrmErrorCode.SOLR_EXCEPTION);
        }
        return ret;
    }

    /**
     * 根据tag查询用户ID
     *
     * @param tag
     * @param orgId
     * @return
     */
    public Set<Long> queryUserIdsByTag(String tag, Long orgId) {
        Set<Long> userIds = new HashSet<>();
        if (tag == null) {
            return userIds;
        }
        StringBuilder sb = new StringBuilder();
        sb.append("id:t_* AND ").append(" org_id:" + orgId).append(" AND content:*" + tag + "*");
        SolrQuery query = new SolrQuery();
        log.info("[Solr] query tag param:" + sb.toString());
        query.setQuery(sb.toString());
        query.setRows(Integer.MAX_VALUE);
        QueryResponse response = null;
        try {
            response = getSolr().query(SolrConstant.CRM_STUDENT_COLLECTION, query, SolrRequest.METHOD.POST);
            SolrDocumentList list = response.getResults();
            log.info("[Solr] query tag result:" + list.getNumFound());
            for (Iterator<SolrDocument> iterator = list.iterator(); iterator.hasNext();) {
                SolrDocument doc = iterator.next();
                Long userId = (Long) doc.get("user_id");
                userIds.add(userId);
            }
        } catch (SolrServerException e) {
            log.error("[Solr] SolrServerException ", e);
            throw new SolrException(CrmErrorCode.SOLR_EXCEPTION);
        } catch (IOException e) {
            log.error("[Solr] IOException ", e);
            throw new SolrException(CrmErrorCode.SOLR_EXCEPTION);
        }
        return userIds;
    }

    /**
     * 查询报班学员
     *
     * @param startTime
     * @param endTime
     * @param orgId
     * @return
     */
    private Set<Long> queryUserIdsByEnrollTime(Date startTime, Date endTime, long orgId) {
        StringBuilder sb = new StringBuilder();
        sb.append("id:e_* AND del_status:0 AND ").append(" org_id:" + orgId);
        if (startTime != null && endTime != null) {
            SimpleDateFormat sdf = new SimpleDateFormat(SOLR_DATE_FORMATTER);
            sb.append(" AND enroll_time:[" + sdf.format(startTime) + " TO " + sdf.format(endTime) + "}");
        }

        Set<Long> userIds = new HashSet<>();
        SolrDocumentList list = getDocumentList(SolrConstant.STUDENT_COURSE_COLLECTION, sb.toString());
        log.info("[Solr] query enroll result:" + list.getNumFound());
        for (Iterator<SolrDocument> iterator = list.iterator(); iterator.hasNext();) {
            SolrDocument doc = iterator.next();
            Long userId = (Long) doc.get("user_id");
            userIds.add(userId);
        }

        return userIds;
    }

    @Override
    public List<StudentDto> queryStudent(StudentQueryParam param, PageDto pageDto) {
        long begin = System.currentTimeMillis();
        log.info("QueryParam=" + param.toString());
        List<StudentDto> retList = new ArrayList<>();

        Set<Long> querySet = param.getStudentIds();
        // 员工按班级查询
        // Set<Long> ownCourseIds = null;

        // 查询当前机构的所有学生
        StringBuilder sb = new StringBuilder();
        sb.append("id:s_* AND del_status:0 AND org_id:" + param.getOrgId());
        SolrQuery query = new SolrQuery();

        try {
            List<Integer> lessonStatus = Lists.newArrayList();
            if ((param.getStatus() != null && param.getStatus() != StudentLessonStatus.ALL)) {
                lessonStatus.addAll(StudentLessonStatus.findSubStatus(param.getStatus().getStatus()));
                sb.append(" AND ").append(SolrUtil.or("lesson_status", lessonStatus));
            }

            TimeRange range = null;
            Set<Long> enrollUserIds = null;

            // 报名日期
            if (param.getEnrollTime() != null) {
                range = param.getEnrollTime().timeRange();
                enrollUserIds = queryUserIdsByEnrollTime(range.getStartTime(), range.getEndTime(), param.getOrgId());
            }

            // 最近7天报名
            if (param.getOpType() != null && param.getOpType() == OpType.ENROLL) {
                enrollUserIds = queryUserIdsByEnrollTime(DateUtil.getStartOfDay(DateUtil.getOffSetDate(new Date(), -6)),
                    DateUtil.getStartOfDay(DateUtil.getOffSetDate(new Date(), 1)), param.getOrgId());
            }

            if (enrollUserIds != null) {
                if (!querySet.isEmpty()) {
                    querySet.retainAll(enrollUserIds);
                } else {
                    querySet = new LinkedHashSet<>();
                    querySet.addAll(enrollUserIds);
                }
                if (querySet.isEmpty()) {
                    return Collections.emptyList();
                }
            }
            log.info("[Solr] querySet=" + querySet + ";StudentIds=" + param.getStudentIds());

            if (param.getLessonStatus() != null) {
                if (param.getLessonStatus() == 1) {// 已排课
                    query.setFilterQueries("{!join from=student_id to=user_id}id:stu_lesson_*");
                } else {// 未排课
                    query.setFilterQueries("-{!join from=student_id to=user_id}id:stu_lesson_*");
                }
            }

            // 按课时筛选
            if (param.getLeftMinClassHour() != null || param.getLeftMaxClassHour() != null) {
                if (param.getLessonStatus() != null && param.getLessonStatus() == 2) {// 未排课和按课时筛选是互斥的
                    return Collections.emptyList();
                }
                
                //{!join from=student_id to=user_id v='leftArrangeTime:[%s TO %s] or left_count:[%s TO %s]'}id:stu_lesson_*
                String fqStrFormat =
                    "{!join from=student_id to=user_id v='left_count:[%s TO %s]'}id:stu_lesson_*";

                Object leftMinClassTimes = null;
                Object maxClassTimes = null;
                if (param.getLeftMinClassHour() != null) {
                    leftMinClassTimes = param.getLeftMinClassHour();
                } else {
                    leftMinClassTimes = "*";
                }

                if (param.getLeftMaxClassHour() != null) {
                    maxClassTimes = param.getLeftMaxClassHour();
                } else {
                    maxClassTimes = "*";
                }

                String fqStr = String.format(fqStrFormat, leftMinClassTimes, maxClassTimes);
                log.info("[Solr] fq = {}", fqStr);
                query.setFilterQueries(fqStr);
            }

            SimpleDateFormat sdf = new SimpleDateFormat(SOLR_DATE_FORMATTER);

            if (param.getOpType() != null) {
                Date start = null;
                Date end = null;
                switch (param.getOpType()) {
                    case FOLLOW:
                        start = DateUtil.getStartOfDay(new Date());
                        end = DateUtil.getEndOfDay(DateUtil.getOffSetDate(start, 6));
                        sb.append(" AND ");
                        sb.append("next_remind_time:");
                        sb.append("[" + sdf.format(start) + " TO " + sdf.format(end) + "]");
                        break;
                    case CREATE:
                        end = DateUtil.getEndOfDay(new Date());
                        start = DateUtil.getStartOfDay(DateUtil.getOffSetDate(new Date(), -6));
                        sb.append(" AND ");
                        sb.append("create_time:");
                        sb.append("[" + sdf.format(start) + " TO " + sdf.format(end) + "]");
                        break;
                    default:
                        log.info("[Solr] OPType=" + param.getOpType());
                        break;
                }
            }

            if (param.getCreateTime() != null) {
                range = param.getCreateTime().timeRange();
                sb.append(" AND ");
                sb.append(
                    "create_time:[" + sdf.format(range.getStartTime()) + " TO " + sdf.format(range.getEndTime()) + "}");
            }
            if (param.getFollowTime() != null) {
                range = param.getFollowTime().timeRange();
                sb.append(" AND ");
                sb.append("next_remind_time:[" + sdf.format(range.getStartTime()) + " TO "
                    + sdf.format(range.getEndTime()) + "}");
            }

            if (param.getGender() != null) {
                sb.append(" AND ").append("gender:").append(param.getGender());
            }

            // 跟进人
            if (GenericsUtils.notNullAndEmpty(param.getAddCascadeIds())) {
                sb.append(" AND ").append(SolrUtil.or("add_cascade_id", param.getAddCascadeIds()));
            }

            // for app search
            if (StringUtils.isNotBlank(param.getSearchKey())) {
                param.setSearchKey(ClientUtils.escapeQueryChars(param.getSearchKey()));

                sb.append(" AND (name:*").append(param.getSearchKey()).append("*");
                sb.append(" OR pinyin:*").append(param.getSearchKey()).append("*");
                if (param.getSearchKey().length() > 3) {
                    try {
                        new BigDecimal(param.getSearchKey());
                        sb.append(" OR mobile:*").append(param.getSearchKey()).append("*");
                    } catch (Exception e) {
                    }
                }

                // .append("* OR parent_mobile:*").append(param.getSearchKey()).append("* OR parent_name:*")
                // .append(param.getSearchKey()).append("*") those code is note by Rezar
                Set<Long> userIdWithTags = queryUserIdsByTag(param.getSearchKey(), param.getOrgId());
                if (userIdWithTags != null && userIdWithTags.size() > 0) {
                    if (GenericsUtils.notNullAndEmpty(userIdWithTags)) {
                        if (userIdWithTags.size() >= 1000) {
                            Set<Long> subSets = new HashSet<>(1000);
                            Iterator<Long> iterator = userIdWithTags.iterator();
                            int index = 0;
                            while (iterator.hasNext()) {
                                subSets.add(iterator.next());
                                index++;
                                // 傻逼,这里直接判断userIdWithTags的大小就行了....
                                if (index > 1000) {
                                    break;
                                }
                            }
                            userIdWithTags = subSets;
                        }
                        sb.append(" OR ").append(SolrUtil.or("user_id", userIdWithTags));
                    }
                }
                sb.append(" )");
            }

            // for pc search
            if (StringUtils.isNotBlank(param.getQueryStr())) {
                param.setQueryValue(ClientUtils.escapeQueryChars(param.getQueryValue()));
                // 对特殊字符进行转义处理
                if (param.getQueryStr().equals("tag")) {
                    Set<Long> ids = queryUserIdsByTag(param.getQueryValue(), param.getOrgId());
                    if (GenericsUtils.notNullAndEmpty(ids)) {
                        sb.append(" AND ").append(SolrUtil.or("user_id", ids));
                    } else {
                        // 如果筛选了标签但没有数据,直接返回空集合
                        return GenericsUtils.emptyList();
                    }
                }
                if (param.getQueryStr().equals("mobile")) {
                    sb.append(" AND mobile:*").append(param.getQueryValue()).append("*");
                }
                if (param.getQueryStr().equals("name")) {
                    sb.append(" AND name:*").append(param.getQueryValue()).append("*");
                }
                if (param.getQueryStr().equals("parentName")) {
                    sb.append(" AND parent_name:*").append(param.getQueryValue()).append("*");
                }
                if (param.getQueryStr().equals("parentMobile")) {
                    sb.append(" AND parent_mobile:*").append(param.getQueryValue()).append("*");
                }
                if (param.getQueryStr().startsWith("customSearchValue")) {
                    sb.append(" AND custom_search_value:")
                        .append(StringEscapeUtils.unescapeJava(param.getQueryValue()));
                }
            }

            int start = (pageDto.getPageNum() - 1) * pageDto.getPageSize();
            int limit = pageDto.getPageSize();
            log.info("[Solr] QueryParam=" + query.toString() + "Start=" + start + ";limit=" + limit);
            query.setStart(start);
            query.setRows(limit);
            SolrQuery.ORDER order = (param.getOrderType() == 0 ? SolrQuery.ORDER.asc : SolrQuery.ORDER.desc);
            log.info("order is :{} , orderField:{} ", param.getOrderType(),
                PCStudentOrderEnum.find(param.getOrderName()));
            if (GenericsUtils.notNullAndEmpty(param.getOrderName())) {
                query.addSort(PCStudentOrderEnum.find(param.getOrderName()), order);
            } else {
                query.addSort("pinyin", SolrQuery.ORDER.asc);
            }

            String queryStr = sb.toString();
            log.info("[Solr] queryStr=" + queryStr);
            long total = 0;
            log.info("[Solr] querySet=" + querySet);
            if (querySet != null && querySet.size() > 0) {
                if (querySet.size() > MAX_QUERY_LIMIT) {
                    ArrayList<Long> list = new ArrayList<Long>(querySet.size());
                    list.addAll(querySet);
                    for (int i = 0; i * MAX_QUERY_LIMIT < querySet.size(); i++) {
                        int from = i * MAX_QUERY_LIMIT;
                        int to = Math.min((i + 1) * MAX_QUERY_LIMIT, querySet.size());
                        query.setStart(0);
                        query.setRows(MAX_QUERY_LIMIT);
                        List<Long> subList = list.subList(from, to);
                        String str = SolrUtil.or("user_id", subList);
                        String executeQueryStr = queryStr + " AND " + str;
                        query.setQuery(executeQueryStr);
                        log.info("[Solr] student query param= " + executeQueryStr);
                        QueryResponse response =
                            getSolr().query(SolrConstant.CRM_STUDENT_COLLECTION, query, SolrRequest.METHOD.POST);
                        SolrDocumentList documentList = response.getResults();
                        for (Iterator<SolrDocument> iterator = documentList.iterator(); iterator.hasNext();) {
                            StudentDto sd = buildStudent(iterator.next());
                            retList.add(sd);
                        }
                    }
                    if (start > retList.size()) {
                        return Collections.emptyList();
                    }
                    total = retList.size();
                    Collections.sort(retList, new Comparator<StudentDto>() {
                        @Override
                        public int compare(StudentDto o1, StudentDto o2) {
                            return o1.getPinyin().compareTo(o2.getPinyin());
                        }
                    });
                    retList = retList.subList(start, Math.min(retList.size(), start + limit));
                    pageDto.setCount((int) total);
                    log.info("[Solr] Query count=" + total);
                    // setClassInfo(retList, param.getOrgId(), ownCourseIds); // 设置班级信息
                    log.info("[Solr] Query List cost={},orgId={}", (System.currentTimeMillis() - begin),
                        param.getOrgId());
                    return retList;
                }
            }
            if (querySet != null && querySet.size() > 0) {
                log.debug("[Solr] Query querySet={}", querySet);
                queryStr = queryStr + " AND " + SolrUtil.or("user_id", querySet);
            }
            log.info("[Solr] student query param= {},collection={}", queryStr, SolrConstant.CRM_STUDENT_COLLECTION);
            query.setQuery(queryStr);

            log.info("[Solr] Query List cost={},orgId={}", (System.currentTimeMillis() - begin), param.getOrgId());

            QueryResponse response =
                getSolr().query(SolrConstant.CRM_STUDENT_COLLECTION, query, SolrRequest.METHOD.POST);

            log.info("[Solr] Query List cost={},orgId={}", (System.currentTimeMillis() - begin), param.getOrgId());

            SolrDocumentList documentList = response.getResults();
            for (Iterator<SolrDocument> iterator = documentList.iterator(); iterator.hasNext();) {
                retList.add(buildStudent(iterator.next()));
            }
            total = documentList.getNumFound();
            pageDto.setCount((int) total);
            log.info("[Solr] Query count=" + total);
            // setClassInfo(retList, param.getOrgId(), ownCourseIds);

        } catch (Exception e) {
            log.error("[Solr] Exception ", e);
            throw new SolrException(CrmErrorCode.SOLR_EXCEPTION);
        }
        log.info("[Solr] Query List cost={},orgId={}", (System.currentTimeMillis() - begin), param.getOrgId());
        return retList;
    }

    /**
     *
     * @param courseIds
     * @param orgId
     * @return
     */
    @Override
    public Set<Long> queryCourseStudentUserIds(Collection<Long> courseIds, Long orgId) {
        Set<Long> studentIds = new HashSet<>();
        if (GenericsUtils.isNullOrEmpty(courseIds)) {
            return studentIds;
        }
        StringBuilder sb = new StringBuilder();
        sb.append("id:e_* AND del_status:0 AND status:0 AND org_id:").append(orgId);
        sb.append(" AND ").append(SolrUtil.or("course_id", courseIds));
        SolrDocumentList documentList = getDocumentList(SolrConstant.STUDENT_COURSE_COLLECTION, sb.toString());
        for (Iterator<SolrDocument> iterator = documentList.iterator(); iterator.hasNext();) {
            SolrDocument doc = iterator.next();
            Long sId = (Long) doc.get("user_id");
            studentIds.add(sId);
        }
        return studentIds;
    }

    @Override
    public TwoTuple<Map<String, StudentClass>, Map<String, StudentClassHour>> findStudentClassAndStudentClassHour(
        Collection<Long> userIds, Long orgId) {
        log.info("[findStudentClassAndStudentClassHour] userIds:{} orgId:{} ", userIds, orgId);
        StringBuilder sb = new StringBuilder();
        Map<String, StudentClassHour> studentClassHourMap = Maps.newHashMap();
        Map<String, StudentClass> statusMap = null;
        try {
            if (GenericsUtils.isNullOrEmpty(userIds)) {
                return null;
            }
            sb.append("id:stu_lesson_* AND org_id:" + orgId).append(" AND ").append(SolrUtil.or("student_id", userIds));
            log.info("query for student_course infos : {} ", sb.toString());
            SolrQuery query = new SolrQuery();
            query.setQuery(sb.toString());
            query.setRows(Integer.MAX_VALUE);
            QueryResponse response =
                getSolr().query(SolrConstant.CRM_STUDENT_COLLECTION, query, SolrRequest.METHOD.POST);
            SolrDocumentList list = response.getResults();
            log.info("[Solr] query class param:" + sb.toString());
            log.info("[Solr] query class result:" + list.size());
            SolrQuery _query = new SolrQuery();
            StringBuilder _sb = new StringBuilder();
            _sb.append("id:e_* AND org_id:").append(orgId).append(" AND ").append(SolrUtil.or("user_id", userIds));
            log.info("_query is :{}", _sb.toString());
            _query.setQuery(_sb.toString());
            _query.setRows(Integer.MAX_VALUE);
            QueryResponse _response =
                getSolr().query(SolrConstant.CRM_STUDENT_COLLECTION, _query, SolrRequest.METHOD.POST);
            statusMap = createStatusMap(_response.getResults());
            for (Iterator<SolrDocument> iterator = list.iterator(); iterator.hasNext();) {
                SolrDocument document = iterator.next();
                StudentClassHour classHour = buildStudentClassHour(document);
                Long userId = classHour.getUserId();
                Long courseId = classHour.getCourseId();
                String key = userId + "_" + courseId;
                studentClassHourMap.put(key, classHour);
            }
        } catch (Exception e) {
            log.error("[Solr] Exception ", e);
        }
        log.info("statusMap is :{} ", statusMap);
        log.info("studentClassHourMap is :{} ", studentClassHourMap);
        return TupleUtil.tuple(statusMap, studentClassHourMap);
    }

    /**
     * @param studentDtos
     * @param orgId
     * @param ownCourseIds 当前登录账号管理的班级
     */
    @Override
    public void setClassInfo(List<StudentDto> studentDtos, final Long orgId, Set<Long> ownCourseIds) {
        long begin = System.currentTimeMillis();
        StringBuilder sb = new StringBuilder();
        Map<String, StudentClassHour> map = new HashMap<>();
        try {
            if (studentDtos == null || studentDtos.size() < 1) {
                return;
            }
            final Map<Long, StudentDto> studentIdMap =
                CollectionUtils.extractMap(studentDtos, new CollectionUtils.Extracter<Long, StudentDto>() {
                    @Override
                    public Long extract(StudentDto studentDto) {
                        return studentDto.getUserId();
                    }
                });
            Future<QueryResponse> future = executorService.submit(new Callable<QueryResponse>() {
                @Override
                public QueryResponse call() throws Exception {
                    SolrQuery _query = new SolrQuery();
                    StringBuilder _sb = new StringBuilder();
                    _sb.append("id:e_* AND org_id:").append(orgId).append(" AND ")
                        .append(SolrUtil.or("user_id", studentIdMap.keySet()));
                    log.info("_query is :{}", _sb.toString());
                    _query.setQuery(_sb.toString());
                    _query.setRows(Integer.MAX_VALUE);
                    QueryResponse _response =
                        getSolr().query(SolrConstant.STUDENT_COURSE_COLLECTION, _query, SolrRequest.METHOD.POST);
                    return _response;
                }
            });

            sb.append("id:stu_lesson_* AND org_id:" + orgId).append(" AND ")
                .append(SolrUtil.or("student_id", studentIdMap.keySet()));
            if (ownCourseIds != null) {
                sb.append(" AND ").append(SolrUtil.or("course_id", ownCourseIds));
            }
            log.info("query for student_course infos : {} ", sb.toString());
            SolrQuery query = new SolrQuery();
            query.setQuery(sb.toString());
            query.setRows(Integer.MAX_VALUE);
            QueryResponse response =
                getSolr().query(SolrConstant.CRM_STUDENT_COLLECTION, query, SolrRequest.METHOD.POST);
            SolrDocumentList list = response.getResults();

            log.info("[Solr] query class param:" + sb.toString());
            log.info("[Solr] query class result:" + list.size());
            log.info("[Solr] Query course info cost={},orgId={}", (System.currentTimeMillis() - begin), orgId);
            Map<String, StudentClass> statusMap = createStatusMap(future.get().getResults());

            Map<Long, List<StudentClassHour>> sscCache = new HashMap<>();
            for (Iterator<SolrDocument> iterator = list.iterator(); iterator.hasNext();) {
                SolrDocument document = iterator.next();
                StudentClassHour classHour = buildStudentClassHour(document);
                Long userId = classHour.getUserId();
                Long courseId = classHour.getCourseId();
                String key = userId + "_" + courseId;
                StudentClass studentClass = statusMap.get(key);
                if (studentClass != null) {
                    if (studentClass.getStatus() != 0) {
                        classHour.setTotalCount(classHour.getFinishCount());
                    } else {
                        classHour.setTotalCount(Math.max(studentClass.getContractCount(), classHour.getTotalCount()));
                    }
                }
                if (map.get(key) == null) {
                    map.put(key, classHour);
                } else {
                    classHour = map.get(key);
                }
                List<StudentClassHour> courseList = sscCache.get(userId);
                if (courseList == null) {
                    courseList = new ArrayList<>();
                    sscCache.put(userId, courseList);
                }
                courseList.add(classHour);
            }

            Map<Long, StudentClassHour> retMap = new HashMap<>();

            // 计算剩余课时最少的（如果剩余课时相同显示最新创建的）
            for (String stuCourseId : map.keySet()) {
                StudentClassHour tempClassHour = map.get(stuCourseId);
                StudentClassHour retClassHour = retMap.get(tempClassHour.getUserId());
                if (retClassHour == null) {
                    retMap.put(tempClassHour.getUserId(), tempClassHour);
                } else {
                    if (tempClassHour.getLeftCount() > 0) {
                        if (retClassHour.getLeftCount() <= 0) {
                            // 显示有剩余课时
                            retMap.put(tempClassHour.getUserId(), tempClassHour);
                        } else if (retClassHour.getLeftCount() > tempClassHour.getLeftCount()) {
                            // 显示剩余课时最小的
                            retMap.put(tempClassHour.getUserId(), tempClassHour);
                        } else if (retClassHour.getLeftCount() == tempClassHour.getLeftCount()) {
                            // 最小的剩余课时一样的话，展示最近一次排课的
                            if (retClassHour.getCourseId() < tempClassHour.getCourseId()) {
                                retMap.put(tempClassHour.getUserId(), tempClassHour);
                            }
                        }
                    } else {
                        // 如果没有剩余课时，显示最近一次排课的课时
                        if (retClassHour.getLeftCount() <= 0) {
                            if (retClassHour.getCourseId() < tempClassHour.getCourseId()) {
                                retMap.put(tempClassHour.getUserId(), tempClassHour);
                            }
                        }
                    }
                }
            }

            log.info("retMap : {} ", retMap);

            for (StudentDto dto : studentDtos) {
                StudentClassHour studentClassHour = retMap.get(dto.getUserId());
                if (studentClassHour != null) {
                    dto.setLeftClassHour(studentClassHour.getLeftCount());
                    dto.setFinishClassHour(studentClassHour.getFinishCount());
                    dto.setCourseId(studentClassHour.getCourseId());
                    dto.setHasLesson(true);
                }
            }

        } catch (IOException e) {
            log.error("[Solr] IOException ", e);
        } catch (SolrServerException e) {
            log.error("[Solr] SolrServerException ", e);
        } catch (InterruptedException e) {
            log.error("[Solr] InterruptedException ", e);
        } catch (ExecutionException e) {
            log.error("[Solr] ExecutionException ", e);
        }
        log.info("[Solr] Query course info cost={},orgId={}", (System.currentTimeMillis() - begin), orgId);
    }

    /**
     * @param solrDocumentList
     * @return
     */
    private Map<String, StudentClass> createStatusMap(SolrDocumentList solrDocumentList) {
        Map<String, StudentClass> statusMap = Maps.newHashMap();
        Iterator<SolrDocument> iterator = solrDocumentList.iterator();
        while (iterator.hasNext()) {
            SolrDocument next = iterator.next();
            Long userId = (Long) next.get("user_id");
            Long courseId = (Long) next.get("course_id");
            Integer lessonCount = (Integer) next.get("lesson_count");
            StudentClass studentClass = new StudentClass();
            String key = userId + "_" + courseId;
            Integer status = (Integer) next.get("status");
            studentClass.setStatus(status);
            studentClass.setContractCount(lessonCount == null ? 0 : lessonCount);
            studentClass.setUserId(userId);
            studentClass.setCourseId(courseId);
            statusMap.put(key, studentClass);
        }

        log.info("statusMap is :{}", statusMap);

        return statusMap;
    }

    private StudentDto buildStudent(SolrDocument doc) {
        StudentDto stu = new StudentDto();
        stu.setName((String) doc.get("name"));
        stu.setMobile((String) doc.get("mobile"));
        stu.setWeixin((String) doc.get("weixin"));
        stu.setStuLessonStatus((Integer) doc.get("lesson_status"));
        stu.setStudentId((Long) doc.get("s_self_id"));
        stu.setUserId((Long) doc.get("user_id"));
        stu.setPinyin((String) doc.get("pinyin"));
        stu.setAvatar((Long) doc.get("avatar"));
        return stu;
    }

    private StudentClassHour buildStudentClassHour(SolrDocument doc) {
        Long studentId = (Long) doc.get("student_id");
        Long courseId = (Long) doc.get("course_id");
        Long totalCount = (Long) doc.get("total");
        Long finishCount = (Long) doc.get("finished");
        Long totalClassHour = (Long) doc.get("totalClassHour");
        Long totalTime = (Long) doc.get("total_time");
        Long leftTime = (Long) doc.get("left_time");
        Long finishedTime = (Long) doc.get("finished_time");
        totalCount = (totalCount == null ? 0L : totalCount);
        finishCount = (finishCount == null ? 0L : finishCount);
        StudentClassHour classHour = new StudentClassHour();
        classHour.setUserId(studentId);
        classHour.setCourseId(courseId);
        classHour.setFinishCount(finishCount.intValue());
        classHour.setTotalCount(totalCount.intValue());
        classHour.setTotalClassHour(totalClassHour==null?0:totalClassHour);
        classHour.setTotalTime(totalTime==null?0:totalTime);
        classHour.setLeftTime(leftTime==null?0:leftTime);
        classHour.setFinishedTime(finishedTime==null?0:finishedTime);
        return classHour;
    }

    @Override
    public StudentDto queryStudentById(Long userId, Long orgId) {
        StudentDto dto = null;
        StringBuilder sb = new StringBuilder();
        sb.append("id:s_* AND del_status:0 AND org_id:" + orgId);
        sb.append(" AND user_id:" + userId);
        SolrDocumentList list = getDocumentList(SolrConstant.CRM_STUDENT_COLLECTION, sb.toString());
        List<StudentDto> dtos = new ArrayList<>();
        for (Iterator<SolrDocument> iterator = list.iterator(); iterator.hasNext();) {
            SolrDocument doc = iterator.next();
            dto = buildStudent(doc);
            dtos.add(dto);
        }
        setClassInfo(dtos, orgId, null);
        return dto;
    }

    @Override
    public List<StudentClassHour> queryStudentClassHourList(Long userId, Long orgId) {
        StringBuilder sb = new StringBuilder();
        sb.append("id:stu_lesson_* AND org_id:" + orgId).append(" AND ").append("student_id:" + userId);
        SolrDocumentList list = getDocumentList(SolrConstant.CRM_STUDENT_COLLECTION, sb.toString());
        List<StudentClassHour> ret = new ArrayList<>();
        for (Iterator<SolrDocument> iterator = list.iterator(); iterator.hasNext();) {
            SolrDocument document = iterator.next();
            StudentClassHour classHour = buildStudentClassHour(document);
            ret.add(classHour);
        }
        return ret;
    }

    @Override
    public void updateOldRow(Map<String, String> rs) throws SolrServerException, IOException {
        rs = mapKeyToUpper(rs);
        Map<String, Object> insertKeyMap = getValueMap(rs);
        log.info("Debug insertKeyMap" + insertKeyMap + "collection:" + SolrConstant.CRM_CONSULT_USER_COLLECTION);
        // this.update(SolrConstant.CRM_STUDENT_COLLECTION, insertKeyMap);
        // this.getSolr().commit(SolrConstant.CRM_STUDENT_COLLECTION, true, true, true);
    }

    @Override
    public void deleteOldRow(Map<String, String> rs) throws SolrServerException, IOException {
        rs = mapKeyToUpper(rs);
        String key = rs.get("ID");
        String query = "id:c_" + key;
        // this.delete(SolrConstant.CRM_STUDENT_COLLECTION, query);
        // this.getSolr().commit(SolrConstant.CRM_STUDENT_COLLECTION, true, true, true);
    }

    @Override
    public void insertNewRow(Map<String, String> rs) throws SolrServerException, IOException {
        rs = mapKeyToUpper(rs);
        Map<String, Object> insertKeyMap = getValueMap(rs);
        // this.add(SolrConstant.CRM_STUDENT_COLLECTION, insertKeyMap);
        // this.getSolr().commit(SolrConstant.CRM_STUDENT_COLLECTION, true, true, true);
    }

    private Map<String, Object> getValueMap(Map<String, String> rs) {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        Map<String, Object> map = new HashMap<String, Object>();
        map.put("id", "s_" + rs.get("ID"));
        map.put("s_self_id", Long.parseLong(rs.get("ID")));
        map.put("user_id", Long.parseLong(rs.get("USER_ID")));
        map.put("org_id", Long.parseLong(rs.get("ORG_ID")));
        map.put("name", rs.get("STUDENT_NAME"));
        map.put("mobile", rs.get("MOBILE"));
        map.put("weixin", rs.get("WEIXIN"));
        map.put("lesson_status", rs.get("LESSON_STATUS"));
        // map.put("add_cascade_id", rs.get("ADD_CASCADE_ID"));
        String pinyin = HanZiPinYinUtils.getLowerCasePinYin(rs.get("STUDENT_NAME"));
        if (StringUtils.isNotBlank(pinyin)) {
            char init = pinyin.charAt(0);
            if (init < 'a' || init > 'z') {
                pinyin = "~";
            }
        } else {
            pinyin = "~";
        }
        map.put("pinyin", pinyin);

        if (StringUtils.isNotBlank(rs.get("GENDER"))) {
            map.put("gender", Integer.parseInt(rs.get("GENDER")));
        }
        if (StringUtils.isNotBlank(rs.get("DEL_STATUS"))) {
            map.put("del_status", Integer.parseInt(rs.get("DEL_STATUS")));
        }
        if (StringUtils.isNotBlank(rs.get("AVATAR"))) {
            map.put("avatar", Long.parseLong(rs.get("AVATAR")));
        }
        try {
            if (StringUtils.isNotBlank(rs.get("CREATE_TIME"))) {
                map.put("create_time",
                    com.baijia.tianxiao.dal.solr.utils.DateUtil.getSolrDate(sdf.parse(rs.get("CREATE_TIME"))));
            }
            if (StringUtils.isNotBlank(rs.get("UPDATE_TIME"))) {
                map.put("update_time",
                    com.baijia.tianxiao.dal.solr.utils.DateUtil.getSolrDate(sdf.parse(rs.get("UPDATE_TIME"))));
            }
            if (StringUtils.isNotBlank(rs.get("NEXT_REMIND_TIME"))) {
                map.put("next_remind_time",
                    com.baijia.tianxiao.dal.solr.utils.DateUtil.getSolrDate(sdf.parse(rs.get("NEXT_REMIND_TIME"))));
            }
        } catch (Exception e) {
            log.error("", e);
        }
        return map;
    }
}