package com.baijia.tianxiao.dal.solr.query.impl;

import com.baijia.tianxiao.dal.solr.constant.SolrConstant;
import com.baijia.tianxiao.dal.solr.dto.ClassRoomQueryParam;
import com.baijia.tianxiao.dal.solr.dto.ClassRoomSolrDto;
import com.baijia.tianxiao.dal.solr.query.ClassRoomQuery;
import com.baijia.tianxiao.dal.solr.utils.SolrUtil;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.date.DateUtil;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.solr.client.solrj.SolrQuery;
import org.apache.solr.client.solrj.SolrRequest;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.apache.solr.common.SolrDocument;
import org.apache.solr.common.SolrDocumentList;
import org.springframework.stereotype.Repository;

import java.io.IOException;
import java.util.Arrays;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TimeZone;

/**
 * Created by hanlaijin on 17/2/16.
 */
@Repository
@Slf4j
public class ClassRoomQueryImpl extends SolrAbstractServiceImpl implements ClassRoomQuery {

    private static final String KEYS =
            "ID,ORG_ID,ROOM_NAME,ROOM_SIZE,DEL_STATUS,CREATE_TIME,RECYCLE_STATUS,RECYCLE_TIME,ARRANGED_COUNT,ARRANGED_MINUTES";

    @Override
    public void insertNewRow(Map<String, String> rs) throws SolrServerException, IOException {
        Map<String, Object> insertMap = getInsertMap(rs);
        this.add(SolrConstant.ERP_CLASS_ROOM_COLLECTION, insertMap);
    }

    private Map<String, Object> getInsertMap(Map<String, String> rowMap) {
        Map<String, Object> upperKeyMap = getUpperKeyMap(rowMap);
        Map<String, Object> result = dealUpperKeyMap(upperKeyMap);
        return result;
    }

    private Map<String, Object> getUpperKeyMap(Map<String, String> rowMap) {
        Map<String, Object> upperKeyMap = Maps.newHashMap();
        List<String> keyList = Lists.newArrayList(Arrays.asList(KEYS.split(",")));
        for (String key : keyList) {
            String value = rowMap.get(key);
            upperKeyMap.put(key, value);
        }

        String roomId = (String) upperKeyMap.get("ID");
        if (roomId != null) {
            upperKeyMap.put("ID", "erp_" + roomId);
            upperKeyMap.put("ROOM_ID", roomId);
            upperKeyMap.put("MONTH", DateUtil.getThisMonth());
        } else {
            log.info("can not find id field ");
        }
        return upperKeyMap;
    }

    private Map<String, Object> dealUpperKeyMap(Map<String, Object> upperKeyMap) {
        Map<String, Object> result = Maps.newHashMap();
        try {
            solveFields("room_size,del_status,recycle_status,arranged_count,arranged_minutes,month".toUpperCase(), upperKeyMap, Integer.class);
            solveFields("room_id,org_id".toUpperCase(), upperKeyMap, Long.class);
            solveFields("create_time,recycle_time".toUpperCase(), upperKeyMap, Date.class);
            for (Map.Entry<String, Object> entry : upperKeyMap.entrySet()) {
                result.put(entry.getKey().toLowerCase(), entry.getValue());
            }
        } catch (Exception e) {
            log.error("can not insert new values cause by : {} ", e);
        }
        log.info("after populate , the valueMap is : {} ", result);
        return result;
    }

    @Override
    public List<ClassRoomSolrDto> queryClassRoom(ClassRoomQueryParam param, PageDto pageDto) throws IOException, SolrServerException {
        List<ClassRoomSolrDto> result = Lists.newArrayList();
        String querySql = getClassRoomListQueryString(param);
        log.info("[query string] is : {} ,param={}", querySql,param);
        SolrQuery query = new SolrQuery();
        query.setQuery(querySql);
        dealOrder(query, param.getOrder(), param.getRoomName());
        int start = 0;
        int limit = Integer.MAX_VALUE;
        if (pageDto != null) {
            start = (pageDto.getPageNum() - 1) * pageDto.getPageSize();
            limit = pageDto.getPageSize();
        }
        query.setStart(start);
        query.setRows(limit);
        QueryResponse response = getSolr().query(SolrConstant.ERP_CLASS_ROOM_COLLECTION, query, SolrRequest.METHOD.POST);
        SolrDocumentList documentList = response.getResults();
        for (Iterator<SolrDocument> iterator = documentList.iterator(); iterator.hasNext(); ) {
            result.add(buildClassRoomSolrDto(iterator.next()));
        }
        long total = documentList.getNumFound();
        log.debug("[solr query pageCount]={}",total);
        if (pageDto == null) {
            pageDto = new PageDto();
        }
        pageDto.setCount((int) total);
        return result;
    }

    private String getClassRoomListQueryString(ClassRoomQueryParam param) {
        StringBuilder sb = new StringBuilder();
        sb.append("id:erp_* AND del_status:0 AND month:"+DateUtil.getThisMonth());
        sb.append(createEquals("org_id", param.getOrgId()));
        if (param.getRecycleStatus() != null) {
            sb.append(createEquals("recycle_status", param.getRecycleStatus()));
        }
        if (param.getRoomName() != null) {
            sb.append(createEquals("room_name", "*" + param.getRoomName() + "*"));
        }
        return sb.toString();
    }

    private String createEquals(String fieldName, Object value) {
        return " AND " + SolrUtil.equal(fieldName, value) + " ";
    }

    private void dealOrder(SolrQuery query, Integer order, String roomName) {
        if (StringUtils.isNotBlank(roomName)) {
            query.addSort("recycle_time", SolrQuery.ORDER.asc);
            return;
        }
        if (order == null) {
            query.addSort("create_time", SolrQuery.ORDER.desc);
        } else {
            switch (order) {
                case 1:
                    query.addSort("create_time", SolrQuery.ORDER.desc);
                    break;
                case 2:
                    query.addSort("create_time", SolrQuery.ORDER.asc);
                    break;
                case 3:
                    query.addSort("arranged_minutes", SolrQuery.ORDER.desc);
                    break;
                case 4:
                    query.addSort("arranged_minutes", SolrQuery.ORDER.asc);
                    break;
                case 5:
                    query.addSort("recycle_time", SolrQuery.ORDER.desc);
                    break;
                case 6:
                    query.addSort("recycle_time", SolrQuery.ORDER.asc);
                    break;
            }
        }
    }

    private ClassRoomSolrDto buildClassRoomSolrDto(SolrDocument doc) {
        ClassRoomSolrDto classRoom = new ClassRoomSolrDto();
        classRoom.setRoomId((Long) doc.get("room_id"));
        classRoom.setOrgId((Long) doc.get("org_id"));
        classRoom.setRoomName((String) doc.get("room_name"));
        classRoom.setRoomSize((Integer) doc.get("room_size"));
        classRoom.setDelStatus((Integer) doc.get("del_status"));
        classRoom.setRecycleStatus((Integer) doc.get("recycle_status"));
        classRoom.setCreateTime(getDateFromSolr((Date) doc.get("create_time")));
        classRoom.setRecycleTime(getDateFromSolr((Date) doc.get("recycle_time")));
        classRoom.setArrangedCount((Integer) doc.get("arranged_count"));
        classRoom.setArrangedMinutes((Integer) doc.get("arranged_minutes"));
        return classRoom;
    }

    private Date getDateFromSolr(Date date) {
        if (date == null) {
            return null;
        }
        Long time = date.getTime();
        int timeDiff = TimeZone.getDefault().getRawOffset() - TimeZone.getTimeZone("GMT").getRawOffset();// 本地时区和目标时区差
        Long time1 = time - timeDiff;
        date = new Date(time1);
        return date;
    }

}
