package com.baijia.tianxiao.dal.solr.query.impl;

import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.solr.client.solrj.SolrQuery;
import org.apache.solr.client.solrj.SolrRequest;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.impl.CloudSolrClient;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.apache.solr.common.SolrDocumentList;
import org.apache.solr.common.SolrInputDocument;

import com.baijia.tianxiao.dal.solr.constant.SolrConfig;
import com.baijia.tianxiao.dal.solr.exceptions.SolrException;
import com.baijia.tianxiao.dal.solr.query.SolrService;
import com.baijia.tianxiao.enums.CrmErrorCode;
import com.baijia.tianxiao.util.GenericsUtils;

import lombok.extern.slf4j.Slf4j;

/**
 * Created by liuxp on 16/4/28.
 */
@Slf4j
public abstract class SolrAbstractServiceImpl implements SolrService {
    private volatile CloudSolrClient solrClient;

    @Override
    public CloudSolrClient getSolr() {
        if (solrClient == null && SolrConfig.ZK != null) {
            synchronized (this) {
                if (solrClient == null && SolrConfig.ZK != null) {
                    log.info("======solrConfig.zk is :{} ", SolrConfig.ZK);
                    solrClient = new CloudSolrClient(SolrConfig.ZK);
                }
            }
        }
        return solrClient;
    }

    @Override
    public void add(String collection, Map<String, Object> valueByField) throws SolrServerException, IOException {
        if (valueByField == null || valueByField.isEmpty()) {
            return;
        }
        SolrInputDocument inputDocument = new SolrInputDocument();
        for (String field : valueByField.keySet()) {
            inputDocument.addField(field, valueByField.get(field));
        }
        if (log.isDebugEnabled()) {
            log.debug("Try to add document:" + inputDocument);
        }
        getSolr().add(collection, inputDocument);
    }

    @Override
    public void update(String collection, Map<String, Object> valueByField) throws SolrServerException, IOException {
        add(collection, valueByField);
    }

    @Override
    public void delete(String collection, String key) throws SolrServerException, IOException {
        log.info("[Solr] collection=" + collection + ";key=" + key);
        getSolr().deleteByQuery(collection, key);
        getSolr().commit(collection);
    }

    public SolrDocumentList getDocumentList(String collection, String queryStr) {
        SolrQuery query = new SolrQuery();
        query.setRows(Integer.MAX_VALUE);
        log.info("[Solr] query param:" + queryStr);
        query.setQuery(queryStr);
        QueryResponse response = null;
        SolrDocumentList list = null;
        try {
            response = getSolr().query(collection, query, SolrRequest.METHOD.POST);
            list = response.getResults();
            log.info("[Solr] query Result:" + list.getNumFound());
        } catch (SolrServerException e) {
            log.error("SolrServerException", e);
            throw new SolrException(CrmErrorCode.SOLR_EXCEPTION);
        } catch (IOException e) {
            log.error("IOException", e);
            throw new SolrException(CrmErrorCode.SOLR_EXCEPTION);
        }
        return list;
    }

    public Map<String, String> mapKeyToUpper(Map<String, String> oldMap) {
        Map<String, String> map = null;
        if (oldMap != null) {
            map = new HashMap<String, String>();
            for (String key : oldMap.keySet()) {
                map.put(key.toUpperCase(), oldMap.get(key));
            }
        }
        return map;
    }

    public void solveFields(String fields, Map<String, Object> newValues, Class<?> changeType) throws ParseException {
        if (GenericsUtils.isNullOrEmpty(fields)) {
            return;
        }
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        List<String> longFields = Arrays.asList(fields.split(","));
        for (String field : longFields) {
            Object value = newValues.get(field);
            if (value != null) {
                String valueStr = value.toString();
                if (GenericsUtils.notNullAndEmpty(valueStr.trim())) {
                    if (changeType == Long.class) {
                        newValues.put(field, Long.parseLong(valueStr));
                    } else if (changeType == Integer.class) {
                        newValues.put(field, Integer.parseInt(valueStr));
                    } else if (changeType == Double.class) {
                        newValues.put(field, Double.parseDouble(valueStr));
                    } else if (changeType == Date.class) {
                        Object dateValue = com.baijia.tianxiao.dal.solr.utils.DateUtil.getSolrDate(sdf.parse(valueStr));
                        if (valueStr.equals("0000-00-00 00:00:00")) {
                            dateValue = null;
                        }
                        newValues.put(field, dateValue);
                    }
                } else {
                    newValues.put(field, null);
                }
            }
        }
    }
}
