package com.baijia.tianxiao.dal.es.dao.impl;

import com.baijia.tianxiao.dal.es.constant.EsClientConfig;
import com.baijia.tianxiao.dal.es.dao.EsBaseDao;
import com.baijia.tianxiao.dal.es.exceptions.EsException;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.elasticsearch.action.admin.indices.create.CreateIndexResponse;
import org.elasticsearch.action.admin.indices.delete.DeleteIndexResponse;
import org.elasticsearch.client.transport.TransportClient;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.transport.InetSocketTransportAddress;
import org.elasticsearch.transport.client.PreBuiltTransportClient;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Map;

/**
 * @author weihongyan
 * @description TODO
 * @date 21/03/2017
 */
@Slf4j
public abstract class AbstractEsBaseDao implements EsBaseDao {
    
    protected static final ObjectMapper objectMapper = new ObjectMapper();
    
    private volatile static TransportClient transportClient;
    
    @Override
    public TransportClient getClient(){
        if (null == transportClient){
            synchronized (AbstractEsBaseDao.class){
                
                if(null == transportClient){
                    // create client
                    long time = System.currentTimeMillis();
                    Settings settings = Settings.builder().put("cluster.name", EsClientConfig.clusterNameString).build();
                    TransportClient creatingClient = new PreBuiltTransportClient(settings);
                    for(Map.Entry<String, Integer> entry : EsClientConfig.getAddressList()){
                        try {
                            creatingClient.addTransportAddress(
                                new InetSocketTransportAddress(InetAddress.getByName(entry.getKey()), entry.getValue())
                            );
                        } catch (UnknownHostException e) {
                            log.error("[AbstractEsBaseDao] invalide ip or port, and ignored! ip:{}, port:{}", entry.getKey(), entry.getValue());
                            continue;
                        }
                    }
                    log.info("[{}] init transportClient done! costs:{}ms", this.getClass().getSimpleName(), System.currentTimeMillis() - time);
                    this.transportClient = creatingClient;
                    
                }
                
            }
        }
        return transportClient;
    }

    @Override
    public void createIndex(String index, int numberOfShards, int numberOfReplicas){
        try {
            CreateIndexResponse createIndexResponse = this.getClient()
                .admin()
                .indices()
                .prepareCreate(index)
                .setSettings(
                    Settings.builder()
                        .put("number_of_shards", numberOfShards)
                        .put("number_of_replicas", numberOfReplicas)
                        .build()
                ).get();
            log.warn("[{}] create es index:{} with {} Shards, {} Replicas result:{}", this.getClass().getSimpleName(),
                index, numberOfShards, numberOfReplicas, createIndexResponse.isAcknowledged());
        }catch (Exception e){
            log.error("[{}] create es index:{} with {} Shards, {} Replicas error! e:{}",
                this.getClass().getSimpleName(), index, numberOfShards, numberOfReplicas, e);
            throw new EsException(CommonErrorCode.SYSTEM_ERROR, e.getMessage());
        }
    }

    @Override
    public void deleteIndex(String index) {
        try {
            DeleteIndexResponse deleteIndexResponse = this.getClient().admin().indices().prepareDelete(index).get();
            log.warn("[{}] delete es index:{} result:{}", this.getClass().getSimpleName(), index,
                deleteIndexResponse.isAcknowledged());
        } catch (Exception e) {
            log.error("[{}] delete es index:{} error! e:{}", this.getClass().getSimpleName(), index, e);
            throw new EsException(CommonErrorCode.SYSTEM_ERROR, e.getMessage());
        }
    }

}
