/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */
package com.baijia.tianxiao.dal.user.dao.impl;

import com.baijia.tianxiao.dal.org.extractors.StatisticsExtractor;
import com.baijia.tianxiao.dal.user.dao.TeacherDao;
import com.baijia.tianxiao.dal.user.po.Teacher;
import com.baijia.tianxiao.sqlbuilder.SingleSqlBuilder;
import com.baijia.tianxiao.sqlbuilder.bean.impl.MatchMode;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.sqlbuilder.support.JdbcTemplateDaoSupport;
import com.baijia.tianxiao.util.CollectorUtil;
import com.baijia.tianxiao.util.query.BatchQueryCallback;
import com.baijia.tianxiao.util.query.BatchQueryTemplate;
import com.baijia.tianxiao.util.query.ListBatchQueryTemplate;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.jdbc.core.BeanPropertyRowMapper;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.stereotype.Repository;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Repository("teacherDao")
@Slf4j
public class TeacherDaoImpl extends JdbcTemplateDaoSupport<Teacher> implements TeacherDao {

    public TeacherDaoImpl() {
        super(Teacher.class);
    }

    @Override
    public Teacher getByUserId(@NonNull Long userId, String... queryProp) {
        SingleSqlBuilder<Teacher> builder = createSqlBuilder();
        builder.select(queryProp);
        builder.eq("userId", userId);
        return uniqueResult(builder);
    }

    @Override
    public Teacher getByNumber(Long userNumber, String... queryProp) {
        StringBuilder sql = new StringBuilder();
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("userNumber", userNumber);
        sql.append(
                "SELECT u.mobile,t.realname realName,t.nickname nickName,t.user_id userId,t.avatar  from cdb.user u join cdb.teacher t on t.user_id = u.id ");
        sql.append("WHERE 1=1 AND ");
        sql.append("u.number = :userNumber limit 1");
        List<Teacher> list = getNamedJdbcTemplate().query(sql.toString(), params, new RowMapper<Teacher>() {
            @Override
            public Teacher mapRow(ResultSet resultSet, int i) throws SQLException {
                Teacher teacher = new Teacher();
                teacher.setMobile(resultSet.getString("mobile"));
                teacher.setRealName(resultSet.getString("realName"));
                teacher.setNickName("nickName");
                teacher.setUserId(resultSet.getLong("userId"));
                teacher.setAvatar(resultSet.getLong("avatar"));
                return teacher;
            }
        });
        if (list != null && list.size() > 0) {
            return list.get(0);
        } else {
            return null;
        }
    }

    @Override
    public List<Teacher> getByUserIds(Collection<Long> userIds, final String... queryProp) {
        if (userIds == null || userIds.size() == 0) {
            return Collections.EMPTY_LIST;
        }
        BatchQueryTemplate<Long, List<Teacher>> queryTemplate = new ListBatchQueryTemplate<>();
        List<Teacher> teachers = queryTemplate.batchQuery(userIds, new BatchQueryCallback<Long, List<Teacher>>() {
            @Override
            public List<Teacher> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<Teacher> builder = createSqlBuilder(queryProp);
                builder.in("userId", querySet);
                return queryList(builder);
            }
        });
        return teachers;

    }

    @Override
    public Map<Long, String> getTeacherRealNameMap(@NonNull Collection<Long> teacherIds) {
        if (teacherIds.isEmpty()) {
            log.warn("teacher ids is empty.");
            return Collections.emptyMap();
        }

        BatchQueryTemplate<Long, List<Teacher>> queryTemplate = new ListBatchQueryTemplate<>();
        List<Teacher> teachers = queryTemplate.batchQuery(teacherIds, new BatchQueryCallback<Long, List<Teacher>>() {
            @Override
            public List<Teacher> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<Teacher> builder = createSqlBuilder();
                builder.select("realName");
                builder.select("nickName");
                builder.select("orgId");
                builder.select("userId");
                builder.in("userId", querySet);
                return queryList(builder);
            }
        });

        if (CollectionUtils.isEmpty(teachers)) {
            return Collections.emptyMap();
        }
        Map<Long, String> teacherRealNameMap = Maps.newHashMap();
        for (Teacher teacher : teachers) {
            if(teacher.getOrgId() == null){
                continue;
            }
            if (StringUtils.isNoneBlank(teacher.getRealName())) {
                teacherRealNameMap.put(teacher.getUserId(), teacher.getRealName());
            } else if (StringUtils.isNoneBlank(teacher.getNickName())) {
                teacherRealNameMap.put(teacher.getUserId(), teacher.getNickName());
            } else {
                teacherRealNameMap.put(teacher.getUserId(), "匿名");
            }
        }
        return teacherRealNameMap;

    }

    @Override
    public List<Teacher> getTeachersByOrgId(Long orgId, PageDto page, String... queryProp) {

        SingleSqlBuilder<Teacher> builder = createSqlBuilder();
        builder.select(queryProp);
        builder.eq("orgId", orgId);
        builder.desc("createTime");
        builder.setPage(page);
        return queryList(builder);
    }

    @Override
    public Map<Long, List<Teacher>> getBranchTeachMap(Long orgId, String... queryProp) {
        SingleSqlBuilder<Teacher> builder = createSqlBuilder();
        builder.select(queryProp);
        builder.eq("orgId", orgId);
        List<Teacher> teachers = queryList(builder);
        return CollectorUtil.group(teachers, new Function<Teacher, Long>() {
            @Override
            public Long apply(Teacher from) {
                return from.getBranchId();
            }
        });
    }

    @Override
    public Integer getTeachersCountByOrgId(Long orgId) {
        SingleSqlBuilder<Teacher> builder = createSqlBuilder();
        builder.count("id");
        builder.eq("orgId", orgId);
        return queryForObject(builder, Integer.class);
    }

    @Override
    public List<Teacher> searchTeacherByMobileAndName(Long orgId, String key, Integer limit) {
        Preconditions.checkArgument(StringUtils.isNoneBlank(key), "search key can not be empty");
        Map<String, Object> params = new HashMap<String, Object>();

        StringBuilder sql = new StringBuilder();
        sql.append(
                "SELECT u.mobile,t.realname realName,t.nickname nickName,t.user_id userId,t.avatar  from cdb.user u join cdb.teacher t on t.user_id = u.id ");
        sql.append("WHERE 1=1 AND ");
        if (orgId != null) {
            sql.append(" t.organization_id=:orgId AND ");
            params.put("orgId", orgId);
        }
        sql.append("( u.mobile like :key or u.number like :key or t.realname like :key or t.nickname like :key )");
        if (limit != null && limit > 0) {
            sql.append(" LIMIT 0,:limit");
            params.put("limit", limit);
        }
        params.put("key", MatchMode.ANYWHERE.toMatchString(key));

        return this.getNamedJdbcTemplate().query(sql.toString(), params,
                new BeanPropertyRowMapper<Teacher>(Teacher.class));

    }

    @Override
    public List<Teacher> getByTeacherIds(Collection<Long> teacherIds, final String... queryProp) {
        if (teacherIds.isEmpty()) {
            log.warn("teacher ids is empty.");
            return Lists.newArrayList();
        }

        BatchQueryTemplate<Long, List<Teacher>> queryTemplate = new ListBatchQueryTemplate<>();
        return queryTemplate.batchQuery(teacherIds, new BatchQueryCallback<Long, List<Teacher>>() {
            @Override
            public List<Teacher> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<Teacher> builder = createSqlBuilder(queryProp);
                builder.in("userId", querySet);
                return queryList(builder);
            }
        });
    }

    @Override
    public List<Teacher> getTeachers(Long orgId, Collection<Long> teacherIds, String key, Date startTime, Date endTime,Integer teacherStatus,PageDto page) {
        Map<String, Object> params = new HashMap<String, Object>();

        StringBuilder countSql = new StringBuilder();
        StringBuilder selectSql = new StringBuilder();
        StringBuilder fromSql = new StringBuilder();
        StringBuilder limitSql = new StringBuilder();
        countSql.append("SELECT count(u.id) ");
        selectSql.append(
                "SELECT u.id,u.mobile,t.realname realName,t.nickname nickName,t.user_id userId,t.avatar,t.school_age schoolAge,t.other_info other_info, t.created_at as createTime ");
        fromSql.append(" from yunying.org_teachers u join cdb.teacher t on t.user_id = u.user_id ");
        fromSql.append("WHERE 1=1  ");
        if (orgId != null) {
            fromSql.append(" AND u.org_id=:orgId  ");
            fromSql.append(" AND t.organization_id=:orgId");
            params.put("orgId", orgId);
        }
        if (CollectionUtils.isNotEmpty(teacherIds)) {
            fromSql.append(" AND u.user_id in (:teacherIds)  ");
            params.put("teacherIds", teacherIds);
        }
        if(teacherStatus != null){
            fromSql.append(" AND u.status=:teacherStatus ");
            params.put("teacherStatus",teacherStatus);
        }
        if (startTime != null && endTime != null) {
            fromSql.append(" AND (t.created_at BETWEEN :startTime AND :endTime) ");
            params.put("startTime", startTime);
            params.put("endTime", endTime);
        }
        if (StringUtils.isNotEmpty(key)) {
            if(key.length()<4 && StringUtils.isNumeric(key)){
                fromSql.append(" AND ( t.realname like :key or t.nickname like :key) ");
            }else {
                fromSql.append(
                        " AND ( u.mobile like :key or t.realname like :key or t.nickname like :key) ");
            }
            params.put("key", MatchMode.ANYWHERE.toMatchString(key.trim()));
        }

        fromSql.append(" ORDER BY t.created_at desc ");
        if (page != null) {
            limitSql.append(" LIMIT :begin,:limit");
            params.put("begin", page.firstNum());
            params.put("limit", page.getPageSize());
        }
        String countsql = countSql.append(fromSql).toString();
        String selectsql = selectSql.append(fromSql.append(limitSql)).toString();
        log.info("query teacher sql:{},params:{}", selectsql, params);
        List<Teacher> teachers = this.getNamedJdbcTemplate().query(selectsql, params, new BeanPropertyRowMapper<Teacher>(Teacher.class));
        if (page != null) {
            log.info("Query teacher debug sql:{},params:{}", countSql.toString(), params);
            int count = getNamedJdbcTemplate().queryForObject(countsql, params, Integer.class);
            page.setCount(count);
            page.setCurPageCount(teachers.size());
        }
        return teachers;
    }

    @Override
    public List<Teacher> getTeachers(Long orgId, Collection<Long> teacherIds, String key, Date startTime, Date endTime,
                                     PageDto page) {
        return getTeachers(orgId,teacherIds,key,startTime,endTime,null,page);
    }


    @Override
    public void refreshCdbTeacher(Long orgId, Long id) {
        StringBuilder sb = new StringBuilder("update cdb.teacher set organization_id =null,updated_at=now() where id > :id and  organization_id = :orgId");
        Map param = Maps.newHashMap();
        param.put("orgId", orgId);
        param.put("id", id);
        this.getNamedJdbcTemplate().update(sb.toString(), param);
    }

    @Override
    public Map<Long, Integer> getPeriodOrgTeacher(Integer category, Date startDate, Date endDate,
                                                  List<Long> TianxiaoOrgIds) {
        Map<String, Object> param = new HashMap<>();
        param.put("orgIds", TianxiaoOrgIds);
        String sql =
                "select count(1) as count,organization_id as org_id from cdb.teacher where organization_id in (:orgIds) ";
        if (category != null) {
            sql += " and category = :category";
            param.put("category", category);
        }
        if (startDate != null && endDate != null) {
            sql += " and created_at between :startDate and :endDate";
            param.put("startDate", startDate);
            param.put("endDate", endDate);
        }
        sql += " GROUP BY organization_id";
        return this.getNamedJdbcTemplate().query(sql, param, new StatisticsExtractor());
    }

    @Override
    public List<Teacher> fuzzyQueryByNameAndMobile(Long orgId, Collection<Long> teacherIds, String key) {
//        Preconditions.checkArgument(StringUtils.isNoneBlank(key), "请输入查询条件");
        Preconditions.checkArgument(orgId != null && orgId > 0, "无法获取机构信息");
        StringBuilder sql = new StringBuilder();
        Map<String, Object> params = new HashMap<String, Object>();
        sql.append(
                "SELECT u.mobile,t.realname realName,t.nickname nickName,t.user_id userId,t.avatar  from cdb.user u join cdb.teacher t on t.user_id = u.id ");
        sql.append("WHERE 1=1 ");
        if (CollectionUtils.isNotEmpty(teacherIds)) {
            sql.append(" and u.id in (:teacherIds) ");
            params.put("teacherIds", teacherIds);
        }
        if (orgId != null && orgId > 0) {
            sql.append(" and t.organization_id=:orgId ");
            params.put("orgId", orgId);
        }
        if (StringUtils.isNotBlank(key)) {
            sql.append("and ( u.mobile like :key or t.realname like :key )");
            params.put("key", MatchMode.ANYWHERE.toMatchString(key));
        }
        return this.getNamedJdbcTemplate().query(sql.toString(), params, new BeanPropertyRowMapper<Teacher>(Teacher.class));
    }

	
	@Override
	public Integer countTeacher(Collection<Long> orgIds, Collection<Long> teacherIds, String teacherName) {
		SingleSqlBuilder<Teacher> builder = createSqlBuilder();
		builder.count("id");
		if(StringUtils.isNotBlank(teacherName)){
			builder.like("realname", teacherName, MatchMode.ANYWHERE);
		}
		
		if(CollectionUtils.isNotEmpty(teacherIds)){
			builder.in("userId", teacherIds);
		}
		
		if(CollectionUtils.isNotEmpty(orgIds)){
			builder.in("orgId", orgIds);
		}
		
		Integer count = queryForObject(builder, Integer.class);
        return count!=null ? count:0;
	}

	@Override
	public List<Teacher> listTeacher(Collection<Long> orgIds, Collection<Long> teacherIds, String teacherName, PageDto pageDto) {
		SingleSqlBuilder<Teacher> builder = createSqlBuilder();
		if(StringUtils.isNotBlank(teacherName)){
			builder.like("realname", teacherName, MatchMode.ANYWHERE);
		}
		
		if(CollectionUtils.isNotEmpty(teacherIds)){
			if(teacherIds.size()==1){
				builder.eq("userId", teacherIds.iterator().next());
			}else{
				builder.in("userId", teacherIds);
			}
		}
		
		if(CollectionUtils.isNotEmpty(orgIds)){
			if(orgIds.size()==1){
				builder.eq("orgId", orgIds.iterator().next());
			}else{
				builder.in("orgId", orgIds);
			}
		}
		
        if(pageDto!=null){
        	builder.setPage(pageDto);
        }
        
        builder.asc("orgId");
        builder.asc("realname");
        return queryList(builder);
	}
}
