package com.baijia.tianxiao.dal.org.dao.impl;

import com.baijia.tianxiao.dal.enums.CourseTypeEnum;
import com.baijia.tianxiao.dal.org.dao.OrgCourseDao;
import com.baijia.tianxiao.dal.org.po.OrgCourse;
import com.baijia.tianxiao.dal.util.SerialNumberUtil;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sqlbuilder.SingleSqlBuilder;
import com.baijia.tianxiao.sqlbuilder.bean.impl.MatchMode;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.sqlbuilder.support.JdbcTemplateDaoSupport;
import com.baijia.tianxiao.util.CollectorUtil;
import com.baijia.tianxiao.util.GenericsUtils;
import com.baijia.tianxiao.util.ListUtil;
import com.baijia.tianxiao.util.query.BatchQueryCallback;
import com.baijia.tianxiao.util.query.BatchQueryTemplate;
import com.baijia.tianxiao.util.query.ListBatchQueryTemplate;
import com.baijia.tianxiao.util.query.MapBatchQueryTemplate;
import com.google.common.base.Function;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.support.lob.DefaultLobHandler;
import org.springframework.jdbc.support.lob.LobHandler;
import org.springframework.stereotype.Repository;

import java.io.Serializable;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.*;

@Slf4j
@Repository
public class OrgCourseDaoImpl extends JdbcTemplateDaoSupport<OrgCourse> implements OrgCourseDao {

    public OrgCourseDaoImpl() {
        super(OrgCourse.class);
    }

    @Override
    public List<OrgCourse> getCoursesByOrgNumber(Long orgNumber, String key, Integer status, Integer isCourse, Integer isClass, Integer courseType, PageDto page,
                                                 String... queryProp) {

        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder(queryProp);
        builder.eq("orgNumber", orgNumber);
        if (StringUtils.isNotEmpty(key)) {
            builder.like("name", key, MatchMode.ANYWHERE);
        }
        if (status != null) {
            builder.eq("status", status);
        }
        buildCourseType(builder, isCourse, isClass, courseType);

        if (page != null) {
            builder.setPage(page);
        }

        builder.eq("isDel", 0);
        builder.desc("beginTime");
        return changeCourseType(queryList(builder));
    }

    @Override
    public List<OrgCourse> getCoursesByOrgNumber(@NonNull Long orgNumber, @NonNull Integer isDel, String... queryProp) {
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder(queryProp);
        builder.eq("orgNumber", orgNumber);
        builder.eq("isDel", isDel);
        return changeCourseType(queryList(builder));
    }

    @Override
    public List<OrgCourse> getCoursesByOrgNumberAndName(Long orgNumber, String courseName, Integer status,
        Integer isCourse, Integer isClass, Integer courseType, PageDto page, String... props) {
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder(props);
        builder.eq("orgNumber", orgNumber);
        if (StringUtils.isNotEmpty(courseName)) {
            builder.eq("name", courseName);
        }
        if (status != null) {
            builder.eq("status", status);
        }
        buildCourseType(builder, isCourse, isClass, courseType);
        if (page != null) {
            builder.setPage(page);
        }
        builder.eq("isDel", 0);
        builder.desc("beginTime");
        return changeCourseType(queryList(builder));
    }

    @Override
    public List<OrgCourse> getCoursesByOrgNumberAndNames(Long orgNumber, List<String> courseNames, Integer status,
        Integer isCourse, Integer isClass, Integer courseType, PageDto page, String... props) {

        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder(props);
        builder.eq("orgNumber", orgNumber);
        if (CollectionUtils.isNotEmpty(courseNames)) {
            builder.in("name", courseNames);
        }
        if (status != null) {
            builder.eq("status", status);
        }
        buildCourseType(builder, isCourse, isClass, courseType);
        if (page != null) {
            builder.setPage(page);
        }

        builder.eq("isDel", 0);
        builder.desc("beginTime");
        return changeCourseType(queryList(builder));
    }

    @Override
    public List<OrgCourse> getCourseList(Long orgNumber, Collection<Long> courseIds, String key, Integer status,
                                         Date startTime, Date endTime, Integer isCourse, Integer isClass, Integer courseType, PageDto page, String... queryProp) {
        OrgCourseListParam param = new OrgCourseListParam();
        param.setKey(key);
        param.setStatus(status);
        param.setStartTime(startTime);
        param.setEndTime(endTime);
        param.setIsCourse(isCourse);
        param.setOrderField("createTime");
        param.setOrderType("DESC");

        param.setIsClass(isClass);
        param.setCourseType(courseType);
        param.setPage(page);
        return changeCourseType(getCourseList(orgNumber, courseIds, param, queryProp));
    }

    @Override
    public List<OrgCourse> getCourseList(Integer cascadeId, Long orgNumber, Collection<Long> courseIds, String key,
                                         Integer status, Date startTime, Date endTime, Integer isCourse, Integer isClass, Integer courseType, PageDto page, String... queryProp) {
        OrgCourseListParam param = new OrgCourseListParam();
        param.setKey(key);
        param.setStatus(status);
        param.setStartTime(startTime);
        param.setEndTime(endTime);
        param.setIsCourse(isCourse);
        param.setIsClass(isClass);
        param.setCourseType(courseType);
        param.setPage(page);
        // param在这里new，判断子帐号逻辑也得放这里
        if (cascadeId != null) {
            param.setCascadeId(cascadeId);
        }
        return changeCourseType(getCourseList(orgNumber, courseIds, param, queryProp));
    }

    @Override
    public List<Long> getCourseIds(Long orgNumber, Integer status, Integer isCourse, Integer isClass, Integer courseType) {
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder("id");
        builder.eq("orgNumber", orgNumber);

        if (status != null) {
            builder.eq("status", status);
        }
        buildCourseType(builder, isCourse, isClass, courseType);
        builder.eq("isDel", 0);
        List<OrgCourse> courses = queryList(builder);
        List<Long> resultList = Lists.newArrayList();
        for (OrgCourse course : courses) {
            resultList.add(course.getId());
        }
        return resultList;
    }

    @Override
    public List<Long> getCourseList(Long orgNumber, Collection<Long> courseIds, String key, Integer status, Integer isCourse, Integer isClass, Integer courseType) {
        if (courseIds != null && CollectionUtils.isEmpty(courseIds)) {
            return Lists.newArrayList();
        }

        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder("id");
        builder.eq("orgNumber", orgNumber);
        if (StringUtils.isNotEmpty(key)) {
            builder.like("name", key, MatchMode.ANYWHERE);
        }
        if (courseIds != null) {
            builder.in("id", courseIds);
        }
        if (status != null) {
            builder.eq("status", status);
        }
        buildCourseType(builder, isCourse, isClass, courseType);
        builder.eq("isDel", 0);
        List<OrgCourse> courses = queryList(builder);
        List<Long> resultList = Lists.newArrayList();
        for (OrgCourse course : courses) {
            resultList.add(course.getId());
        }
        return resultList;
    }

    @Override
    public List<OrgCourse> getCoursesByCascadeIds(Long orgNumber, Collection<Long> cascadeIds, String key, Integer status, Integer isCourse, Integer isClass, Integer courseType) {
        if (cascadeIds != null && CollectionUtils.isEmpty(cascadeIds)) {
            return Lists.newArrayList();
        }

        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder();
        builder.eq("orgNumber", orgNumber);
        if (StringUtils.isNotEmpty(key)) {
            builder.like("name", key, MatchMode.ANYWHERE);
        }
        if (cascadeIds != null) {
            builder.in("cascadeId", cascadeIds);
        }
        if (status != null) {
            builder.eq("status", status);
        }
        buildCourseType(builder, isCourse, isClass, courseType);
        builder.eq("isDel", 0);
        return changeCourseType(queryList(builder));
    }

    @Override
    public List<Long> getCourseList(Long orgNumber, Collection<Long> courseIds, Double lPrice, Double hPrice,
                                    Collection<Long> subjectIds, String key, Integer status, Integer isCourse, Integer isClass, Integer courseType, Integer chargeType) {
        if (courseIds != null && CollectionUtils.isEmpty(courseIds)) {
            return Lists.newArrayList();
        }
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder("id");
        if (courseIds != null) {
            builder.in("id", courseIds);
        }
        builder.eq("orgNumber", orgNumber);
        if (StringUtils.isNotEmpty(key)) {
            builder.like("name", key, MatchMode.ANYWHERE);
        }
        if (status != null) {
            builder.eq("status", status);
        }
        buildCourseType(builder, isCourse, isClass, courseType);
        if (lPrice != null) {
            builder.ge("price", lPrice);
        }

        if (hPrice != null) {
            builder.le("price", hPrice);
        }

        if (subjectIds != null) {
            builder.in("subjectId", subjectIds);
        }
        if (chargeType != null) {
            builder.eq("chargeType", chargeType);
        }
        builder.desc("beginTime");
        builder.eq("isDel", 0);
        List<OrgCourse> courses = queryList(builder);
        List<Long> resultList = Lists.newArrayList();
        for (OrgCourse course : courses) {
            resultList.add(course.getId());
        }
        return resultList;
    }

    @Override
    public List<OrgCourse> getCourseList(Long orgNumber, Collection<Long> courseIds, OrgCourseListParam param,
                                         String... queryProp) {
        log.debug("*********orgNumber:{},courseIds:{},param={}", orgNumber, courseIds, param);
        if (courseIds != null && CollectionUtils.isEmpty(courseIds)) {
            return Lists.newArrayList();
        }
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder(queryProp);
        if (CollectionUtils.isNotEmpty(courseIds)) {
            builder.in("id", courseIds);
        }
        builder.eq("orgNumber", orgNumber);
        if (StringUtils.isNotEmpty(param.getKey())) {
            builder.like("name", param.getKey(), MatchMode.ANYWHERE);
        }
        if (param.getCascadeId() != null && param.getCascadeId() > 0) {
            builder.eq("cascadeId", param.getCascadeId());
        }
        if (param.getStatus() != null) {
            builder.eq("status", param.getStatus());
        }
        if (param.getStartTime() != null) {
            builder.ge("createTime", param.getStartTime());
        }
        if (param.getEndTime() != null) {
            builder.le("createTime", param.getEndTime());
        }
        if (param.getPage() != null) {
            builder.setPage(param.getPage());
        }
        buildCourseType(builder, param.getIsCourse(), param.getIsClass(), param.getCourseType());
        if (param.getChargeType() != null) {
            builder.eq("chargeType", param.getChargeType());
        }
        if (param.getArranged() != null && param.getArranged() != -1) {
            builder.eq("arrange_lesson", param.getArranged());
        }
        if (param.getQueryCondition() != null) {
            builder.add(param.getQueryCondition());
        }
        if (StringUtils.isNoneBlank(param.getOrderField(), param.getOrderType())) {
            if (param.getOrderType().equals("DESC")) {
                builder.desc(param.getOrderField(), param.getOrderField().equals("name"));
            } else if (param.getOrderType().equals("ASC")) {
                builder.asc(param.getOrderField(), param.getOrderField().equals("name"));
            }
        }
        if (CollectionUtils.isNotEmpty(param.getChargeTypes())) {
            //根据cascadeId判断是人为补充还是系统补充
            int size = param.getChargeTypes().size();
            if(size>1){
                builder.eq("kexiaoCascadeId", 0);
            }else{
                builder.eq("kexiaoCascadeId", -1);
            }
            builder.in("chargeType", param.getChargeTypes());
        }
        builder.eq("isDel", param.getIsDel());
        log.debug("course list－－－－－－－－－sql={},param={}", builder.toSql(), builder.collectConditionValue());
        return changeCourseType(queryList(builder));
    }

    @Override
    public Map<Long, OrgCourse> getOrgCourseMap(Collection<Long> courseIds, String... props) {
        if (CollectionUtils.isEmpty(courseIds)) {
            return Collections.emptyMap();
        }

        List<OrgCourse> courses = changeCourseType(this.getByIds(courseIds, props));
        if (CollectionUtils.isNotEmpty(courses)) {
            Map<Long, OrgCourse> result = Maps.newHashMap();
            for (OrgCourse course : courses) {
                result.put(course.getId(), course);
            }
            return result;
        }
        return Collections.emptyMap();
    }

    @Override
    public List<Long> getCourseIdsByOrgNumberAndCourseName(Long orgNumber, String name, Integer status, Integer isCourse, Integer isClass, Integer courseType) {
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder();
        builder.select("id");
        builder.eq("orgNumber", orgNumber);
        if (status != null) {
            builder.eq("status", status);
        }
        buildCourseType(builder, isCourse, isClass, courseType);
        if (StringUtils.isNotBlank(name)) {
            builder.like("name", name, MatchMode.ANYWHERE);
        }
        builder.eq("isDel", 0);
        return queryForList(builder, Long.class);
    }

    @Override
    public List<OrgCourse> getCourseByOrgNumberAndCourseName(Long orgNumber, String name, Integer status,
                                                             Collection<Long> courseIds, Integer isCourse, Integer isClass, Integer courseType, Integer limit) {
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder();

        builder.eq("orgNumber", orgNumber);
        if (status != null) {
            builder.eq("status", status);
        }
        if (StringUtils.isNotBlank(name)) {
            builder.like("name", name, MatchMode.ANYWHERE);
        }
        builder.eq("isDel", 0);
        if (courseIds != null) {
            builder.in("id", courseIds);
        }
        buildCourseType(builder, isCourse, isClass, courseType);
        builder.eq("isClass", 1);
        if (null != limit) {
            builder.setMaxSize(limit);
        }
        builder.desc("createTime");
        List<OrgCourse> list = changeCourseType(this.queryList(builder));
        log.debug("list = {}", list);
        return list;
    }

    @Override
    public OrgCourse getByCourseId(Long orgCourseId, String... queryProp) {
        return changeCourseType(this.getById(orgCourseId, queryProp));
    }

    @Override
    public OrgCourse getByCourseNumber(Long orgCourseNumber, String... queryProp) {
        SingleSqlBuilder<OrgCourse> builder = createSqlBuilder(queryProp);
        builder.eq("number", orgCourseNumber);
        return changeCourseType(uniqueResult(builder));
    }

    @Override
    public List<OrgCourse> getByIdsOrderByIn(Collection<Long> courseIds, Integer limit) {

        if (CollectionUtils.isEmpty(courseIds)) {
            return Collections.emptyList();
        }
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder();
        builder.ne("status", 1);
        builder.in("id", courseIds);
        builder.setMaxSize(limit);
        List<OrgCourse> list = changeCourseType(this.queryList(builder));
        log.debug("list = {}", list);
        return list;
    }

    @SuppressWarnings("deprecation")
    @Override
    public int getValidOrgCourseCount(Integer orgNumber, Integer isCourse, Integer isClass, Integer courseType) {
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder();
        builder.count("id");
        builder.eq("status", 0);
        builder.eq("verifyStatus", 1);
        builder.eq("isDel", 0);
        builder.eq("orgNumber", orgNumber);
        if (null != isCourse) {
            builder.eq("is_course", isCourse);
        }
        if (null != isClass) {
            builder.eq("is_class", isClass);
        }
        if (null != courseType) {
            builder.eq("course_type", courseType);
        }
        return queryForObject(builder, Integer.class);
    }

    @Override
    public Map<Long, String> getCourseNameMap(Collection<Long> courseIds) {
        if (CollectionUtils.isEmpty(courseIds)) {
            return Collections.emptyMap();
        }
        MapBatchQueryTemplate<Long, Long, String> queryTemplate = new MapBatchQueryTemplate<>();
        return queryTemplate.batchQuery(courseIds, new BatchQueryCallback<Long, Map<Long, String>>() {
            @Override
            public Map<Long, String> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<OrgCourse> builder = createSqlBuilder("id", "name");
                builder.in("id", querySet);
                List<OrgCourse> courses = queryList(builder);
                return CollectorUtil.collectMap(courses, new Function<OrgCourse, Long>() {
                    @Override
                    public Long apply(OrgCourse from) {
                        return from.getId();
                    }
                }, new Function<OrgCourse, String>() {
                    @Override
                    public String apply(OrgCourse from) {
                        return from.getName();
                    }
                });
            }
        });
    }

    @Override
    public Map<Long, Integer> getCourseTypeMap(Collection<Long> courseIds) {
        if (CollectionUtils.isEmpty(courseIds)) {
            return Collections.emptyMap();
        }
        MapBatchQueryTemplate<Long, Long, Integer> queryTemplate = new MapBatchQueryTemplate<>();
        return queryTemplate.batchQuery(courseIds, new BatchQueryCallback<Long, Map<Long, Integer>>() {
            @Override
            public Map<Long, Integer> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<OrgCourse> builder = createSqlBuilder("id", "courseType");
                builder.in("id", querySet);
                List<OrgCourse> courses = queryList(builder);
                return CollectorUtil.collectMap(courses, new Function<OrgCourse, Long>() {
                    @Override
                    public Long apply(OrgCourse from) {
                        return from.getId();
                    }
                }, new Function<OrgCourse, Integer>() {
                    @Override
                    public Integer apply(OrgCourse from) {
                        return from.getCourseType();
                    }
                });
            }
        });
    }

    @Deprecated
    @Override
    public Map<Integer, Integer> allCourseOrgIdMap() {

        Map<String, Object> param = new HashMap<>();
        String sql =
                "select  c.id as c_course_id ,a.id as c_org_id  from cdb.org_course c,yunying.org_account a where a.number = c. organization_number";

        return this.getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<Map<Integer, Integer>>() {
            @Override
            public Map<Integer, Integer> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<Integer, Integer> map = new HashMap<>();
                while (rs.next()) {
                    Integer courseId = rs.getInt("c_course_id");
                    Integer orgId = rs.getInt("c_org_id");
                    map.put(courseId, orgId);
                }
                return map;
            }
        });
    }

    @Override
    public OrgCourse getCourseByCourseNumberAndOrgNumber(Long orgNumber, Long courseNumber, String... queryProp) {
        SingleSqlBuilder<OrgCourse> builder = createSqlBuilder(queryProp);
        builder.eq("orgNumber", orgNumber);
        builder.eq("number", courseNumber);
        builder.eq("isDel", 0);
        return changeCourseType(uniqueResult(builder));

    }

    @Override
    public List<OrgCourse> getOrgCourseList(Long orgNumber, Collection<Long> courseNumbers) {
        SingleSqlBuilder<OrgCourse> builder = createSqlBuilder();
        builder.eq("orgNumber", orgNumber);
        builder.in("number", courseNumbers);
        builder.eq("isDel", 0);
        return changeCourseType(queryList(builder));
    }

    @Override
    public List<Integer> getOrgCourseSubjectIds(Integer orgNumber, Integer isCourse, Integer isClass, Integer courseType) {
        String sql = "select distinct subject_id from cdb.org_course where status=0 and organization_number=? ";
        if (null != isCourse) {
            sql += " and is_course=? ";
        }
        if (null != isClass) {
            sql += " and is_class=? ";
        }
        if (null != courseType) {
            sql += " and course_type=? ";
        }
        List<Integer> subjectIds = this.getNamedJdbcTemplate().getJdbcOperations().query(sql,
                new Object[]{orgNumber}, new RowMapper<Integer>() {

                    @Override
                    public Integer mapRow(ResultSet rs, int rowNum) throws SQLException {
                        return rs.getInt("subject_id");
                    }
                });
        return subjectIds;
    }

    @Override
    public List<Long> getCourseNumberByOrg(Long orgNumber, Integer isCourse, Integer isClass, Integer courseType) {
        String sql = "select number from cdb.org_course where status=0 and is_del=0 and organization_number=? ";
        if (null != isCourse) {
            sql += " and is_course=? ";
        }
        if (null != isClass) {
            sql += " and is_class=? ";
        }
        if (null != courseType) {
            sql += " and course_type=? ";
        }
        List<Long> numbers = this.getNamedJdbcTemplate().getJdbcOperations().query(sql, new Object[]{orgNumber},
                new RowMapper<Long>() {

                    @Override
                    public Long mapRow(ResultSet rs, int rowNum) throws SQLException {
                        return rs.getLong("number");
                    }
                });
        return numbers;
    }

    @Override
    public List<Long> getCourseIdsByCourseNumberAndOrgNumber(final Long orgNumber, Collection<Long> courseNumbers, Integer courseType) {
        if (CollectionUtils.isEmpty(courseNumbers)) {
            return Lists.newArrayList();
        }
        BatchQueryTemplate<Long, List<Long>> batchQueryTemplate = new ListBatchQueryTemplate<>();
        return batchQueryTemplate.batchQuery(courseNumbers, new BatchQueryCallback<Long, List<Long>>() {
            @Override
            public List<Long> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<OrgCourse> builder = createSqlBuilder("id");
                builder.eq("orgNumber", orgNumber);
                builder.in("number", querySet);
                builder.eq("isDel", 0);
                return queryForList(builder, Long.class);
            }
        });
    }

    @Override
    public List<OrgCourse> getAndFilterOrgCourse(Integer orgNumber, Collection<Long> excludeIds, Integer isCourse, Integer isClass, Integer courseType, PageDto pageDto) {
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder();
        builder.eq("orgNumber", orgNumber);
        builder.eq("status", 0);
        if (excludeIds != null && !excludeIds.isEmpty()) {
            builder.notin("id", excludeIds);
        }
        buildCourseType(builder, isCourse, isClass, courseType);
        builder.eq("isDel", 0);
        builder.setPage(pageDto);
        return changeCourseType(this.queryList(builder));
    }

    @Override
    public List<OrgCourse> getAndFilterOrgCourseByIn(Integer orgNumber, Collection<Long> includeIds, Integer isCourse, Integer isClass, Integer courseType, PageDto pageDto) {
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder();
        builder.eq("orgNumber", orgNumber);
        builder.eq("status", 0);
        builder.in("id", includeIds);
        buildCourseType(builder, isCourse, isClass, courseType);
        builder.setPage(pageDto);
        builder.desc("createTime");
        builder.eq("isDel", 0);
        return changeCourseType(this.queryList(builder));

    }

    @Override
    public List<OrgCourse> getAndFilterOrgCourseByIn(Integer orgNumber, Collection<Long> includeIds, Integer status, Integer isCourse, Integer isClass, Integer courseType,
                                                     PageDto pageDto) {
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder();
        builder.eq("orgNumber", orgNumber);
        if (status != null) {
            builder.eq("status", status);
        }
        builder.in("id", includeIds);
        buildCourseType(builder, isCourse, isClass, courseType);
        builder.setPage(pageDto);
        builder.eq("isDel", 0);
        builder.desc("createTime");
        return changeCourseType(this.queryList(builder));

    }

    @Override
    public int getValidOrgCourseCount(Integer orgNumber, Collection<Long> ids) {
        SingleSqlBuilder<OrgCourse> builder = createSqlBuilder();
        builder.count("id");
        builder.in("id", ids);
        builder.eq("orgNumber", orgNumber);
        builder.eq("status", 0);
        builder.eq("isDel", 0);
        return queryForObject(builder, Integer.class);
    }

    /**
     * color是以#开始的十六进制字符串 #AA33BC
     */
    @Override
    public void changeCourseColor(Integer orgNumber, Long courseId, String color) {
        OrgCourse course = this.getByCourseId(courseId);
        if (course != null) {
            course.setColor(color);
            this.update(course);
        } else {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "course not found!");
        }
    }

    @SuppressWarnings("unchecked")
    @Override
    public void changeCourseColor(Collection<OrgCourse> courses) {
        if (CollectionUtils.isNotEmpty(courses)) {
            List<Map<String, Object>> batchs = Lists.newArrayList();
            for (OrgCourse orgCourse : courses) {
                Map<String, Object> map = Maps.newHashMap();
                map.put("color", orgCourse.getColor());
                map.put("id", orgCourse.getId());
                batchs.add(map);
            }
            String sql = "update cdb.org_course set color =:color where id=:id";
            this.getNamedJdbcTemplate().batchUpdate(sql, batchs.toArray(new HashMap[courses.size()]));
        }
    }

    @Override
    public String getCourseNameById(Long orgCourseId) {

        OrgCourse course = this.getByCourseId(orgCourseId, "name");
        if (course != null) {
            return course.getName();
        }
        return "";
    }

    @Override
    public List<OrgCourse> getVisualOrgCourseByOrgNumber(Integer orgNumber, String subjectIds, int orderType, Integer isCourse, Integer isClass, Integer courseType) {
        Map<String, Object> paramMap = Maps.newHashMap();
        paramMap.put("orgNumber", orgNumber);
        String sql =
                "SELECT id,number,name,subject_id,rank,extend_status,transform,page_view FROM cdb.org_course where organization_number=:orgNumber and status=0  and is_del=0 ";
        if (StringUtils.isNotEmpty(subjectIds)) {
            String[] subArray = StringUtils.split(subjectIds, "|");
            List<Integer> subjectList = Lists.newArrayList();
            for (String s : subArray) {
                subjectList.add(Integer.valueOf(s));
            }
            paramMap.put("subjectIds", subjectList);
            sql = sql + " and subject_id in(:subjectIds) ";
        }
        if (null != isCourse) {
            paramMap.put("isCourse", isCourse);
            sql += " and is_course=:isCourse ";
        }
        if (null != isClass) {
            paramMap.put("isClass", isClass);
            sql += " and is_class=:isClass ";
        }
        if (null != courseType) {
            paramMap.put("courseType", courseType);
            sql += " and course_type=:courseType  ";
        }
        if (orderType == 0) {
            sql = sql + " order by extend_status desc, rank desc ";
        } else if (orderType == 1) {
            sql = sql + " order by rank desc ";
        } else if (orderType == 2) {
            sql = sql + " order by rank asc ";
        } else if (orderType == 3) {
            sql = sql + " order by page_view desc ";
        } else if (orderType == 4) {
            sql = sql + " order by page_view asc ";
        } else if (orderType == 5) {
            sql = sql + " order by transform desc ";
        } else if (orderType == 6) {
            sql = sql + " order by transform asc ";
        }
        log.info("orgCourseQuery,sql=" + sql);
        final List<OrgCourse> result = Lists.newArrayList();
        getNamedJdbcTemplate().query(sql, paramMap, new ResultSetExtractor<List<OrgCourse>>() {

            @Override
            public List<OrgCourse> extractData(ResultSet rs) throws SQLException, DataAccessException {
                while (rs.next()) {
                    OrgCourse orgCourse = new OrgCourse();
                    orgCourse.setId(rs.getLong("id"));
                    orgCourse.setNumber(rs.getLong("number"));
                    orgCourse.setName(rs.getString("name"));
                    orgCourse.setSubjectId(rs.getInt("subject_id"));
                    orgCourse.setRank(rs.getInt("rank"));
                    orgCourse.setExtendStatus(rs.getInt("extend_status"));
                    orgCourse.setTransform(rs.getInt("transform"));
                    orgCourse.setPageView(rs.getInt("page_view"));
                    result.add(orgCourse);
                }
                return result;
            }
        });
        return changeCourseType(result);
    }

    @Override
    public OrgCourse getOrgCourse(Integer orgNumber, Long courseNum) {
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder();
        builder.eq("orgNumber", orgNumber);
        builder.eq("number", courseNum);
        return changeCourseType(this.uniqueResult(builder));
    }

    @Override
    public int getAdvertiseCourseCount(Integer orgNumber, Integer isCourse, Integer isClass, Integer courseType) {
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder();
        builder.eq("orgNumber", orgNumber);
        buildCourseType(builder, isCourse, isClass, courseType);
        builder.eq("extendStatus", 1);
        builder.count("id");
        builder.eq("isDel", 0);
        Integer result = this.queryForObject(builder, Integer.class);
        if (result == null) {
            return 0;
        }
        return result;
    }

    @Override
    public int getMaxId() {
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder();
        builder.eq("isDel", 0);
        builder.max("id");
        Integer maxId = queryForObject(builder, Integer.class);
        if (maxId != null) {
            return maxId;
        } else {
            return 0;
        }
    }

    @Override
    public List<OrgCourse> getCourseList(Collection<Long> courseIds, Long orgNumber, Integer isCourse, Integer isClass, Integer courseType, PageDto page, String... queryProp) {

        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder(queryProp);
        builder.in("id", courseIds);
        if (page != null) {
            builder.setPage(page);
        }
        if (orgNumber != null) {
            builder.eq("orgNumber", orgNumber);
        }
        buildCourseType(builder, isCourse, isClass, courseType);
        builder.eq("isDel", 0);
        builder.desc("createTime");
        return changeCourseType(this.queryList(builder));
    }

    @Override
    public List<String> getBlobString(Long orgNumber, Long courseNumber) {
        String sql =
                "SELECT introduction FROM cdb.org_course where organization_number=:orgNumber and number=:courseNumber";
        Map<String, Object> paramMap = Maps.newHashMap();
        paramMap.put("orgNumber", orgNumber);
        paramMap.put("courseNumber", courseNumber);

        final LobHandler lobHandler = new DefaultLobHandler();

        return getNamedJdbcTemplate().query(sql, paramMap, new RowMapper<String>() {

            @Override
            public String mapRow(ResultSet rs, int rowNum) throws SQLException {
                byte[] bytes = lobHandler.getBlobAsBytes(rs, 1);
                if (bytes != null) {
                    String s = new String(bytes);
                    return s;
                } else {
                    return null;
                }

            }

        });

    }

    @Override
    public void repairNumber() {
        String sql = "select id,number from cdb.org_course where length(number)>10";
        List<OrgCourse> list = this.getNamedJdbcTemplate().query(sql, new RowMapper<OrgCourse>() {

            @Override
            public OrgCourse mapRow(ResultSet rs, int rowNum) throws SQLException {

                OrgCourse orgCourse = new OrgCourse();
                orgCourse.setNumber(rs.getLong("number"));
                orgCourse.setId(rs.getLong("id"));
                return orgCourse;
            }

        });
        int count = 0;
        for (OrgCourse course : list) {
            long number = SerialNumberUtil.generateNumber(course.getId().intValue(), SerialNumberUtil.COURSE_MAX, 4);
            if (!course.getNumber().equals(number)) {
                course.setNumber(number);
                this.saveOrUpdate(course, "number");
                count++;
            }
        }
        log.info("repair {} records.", count);

    }

    @Override
    public Map<Integer, Integer> getCourseTotal(Date startTime, Date endTime, List<Integer> TianxiaoOrgNumbers, Integer isCourse, Integer isClass, Integer courseType) {
        Map<String, Object> param = new HashMap<>();
        param.put("orgNumbers", TianxiaoOrgNumbers);

        StringBuilder sb = new StringBuilder();
        sb.append("select organization_number, count(id) count from cdb.org_course where organization_number in (:orgNumbers)");
        if (isCourse != null) {
            param.put("isCourse", isCourse);
            sb.append(" and is_course= :isCourse");
        }

        if (isClass != null) {
            param.put("isClass", isClass);
            sb.append(" and is_class= :isClass");
        }

        if (courseType != null) {
            param.put("courseType", courseType);
            sb.append(" and course_type= :courseType");
        }
        if (startTime != null && endTime != null) {
            sb.append(" and create_time between :startTime and :endTime ");
            param.put("endTime", endTime);
            param.put("startTime", startTime);
        }

        sb.append(" group by organization_number");

        return this.getNamedJdbcTemplate().query(sb.toString(), param, new ResultSetExtractor<Map<Integer, Integer>>() {

            @Override
            public Map<Integer, Integer> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<Integer, Integer> map = new HashMap<>();
                while (rs.next()) {
                    Integer orgNamber = rs.getInt("organization_number");
                    Integer count = rs.getInt("count");
                    map.put(orgNamber, count);
                }
                return map;
            }
        });
    }

    @Override
    public Map<String, Integer> getCourseTotalMap(Date startTime, Date endTime, List<Integer> TianxiaoOrgNumbers, Integer isCourse, Integer isClass, Integer courseType) {
        Map<String, Object> param = new HashMap<>();
        param.put("orgNumbers", TianxiaoOrgNumbers);
        StringBuilder sb = new StringBuilder();
        sb.append("select count(id) count, DATE_FORMAT(create_time,'%y-%m-%d') createTime from cdb.org_course where organization_number in (:orgNumbers)");

        if (isCourse != null) {
            param.put("isCourse", isCourse);
            sb.append(" and is_course= :isCourse");
        }

        if (isClass != null) {
            param.put("isClass", isClass);
            sb.append(" and is_class= :isClass");
        }

        if (courseType != null) {
            param.put("courseType", courseType);
            sb.append(" and course_type= :courseType");
        }
        if (startTime != null && endTime != null) {
            sb.append(" and create_time between :startTime and :endTime ");
            param.put("endTime", endTime);
            param.put("startTime", startTime);
        }
        sb.append(" group by MONTH(create_time), DAY(create_time)");

        return this.getNamedJdbcTemplate().query(sb.toString(), param, new ResultSetExtractor<Map<String, Integer>>() {

            @Override
            public Map<String, Integer> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<String, Integer> map = new HashMap<>();
                while (rs.next()) {
                    String createTime = rs.getString("createTime");
                    Integer count = rs.getInt("count");
                    map.put(createTime, count);
                }
                return map;
            }
        });
    }

    @Override
    public void refreshOrgCourse(Long organizationNumber, Long id) {
        StringBuilder sb = new StringBuilder(
                "update cdb.org_course set is_del =1 where id > :id and  organization_number = :organizationNumber");
        Map param = Maps.newHashMap();
        param.put("organizationNumber", organizationNumber);
        param.put("id", id);
        this.getNamedJdbcTemplate().update(sb.toString(), param);
    }

    @Override
    public List<OrgCourse> getOrgCourseList(Collection<Integer> orgNumbers, Integer isCourse, Integer isClass, Integer courseType) {
        SingleSqlBuilder<OrgCourse> builder = createSqlBuilder();
        builder.in("orgNumber", orgNumbers);
        buildCourseType(builder, isCourse, isClass, courseType);

        return changeCourseType(this.queryList(builder));
    }

    @Override
    public List<OrgCourse> getNormalCourseList(Collection<Long> courseIds, String... queryProp) {
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder(queryProp);
        builder.in("id", courseIds);
        builder.eq("isDel", 0);
        builder.desc("createTime");
        return changeCourseType(this.queryList(builder));

    }

    @Override
    public List<OrgCourse> getCourseByCasCadeIds(Integer orgNumber, List<Integer> casCadeIds, Integer isCourse, Integer isClass, Integer courseType, String... queryProp) {
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder(queryProp);
        builder.eq("orgNumber", orgNumber);
        builder.in("cascadeId", casCadeIds);
        buildCourseType(builder, isCourse, isClass, courseType);
        return changeCourseType(queryList(builder));
    }

    @Override
    public List<Long> getCourseIdsByCascadeId(Integer cascadeId, Integer isCourse, Integer isClass, Integer courseType) {
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder("id");
        builder.eq("cascadeId", cascadeId);
        buildCourseType(builder, isCourse, isClass, courseType);
        builder.eq("isDel", 0);
        builder.desc("createTime");
        List<Long> ids = Lists.newArrayList();
        List<OrgCourse> orgCourses = queryList(builder);
        if (CollectionUtils.isNotEmpty(orgCourses)) {
            for (OrgCourse orgCourse : orgCourses) {
                ids.add(orgCourse.getId());
            }
        }
        return ids;
    }

    @Override
    public OrgCourse getOrgCourse(Integer orgNumber, Long courseNum, Integer cascadeId) {
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder();
        builder.eq("orgNumber", orgNumber);
        builder.eq("number", courseNum);
        builder.eq("cascadeId", cascadeId);
        OrgCourse result = this.uniqueResult(builder);
        if (result != null && result.getCourseType() == CourseTypeEnum.COURSE_TYPE_NEW_CLASS.getCode()) {
            result.setCourseType(CourseTypeEnum.COURSE_TYPE_CLASS.getCode());
        }
        return result;
    }

    @Override
    public void batchUpdateHeadTeacher(Integer orgNumber, Integer oldCascadeId, Integer newCascadeId) {
        String sql =
                "update cdb.org_course set cascade_id=:newCascadeId where organization_number=:number and cascade_id=:oldCascadeId";
        Map<String, Object> map = new HashMap<>();
        map.put("number", orgNumber);
        map.put("oldCascadeId", oldCascadeId);
        map.put("newCascadeId", newCascadeId);
        log.info("[HeadTeacher] sql={},param={}", sql, map);
        getNamedJdbcTemplate().update(sql, map);
    }

    @Override
    public List<OrgCourse> searchCourses(Long number, Set<Long> filterIds, boolean allCourse, Integer isCourse, Integer isClass, Integer courseType, PageDto pageDto) {
        log.info("orgNumber is : {} ,filterIds is : {} ,isAllcourse is:{} ,pageDto is : {} ", number, filterIds,
                allCourse, pageDto);
        SingleSqlBuilder<OrgCourse> createSqlBuilder = this.createSqlBuilder();
        createSqlBuilder.eq("orgNumber", number);

        if (allCourse) {
            if (GenericsUtils.notNullAndEmpty(filterIds)) {
                createSqlBuilder.notin("id", filterIds);
            }
        } else {
            if (GenericsUtils.isNullOrEmpty(filterIds)) {
                return Collections.emptyList();
            }
            createSqlBuilder.in("id", filterIds);
        }
        buildCourseType(createSqlBuilder, isCourse, isClass, courseType);
        if (pageDto != null) {
            createSqlBuilder.setPage(pageDto);
        }
        String sql = createSqlBuilder.toSql();
        log.info("sql is : {} ", sql);
        return this.changeCourseType(queryList(createSqlBuilder));
    }

    @Override
    public List<OrgCourse> fuzzyQuery(Integer orgNumber, String courseName, Collection<Long> courseIds, PageDto pageDto) {
        SingleSqlBuilder<OrgCourse> builder = createSqlBuilder();
        builder.eq("orgNumber", orgNumber);
        if (CollectionUtils.isNotEmpty(courseIds)) {
            builder.in("id", courseIds);
        }
        builder.eq("isClass", 1);
        builder.eq("isDel", 0);
        if (StringUtils.isNotBlank(courseName)) {
            builder.like("name", courseName, MatchMode.ANYWHERE);
        }
        if (pageDto != null) {
            builder.setPage(pageDto);
        }
        return changeCourseType(queryList(builder));
    }

    @Override
    public List<OrgCourse> fuzzyQueryByCourseType(Integer orgNumber, String courseName, Collection<Long> courseIds, Integer type, PageDto pageDto) {
        SingleSqlBuilder<OrgCourse> builder = createSqlBuilder();
        builder.eq("orgNumber", orgNumber);
        if (CollectionUtils.isNotEmpty(courseIds)) {
            builder.in("id", courseIds);
        }
        builder.eq("isClass", 1);
        builder.eq("isDel", 0);
        builder.asc("name", true);
        if (StringUtils.isNotBlank(courseName)) {
            builder.like("name", courseName, MatchMode.ANYWHERE);
        }
        if (pageDto != null) {
            builder.setPage(pageDto);
        }
        return changeCourseType(queryList(builder));
    }

    @Override
    public List<OrgCourse> getCoursesByParentIds(Long orgNumber, Collection<Long> parentIds, String... props) {
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder(props);
        if (CollectionUtils.isEmpty(parentIds)) {
            return Lists.newArrayList();
        }
        if (null != orgNumber) {
            builder.eq("orgNumber", orgNumber);
        }
        builder.in("parentId", parentIds);
        builder.eq("isDel", 0);
        return changeCourseType(queryList(builder));
    }

    @Override
    public List<Long> getClassIdsByParentId(Long parentId) {
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder("id");
        builder.eq("parentId", parentId);
        List<OrgCourse> courses = queryList(builder);
        return ListUtil.toKeyList(courses, "id", OrgCourse.class);
    }

    @Override
    public void updateFinish(Long id, Integer isFinish) {
        Map<String, Object> updateCondtion = Maps.newHashMap();
        updateCondtion.put("id", id);
        updateCondtion.put("isFinish", isFinish);
        this.update(updateCondtion, "isFinish");
    }

    @Override
    public int getAllCourseCountByOrg(Long orgNumber) {
        SingleSqlBuilder<OrgCourse> builder = createSqlBuilder();
        builder.count("id");
        builder.eq("status", 0);
        builder.eq("isDel", 0);
        builder.eq("isCourse", 1);
        builder.eq("orgNumber", orgNumber);
        Integer count = queryForObject(builder, Integer.class);
        if (count == null) {
            return 0;
        }
        return count;
    }

    @Override
    public List<OrgCourse> getValidClassCourseCountsByIds(Integer number, List<Integer> orgCourseIds) {
        SingleSqlBuilder<OrgCourse> builder = createSqlBuilder();
        builder.in("id", orgCourseIds);
        builder.eq("orgNumber", number);
        builder.eq("status", 0);
        builder.eq("isDel", 0);
        return changeCourseType(queryList(builder, OrgCourse.class));
    }

    @Override
    public List<OrgCourse> getByCourseIds(Collection<Long> courseIds, PageDto pageDto, String... props) {
        if (CollectionUtils.isEmpty(courseIds)) {
            return Lists.newArrayList();
        }
        SingleSqlBuilder<OrgCourse> builder = createSqlBuilder(props);
        builder.in("id", courseIds);
        builder.setPage(pageDto);
        return changeCourseType(queryList(builder));
    }

    private void buildCourseType(SingleSqlBuilder<OrgCourse> builder, Integer isCourse, Integer isClass, Integer courseType) {

        if (isCourse != null) {
            builder.eq("isCourse", isCourse);
        }
        if (isClass != null) {
            builder.eq("isClass", isClass);
        }
        if (courseType != null) {
            if (courseType == 1) {
                List<Integer> ct = Lists.newArrayList();
                ct.add(1);
                ct.add(3);
                builder.in("courseType", ct);
            } else {
                builder.eq("courseType", courseType);
            }
        }
    }

    //将所有的新班课查询出来后course_type改为班课返回给前端
    public List<OrgCourse> changeCourseType(List<OrgCourse> list) {
        if (CollectionUtils.isEmpty(list)) {
            return Collections.emptyList();
        }

        for (OrgCourse orgCourse : list) {
            changeCourseType(orgCourse);
        }
        return list;
    }

    private OrgCourse changeCourseType(OrgCourse orgCourse) {
        if (orgCourse != null && orgCourse.getCourseType() != null && orgCourse.getCourseType() == CourseTypeEnum.COURSE_TYPE_NEW_CLASS.getCode()) {
            orgCourse.setCourseType(CourseTypeEnum.COURSE_TYPE_CLASS.getCode());
        }
        return orgCourse;
    }

    @Override
    public <PK extends Serializable> OrgCourse getById(PK id, String... queryProps) {
        OrgCourse course = super.getById(id, queryProps);
        return changeCourseType(course);
    }

    @Override
    public <PK extends Serializable> List<OrgCourse> getByIds(Collection<PK> ids, String... queryProps) {
        List<OrgCourse> list = super.getByIds(ids, queryProps);
        return changeCourseType(list);
    }

	@Override
	public Map<Long, OrgCourse> mapKeyCourseId(Long orgNumber, String keyword, Integer courseType, Integer chargeType, Integer chargeUnit) {
		Map<Long, OrgCourse> map = new HashMap<Long, OrgCourse>();
		SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder();
        builder.eq("orgNumber", orgNumber);
        if (StringUtils.isNotBlank(keyword)) {
            builder.like("name", keyword, MatchMode.ANYWHERE);
        }
        if (courseType != null) {
            builder.eq("courseType", courseType);
        }
        if (courseType != null) {
            builder.eq("chargeType", chargeType);
        }
        if (courseType != null) {
            builder.eq("chargeUnit", chargeUnit);
        }
        builder.eq("isDel", 0);

        List<OrgCourse> list = queryList(builder);
        if(CollectionUtils.isNotEmpty(list)){
        	for(OrgCourse oc:list){
        		map.put(oc.getId(), oc);
        	}
        }
        return map;
	}
}
