package com.baijia.tianxiao.biz.www.impl;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataAccessException;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.stereotype.Service;

import com.baijia.tianxiao.biz.www.IllegalAccessCtrlService;
import com.baijia.tianxiao.dto.WebResponse;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.util.WebResponseHelper;

@Service
public class IllegalAccessCtrlServiceImpl implements IllegalAccessCtrlService {

    // 验证码、登录密码错误最大失败次数，超过账户会被锁定
    private static final int INVALID_ACCESS_LIMIT = 20;

    // 每失败一次，失败阈值增加量
    private static final long INCRBY_OFFSET = 1;

    // 账户被锁定时间，单位：秒
    private static final long ACCOUNT_LOCKED_TIME = 3600;

    @Autowired
    private RedisTemplate<String, Object> redisTemplate;

    @Override
    public WebResponse<Object> detect(final String key, final long offset, 
                               final long expire) {
        final RedisSerializer<String> serializer = getRedisSerializer(); 
        return redisTemplate.execute(new RedisCallback<WebResponse<Object>>() {
            @Override
            public WebResponse<Object> doInRedis(RedisConnection connection)
                    throws DataAccessException {
                
                Long currTime = System.currentTimeMillis();
                byte time[] = serializer.serialize(currTime.toString());
                byte keyByte[] = serializer.serialize(key);
                if (connection.exists(key.getBytes())) {
                    if (isAccountLocked(connection, keyByte, "locked".getBytes())) {
                            return WebResponseHelper.error(CommonErrorCode.BUSINESS_ERROR, "用户名或密码错误,账户将被锁定一小时");
                    }
                    
                    long invalidCount = hincrBy(connection, keyByte, "count".getBytes(), offset);
                    if (invalidCount >= INVALID_ACCESS_LIMIT) {
                        connection.hSet(keyByte, "etime".getBytes(), time);
                        connection.hSet(keyByte, "locked".getBytes(), "true".getBytes());
                        return WebResponseHelper.error(CommonErrorCode.BUSINESS_ERROR, "您的账户已被锁定，请1小时后再试");
                    } else {
                        if (invalidCount >= 17) {
                            connection.hSet(keyByte, "etime".getBytes(), time);
                                return WebResponseHelper.error(CommonErrorCode.BUSINESS_ERROR, "用户名或密码错误，还可输入"
                                    + (INVALID_ACCESS_LIMIT - invalidCount) +"次");
                        } else {
                                return WebResponseHelper.error(CommonErrorCode.BUSINESS_ERROR, "用户名或密码错误，请重新输入");
                        }
                    }
                } else {
                    connection.hSet(keyByte, "stime".getBytes(), time);
                    connection.hSet(keyByte, "etime".getBytes(), time);
                    connection.hSet(keyByte, "locked".getBytes(), "false".getBytes());
                    hincrBy(connection, keyByte, "count".getBytes(), offset);
                    connection.expire(keyByte, ACCOUNT_LOCKED_TIME);
                    return WebResponseHelper.error(CommonErrorCode.BUSINESS_ERROR, "用户名或密码错误，请重新输入");
                }
            }

    boolean isAccountLocked(RedisConnection connection, byte key[], byte filed[]) {
        byte[] locked = connection.hGet(key, filed);
        if (locked != null) {
            return Boolean.parseBoolean(serializer.deserialize(locked));
        }

        return false;
    }

    long hincrBy(RedisConnection connection, byte key[], byte filed[], long offset) {
        return connection.hIncrBy(key, filed, offset);
    }

    });}

    @Override
    public WebResponse<Object> detect(String key) {
        return detect(key, INCRBY_OFFSET, ACCOUNT_LOCKED_TIME);
    }

    /**
     * 获取 RedisSerializer
     */
    protected RedisSerializer<String> getRedisSerializer() {
        return redisTemplate.getStringSerializer();
    }

    @Override
    public boolean isLocked(final String key) {
        final RedisSerializer<String> serializer = getRedisSerializer();
        return redisTemplate.execute(new RedisCallback<Boolean>() {
            @Override
            public Boolean doInRedis(RedisConnection connection) throws DataAccessException {

                byte[] locked = connection.hGet(serializer.serialize(key), "locked".getBytes());
                if (locked != null) {
                    return Boolean.parseBoolean(serializer.deserialize(locked));
                }
                return false;
            }
        });
    }
}
