
/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2016 All Rights Reserved.
 */

package com.baijia.tianxiao.biz.www.authentication.handler;

import java.security.NoSuchAlgorithmException;
import java.security.spec.InvalidKeySpecException;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections4.CollectionUtils;
import org.joda.time.DateTime;
import org.joda.time.Days;
import org.springframework.beans.factory.annotation.Autowired;

import com.baijia.tianxiao.biz.www.LoginAccountDtoHelper;
import com.baijia.tianxiao.biz.www.authentication.AbstractAuthenticateHandler;
import com.baijia.tianxiao.biz.www.authentication.Credential;
import com.baijia.tianxiao.biz.www.authentication.dto.RoleDto;
import com.baijia.tianxiao.biz.www.authentication.dto.TXLoginAccountDto;
import com.baijia.tianxiao.biz.www.constant.BizConstant;
import com.baijia.tianxiao.biz.www.constant.ExpTxPermission;
import com.baijia.tianxiao.constants.org.BizConf;
import com.baijia.tianxiao.dal.index.constant.TxVisitorStatus;
import com.baijia.tianxiao.dal.index.po.TxVisitor;
import com.baijia.tianxiao.dal.org.po.OrgAccount;
import com.baijia.tianxiao.dal.org.po.OrgInfo;
import com.baijia.tianxiao.dal.org.po.TXAccount;
import com.baijia.tianxiao.dal.pcAuthority.dao.TXPermissionDao;
import com.baijia.tianxiao.dal.pcAuthority.po.TXPermission;
import com.baijia.tianxiao.dto.hag.HagDownload;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.organization.api.OrgAccountService;
import com.baijia.tianxiao.sal.organization.constant.TXPermissionHasType;
import com.baijia.tianxiao.sal.organization.index.service.TxVisitorService;
import com.baijia.tianxiao.sal.organization.org.dto.pcAuthority.PermissionDto;
import com.baijia.tianxiao.sal.organization.org.service.OrgInfoService;
import com.baijia.tianxiao.sal.organization.org.service.TXAccountService;
import com.baijia.tianxiao.util.encrypt.EncryptUtils;
import com.baijia.tianxiao.util.encrypt.PasswordUtil;
import com.baijia.tianxiao.util.httpclient.HttpClientUtils;
import com.baijia.tianxiao.util.json.JacksonUtil;
import com.baijia.tianxiao.util.properties.PropertiesReader;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import lombok.extern.slf4j.Slf4j;

/**
 * @title TxVisitorAuthenticateHandler
 * @desc O(∩_∩)O~
 * @author caoliang
 * @date 2016年8月25日
 * @version 1.0
 */
@Slf4j
public class TxVisitorAuthenticateHandler extends AbstractAuthenticateHandler {

    private static final int EXP_USED = 1;

    // 体验帐号关闭的权限
    private static List<ExpTxPermission> FORBID_PERMISSIONS = Lists.newArrayList(ExpTxPermission.PAY_PWD,
        ExpTxPermission.DRAW, ExpTxPermission.GATHERING_H5, ExpTxPermission.ORG_TEL_ADD, ExpTxPermission.ORG_TEL_DEL,
        ExpTxPermission.ORG_TEL_EDIT, ExpTxPermission.SET_PAY_PWD);

    @Autowired
    private TxVisitorService txVisitorService;
    @Autowired
    private OrgAccountService orgAccountService;
    @Autowired
    private OrgInfoService orgInfoService;
    @Autowired
    private LoginAccountDtoHelper loginAccountDtoHelper;
    @Autowired
    private TXAccountService txAccountService;
    @Autowired
    private TXPermissionDao txPermissionDao;

    @Override
    protected boolean preAuthenticate(Credential credential, Map<String, Object> options) throws BussinessException {
        if (!super.preAuthenticate(credential, options)) {
            return false;
        }
        String vReq = options.get(BizConstant.VERSION).toString().replace(".", "");
        String vBase = "131";
        for (int index = 0; index < vBase.length(); index++) {
            if (Integer.valueOf(String.valueOf(vBase.charAt(0))) < Integer.valueOf(String.valueOf(vReq.charAt(0)))) {
                // 首尾版本号已经大于1时，不做判断
                break;
            }
            if (Integer.valueOf(String.valueOf(vBase.charAt(1))) < Integer.valueOf(String.valueOf(vReq.charAt(1)))) {
                // 第二位尾版本号已经大于3时，不做判断
                break;
            }
            if (Integer.valueOf(String.valueOf(vBase.charAt(1))) < Integer.valueOf(String.valueOf(vReq.charAt(1)))) {
                // 第二位版本号已经大于3时，不做判断
                break;
            }
            Integer base = Integer.valueOf(String.valueOf(vBase.charAt(index)));
            Integer req = Integer.valueOf(String.valueOf(vReq.charAt(index)));
            if (base > req) {
                throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "请升级至最新版本");
            }
        }
        return true;
    }

    @Override
    protected boolean doAuthentication(Credential credential, Map<String, Object> options) throws BussinessException {
        List<TxVisitor> txVisitors = txVisitorService.getByMobile(credential.getMobile());
        if (CollectionUtils.isEmpty(txVisitors)) {
            return false;
        }
        // 老数据有重复mobile情况
        TxVisitor txVisitor = txVisitors.get(0);
        if(txVisitor.getStatus() != TxVisitorStatus.ACTIVE.getCode()){
            return false;
        }
        // 过期校验
        if (txVisitor.getStartTime() == null) {
            // 未开通时，按照过期处理
            return false;
        } else if (Days.daysBetween(new DateTime(txVisitor.getStartTime().getTime()), DateTime.now())
            .isGreaterThan(Days.days(txVisitor.getRemainDay()))) {
            return false;
        }
        try {
            if (!PasswordUtil.validatePassword(credential.getPassword(), txVisitor.getPassword())) {
                return false;
            }
        } catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
            log.info("login valid password error! {}", e);
            return false;
        }
        options.put(TX_VISITOR_KEY, txVisitor);
        return true;
    }

    /**
     * 构建体验帐号返回信息
     * 
     * @param credential
     * @param authenticated
     * @param loginDtos
     * @param options
     * @return
     * @throws BussinessException
     * @throws Exception
     */
    @Override
    protected boolean postAuthenticate(Credential credential, boolean authenticated, List<TXLoginAccountDto> loginDtos,
        Map<String, Object> options) throws BussinessException, Exception {
        Map<String, String> params = Maps.newHashMap();
        params.put("resource", BizConf.HAG_RESOURSE_TIANXIAO_TRIAL_ACCOUNT);
        String url = PropertiesReader.getValue("hag", "hag.http.url.downloadList");
        String response = HttpClientUtils.doPost(url, params);
        HagDownload hagDto = JacksonUtil.str2Obj(response, HagDownload.class);
        log.info("hagDto:{}", hagDto);
        if (hagDto == null || CollectionUtils.isEmpty(hagDto.getData())) {
            return false;
        }
        String numberStr = hagDto.getData().get(0).split("#")[0];

        OrgAccount orgAccount = orgAccountService.getOrgAccountByNumber(Integer.valueOf(numberStr));
        if (orgAccount == null) {
            return false;
        }
        TxVisitor txVisitor = (TxVisitor) options.get(TX_VISITOR_KEY);
        OrgInfo orgInfo = orgInfoService.getOrgInfoByOrgId(orgAccount.getId());
        TXLoginAccountDto txLoginAccountDto = loginAccountDtoHelper.buildLoginData(orgAccount, orgInfo);
        txLoginAccountDto.setDataAccountType(BizConf.EXPERIENCE_ACCOUNT);
        List<RoleDto> roleDtos = Lists.newArrayList();
        RoleDto roleDto = new RoleDto();
        roleDto.setNickName(orgInfo.getShortName());
        roleDto.setName(credential.getMobile());
        roleDto.setMobile(credential.getMobile());
        roleDto.setType(BizConf.EXPERIENCE_ACCOUNT);
        roleDto.setId(orgAccount.getNumber());
        TXAccount txAccount = txAccountService.getByOrgId(orgAccount.getId());
        String auth_token = EncryptUtils.txStrEncode(orgAccount.getId(), null, txAccount.getId(), null, null);
        roleDto.setAuth_token(auth_token);
        List<PermissionDto> permissions = Lists.newArrayList();
        // 体验帐号关闭的权限
        for (ExpTxPermission tp : FORBID_PERMISSIONS) {
            PermissionDto pd = new PermissionDto();
            pd.setName(tp.getLabel());
            pd.setTag(ExpTxPermission.getCompleteValue(tp));
            pd.setType(TXPermissionHasType.LOST.getCode());
            permissions.add(pd);
        }
        // 拼所有权限
        List<TXPermission> txPermissionsInDB = txPermissionDao.getAll();
        for (TXPermission txPermission : txPermissionsInDB) {
            PermissionDto pd = new PermissionDto();
            pd.setName(txPermission.getName());
            pd.setTag(txPermission.getCode().toString());
            pd.setType(TXPermissionHasType.OWN.getCode());
            permissions.add(pd);
        }
        roleDto.setHasPermissions(permissions);
        roleDtos.add(roleDto);
        txLoginAccountDto.setHasRoles(roleDtos);
        // 1-登录 0-未登录
        if (txVisitor.getIsUsed() != null && txVisitor.getIsUsed() != EXP_USED) {
            txVisitor.setIsUsed(EXP_USED);
            txVisitorService.updateVisitor(txVisitor);
        }
        loginDtos.add(txLoginAccountDto);
        return true;
    }
}
