
/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2016 All Rights Reserved.
 */

package com.baijia.tianxiao.biz.www.authentication.handler;

import java.security.NoSuchAlgorithmException;
import java.security.spec.InvalidKeySpecException;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeoutException;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;

import com.baijia.tianxiao.biz.www.LoginAccountDtoHelper;
import com.baijia.tianxiao.biz.www.Util.PermissionTransformUtil;
import com.baijia.tianxiao.biz.www.authentication.AbstractAuthenticateHandler;
import com.baijia.tianxiao.biz.www.authentication.Credential;
import com.baijia.tianxiao.biz.www.authentication.dto.RoleDto;
import com.baijia.tianxiao.biz.www.authentication.dto.TXLoginAccountDto;
import com.baijia.tianxiao.biz.www.constant.BizConstant;
import com.baijia.tianxiao.constants.org.BizConf;
import com.baijia.tianxiao.dal.org.constant.CampusAccountType;
import com.baijia.tianxiao.dal.org.constant.TXAccountType;
import com.baijia.tianxiao.dal.org.po.OrgAccount;
import com.baijia.tianxiao.dal.org.po.OrgInfo;
import com.baijia.tianxiao.dal.org.po.OrgSubAccount;
import com.baijia.tianxiao.dal.org.po.TXAccount;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.organization.api.OrgAccountService;
import com.baijia.tianxiao.sal.organization.org.dto.pcAuthority.TxAccountPermissionsDto;
import com.baijia.tianxiao.sal.organization.org.service.OrgInfoService;
import com.baijia.tianxiao.sal.organization.org.service.OrgSubAccountService;
import com.baijia.tianxiao.sal.organization.org.service.TXAccountService;
import com.baijia.tianxiao.sal.organization.org.service.TxAccountPermissionService;
import com.baijia.tianxiao.util.encrypt.EncryptUtils;
import com.baijia.tianxiao.util.encrypt.PasswordUtil;
import com.baijia.tianxiao.util.memcached.MemcachedUtil;
import com.baijia.yunying.hag.service.HagService;
import com.google.common.collect.Lists;

import lombok.extern.slf4j.Slf4j;
import net.rubyeye.xmemcached.exception.MemcachedException;

/**
 * @title TxAccountAuthenticateHandler
 * @desc O(∩_∩)O~
 * @author caoliang
 * @date 2016年8月24日
 * @version 1.0
 */
@Slf4j
public class TxAccountAuthenticateHandler extends AbstractAuthenticateHandler {

    @Autowired
    private OrgInfoService orgInfoService;
    @Autowired
    private LoginAccountDtoHelper loginAccountDtoHelper;
    @Autowired
    private TXAccountService txAccountService;
    @Autowired
    private OrgAccountService orgAccountService;
    @Autowired
    private HagService hagService;
    @Autowired
    private OrgSubAccountService orgSubAccountService;
    @Autowired
    private TxAccountPermissionService txAccountPermissionService;

    @Override
    protected boolean doAuthentication(Credential credential, Map<String, Object> options) throws BussinessException {
        OrgAccount orgAccount = orgAccountService.getOrgAccountByMobile(credential.getMobile());
        if (orgAccount == null) {
            log.info("can not find mobile");
            return false;
        }
        options.put(ORG_ACCOUNT_KEY, orgAccount);
        // 临时密码
        Boolean isTemp = false;
        try {
            String tempPwd = MemcachedUtil.get("uniq_temp_tx_password_" + credential.getMobile());
            if (StringUtils.isNoneBlank(tempPwd)) {
                if (tempPwd.equals(credential.getPassword())) {
                    MemcachedUtil.delete("uniq_temp_tx_password_" + credential.getMobile());
                    isTemp = true;
                }
            }
        } catch (TimeoutException | InterruptedException | MemcachedException e) {
            log.error("get temp pwd error!mobile:{}", credential.getMobile(), e);
        }

        try {
            if (!isTemp) {
                if (!PasswordUtil.validatePassword(credential.getPassword(), orgAccount.getPassword())) {
                    log.info("orgaccount pwd error!");
                    return false;
                }
            }
        } catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
            log.info("login error!:{}", e);
            return false;
        }

        boolean hasTXPermission = hagService.hasPermission(orgAccount.getNumber(), BizConf.DEFAULT_TYPE,
            BizConf.HAG_RESOURSE_ORG_TIANXIAO_NUMBER);
        if (!hasTXPermission) {
            log.info("hag not permission");
            return false;
        }

        TXAccount txAccount = txAccountService.getByOrgId(orgAccount.getId());
        options.put(TX_ACCOUNT_KEY, txAccount);
        if (txAccount == null) {
            // 找主帐号版本
            OrgSubAccount orgSubAccount = orgSubAccountService.getByOrgId(orgAccount.getId());
            txAccount = txAccountService.getByOrgId(orgSubAccount.getPid());
            if (txAccount.getVipLevel().intValue() == TXAccountType.DAZHONG.getCode()) {
                // 大众版不能有分校，有的话算脏数据，不给登录
                log.info("fenxiaoqu");
                return false;
            }
            options.put(MASTER_FLAG, Boolean.FALSE);
        } else {
            // 检查主校区
            if (!loginAccountDtoHelper.validTXVersion(txAccount, options.get(BizConstant.VERSION).toString())) {
                throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "大众版用户请升级至最新版本");
            }
            options.put(MASTER_FLAG, Boolean.TRUE);
        }

        return true;

    }

    @Override
    protected boolean postAuthenticate(Credential credential, boolean authenticated, List<TXLoginAccountDto> loginDtos,
        Map<String, Object> options) throws BussinessException, Exception {
        OrgAccount orgAccount = (OrgAccount) options.get(ORG_ACCOUNT_KEY);
        TXAccount txAccount = (TXAccount) options.get(TX_ACCOUNT_KEY);
        OrgInfo orgInfo = orgInfoService.getOrgInfoByOrgId(orgAccount.getId());
        TXLoginAccountDto dto = loginAccountDtoHelper.buildLoginData(orgAccount, orgInfo);
        dto.setDataAccountType(BizConf.NORMAL_ACCOUNT);
        List<RoleDto> roleDtos = Lists.newArrayList();
        RoleDto roleDto = new RoleDto();
        roleDto.setId(orgAccount.getNumber());
        roleDto.setNickName(orgInfo.getContacts());
        roleDto.setName("校长");
        roleDto.setMobile(orgAccount.getMobile());
        roleDtos.add(roleDto);
        String auth_token = null;
        Boolean masterFlag = (Boolean) options.get(MASTER_FLAG);
        if (masterFlag) {
            roleDto.setType(CampusAccountType.MASTER_PRINCIPAL.getCode());
            auth_token = EncryptUtils.txStrEncode(orgAccount.getId(), null, txAccount.getId(), null, null);
        } else {
            roleDto.setType(CampusAccountType.SLAVE_PRINCIPAL.getCode());
            auth_token = EncryptUtils.txStrEncode(orgAccount.getId(), null, null, null, null);
            roleDto.setType(CampusAccountType.SLAVE_PRINCIPAL.getCode());
        }
        TxAccountPermissionsDto txAccountPermissionDto =
            txAccountPermissionService.universalGetPermissions(orgAccount.getId(), null);
        roleDto.setHasPermissions(PermissionTransformUtil.trans(txAccountPermissionDto.getAPPps()));
        roleDto.setAuth_token(auth_token);
        // 天校写死的常量
        roleDto.setRoleAvatar(dto.getAvatar());
        dto.setHasRoles(roleDtos);
        loginDtos.add(dto);
        return true;
    }

}
