/**
 * Baijiahulian.com Inc. Copyright (c) 2015-2015 All Rights Reserved.
 */
package com.baijia.tianxiao.biz.wechat.service.impl;

import javax.servlet.http.HttpServletResponse;

import com.baijia.tianxiao.dal.org.dao.OrgAccountDao;
import com.baijia.tianxiao.dal.org.po.OrgAccount;

import com.baijia.tianxiao.enums.RedisKeyEnums;
import com.baijia.tianxiao.util.date.DateUtil;
import lombok.extern.slf4j.Slf4j;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataAccessException;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;

import com.baijia.tianxiao.biz.wechat.dto.WechatCallbackResponseDto;
import com.baijia.tianxiao.biz.wechat.service.WechatWebAuthService;
import com.baijia.tianxiao.biz.wechat.util.WechatWebAuthCookieUtil;
import com.baijia.tianxiao.dal.wechat.po.AuthorizationInfo;
import com.baijia.tianxiao.dal.wechat.po.Fans;
import com.baijia.tianxiao.dto.smstoken.WechatSmsTokenDto;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.exception.RedirectException;
import com.baijia.tianxiao.sal.wechat.api.AuthorizationInfoService;
import com.baijia.tianxiao.sal.wechat.api.FansService;
import com.baijia.tianxiao.sal.wechat.api.WebAuthService;
import com.baijia.tianxiao.enums.SalWechatErrorCode;
import com.baijia.tianxiao.sal.wechat.constant.webauth.WebAuthScope;
import com.baijia.tianxiao.sal.wechat.dto.fans.FansDetailDto;
import com.baijia.tianxiao.sal.wechat.dto.webauth.WebAuthDto;
import com.baijia.tianxiao.sal.wechat.dto.webauth.WebAuthorizationDto;
import com.baijia.tianxiao.util.SerializeUtil;

import java.util.Date;

/**
 * @author : zhenyujian
 * @title : WechatWebAuthServiceImpl
 * @description : 网页授权
 * @date : 2016年12月9日 上午10:39:04
 */
@Service
@Slf4j
public class WechatWebAuthServiceImpl implements WechatWebAuthService {

    @Autowired
    private WebAuthService webAuthService;
    @Autowired
    private FansService fansService;
    @Autowired
    private AuthorizationInfoService authorizationInfoService;
    @Autowired
    private OrgAccountDao orgAccountDao;


    @Autowired
    private RedisTemplate<String, String> redisTemplate;

    /*REDIS CACHE*/
    String getWechatCallbackResponseKey(WebAuthDto webAuthDto) {
        return new StringBuilder().append(RedisKeyEnums.CRM.WECHAT_CALLBACK_PREFIX.getRedisKey()).append(webAuthDto.getCode()).toString();
    }

    private void cacheWechatCallbackResponse(final WebAuthDto webAuthDto, final WechatCallbackResponseDto responseDto) {
        try {
            redisTemplate.execute(new RedisCallback<Object>() {
                @Override
                public Object doInRedis(RedisConnection connection) throws DataAccessException {
                    String key = getWechatCallbackResponseKey(webAuthDto);
                    connection.set(key.getBytes(), SerializeUtil.serialize(responseDto));
                    connection.expire(key.getBytes(), 60 * 30);//30分钟
                    return null;
                }
            });

        } catch (Exception e) {
            log.error("cacheWechatCallbackResponse - exception:", e);
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "微信网页授权缓存失败");
        }
    }


    private WechatCallbackResponseDto getWechatCallbackResponse(final WebAuthDto webAuthDto) {
        try {
            return redisTemplate.execute(new RedisCallback<WechatCallbackResponseDto>() {
                @Override
                public WechatCallbackResponseDto doInRedis(RedisConnection connection) throws DataAccessException {
                    String key = getWechatCallbackResponseKey(webAuthDto);
                    if (connection.exists(key.getBytes())) {
                        byte[] value = connection.get(key.getBytes());
                        Object obj = SerializeUtil.unserialize(value);
                        if (obj != null) {
                            return (WechatCallbackResponseDto) obj;
                        }
                        return null;
                    }
                    return null;
                }
            });
        } catch (Exception e) {
            log.error("getWechatCallbackResponse - exception:", e);
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "查询 微信网页授权缓存失败");
        }
    }


    @Override
    public WechatCallbackResponseDto getInfoByWebAuthorization(WebAuthDto webAuthDto, String webAuthLink4UserInfo) throws RedirectException, BussinessException {
        WechatCallbackResponseDto cacheDto = getWechatCallbackResponse(webAuthDto);
        if (cacheDto != null) {
            log.debug("hit cache! cacheDto:{} ", cacheDto);
            return cacheDto;
        }

        AuthorizationInfo authorizationInfo = authorizationInfoService.getByAuthorizerAppId(webAuthDto.getAppid());
        if (authorizationInfo == null) {
            throw new BussinessException(SalWechatErrorCode.WECHAT_UNBIND_ORG);
        }

        String openId = null;
        try {
            if (webAuthDto.getScope().equals(WebAuthScope.BASE.getValue())) {
                //基础授权
                WebAuthorizationDto dto = webAuthService.getWebAuthorizationDtoByWebAuthCode(webAuthDto);
                FansDetailDto fansDto = fansService.getFansDetail(dto.getOpenid());
                if (fansDto == null || DateUtil.getDateDiff(fansDto.getUpdateTime(), new Date()) > 7) {
                    throw new RedirectException("不存在用户信息 302跳转申请用户信息授权", webAuthLink4UserInfo);
                } else {
                    openId = fansDto.getOpenId();
                }

            } else if (webAuthDto.getScope().equals(WebAuthScope.USERINFO.getValue())) {
                //用户信息授权
                Fans fans = webAuthService.getWechatUserByWebAuthCode(webAuthDto);
                if (fans != null) {
                    openId = fans.getOpenId();
                    fans.setAuthorizerAppId(webAuthDto.getAppid());
                    Fans fansInDB = fansService.getFans(openId);
                    if (fansInDB != null) {
                        fans.setLastCommunicationTime(fansInDB.getLastCommunicationTime());
                    }
                    fansService.saveOrUpdateFans(fans);
                }
            } else {
                throw new BussinessException(SalWechatErrorCode.WECHAT_WEB_AUTH_FAIL, "非法的scope:" + webAuthDto.getScope());
            }

        } catch (RedirectException e) {
            throw e;
        } catch (Exception e) {
            log.warn("wechat - webauth - getInfoByWebAuthorization - exception ", e);
            throw new BussinessException(SalWechatErrorCode.WECHAT_WEB_AUTH_FAIL);
        }


        WechatCallbackResponseDto dto = new WechatCallbackResponseDto(Long.parseLong(authorizationInfo.getOrgId() + ""), openId, webAuthDto.getAppid());
        //免费版处理
        if (StringUtils.isNotBlank(webAuthDto.getSms_token())) {
            WechatSmsTokenDto token = null;
            try {
                token = WechatSmsTokenDto.fromTokenStr(webAuthDto.getSms_token());
                dto.setOrgId(token.getOrgId());

//                if (!authorizationInfo.getAuthorizerAppId().equals(token.getWechatAppId()) || !freeVersionService.isFreeAccount(authorizationInfo.getAuthorizerAppId())) {
//                    throw new RuntimeException(" wrong smstoken ");
//                }
            } catch (Exception e) {
                log.error("", e);
            }

        }

        cacheWechatCallbackResponse(webAuthDto, dto);
        return dto;
    }


    @Override
    public Fans getFansByOrgNumberWebAuth(Long number, String webAuthCode) {
        log.info("WechatWebAuthServiceImpl ==>getFansByOrgNumberWebAuth params={},{}", number, webAuthCode);
        OrgAccount orgAccount = orgAccountDao.getAccountByNumber(number.intValue());
        if (orgAccount != null) {
            AuthorizationInfo authorizationInfo = authorizationInfoService.getByOrgId(orgAccount.getId());
            if (authorizationInfo != null) {
                WebAuthDto webAuthDto = new WebAuthDto();
                webAuthDto.setCode(webAuthCode);
                webAuthDto.setAppid(authorizationInfo.getAuthorizerAppId());
                WebAuthorizationDto dto = webAuthService.getWebAuthorizationDtoByWebAuthCode(webAuthDto);
                log.info("WechatWebAuthServiceImpl ==parasm={},result= {}",webAuthDto,dto);
                if (dto != null) {
                    return fansService.getFans(dto.getOpenid());
                }
            }
        }
        return null;
    }

	@Override
	public WechatCallbackResponseDto getInfoByWebAuthorization(WebAuthDto webAuthDto, String webAuthLink4UserInfo, HttpServletResponse response) throws RedirectException, BussinessException {
		WechatCallbackResponseDto dto = getInfoByWebAuthorization(webAuthDto, webAuthLink4UserInfo);
		WechatWebAuthCookieUtil.addCookieOpenId(dto.getWechatAppId(), dto.getWeixinOpenId(), response);
		return dto;
	}
}

