package com.baijia.tianxiao.biz.student.syn.service.impl;

import java.io.IOException;
import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Semaphore;

import com.baijia.commons.lang.utils.collection.CollectionUtils;
import com.baijia.tianxiao.biz.student.syn.service.StudentKexiaoService;
import com.baijia.tianxiao.constant.SignStatus;
import com.baijia.tianxiao.dal.enums.CourseTypeEnum;
import org.apache.commons.lang.StringUtils;
import org.apache.solr.client.solrj.SolrServerException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataAccessException;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.baijia.tianxiao.constants.TianXiaoConstant;
import com.baijia.tianxiao.dal.org.dao.*;
import com.baijia.tianxiao.dal.org.po.*;
import com.baijia.tianxiao.dal.solr.constant.SolrConstant;
import com.baijia.tianxiao.dal.solr.enums.StudentLessonStatus;
import com.baijia.tianxiao.dal.solr.query.CrmStudentQuery;
import com.baijia.tianxiao.redis.AbstractBaseRedisDao;
import com.baijia.tianxiao.sal.student.enums.LessonStatus;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.ListUtil;

import lombok.Data;
import lombok.extern.slf4j.Slf4j;

/**
 * Created by liuxp on 16/7/22.
 */
@Service
@Slf4j
public class StudentKexiaoServiceImpl extends AbstractBaseRedisDao<String, Object> implements StudentKexiaoService {

    private static final int PAGE_SIZE = 1000;
    private static final int UPDATE_TIME_TYPE = 1;
    private static final int START_TIME_TYPE = 2;
    private static final String TX_ORG_KEXIAO_RULE = "#TX#ORG#KEXIAO#RULE";

    @Autowired
    private OrgClassLessonDao classLessonDao;
    @Autowired
    private OrgStudentLessonDao studentLessonDao;
    @Autowired
    private OrgCourseConsumeRuleDao courseConsumeRuleDao;
    @Autowired
    private OrgLessonSignDao orgLessonSignDao;
    @Autowired
    private OrgCourseDao courseDao;

    private ExecutorService threadPool = Executors.newFixedThreadPool(10);

    private static final Semaphore updateTimeStuLessonStatusSemaphore = new Semaphore(1);
    private static final Semaphore startTimeStuLessonStatusSemaphore = new Semaphore(1);

    /**
     * 监控签到变化 获取最新签到变更的学生和课程
     * 
     * @return
     */
    private List<OrgStudentCourse> getStudentCourseListBySignUp() {
        return null;
    }

    @Override
    public void refreshKexiaoRule(int beforeMinutes) {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.MINUTE, -1 * beforeMinutes);
        List<OrgCourseConsumeRule> rules = courseConsumeRuleDao.queryConsumRuleListByUpdateTime(calendar.getTime());
        if (rules != null) {
            for (OrgCourseConsumeRule rule : rules) {
                Integer oldRule = getKexiaoRule(rule.getCourseId());
                if (!rule.getRuleValue().equals(oldRule)) {
                    //课消规则如果发生了改变，需要重新计算该门课下所有学生的课消
                    threadPool.submit(new KexiaoRunnable(rule.getCourseId(),rule.getRuleValue()));
                }
                setKexiaoRule(rule.getCourseId(), rule.getRuleValue());
            }
        }
        log.info("[Kexiao] Refresh kexiao rule.");
    }

    /**
     * 按签到算课消
     */
    public void updateClassLessonStatusBySignUp(List<OrgLessonSign> lessonSigns) {
        if (lessonSigns != null && lessonSigns.size() < 1) {
            return;
        }

        List<Long> list = ListUtil.toKeyList(lessonSigns, "courseId", OrgLessonSign.class);
        Map<Long,Long> classCourseMap = getClassCourseIdMap(list);
        List<OrgLessonSign> unSignList = new ArrayList<>();
        List<OrgLessonSign> finishedList = new ArrayList<>();
        for (OrgLessonSign sign : lessonSigns) {
            Integer rule = getKexiaoRule(classCourseMap.get(sign.getCourseId()));
            if (rule != null && rule>0) {
                LessonStatus status = getKexiaoStatus(sign, rule);
                if (status == LessonStatus.FINISHED) {
                    finishedList.add(sign);
                } else {
                    unSignList.add(sign);
                }
            }
        }

        if (unSignList.size() > 0) {
            studentLessonDao.batchUpdateKexiaoStatusBySign(unSignList, LessonStatus.UN_START.getStatus());
        }
        if (finishedList.size() > 0) {
            studentLessonDao.batchUpdateKexiaoStatusBySign(finishedList, LessonStatus.FINISHED.getStatus());
        }

    }

    private Map<Long,Long> getClassCourseIdMap(Collection<Long> clazzIds){
        List<OrgCourse> classList = courseDao.getByIds(clazzIds, "id", "parentId","courseType");
        Map<Long,Long> classCourseMap = new HashMap<>();
        for (OrgCourse clazz:classList){
            if(CourseTypeEnum.isOneToOne(clazz.getCourseType())){
                classCourseMap.put(clazz.getId(),clazz.getParentId());
            }else {
                classCourseMap.put(clazz.getId(),clazz.getId());
            }
        }
        return classCourseMap;
    }

    private LessonStatus getKexiaoStatus(OrgLessonSign sign, int rule) {
        int byteCode = 0;
        // 1，签到；2，请假；3，旷课
        if (sign.getStatus() == SignStatus.SIGNED.getCode()) {
            byteCode = 1;
        } else if (sign.getStatus() == SignStatus.LEAVE.getCode()) {
            byteCode = 2;
        } else if (sign.getStatus() == SignStatus.ABSENT.getCode()) {
            byteCode = 4;
        } else {
            log.info("[Kexiao] UnSign.sign={}", sign);
        }
        int result = byteCode & rule;
        if (result > 0) {
            return LessonStatus.FINISHED;
        } else {
            return LessonStatus.UN_START;
        }
    }

    public void setKexiaoRule(final long courseId, final int rule) {
        redisTemplate.execute(new RedisCallback<Integer>() {
            @Override
            public Integer doInRedis(RedisConnection connection) throws DataAccessException {
                connection.select(TianXiaoConstant.TX_PV_REDIS_DB);
                RedisSerializer<String> serializer = getRedisSerializer();
                byte[] key = serializer.serialize(TX_ORG_KEXIAO_RULE);
                byte[] field = serializer.serialize(String.valueOf(courseId));
                byte[] value = serializer.serialize(String.valueOf(rule));
                connection.hSet(key, field, value);
                return null;
            }
        });
    }

    public Integer getKexiaoRule(final long courseId) {
        String ruleStr = redisTemplate.execute(new RedisCallback<String>() {
            @Override
            public String doInRedis(RedisConnection connection) throws DataAccessException {
                connection.select(TianXiaoConstant.TX_PV_REDIS_DB);
                RedisSerializer<String> serializer = getRedisSerializer();
                byte[] keyBytes = serializer.serialize(TX_ORG_KEXIAO_RULE);
                byte[] field = serializer.serialize(String.valueOf(courseId));
                byte[] value = connection.hGet(keyBytes, field);
                String rule = serializer.deserialize(value);
                log.info("[Kexiao] kexiao rule={},courseId={}", rule, courseId);
                return rule;
            }
        });
        if (StringUtils.isNotBlank(ruleStr)) {
            return Integer.parseInt(ruleStr);
        } else {
            return null;
        }
    }

    /**
     * 同步排课状态
     * 
     * @param beforeMinutes
     */
    @Transactional
    public void syncClassLessonStatus(int beforeMinutes, int type) {
        log.info("[Kexiao] Start===============");
        long begin = System.currentTimeMillis();
        if (type == UPDATE_TIME_TYPE) {
            boolean isUpdateTimeAllowed = updateTimeStuLessonStatusSemaphore.tryAcquire();
            if (!isUpdateTimeAllowed) {
                log.info("[Kexiao] Other thread is execute.");
                return;
            }
        } else {
            boolean isStartTimeAllowed = startTimeStuLessonStatusSemaphore.tryAcquire();
            if (!isStartTimeAllowed) {
                log.info("[Kexiao] Other thread is execute.");
                return;
            }
        }
        try {
            Calendar calendar = Calendar.getInstance();
            calendar.add(Calendar.MINUTE, -1 * beforeMinutes);
            PageDto pageDto = new PageDto();
            pageDto.setPageSize(PAGE_SIZE);
            Date now = new Date();
            List<OrgClassLesson> lessons = null;
            if (type == UPDATE_TIME_TYPE) {
                lessons = classLessonDao.getOrgClassLessonsByUpdateTime(calendar.getTime(), now, pageDto);// 按跟新时间同步数据
            } else {
                lessons = classLessonDao.getOrgClassLessonsByStartTime(calendar.getTime(), now, pageDto);// 按修改时间同步数据
            }
            log.info("[Kexiao]Executing===============");
            while (lessons != null && lessons.size() > 0) {
                updateClassLessonStatus(lessons);
                pageDto.setPageNum(pageDto.getPageNum() + 1);
                if (type == UPDATE_TIME_TYPE) {
                    lessons = classLessonDao.getOrgClassLessonsByUpdateTime(calendar.getTime(), now, pageDto);
                } else {
                    lessons = classLessonDao.getOrgClassLessonsByStartTime(calendar.getTime(), now, pageDto);
                }
            }
        } finally {
            if (type == UPDATE_TIME_TYPE) {
                updateTimeStuLessonStatusSemaphore.release();
            } else {
                startTimeStuLessonStatusSemaphore.release();
            }
        }
        log.info("[Kexiao] cost={}", (System.currentTimeMillis() - begin));
    }

    private void updateClassLessonStatus(List<OrgClassLesson> lessons) {

        List<Long> list = ListUtil.toKeyList(lessons, "courseId", OrgClassLesson.class);
        Map<Long,Long> classCourseMap = getClassCourseIdMap(list);

        Date now = new Date();
        Set<Long> unStartIds = new HashSet<>();
        Set<Long> finishedIds = new HashSet<>();
        log.info("[Kexiao] classCourseMap={},courseId={}",classCourseMap,list);
        for (OrgClassLesson lesson : lessons) {
            Long courseId = classCourseMap.get(lesson.getCourseId());
            if(courseId==null){
                log.warn("[Kexiao] classId is {},courseId is null",lesson.getCourseId());
                continue;
            }
            Integer rule = getKexiaoRule(courseId);
            if (rule != null && rule > 0) {//>0为按签到算课消
                log.info("[Kexiao]Kexiao Rule={},courseId={}", rule, lesson.getCourseId());
                continue;
            }

            if (lesson.getStartTime().compareTo(now) <= 0) {
                finishedIds.add(lesson.getId());
            } else {
                unStartIds.add(lesson.getId());
            }
        }
        if (unStartIds.size() > 0) {
            studentLessonDao.batchUpdateKexiaoStatus(unStartIds, LessonStatus.UN_START.getStatus());
        }
        if (finishedIds.size() > 0) {
            studentLessonDao.batchUpdateKexiaoStatus(finishedIds, LessonStatus.FINISHED.getStatus());
        }
    }

    private class KexiaoRunnable implements Runnable{
        private long courseId;
        private int rule;

        public KexiaoRunnable(long courseId, int rule) {
            this.courseId = courseId;
            this.rule = rule;
        }

        @Override
        public void run() {
            log.info("[Kexiao] courseId={},rule={}", courseId, rule);
            List<Long> classIds = new ArrayList<>();
            OrgCourse course = courseDao.getById(courseId, "id", "parentId", "courseType","isClass","isCourse");
            if(CourseTypeEnum.IS_CLASS_FALSE.getCode()==course.getIsClass()
                    && CourseTypeEnum.IS_COURSE_TRUE.getCode()==course.getIsCourse()){
                classIds = courseDao.getClassIdsByParentId(courseId);
            }else {
                classIds.add(courseId);
            }

            log.info("[Kexiao] classIds is {}",classIds);

            if(classIds==null || classIds.size()<1){
                return;
            }

            if(rule>0){//按签到算课消
                List<OrgLessonSign> signs = orgLessonSignDao.getCourseLessonSignIn(classIds);
                List<OrgLessonSign> finishedSigns = new ArrayList<>();
                List<OrgStudentLesson> lessons = studentLessonDao.getOrgStudentLessonsByCourseIds(classIds);
                if(signs!=null){
                    for (Iterator<OrgLessonSign> iterator = signs.iterator();iterator.hasNext();){
                        OrgLessonSign sign = iterator.next();
                        LessonStatus status = getKexiaoStatus(sign,rule);
                        if(status==LessonStatus.FINISHED){
                            finishedSigns.add(sign);
                        }
                    }
                }
                removeSign(lessons, finishedSigns);
                List<OrgLessonSign> unSigns = convertToLessonSigns(lessons);
                log.info("[Kexiao] un_start={},finished={}",unSigns,finishedSigns);
                studentLessonDao.batchUpdateKexiaoStatusBySign(unSigns, LessonStatus.UN_START.getStatus());
                studentLessonDao.batchUpdateKexiaoStatusBySign(finishedSigns, LessonStatus.FINISHED.getStatus());
            }else {//按开课时间算课消
                List<OrgClassLesson> classLessons = classLessonDao.getLessonByCourseIds(classIds);
                updateClassLessonStatus(classLessons);
            }
        }

        /**
         * 从学生课节中移除已计算课消的课节
         * @param lessons
         * @param finishedSigns
         */
        private void removeSign(List<OrgStudentLesson> lessons,List<OrgLessonSign> finishedSigns){
            if(lessons==null || lessons.isEmpty() || finishedSigns==null || finishedSigns.isEmpty()){
                return;
            }
            Map<String,OrgStudentLesson> lessonMap = new HashMap<>();
            for (OrgStudentLesson lesson:lessons){
                lessonMap.put(lesson.getUserId()+"_"+lesson.getLessonId(),lesson);
            }
            for (OrgLessonSign sign:finishedSigns){
                OrgStudentLesson lessonSign = lessonMap.get(sign.getUserId()+"_"+sign.getLessonId());
                if(lessonSign!=null){
                    lessons.remove(lessonSign);
                }
            }
        }

        private List<OrgLessonSign> convertToLessonSigns(List<OrgStudentLesson> lessons){
            List<OrgLessonSign> result = new ArrayList<>();
            if(lessons!=null) {
                for (OrgStudentLesson lesson : lessons) {
                    OrgLessonSign sign = new OrgLessonSign();
                    sign.setLessonId(lesson.getLessonId());
                    sign.setUserId(lesson.getUserId());
                    result.add(sign);
                }
            }
            return result;
        }

    }
}