
/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2016 All Rights Reserved.
 */

package com.baijia.tianxiao.biz.message.service.impl;

import com.baijia.tianxiao.biz.message.dto.request.GroupMsgSendRequestDto;
import com.baijia.tianxiao.biz.message.dto.response.GroupMsgSendResponseDto;
import com.baijia.tianxiao.biz.message.service.OrgGroupMsgService;
import com.baijia.tianxiao.constant.GroupMessageReveiverType;
import com.baijia.tianxiao.dal.org.dao.OrgGroupMsgDao;
import com.baijia.tianxiao.dal.org.dao.OrgGroupMsgReceiverDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentCourseDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentDao;
import com.baijia.tianxiao.dal.org.po.OrgGroupMsg;
import com.baijia.tianxiao.dal.org.po.OrgGroupMsgReceiver;
import com.baijia.tianxiao.dal.org.po.OrgStudent;
import com.baijia.tianxiao.dal.wechat.dao.FansDao;
import com.baijia.tianxiao.dal.wechat.po.Fans;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.course.constant.CourseErrorCode;
import com.baijia.tianxiao.sal.wechat.api.TemplateMsgService;
import com.baijia.tianxiao.sal.wechat.api.WechatQRCodeService;
import com.baijia.tianxiao.sal.wechat.dto.templatemsg.batch.BatchMsg;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.CollectorUtil;
import com.baijia.tianxiao.util.CourseSmsTokenUtil;
import com.baijia.tianxiao.util.GenericsUtils;
import com.baijia.tianxiao.util.date.DateUtil;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import javax.annotation.Resource;

import lombok.extern.slf4j.Slf4j;

/**
 * @title OrgGroupMsgServiceImpl
 * @desc TODO
 * @author shanyu
 * @date 2016年2月24日
 * @version 1.0
 */
@Slf4j
@Service
public class OrgGroupMsgServiceImpl implements OrgGroupMsgService {

    private static String suffix = "...";

    private static String first = "您收到一条消息，/n";

    private static String remark = "/n请点击查看";

    private static String picRegex = "\\[img[^\\[\\]]*\\]"; // 图片正则

    private static String picReplacement = "[图片]"; // 图片正则

    private static String auditRegex = "\\[audio[^\\[\\]]*\\]"; // 声音正则

    private static String auditReplacement = "[语音]"; // 声音正则

    @Resource
    private FansDao fansDao;

    @Resource
    private OrgStudentDao orgStudentDao;

    @Resource
    private OrgGroupMsgDao orgGroupMsgDao;

    @Resource
    private OrgGroupMsgReceiverDao orgGroupMsgReceiverDao;

    @Resource
    private OrgStudentCourseDao orgStudentCourseDao;

    @Resource
    private TemplateMsgService templateMsgService;

    @Resource
    private WechatQRCodeService wechatQRCodeService;

    private ExecutorService threadPool = Executors.newCachedThreadPool();

    @Override
    @Transactional(rollbackFor = Exception.class)
    public OrgGroupMsg sendGroupMsg(GroupMsgSendRequestDto groupMsgSendRequestDto, final Long orgId, final String url) {
        log.info("send group wechat message params={},orgId={},url={}", groupMsgSendRequestDto, orgId, url);
        Preconditions.checkArgument(orgId != null, "orgId is null!");
        int receiverSize = 0;
        // 粉丝
        final List<Fans> fans = fansDao.getByIds(groupMsgSendRequestDto.getFanIdList());
        receiverSize += fans.size();
        log.info("fans size={}", fans.size());
        List<Long> userIds =
            orgStudentCourseDao.getStudentIdsByCourseIds(orgId, groupMsgSendRequestDto.getCourseIdList());
        final List<OrgStudent> students = orgStudentDao.getStudentByUserIds(orgId, userIds);
        Set<Long> receiverUserIds = Sets.newHashSet();
        receiverUserIds.addAll(userIds);
        students.addAll(orgStudentDao.getByIds(groupMsgSendRequestDto.getStudentIdList()));

        receiverSize += students.size();

        log.info("receiver size ={}", receiverSize);
        if (receiverSize == 0) {
            throw new BussinessException(CourseErrorCode.RECEIVER_NOT_EXIST);
        }
        final OrgGroupMsg msg = new OrgGroupMsg();
        msg.setContent(groupMsgSendRequestDto.getContent());
        msg.setMsgType(groupMsgSendRequestDto.getMsgType());
        msg.setOrgId(orgId);
        msg.setSubject(groupMsgSendRequestDto.getSubject());
        msg.setStudentIds(GenericsUtils.isNullOrEmpty(groupMsgSendRequestDto.getStudentIds()) ? ""
            : groupMsgSendRequestDto.getStudentIds());
        msg.setFanIds(
            GenericsUtils.isNullOrEmpty(groupMsgSendRequestDto.getFanIds()) ? "" : groupMsgSendRequestDto.getFanIds());
        msg.setCourseIds(GenericsUtils.isNullOrEmpty(groupMsgSendRequestDto.getCourseIds()) ? ""
            : groupMsgSendRequestDto.getCourseIds());
        msg.setReceiverSize(receiverSize);
        this.orgGroupMsgDao.save(msg, false);
        log.info("msgId={}", msg.getId());
        threadPool.submit(new Runnable() {
            @Override
            public void run() {
                List<OrgGroupMsgReceiver> receivers = Lists.newArrayList();
                for (Fans fan : fans) {
                    receivers.add(buildReceiver(fan, msg.getId(), orgId));
                }
                Map<Long, OrgStudent> studentMap = Maps.newHashMap();
                if (CollectionUtils.isNotEmpty(students)) {
                    studentMap = CollectorUtil.collectMap(students, new Function<OrgStudent, Long>() {
                        @Override
                        public Long apply(OrgStudent arg0) {
                            return arg0.getUserId();
                        }
                    });
                    log.info("student's size={}", studentMap.size());
                }
                for (OrgStudent student : studentMap.values()) {
                    receivers.add(buildReceiver(student, msg.getId(), orgId));
                }
                orgGroupMsgReceiverDao.saveAll(receivers, "msgId", "openId", "orgId", "type", "receiverId");
                long begin = System.currentTimeMillis();
                // 发送消息
                for (OrgGroupMsgReceiver receiver : receivers) {
                    sendMsg(receiver, msg, url);
                }
                log.info("the time cost of create msg is:{},msgId={}", (System.currentTimeMillis() - begin),
                    msg.getId());
            }
        });
        return msg;
    }

    /**
     * 发送消息
     *
     * @param receiver
     * @param msg
     * @param url
     */
    private void sendMsg(OrgGroupMsgReceiver receiver, OrgGroupMsg msg, String url) {
        log.info("receiver={},msg={}", receiver, msg);
        BatchMsg batchMsg = new BatchMsg();
        batchMsg.setOrgId(receiver.getOrgId().intValue());
        batchMsg.setOpenId(receiver.getOpenId());
        batchMsg.setFirst(first);
        batchMsg.setRemark(remark);
        batchMsg.setUrl(String.format(url, getSmsToken(receiver.getMsgId(), receiver.getOrgId())));
        String content = null;
        if (StringUtils.isNoneBlank(msg.getContent())) {
            content = msg.getContent().replaceAll(picRegex, picReplacement).replaceAll(auditRegex, auditReplacement);
        }
        if (StringUtils.isNoneBlank(content) && content.length() > 19) {
            batchMsg.setNoticeDesc(content.substring(0, 19) + suffix);
        } else {
            batchMsg.setNoticeDesc(content + suffix);
        }
        batchMsg.setNoticeNo(String.valueOf(msg.getId()));
        batchMsg.setNoticeName(msg.getSubject());
        try {
            templateMsgService.sendTemplateMsg(batchMsg.toJsonStr());
        } catch (Exception e) {
            log.warn("send message faild", e);
        }
    }

    /**
     * 生成token
     *
     * @param msgId
     * @param orgId
     * @return
     */
    private String getSmsToken(Long msgId, Long orgId) {
        Map<String, Object> params = Maps.newHashMap();
        params.put("msgId", msgId);
        try {
            return CourseSmsTokenUtil.encodeExpireToken(params, orgId, null);
        } catch (Exception e) {
            log.error(e.getMessage());
            return null;
        }
    }

    private OrgGroupMsgReceiver buildReceiver(Fans fans, Long msgId, Long orgId) {
        OrgGroupMsgReceiver reciever = new OrgGroupMsgReceiver();
        reciever.setMsgId(msgId);
        reciever.setOpenId(fans.getOpenId());
        reciever.setOrgId(orgId);
        reciever.setType(GroupMessageReveiverType.FANS.getType());
        reciever.setReceiverId(fans.getId().longValue());
        return reciever;
    }

    private OrgGroupMsgReceiver buildReceiver(OrgStudent orgStudent, Long msgId, Long orgId) {
        OrgGroupMsgReceiver reciever = new OrgGroupMsgReceiver();
        reciever.setMsgId(msgId);
        reciever.setOpenId(orgStudent.getWeixin());
        reciever.setOrgId(orgId);
        reciever.setType(GroupMessageReveiverType.FANS.getType());
        reciever.setReceiverId(orgStudent.getUserId());
        return reciever;
    }

    @Override
    public List<GroupMsgSendResponseDto> groupMsgList(Long orgId, PageDto pageDto, String url) {
        List<OrgGroupMsg> orgGroupMsgList = orgGroupMsgDao.getByOrgId(orgId, pageDto);
        List<GroupMsgSendResponseDto> result = Lists.newArrayList();
        for (OrgGroupMsg orgGroupMsg : orgGroupMsgList) {
            result.add(buildGroupMsgSendResponseDto(orgGroupMsg, url));
        }
        return result;
    }

    private GroupMsgSendResponseDto buildGroupMsgSendResponseDto(OrgGroupMsg orgGroupMsg, String url) {
        GroupMsgSendResponseDto dto = new GroupMsgSendResponseDto();
        dto.setMsgId(orgGroupMsg.getId());
        dto.setContent(orgGroupMsg.getContent());
        dto.setSubject(orgGroupMsg.getSubject());
        dto.setSendTimeStamp(orgGroupMsg.getCreateTime().getTime());
        dto.setReceiverSize(orgGroupMsg.getReceiverSize());
        dto.setSendTime(DateUtil.getStrByDateFormate(orgGroupMsg.getCreateTime(), "yyyy-MM-dd"));
        if (url != null) {
            dto.setUrl(String.format(url, getSmsToken(orgGroupMsg.getId(), orgGroupMsg.getOrgId())));
        }
        return dto;
    }


    @Override
    public GroupMsgSendResponseDto groupMsgDetail(Long orgId, Long msgId, Long reveiverId, Integer receiverType) {
        Preconditions.checkArgument(orgId != null, "orgId is null!");
        Preconditions.checkArgument(msgId != null, "msgId is null!");
        // Preconditions.checkArgument(reveiverId != null, "reveiverId is null!");

        OrgGroupMsg msg = orgGroupMsgDao.getById(msgId);
        
        msg.setContent( msg.getContent().replaceAll(" ","  ") );//word文档粘贴问题处理
        if (msg.getOrgId().longValue() != orgId) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "消息不存在！");
        }
        
        // OrgGroupMsgReceiver receiver = orgGroupMsgReceiverDao.queryReceiver(orgId, msgId, reveiverId);
        // if (receiver == null || receiver.getType().intValue() != receiverType.intValue()) {
        // throw new BussinessException(CommonErrorCode.PARAM_ERROR, "接收人不存在！");
        // }
        return buildGroupMsgSendResponseDto(msg, null);
    }

    @Override
    public String getWechatUrl(Long orgId) {
        return wechatQRCodeService.getQRCodeHeadImgUrl(orgId.intValue());
    }

    @Override
    public void sendGroupMsg(final Long msgId, final Long orgId, final String url) {
        log.info("sendGroupMsg - msgId:{}, orgId:{}, url:{}", msgId, orgId, url);
        List<OrgGroupMsgReceiver> receivers = orgGroupMsgReceiverDao.queryReceivers(orgId, msgId, null);
        OrgGroupMsg orgGroupMsg = orgGroupMsgDao.getById(msgId);

        if (CollectionUtils.isNotEmpty(receivers) && orgGroupMsg != null) {
            long begin = System.currentTimeMillis();
            // 发送消息
            for (OrgGroupMsgReceiver receiver : receivers) {
                sendMsg(receiver, orgGroupMsg, url);
            }
            log.info("[sendGroupMsg] the time cost of sendGroupMsg msg is:{},msgId={}",
                (System.currentTimeMillis() - begin), msgId);
        } else {
            log.warn("[sendGroupMsg] msgId is invalidate.msgId={}", msgId);
            if (CollectionUtils.isEmpty(receivers)) {
                log.warn("[sendGroupMsg] Receivers is null!");
            }
            if (orgGroupMsg == null) {
                log.warn("[sendGroupMsg] OrgGroupMsg is null!");
            }

            new Thread(new Runnable() {
                @Override
                public void run() {
                    for (int i = 0; i < 10; i++) {
                        try {
                            log.info("[sendGroupMsg] repeat count={},msgId={}", i + 1, msgId);
                            Thread.sleep(5000 * (i + 1));
                        } catch (InterruptedException e) {
                            e.printStackTrace();
                        }
                        List<OrgGroupMsgReceiver> receivers = orgGroupMsgReceiverDao.queryReceivers(orgId, msgId, null);
                        OrgGroupMsg orgGroupMsg = orgGroupMsgDao.getById(msgId);
                        if (CollectionUtils.isNotEmpty(receivers) && orgGroupMsg != null) {
                            for (OrgGroupMsgReceiver receiver : receivers) {
                                sendMsg(receiver, orgGroupMsg, url);
                            }
                            log.info("[sendGroupMsg] the time cost of sendGroupMsg msgId={}", msgId);
                            break;
                        }
                    }
                }
            }).start();
        }
    }

}
