package com.baijia.tianxiao.biz.service.impl;

import com.baijia.tianxiao.biz.constants.BindCardOpType;
import com.baijia.tianxiao.biz.service.BankCardService;
import com.baijia.tianxiao.consants.UserRole;
import com.baijia.tianxiao.constants.sms.TxSmsCodeType;
import com.baijia.tianxiao.dal.finance.po.OrgBankCard;
import com.baijia.tianxiao.dal.org.po.OrgAccount;
import com.baijia.tianxiao.dal.org.po.OrgInfo;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.common.api.CommonMsgService;
import com.baijia.tianxiao.sal.organization.api.OrgAccountService;
import com.baijia.tianxiao.sal.organization.finance.dto.response.TxWithdrawBankDto;
import com.baijia.tianxiao.sal.organization.finance.service.TxBankCardService;
import com.baijia.tianxiao.sal.organization.org.service.OrgInfoService;
import com.baijia.tianxiao.util.SmsContentHelper;
import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.IOException;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * Created by wengshengli on 2017/5/22.
 */
@Service
@Slf4j
public class BankCardServiceImpl implements BankCardService {

    @Autowired
    private OrgInfoService orgInfoService;

    @Autowired
    private OrgAccountService orgAccountService;

    @Autowired
    private CommonMsgService commonMsgService;

    @Autowired
    private TxBankCardService txBankCardService;

    private Map<Long, Object> lockMap = new HashMap<Long, Object>();

    @Override
    public void bindCorBank(Long orgId, OrgBankCard bankCard) throws IOException {
        log.info("bind core bank params={},{}", orgId, bankCard);
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is null!");
        Preconditions.checkArgument(StringUtils.isNotBlank(bankCard.getCardOwnerName()), "银行开户姓名为空!");
        Preconditions.checkArgument(bankCard.getCardOwnerName().length() < 30, "企业开户名称过长!");
        bankCard.setCardOwnerName(bankCard.getCardOwnerName().trim());
        Preconditions.checkArgument(StringUtils.isNotBlank(bankCard.getBankNo()), "所在银行为空!");
        Preconditions.checkArgument(StringUtils.isNotBlank(bankCard.getRegion()), "开户行地区信息为空!");
        Preconditions.checkArgument(StringUtils.isNotBlank(bankCard.getCardNum()), "卡号为空!");

        bankCard.setRegion(bankCard.getRegion().trim());
        if (bankCard.getCardOpenLocation() == null) {
            bankCard.setCardOpenLocation("");
        } else {
            bankCard.setCardOpenLocation(bankCard.getCardOpenLocation().trim());
        }
        bankCard.setCardNum(bankCard.getCardNum().trim());
        bankCard.setOrgId(orgId);

        OrgAccount orgAccount = orgAccountService.getOrgAccountById(orgId.intValue());
        if (orgAccount == null) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "机构不存在");
        }

        Object objLock = null;
        synchronized (this) {
            objLock = lockMap.get(orgId);
            if (objLock == null) {
                objLock = new Object();
                lockMap.put(orgId, objLock);
            }
        }
        synchronized (objLock) {
            TxWithdrawBankDto withdrawBankDto = txBankCardService.getOrgBankInfo(orgId);
            if (withdrawBankDto != null) {
                throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "已经绑过卡了");
            }
            bankCard.setCreatTime(new Date());
            txBankCardService.bindCotCard(bankCard);
            sendBindCardSms(orgId.longValue(), BindCardOpType.BIND_CARD, orgAccount.getMobile(), bankCard.getCardNum());
        }
    }

    @Override
    public boolean sendBindCardSms(Long orgId, BindCardOpType bindCardOpType, String mobile, String cardNo) {
        log.info("Bankcard service send message== {},{},{},{}", orgId, bindCardOpType, mobile, cardNo);
        OrgInfo orgInfo = orgInfoService.getOrgInfoByOrgId(orgId.intValue());
        if (orgInfo == null) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "机构不存在");
        }
        String content = "";
        cardNo = cardNo.substring(cardNo.length() - 4);
        TxSmsCodeType txSmsCodeType = null;
        if (bindCardOpType == BindCardOpType.BIND_CARD) {
            txSmsCodeType = TxSmsCodeType.ORG_BANK_BIND_CARD;
            content = SmsContentHelper.createBindBankCard(orgInfo.getShortName(), cardNo);
        } else if (bindCardOpType == BindCardOpType.UNBIND_CARD) {
            txSmsCodeType = TxSmsCodeType.ORG_BANK_UNBIND_CARD;
            content = SmsContentHelper.createUnBindBankCard(orgInfo.getShortName(), cardNo);
        } else {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "绑卡操作类型错误");
        }

        return commonMsgService.sendTxSms(orgId, orgId, UserRole.ORGANIZATION.getRole(), null, null, txSmsCodeType, null, mobile, content);

    }
}
