
/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2016 All Rights Reserved.
 */

package com.baijia.tianxiao.biz.erp.service.impl;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.regex.Pattern;

import javax.annotation.Resource;

import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataAccessException;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import com.baijia.tianxiao.biz.erp.constant.ErpBizConf;
import com.baijia.tianxiao.biz.erp.constant.UploadFileStatus;
import com.baijia.tianxiao.biz.erp.service.CourseClassService;
import com.baijia.tianxiao.biz.erp.service.ErpUploadService;
import com.baijia.tianxiao.biz.erp.thread.SaveStudentsThread;
import com.baijia.tianxiao.biz.erp.util.ErpUploadFileUtil;
import com.baijia.tianxiao.biz.erp.util.ErpUploadFileUtil.StudentFormat;
import com.baijia.tianxiao.biz.erp.vo.ImportCourse;
import com.baijia.tianxiao.biz.erp.vo.ImportStudent;
import com.baijia.tianxiao.constants.OrgCourseStatus;
import com.baijia.tianxiao.dal.enums.CourseTypeEnum;
import com.baijia.tianxiao.dal.org.dao.OrgClassLessonDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentLessonDao;
import com.baijia.tianxiao.dal.org.po.OrgCourse;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.filter.TianxiaoPCContext;
import com.baijia.tianxiao.sal.course.dto.OrgCourseInfoDto;
import com.baijia.tianxiao.sal.course.service.OrgCourseService;
import com.baijia.tianxiao.sal.course.service.impl.ImportOrgCourseProcessServiceImpl.CourseFormat;
import com.baijia.tianxiao.sal.student.api.OrgStudentCourseService;
import com.baijia.tianxiao.sal.student.api.OrgStudentService;
import com.baijia.tianxiao.sal.student.dto.StudentInfoDto;
import com.baijia.tianxiao.sal.student.dto.response.OrgStudentAddresponseDto;
import com.baijia.tianxiao.sal.upload.service.UploadFileReaderService;
import com.baijia.tianxiao.util.json.JacksonUtil;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import lombok.extern.slf4j.Slf4j;

/**
 * @title ErpUploadServiceImpl
 * @desc O(∩_∩)O~
 * @author caoliang
 * @date 2016年7月5日
 * @version 1.0
 */
@Slf4j
@Service
public class ErpUploadServiceImpl implements ErpUploadService {

    private static final String TASK_SUFFIX = "_task_tx";
    public static final String STATUS_SUFFIX = "_status_tx";
    private static final String VALID_LIST_SUFFIX = "_valid_list_tx";
    public static final String INVALID_LIST_SUFFIX = "_invalid_list_tx";
    public static final String VALID_LIST_SIZE_SUFFIX = "_valid_list_size_tx";
    public static final String INVALID_LIST_SIZE_SUFFIX = "_invalid_list_size_tx";
    public static final Long EXPIRE = 60 * 120L;

    @Autowired
    private StringRedisTemplate redisTemplate;

    @Autowired(required = false)
    private ThreadPoolTaskExecutor threadPoolTaskExecutor;
    @Resource
    private OrgCourseService orgCourseService;
    @Autowired
    private OrgClassLessonDao orgClassLessonDao;
    @Autowired
    private OrgStudentService orgStudentService;
    @Autowired
    private OrgStudentCourseService orgStudentCourseService;
    @Autowired
    private OrgStudentLessonDao orgStudentLessonDao;
    @Autowired
    private CourseClassService courseClassService;

    /**
     * 校验excel是否为系统模板 校验数据格式 正确数据存redis 错误数据存redis
     * 
     * @throws IOException
     */
    @Override
    public String uploadCourse(MultipartFile file) throws Exception {
        final Integer orgId = TianxiaoPCContext.getOrgId();
        boolean isProcessing = isTaskProcessing(orgId);

        if (isProcessing) {
            throw new BussinessException(CommonErrorCode.PERMISSION_DENY, "上一个任务仍在处理中，请等待完成后，再次上传");
        }
        final String taskId = UUID.randomUUID().toString();
        final UploadFileReaderService uploadFileService = ErpUploadFileUtil.getUploadFileService(file);
        List<String> templateHeader = ErpUploadFileUtil.getTemplateHeader(ErpBizConf.TEMPLATE_COURSE);
        final List<String> headers = ErpUploadFileUtil.tranHeader(uploadFileService.readData());
        if (!headers.contains("错误原因")) {
            headers.add("错误原因");
        }
        log.debug("*******************temp:{},file:{}", templateHeader, headers);
        // 和系统提供header一致
        if (!CollectionUtils.isEqualCollection(templateHeader, headers)) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "请使用系统模板");
        }
        // 校验过程丢线程池
        threadPoolTaskExecutor.execute(new Runnable() {

            @Override
            public void run() {
                try {
                    // 初始化任务状态
                    setTaskStatusProcessing(taskId);
                    // 一行一行校验数据，正确、错误都存
                    final List<ImportCourse> errList = Lists.newArrayList();
                    final List<ImportCourse> validList = Lists.newArrayList();
                    Object[] lineData = null;
                    headers.remove("错误原因");
                    try {
                        while ((lineData = uploadFileService.readData()) != null) {
                            boolean invalid = false;
                            ImportCourse importCourse = new ImportCourse();
                            for (int i = 0; i < headers.size(); i++) {
                                Object value = lineData[i];
                                CourseFormat courseFormat = ErpUploadFileUtil.colMap.get(headers.get(i));
                                if (value == null || StringUtils.isBlank(value.toString().trim())) {
                                    invalid = true;
                                    BeanUtils.setProperty(importCourse, "errReason", "【" + headers.get(i) + "】不能为空");
                                } else {
                                    BeanUtils.setProperty(importCourse, courseFormat.getFieldName(),
                                        value.toString().trim());
                                    // 取值不合法
                                    if (!Pattern.matches(courseFormat.getFormatRegex(), value.toString())) {
                                        BeanUtils.setProperty(importCourse, "errReason", courseFormat.getFormatMsg());
                                        invalid = true;
                                    }
                                }
                            }
                            if (invalid) {
                                errList.add(importCourse);
                            } else {
                                validList.add(importCourse);
                            }
                        }
                    } catch (Exception e) {
                        log.error("valid upload course error~!", e);
                        setTaskStatusError(taskId);
                        return;
                    }
                    // 把正确的和错误的数据，存入缓存
                    cacheCourseData(taskId, validList, errList);
                    // 设置任务完成
                    setTaskStatusProcessed(taskId, validList.size(), errList.size());

                } catch (Throwable t) {
                    log.error("upload course error!", t);
                }
            }
        });

        return taskId;
    }

    /**
     * @param orgId
     */

    protected void clearOrgTask(final Integer orgId) {
        redisTemplate.execute(new RedisCallback<Boolean>() {

            @Override
            public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                connection.del(getTaskKey(orgId).getBytes());
                return true;
            }
        });
    }

    /**
     * @param taskId
     * @param
     * @param
     */

    protected void setTaskStatusProcessed(final String taskId, final Integer validSize, final Integer errSize) {
        redisTemplate.execute(new RedisCallback<Boolean>() {

            @Override
            public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                connection.setEx(getStatusKey(taskId).getBytes(), EXPIRE,
                    UploadFileStatus.PROCESSED.getCode().toString().getBytes());
                if (errSize != null) {
                    connection.setEx(getErrDataSizeKey(taskId).getBytes(), EXPIRE, String.valueOf(errSize).getBytes());
                }
                if (validSize != null) {
                    connection.setEx(getValidDataSizeKey(taskId).getBytes(), EXPIRE,
                        String.valueOf(validSize).getBytes());
                }
                return true;
            }
        });
    }

    /**
     * @param validList
     * @param errList
     */

    protected void cacheCourseData(final String taskId, final List<ImportCourse> validList,
        final List<ImportCourse> errList) {
        redisTemplate.execute(new RedisCallback<Boolean>() {

            @Override
            public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                if (CollectionUtils.isNotEmpty(validList)) {
                    for (ImportCourse importCourse : validList) {
                        redisTemplate.opsForList().leftPush(getValidDataKey(taskId), JacksonUtil.obj2Str(importCourse));
                    }
                    connection.expire(getValidDataKey(taskId).getBytes(), EXPIRE);
                }
                if (CollectionUtils.isNotEmpty(errList)) {
                    for (ImportCourse importCourse : errList) {
                        redisTemplate.opsForList().leftPush(getErrDataKey(taskId), JacksonUtil.obj2Str(importCourse));
                    }
                    connection.expire(getErrDataKey(taskId).getBytes(), EXPIRE);
                }
                return true;
            }
        });
    }

    /**
     * @param taskId
     */

    protected void setTaskStatusError(final String taskId) {
        redisTemplate.execute(new RedisCallback<Boolean>() {

            @Override
            public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                connection.setEx(getStatusKey(taskId).getBytes(), EXPIRE,
                    UploadFileStatus.ERROR.getCode().toString().getBytes());
                return true;
            }
        });
    }

    /**
     * @param taskId
     */

    protected void setTaskStatusProcessing(final String taskId) {
        redisTemplate.execute(new RedisCallback<Boolean>() {

            @Override
            public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                connection.setEx(getStatusKey(taskId).getBytes(), EXPIRE,
                    UploadFileStatus.PROCESSING.getCode().toString().getBytes());
                return true;
            }
        });
    }

    private final String getTaskKey(Integer orgId) {
        return orgId + TASK_SUFFIX;
    }

    private final String getStatusKey(String taskId) {
        return taskId + STATUS_SUFFIX;
    }

    private final String getErrDataKey(String taskId) {
        return taskId + INVALID_LIST_SUFFIX;
    }

    private final String getValidDataKey(String taskId) {
        return taskId + VALID_LIST_SUFFIX;
    }

    private final String getErrDataSizeKey(String taskId) {
        return taskId + INVALID_LIST_SIZE_SUFFIX;
    }

    private final String getValidDataSizeKey(String taskId) {
        return taskId + VALID_LIST_SIZE_SUFFIX;
    }

    @Override
    public List<ImportCourse> getValidCourseList(final String taskId) {
        return redisTemplate.execute(new RedisCallback<List<ImportCourse>>() {

            @Override
            public List<ImportCourse> doInRedis(RedisConnection connection) throws DataAccessException {
                List<ImportCourse> data = Lists.newArrayList();
                Long size = redisTemplate.opsForList().size(getValidDataKey(taskId));
                if (size != null && size > 0) {
                    for (int i = 0; i < size; i++) {
                        try {
                            data.add(JacksonUtil.str2Obj(redisTemplate.opsForList().rightPop(getValidDataKey(taskId)),
                                ImportCourse.class));
                        } catch (Exception e) {
                            e.printStackTrace();
                            log.error("uploadcourse getValidList error!", e);
                        }
                    }
                }
                if (CollectionUtils.isNotEmpty(data)) {
                    for (ImportCourse importCourse : data) {
                        redisTemplate.opsForList().leftPush(getValidDataKey(taskId), JacksonUtil.obj2Str(importCourse));
                    }
                }
                connection.expire(getValidDataKey(taskId).getBytes(), EXPIRE);
                return data;

            }
        });
    }

    @Override
    public List<ImportCourse> getInValidCourseList(final String taskId) {
        return redisTemplate.execute(new RedisCallback<List<ImportCourse>>() {

            @Override
            public List<ImportCourse> doInRedis(RedisConnection connection) throws DataAccessException {
                List<ImportCourse> data = Lists.newArrayList();
                Long size = redisTemplate.opsForList().size(getErrDataKey(taskId));
                if (size != null && size > 0) {
                    for (int i = 0; i < size; i++) {
                        try {
                            data.add(JacksonUtil.str2Obj(redisTemplate.opsForList().rightPop(getErrDataKey(taskId)),
                                ImportCourse.class));
                        } catch (Exception e) {
                            e.printStackTrace();
                            log.error("uploadcourse getValidList error!", e);
                        }
                    }
                }
                if (CollectionUtils.isNotEmpty(data)) {
                    for (ImportCourse importCourse : data) {
                        redisTemplate.opsForList().leftPush(getErrDataKey(taskId), JacksonUtil.obj2Str(importCourse));
                    }
                }
                connection.expire(getValidDataKey(taskId).getBytes(), EXPIRE);
                return data;

            }
        });
    }

    @Override
    public Map<String, String> getTaskStatus(final String taskId) {
        return redisTemplate.execute(new RedisCallback<Map<String, String>>() {

            @Override
            public Map<String, String> doInRedis(RedisConnection connection) throws DataAccessException {
                byte[] statusBytes = connection.get(getStatusKey(taskId).getBytes());
                byte[] invalidDataSizeBytes = connection.get(getErrDataSizeKey(taskId).getBytes());
                byte[] validDataSizeBytes = connection.get(getValidDataSizeKey(taskId).getBytes());
                String status = UploadFileStatus.PROCESSING.getCode().toString();
                Map<String, String> data = Maps.newHashMap();
                if (statusBytes != null) {
                    status = redisTemplate.getStringSerializer().deserialize(statusBytes);
                }
                if (invalidDataSizeBytes != null) {
                    data.put("invalidDataSize", redisTemplate.getStringSerializer().deserialize(invalidDataSizeBytes));
                }
                if (validDataSizeBytes != null) {
                    data.put("validDataSize", redisTemplate.getStringSerializer().deserialize(validDataSizeBytes));
                }
                data.put("status", status);
                data.put("taskId", taskId);
                return data;
            }
        });
    }

    @Override
    public void saveCourse(final String taskId) {
        final List<ImportCourse> importCourses = getValidCourseList(taskId);
        final Integer orgId = TianxiaoPCContext.getOrgId();
        if (orgId == null) {
            log.error("saveCourse error!orgId is null");
            return;
        }
        if (CollectionUtils.isEmpty(importCourses)) {
            setTaskStatusProcessed(taskId, 0, null);
            return;
        }
        boolean isProcessing = isTaskProcessing(orgId);

        if (isProcessing) {
            throw new BussinessException(CommonErrorCode.PERMISSION_DENY, "上一个任务仍在处理中，请等待完成后，再次上传");
        }

        threadPoolTaskExecutor.execute(new Runnable() {

            @Override
            public void run() {
                try {
                    setTaskStatusProcessing(taskId);
                    initOrgTask(orgId);
                    if (CollectionUtils.isNotEmpty(importCourses)) {
                        for (final ImportCourse importCourse : importCourses) {
                            try {
                                final OrgCourseInfoDto course = buildCourse(importCourse);
                                if (course == null) {
                                    throw new Exception("非法值");
                                }
                                orgCourseService.saveOrUpdateCourse(orgId.longValue(), course, false, true);
                            } catch (Exception e) {
                                log.error("save importCourse error!,importCourse:{}", importCourse, e);
                                // 错误追加
                                importCourse.setErrReason("系统错误，无法插入，请联系天校工作人员为您排查");
                                appendErrorData(taskId, importCourse);
                            }
                        }
                    }
                    setTaskStatusProcessed(taskId, null, null);
                } catch (Throwable t) {
                    log.error("save couers error!", t);
                    setTaskStatusError(taskId);
                } finally {
                    clearOrgTask(orgId);
                }

            }
        });

    }

    /**
     * @param taskId
     * @param
     */

    protected void appendErrorData(final String taskId, final Object data) {
        redisTemplate.execute(new RedisCallback<Boolean>() {

            @Override
            public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                redisTemplate.opsForList().leftPush(getErrDataKey(taskId), JacksonUtil.obj2Str(data));
                connection.expire(getErrDataKey(taskId).getBytes(), EXPIRE);
                connection.incr(getErrDataSizeKey(taskId).getBytes());
                connection.expire(getErrDataSizeKey(taskId).getBytes(), EXPIRE);
                connection.decr(getValidDataSizeKey(taskId).getBytes());
                connection.expire(getValidDataSizeKey(taskId).getBytes(), EXPIRE);
                return true;
            }
        });
    }

    /**
     * @param orgId
     */

    protected void initOrgTask(final Integer orgId) {
        redisTemplate.execute(new RedisCallback<Boolean>() {

            @Override
            public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                connection.setEx(getTaskKey(orgId).getBytes(), EXPIRE, "1".getBytes());
                return true;
            }
        });
    }

    private OrgCourseInfoDto buildCourse(ImportCourse importCourse) {
        OrgCourseInfoDto course = new OrgCourseInfoDto();
        if (importCourse.getCourseName() == null || importCourse.getCoursePrice() == null
            || importCourse.getFreq() == null || importCourse.getMaxStudent() == null) {
            return null;
        }
        course.setCourseName(importCourse.getCourseName().toString());
        Double inputCoursePrice = Double.valueOf(importCourse.getCoursePrice().toString());
        Double coursePrice = inputCoursePrice > ErpBizConf.MAX_COURSE_PRICE ? ErpBizConf.MAX_COURSE_PRICE : inputCoursePrice;
        course.setCoursePrice(coursePrice);
        course.setFreq(Integer.valueOf(importCourse.getFreq().toString()));
        course.setMaxStudent(Integer.valueOf(importCourse.getMaxStudent().toString()));
        // 上架
        course.setStatus(OrgCourseStatus.IN_PROGRESS.getCode());
        course.setIsClass(CourseTypeEnum.IS_CLASS_TRUE.getCode());
        course.setIsCourse(CourseTypeEnum.IS_COURSE_TRUE.getCode());
        course.setCourseType(CourseTypeEnum.COURSE_TYPE_CLASS.getCode());
        return course;
    }

    private OrgCourseInfoDto build1V1Course(ImportCourse importCourse) {
        OrgCourseInfoDto course = new OrgCourseInfoDto();
        if (importCourse.getCourseName() == null || importCourse.getCoursePrice() == null) {
            return null;
        }
        course.setCourseName(importCourse.getCourseName().toString());
        Double inputCoursePrice = Double.valueOf(importCourse.getCoursePrice().toString());
        Double coursePrice = inputCoursePrice > ErpBizConf.MAX_COURSE_PRICE ? ErpBizConf.MAX_COURSE_PRICE : inputCoursePrice;
        course.setCoursePrice(coursePrice);
        // 上架
        course.setStatus(OrgCourseStatus.IN_PROGRESS.getCode());
        course.setIsClass(CourseTypeEnum.IS_CLASS_FALSE.getCode());
        course.setIsCourse(CourseTypeEnum.IS_COURSE_TRUE.getCode());
        course.setCourseType(CourseTypeEnum.COURSE_TYPE_1v1.getCode());
        return course;
    }

    @Override
    public String uploadStudent(MultipartFile file) throws Exception {
        final Integer orgId = TianxiaoPCContext.getOrgId();
        boolean isProcessing = isTaskProcessing(orgId);

        if (isProcessing) {
            throw new BussinessException(CommonErrorCode.PERMISSION_DENY, "上一个任务仍在处理中，请等待完成后，再次上传");
        }
        final String taskId = UUID.randomUUID().toString();
        final UploadFileReaderService uploadFileService = ErpUploadFileUtil.getUploadFileService(file);
        List<String> templateHeader = ErpUploadFileUtil.getTemplateHeader(ErpBizConf.TEMPLATE_STUDENT);
        final List<String> headers = ErpUploadFileUtil.tranHeader(uploadFileService.readData());
        if (!headers.contains("错误原因")) {
            headers.add("错误原因");
        }
        // 和系统提供header一致
        if (!CollectionUtils.isEqualCollection(templateHeader, headers)) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "请使用系统模板");
        }
        // 校验过程丢线程池
        threadPoolTaskExecutor.execute(new Runnable() {

            @Override
            public void run() {
                try {
                    // 初始化任务状态
                    setTaskStatusProcessing(taskId);
                    // 一行一行校验数据，正确、错误都存
                    final List<ImportStudent> errList = Lists.newArrayList();
                    final List<ImportStudent> validList = Lists.newArrayList();
                    final List<String> validMobiles = Lists.newArrayList();
                    final List<String> validCourseNames = Lists.newArrayList();
                    final Map<String, ImportStudent> mobileMap = Maps.newHashMap();
                    Object[] lineData = null;
                    headers.remove("错误原因");
                    try {
                        while ((lineData = uploadFileService.readData()) != null) {
                            boolean invalid = false;
                            ImportStudent importStudent = new ImportStudent();
                            for (int i = 0; i < headers.size(); i++) {
                                Object value = lineData[i];
                                StudentFormat studentFormat = ErpUploadFileUtil.studentMap.get(headers.get(i));
                                if (studentFormat.getFieldName().equals("lessonCount")
                                    || studentFormat.getFieldName().equals("insertCount")) {
                                    if (value != null && StringUtils.isNoneBlank(value.toString().trim())) {
                                        if (!Pattern.matches(studentFormat.getFormatRegex(), value.toString())) {
                                            BeanUtils.setProperty(importStudent, "errReason",
                                                studentFormat.getFormatMsg());
                                            invalid = true;
                                        }
                                        BeanUtils.setProperty(importStudent, studentFormat.getFieldName(),
                                            value.toString().trim());
                                    }

                                } else if (value == null || StringUtils.isBlank(value.toString().trim())) {
                                    invalid = true;
                                    BeanUtils.setProperty(importStudent, "errReason", "【" + headers.get(i) + "】不能为空");
                                } else {
                                    BeanUtils.setProperty(importStudent, studentFormat.getFieldName(),
                                        value.toString().trim());
                                    // 取值不合法
                                    if (!Pattern.matches(studentFormat.getFormatRegex(), value.toString().trim())) {
                                        BeanUtils.setProperty(importStudent, "errReason", studentFormat.getFormatMsg());
                                        invalid = true;
                                    }
                                }
                                if (!invalid) {
                                    if (importStudent.getStudentMobile() != null
                                        && mobileMap.containsKey(importStudent.getStudentMobile())) {
                                        ImportStudent exist =
                                            mobileMap.get(importStudent.getStudentMobile().toString());
                                        if (!exist.getStudentName().toString()
                                            .equals(importStudent.getStudentName().toString())) {
                                            if(orgId!=18925 && orgId!=24909){
                                            invalid = true;
                                            importStudent.setErrReason("与表格内其他行数据【手机号】相同，但姓名不同，无法导入");
                                            }
                                        } else {
                                            if (exist.getCourseName() != null && importStudent.getCourseName() != null
                                                && exist.getCourseName().toString()
                                                    .equals(importStudent.getCourseName().toString())) {
                                                importStudent.setErrReason("该行与表格内其他行数据重复，无法导入");
                                                invalid = true;
                                            } else {
                                            }
                                        }
                                    }
                                }

                            }

                            if (invalid) {
                                errList.add(importStudent);
                            } else {
                                validList.add(importStudent);
                                mobileMap.put(importStudent.getStudentMobile().toString(), importStudent);
                                validMobiles.add(importStudent.getStudentMobile().toString());
                            }
                        }
                        // if validlist不为空
                        // 拿机构所有学生，放Map<mobile,name>
                        // 第一次遍历validList
                        // if 手机号在map里，但name不一致的，放errList里,remove
                        if (CollectionUtils.isNotEmpty(validList)) {
                            List<StudentInfoDto> tempDtos =
                                orgStudentService.listOrgStudentByMobiles(orgId.longValue(), validMobiles);
                            if (CollectionUtils.isNotEmpty(tempDtos)) {
                                ArrayListMultimap<String, String> mobileNameMap = ArrayListMultimap.create();
                                // Map<String, String> mobileNameMap = Maps.newHashMap();
                                for (StudentInfoDto dto : tempDtos) {
                                    mobileNameMap.put(dto.getMobile(), dto.getName());
                                }
                                Iterator<ImportStudent> iterator = validList.iterator();
                                while (iterator.hasNext()) {
                                    ImportStudent importStudent = iterator.next();
                                    if(orgId == 18925 || orgId == 24909){
                                        validCourseNames.add(importStudent.getCourseName());
                                    }else{
                                    if (mobileNameMap.containsKey(importStudent.getStudentMobile())) {
                                        List<String> names = mobileNameMap.get(importStudent.getStudentMobile());
                                        if (CollectionUtils.isNotEmpty(names)
                                            && !names.contains(importStudent.getStudentName())) {
                                            importStudent.setErrReason("与系统内学员【手机号】相同，但姓名不同，无法导入");
                                            errList.add(importStudent);
                                            iterator.remove();
                                        } else {
                                            validCourseNames.add(importStudent.getCourseName());
                                        }
                                    } else {
                                        validCourseNames.add(importStudent.getCourseName());
                                    }
                                    }
                                }
                            }
                        }
                        // if validlist不为空
                        // 把courseName存list，按照list取出所有OrgCourse
                        // 遍历List<OrgCourse>，得到 <CourseName,id> existCourse,
                        // 得到list<id>，拿到<courseid,List<OrgClassLesson>.size()>
                        // 遍历validlist，
                        // if 不在existCourse key中的，放errlist
                        // else 通过name拿到id，如果<courseid,List<OrgClassLesson>.size()>中没有，则没排课，放errlist。validlist.remove
                        if (CollectionUtils.isNotEmpty(validList)) {
                            Map<String, Long> courseNameIdMap = Maps.newHashMap();
                            Map<String, Integer> courseNameCountMap = Maps.newHashMap();
                            List<Long> courseIds = Lists.newArrayList();
                            List<OrgCourse> orgCourses =
                                orgCourseService.getAllOrgCoursesByNames(orgId.longValue(), validCourseNames);
                            if (CollectionUtils.isNotEmpty(orgCourses)) {
                                for (OrgCourse orgCourse : orgCourses) {
                                    courseNameIdMap.put(orgCourse.getName(), orgCourse.getId());
                                    if (courseNameCountMap.containsKey(orgCourse.getName())) {
                                        Integer count = courseNameCountMap.get(orgCourse.getName());
                                        courseNameCountMap.put(orgCourse.getName(), ++count);
                                    } else {
                                        courseNameCountMap.put(orgCourse.getName(), 1);
                                    }
                                    courseIds.add(orgCourse.getId());
                                }
                            }
                            Iterator<ImportStudent> iterator = validList.iterator();
                            while (iterator.hasNext()) {
                                ImportStudent importStudent = iterator.next();
                                if (!courseNameIdMap.containsKey(importStudent.getCourseName())) {
                                    importStudent.setErrReason("【班课课程名称】在系统不存在，请先手动录入班级，再批导班级学员");
                                    errList.add(importStudent);
                                    iterator.remove();
                                } else {
                                    if (courseNameCountMap.get(importStudent.getCourseName()) > 1) {
                                        importStudent.setErrReason("【班课课程名称】在系统有多个重名班级，重名班级的学员请手动录入");
                                        errList.add(importStudent);
                                        iterator.remove();
                                    }
                                }
                            }
                        }

                    } catch (Exception e) {
                        log.error("valid upload student error~!", e);
                        setTaskStatusError(taskId);
                        return;
                    }
                    log.debug("*********************validstudentlist:{}", validList);
                    log.debug("*********************invalidstudentlist:{}", errList);
                    // 把正确的和错误的数据，存入缓存
                    cacheStudentData(taskId, validList, errList);
                    // 设置任务完成
                    setTaskStatusProcessed(taskId, validList.size(), errList.size());
                } catch (Exception e) {
                    log.error("valid upload student error~!", e);
                }

            }
        });

        return taskId;

    }

    /**
     * @param taskId
     * @param validList
     * @param errList
     */

    protected void cacheStudentData(final String taskId, final List<ImportStudent> validList,
        final List<ImportStudent> errList) {
        redisTemplate.execute(new RedisCallback<Boolean>() {

            @Override
            public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                if (CollectionUtils.isNotEmpty(validList)) {
                    for (ImportStudent importStudent : validList) {
                        redisTemplate.opsForList().leftPush(getValidDataKey(taskId),
                            JacksonUtil.obj2Str(importStudent));
                    }
                    connection.expire(getValidDataKey(taskId).getBytes(), EXPIRE);
                }
                if (CollectionUtils.isNotEmpty(errList)) {
                    for (ImportStudent importStudent : errList) {
                        redisTemplate.opsForList().leftPush(getErrDataKey(taskId), JacksonUtil.obj2Str(importStudent));
                    }
                    connection.expire(getErrDataKey(taskId).getBytes(), EXPIRE);
                }
                return true;
            }
        });
    }

    @Override
    public void saveStudents(final String taskId) {
        final List<ImportStudent> importStudents = getValidStudentList(taskId);
        final Integer orgId = TianxiaoPCContext.getOrgId();
        final Integer cascadeId = TianxiaoPCContext.getTXCascadeId();
        if (orgId == null) {
            log.warn("saveStudents error!orgId is null");
            return;
        }
        if (CollectionUtils.isEmpty(importStudents)) {
            setTaskStatusProcessed(taskId, 0, null);
            return;
        }
        boolean isProcessing = isTaskProcessing(orgId);
        if (isProcessing) {
            throw new BussinessException(CommonErrorCode.PERMISSION_DENY, "上一个任务仍在处理中，请等待完成后，再次上传");
        }
        log.info("main thread start……taskId:{}",taskId);
        threadPoolTaskExecutor.execute(new Runnable() {
			@Override
			public void run() {
				try {
					initOrgTask(orgId);
                    // 初始化任务状态
                    setTaskStatusProcessing(taskId);
                    int n = 2;
			        final List<List<ImportStudent>> partImportStudents = Lists.newArrayList();
			        for(int i = 0; i< n;i++) {
			        	partImportStudents.add(new ArrayList<ImportStudent>());
			        }
			        for(int i = 0;i < importStudents.size();i++){
			            int mod = i % n;
			            partImportStudents.get(mod).add(importStudents.get(i));
			        }
			        Map<Integer, Boolean> map = Maps.newHashMap();
			        for(int i = 0; i < partImportStudents.size(); i++) {
			        	map.put(i, false);
			        }
			        Map<Integer, Future> futureMap = Maps.newHashMap();
			        for(int i = 0; i < partImportStudents.size(); i++) {
			        	List<ImportStudent> importStudentList = partImportStudents.get(i);
			        	SaveStudentsThread saveStudentsThread = new SaveStudentsThread(); 
			        	saveStudentsThread.setOrgId(orgId);
			        	saveStudentsThread.setCascadeId(cascadeId);
			        	saveStudentsThread.setTaskId(taskId);
			        	saveStudentsThread.setImportStudents(importStudentList);
			        	saveStudentsThread.setRedisTemplate(redisTemplate);
			        	saveStudentsThread.setOrgStudentService(orgStudentService);
			        	saveStudentsThread.setOrgStudentCourseService(orgStudentCourseService);
			        	saveStudentsThread.setOrgStudentLessonDao(orgStudentLessonDao);
			        	saveStudentsThread.setOrgCourseService(orgCourseService);
			        	saveStudentsThread.setOrgClassLessonDao(orgClassLessonDao);
			        	saveStudentsThread.setId(i);
			        	saveStudentsThread.setMap(map);
			        	Future future =threadPoolTaskExecutor.submit(saveStudentsThread);
			        	futureMap.put(i, future);
			        }
			        for(Future future:futureMap.values()){
			        	try {
							future.get();
						} catch (InterruptedException e) {
							log.error("save Students thread end interrupted error",e);
						} catch (ExecutionException e) {
							log.error("save Students thread end execution error", e);
						}
			        }
			        
			        for(Integer key: map.keySet()) {
			        	if(map.get(key) == false) {
			        		List<ImportStudent> importStudentList = partImportStudents.get(key);
			        		for(ImportStudent importStudent: importStudentList) {
			        			appendErrorData(taskId, importStudent);
			        		}
			        	}
			        }
			        log.info("main thread excute save student success……taskId:{}",taskId);
			        setTaskStatusProcessed(taskId, null, null);   
				} catch (Throwable t) {
	                log.error("save student to orgcourse error!", t);
	                setTaskStatusError(taskId);
	            } finally {
	                clearOrgTask(orgId);
	            }
			}
		});
    }

    @Override
    public List<ImportStudent> getValidStudentList(final String taskId) {
        return redisTemplate.execute(new RedisCallback<List<ImportStudent>>() {

            @Override
            public List<ImportStudent> doInRedis(RedisConnection connection) throws DataAccessException {
                List<ImportStudent> data = Lists.newArrayList();
                Long size = redisTemplate.opsForList().size(getValidDataKey(taskId));
                if (size != null && size > 0) {
                    for (int i = 0; i < size; i++) {
                        try {
                            data.add(JacksonUtil.str2Obj(redisTemplate.opsForList().rightPop(getValidDataKey(taskId)),
                                ImportStudent.class));
                        } catch (Exception e) {
                            e.printStackTrace();
                            log.error("getValidStudentList getValidList error!", e);
                        }
                    }
                }
                log.debug("getValidList :{}", data);
                if (CollectionUtils.isNotEmpty(data)) {
                    for (ImportStudent importStudent : data) {
                        redisTemplate.opsForList().leftPush(getValidDataKey(taskId),
                            JacksonUtil.obj2Str(importStudent));
                    }
                }
                connection.expire(getValidDataKey(taskId).getBytes(), EXPIRE);
                return data;

            }
        });
    }

    @Override
    public List<ImportStudent> getInValidStudentList(final String taskId) {
        return redisTemplate.execute(new RedisCallback<List<ImportStudent>>() {

            @Override
            public List<ImportStudent> doInRedis(RedisConnection connection) throws DataAccessException {
                List<ImportStudent> data = Lists.newArrayList();
                Long size = redisTemplate.opsForList().size(getErrDataKey(taskId));
                if (size != null && size > 0) {
                    for (int i = 0; i < size; i++) {
                        try {
                            data.add(JacksonUtil.str2Obj(redisTemplate.opsForList().rightPop(getErrDataKey(taskId)),
                                ImportStudent.class));
                        } catch (Exception e) {
                            e.printStackTrace();
                            log.error("getInValidStudentList getValidList error!", e);
                        }
                    }
                }
                log.debug("getInValidStudentList:{}", data);
                if (CollectionUtils.isNotEmpty(data)) {
                    for (ImportStudent importStudent : data) {
                        redisTemplate.opsForList().leftPush(getErrDataKey(taskId), JacksonUtil.obj2Str(importStudent));
                    }
                }
                connection.expire(getValidDataKey(taskId).getBytes(), EXPIRE);
                return data;

            }
        });
    }

    private boolean isTaskProcessing(final Integer orgId) {
        return redisTemplate.execute(new RedisCallback<Boolean>() {

            @Override
            public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                return connection.exists(getTaskKey(orgId).getBytes());
            }
        });

    }

    @Override
    public String upload1V1Course(MultipartFile file) throws Exception {

        final Integer orgId = TianxiaoPCContext.getOrgId();
        boolean isProcessing = isTaskProcessing(orgId);

        if (isProcessing) {
            throw new BussinessException(CommonErrorCode.PERMISSION_DENY, "上一个任务仍在处理中，请等待完成后，再次上传");
        }
        final String taskId = UUID.randomUUID().toString();
        final UploadFileReaderService uploadFileService = ErpUploadFileUtil.getUploadFileService(file);
        List<String> templateHeader = ErpUploadFileUtil.getTemplateHeader(ErpBizConf.TEMPLATE_1V1_COURSE);
        final List<String> headers = ErpUploadFileUtil.tranHeader(uploadFileService.readData());
        if (!headers.contains("错误原因")) {
            headers.add("错误原因");
        }
        // 和系统提供header一致
        if (!CollectionUtils.isEqualCollection(templateHeader, headers)) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "请使用系统模板");
        }
        // 校验过程丢线程池
        threadPoolTaskExecutor.execute(new Runnable() {

            @Override
            public void run() {
                try {
                    // 初始化任务状态
                    setTaskStatusProcessing(taskId);
                    // 一行一行校验数据，正确、错误都存
                    final List<ImportCourse> errList = Lists.newArrayList();
                    final List<ImportCourse> validList = Lists.newArrayList();
                    Object[] lineData = null;
                    headers.remove("错误原因");
                    try {
                        while ((lineData = uploadFileService.readData()) != null) {
                            boolean invalid = false;
                            ImportCourse importCourse = new ImportCourse();
                            for (int i = 0; i < headers.size(); i++) {
                                Object value = lineData[i];
                                CourseFormat courseFormat = ErpUploadFileUtil._1V1CourseMap.get(headers.get(i));
                                if (value == null || StringUtils.isBlank(value.toString().trim())) {
                                    invalid = true;
                                    BeanUtils.setProperty(importCourse, "errReason", "【" + headers.get(i) + "】不能为空");
                                } else {
                                    BeanUtils.setProperty(importCourse, courseFormat.getFieldName(),
                                        value.toString().trim());
                                    // 取值不合法
                                    if (!Pattern.matches(courseFormat.getFormatRegex(), value.toString())) {
                                        BeanUtils.setProperty(importCourse, "errReason", courseFormat.getFormatMsg());
                                        invalid = true;
                                    }
                                }
                            }
                            if (invalid) {
                                errList.add(importCourse);
                            } else {
                                validList.add(importCourse);
                            }
                        }
                    } catch (Exception e) {
                        log.error("valid upload 1V1 course error~!", e);
                        setTaskStatusError(taskId);
                        return;
                    }

                    // 把正确的和错误的数据，存入缓存
                    cacheCourseData(taskId, validList, errList);

                    // 设置任务完成
                    setTaskStatusProcessed(taskId, validList.size(), errList.size());
                } catch (Throwable t) {
                    log.error("upload 1v1 course error!", t);
                    setTaskStatusError(taskId);
                }
            }
        });

        return taskId;
    }

    @Override
    public List<ImportCourse> getValid1V1CourseList(final String taskId) {
        return redisTemplate.execute(new RedisCallback<List<ImportCourse>>() {

            @Override
            public List<ImportCourse> doInRedis(RedisConnection connection) throws DataAccessException {
                List<ImportCourse> data = Lists.newArrayList();
                Long size = redisTemplate.opsForList().size(getValidDataKey(taskId));
                if (size != null && size > 0) {
                    for (int i = 0; i < size; i++) {
                        try {
                            data.add(JacksonUtil.str2Obj(redisTemplate.opsForList().rightPop(getValidDataKey(taskId)),
                                ImportCourse.class));
                        } catch (Exception e) {
                            log.error("uploadcourse getValidList error!", e);
                        }
                    }
                }
                if (CollectionUtils.isNotEmpty(data)) {
                    for (ImportCourse importCourse : data) {
                        redisTemplate.opsForList().leftPush(getValidDataKey(taskId), JacksonUtil.obj2Str(importCourse));
                    }
                }
                connection.expire(getValidDataKey(taskId).getBytes(), EXPIRE);
                return data;

            }
        });
    }

    @Override
    public List<ImportCourse> getInValid1V1CourseList(final String taskId) {
        return redisTemplate.execute(new RedisCallback<List<ImportCourse>>() {

            @Override
            public List<ImportCourse> doInRedis(RedisConnection connection) throws DataAccessException {
                List<ImportCourse> data = Lists.newArrayList();
                Long size = redisTemplate.opsForList().size(getErrDataKey(taskId));
                if (size != null && size > 0) {
                    for (int i = 0; i < size; i++) {
                        try {
                            data.add(JacksonUtil.str2Obj(redisTemplate.opsForList().rightPop(getErrDataKey(taskId)),
                                ImportCourse.class));
                        } catch (Exception e) {
                            log.error("uploadcourse getInValidList error!", e);
                        }
                    }
                }
                if (CollectionUtils.isNotEmpty(data)) {
                    for (ImportCourse importCourse : data) {
                        redisTemplate.opsForList().leftPush(getErrDataKey(taskId), JacksonUtil.obj2Str(importCourse));
                    }
                }
                connection.expire(getValidDataKey(taskId).getBytes(), EXPIRE);
                return data;

            }
        });
    }

    @Override
    public void save1V1Course(final String taskId) {
        final List<ImportCourse> importCourses = getValid1V1CourseList(taskId);
        final Integer orgId = TianxiaoPCContext.getOrgId();
        if (orgId == null) {
            log.error("save1V1Course error!orgId is null");
            return;
        }
        if (CollectionUtils.isEmpty(importCourses)) {
            setTaskStatusProcessed(taskId, 0, null);
            return;
        }
        boolean isProcessing = isTaskProcessing(orgId);

        if (isProcessing) {
            throw new BussinessException(CommonErrorCode.PERMISSION_DENY, "上一个任务仍在处理中，请等待完成后，再次上传");
        }
        threadPoolTaskExecutor.execute(new Runnable() {

            @Override
            public void run() {
                try {
                    initOrgTask(orgId);
                    setTaskStatusProcessing(taskId);
                    if (CollectionUtils.isNotEmpty(importCourses)) {
                        for (final ImportCourse importCourse : importCourses) {
                            final OrgCourseInfoDto course = build1V1Course(importCourse);
                            try {
                                if (course == null) {
                                    throw new Exception("非法值");
                                }
                                orgCourseService.saveOrUpdateCourse(orgId.longValue(), course, false, true);
                            } catch (Exception e) {
                                log.error("save importCourse error!,importCourse:{}", importCourse, e);
                                // 错误追加
                                importCourse.setErrReason("系统错误，无法插入，请联系天校工作人员为您排查");
                                appendErrorData(taskId, importCourse);
                            }
                        }
                    }
                    setTaskStatusProcessed(taskId, null, null);
                } catch (Throwable t) {
                    log.error("save couers error!", t);
                    setTaskStatusError(taskId);
                } finally {
                    clearOrgTask(orgId);
                }

            }
        });
    }

    @Override
    public String upload1V1Student(MultipartFile file) throws Exception {
        final Integer orgId = TianxiaoPCContext.getOrgId();
        boolean isProcessing = isTaskProcessing(orgId);

        if (isProcessing) {
            throw new BussinessException(CommonErrorCode.PERMISSION_DENY, "上一个任务仍在处理中，请等待完成后，再次上传");
        }
        final String taskId = UUID.randomUUID().toString();
        final UploadFileReaderService uploadFileService = ErpUploadFileUtil.getUploadFileService(file);
        List<String> templateHeader = ErpUploadFileUtil.getTemplateHeader(ErpBizConf.TEMPLATE_1V1_STUDENT);
        final List<String> headers = ErpUploadFileUtil.tranHeader(uploadFileService.readData());
        if (!headers.contains("错误原因")) {
            headers.add("错误原因");
        }
        // 和系统提供header一致
        if (!CollectionUtils.isEqualCollection(templateHeader, headers)) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "请使用系统模板");
        }
        // 校验过程丢线程池
        threadPoolTaskExecutor.execute(new Runnable() {

            @Override
            public void run() {
                try {
                    // 初始化任务状态
                    setTaskStatusProcessing(taskId);
                    // 一行一行校验数据，正确、错误都存
                    final List<ImportStudent> errList = Lists.newArrayList();
                    final List<ImportStudent> validList = Lists.newArrayList();
                    final List<String> validMobiles = Lists.newArrayList();
                    final List<String> validCourseNames = Lists.newArrayList();
                    final Map<String, ImportStudent> mobileMap = Maps.newHashMap();
                    Object[] lineData = null;
                    headers.remove("错误原因");
                    try {
                        while ((lineData = uploadFileService.readData()) != null) {
                            boolean invalid = false;
                            ImportStudent importStudent = new ImportStudent();
                            for (int i = 0; i < headers.size(); i++) {
                                Object value = lineData[i];
                                StudentFormat studentFormat = ErpUploadFileUtil._1V1StudentMap.get(headers.get(i));
                                if (value == null || StringUtils.isBlank(value.toString().trim())) {
                                    invalid = true;
                                    BeanUtils.setProperty(importStudent, "errReason", "【" + headers.get(i) + "】不能为空");
                                } else {
                                    BeanUtils.setProperty(importStudent, studentFormat.getFieldName(),
                                        value.toString().trim());
                                    // 取值不合法
                                    if (!Pattern.matches(studentFormat.getFormatRegex(), value.toString().trim())) {
                                        BeanUtils.setProperty(importStudent, "errReason", studentFormat.getFormatMsg());
                                        invalid = true;
                                    }
                                }
                                if (!invalid) {
                                    if (importStudent.getStudentMobile() != null
                                        && mobileMap.containsKey(importStudent.getStudentMobile())) {
                                        ImportStudent exist =
                                            mobileMap.get(importStudent.getStudentMobile().toString());
                                        if (!exist.getStudentName().toString()
                                            .equals(importStudent.getStudentName().toString())) {
                                            if(orgId != 18925 && orgId!=24909){
                                                invalid = true;
                                                importStudent.setErrReason("与表格内其他行数据【手机号】相同，但姓名不同，无法导入");
                                            }
                                        } else {
                                            if (exist.getCourseName() != null && importStudent.getCourseName() != null
                                                && exist.getCourseName().toString()
                                                    .equals(importStudent.getCourseName().toString())) {
                                                importStudent.setErrReason("该行与表格内其他行数据重复，无法导入");
                                                invalid = true;
                                            }
                                        }
                                    }
                                }

                            }

                            if (invalid) {
                                errList.add(importStudent);
                            } else {
                                validList.add(importStudent);
                                mobileMap.put(importStudent.getStudentMobile().toString(), importStudent);
                                validMobiles.add(importStudent.getStudentMobile().toString());
                            }
                        }
                        List<StudentInfoDto> tempDtos = null;
                        if (CollectionUtils.isNotEmpty(validList)) {
                            tempDtos =
                                orgStudentService.listOrgStudentByMobiles(orgId.longValue(), validMobiles);
                            ArrayListMultimap<String, String> mobileNameMap = ArrayListMultimap.create();
                            if (CollectionUtils.isNotEmpty(tempDtos)) {
                                for (StudentInfoDto dto : tempDtos) {
                                    mobileNameMap.put(dto.getMobile(), dto.getName());
                                }
                                Iterator<ImportStudent> iterator = validList.iterator();
                                while (iterator.hasNext()) {
                                    ImportStudent importStudent = iterator.next();
                                    if(orgId == 18925 || orgId == 24909){
                                        validCourseNames.add(importStudent.getCourseName());
                                    }else{
                                    if (mobileNameMap.containsKey(importStudent.getStudentMobile())) {
                                        List<String> names = mobileNameMap.get(importStudent.getStudentMobile());
                                        if(CollectionUtils.isNotEmpty(names) && !names.contains(importStudent.getStudentName())){
                                            importStudent.setErrReason("与系统内学员【手机号】相同，但姓名不同，无法导入");
                                            errList.add(importStudent);
                                            iterator.remove();
                                        } else {
                                            validCourseNames.add(importStudent.getCourseName());
                                        }
                                    } else {
                                        validCourseNames.add(importStudent.getCourseName());
                                    }
                                    }
                                }
                            }
                        }
                        List<OrgCourse> orgCourses = null;
                        if (CollectionUtils.isNotEmpty(validList)) {
                            Map<String, Long> courseNameIdMap = Maps.newHashMap();
                            Map<String, Integer> courseNameCountMap = Maps.newHashMap();
                            List<Long> courseIds = Lists.newArrayList();
                            orgCourses =
                                orgCourseService.getAll1V1CoursesByNames(orgId.longValue(), validCourseNames);
                            if (CollectionUtils.isNotEmpty(orgCourses)) {
                                for (OrgCourse orgCourse : orgCourses) {
                                    courseNameIdMap.put(orgCourse.getName(), orgCourse.getId());
                                    if (courseNameCountMap.containsKey(orgCourse.getName())) {
                                        Integer count = courseNameCountMap.get(orgCourse.getName());
                                        courseNameCountMap.put(orgCourse.getName(), ++count);
                                    } else {
                                        courseNameCountMap.put(orgCourse.getName(), 1);
                                    }
                                    courseIds.add(orgCourse.getId());
                                }
                            }
                            Iterator<ImportStudent> iterator = validList.iterator();
                            while (iterator.hasNext()) {
                                ImportStudent importStudent = iterator.next();
                                if (!courseNameIdMap.containsKey(importStudent.getCourseName())) {
                                    importStudent.setErrReason("【1对1课程名称】在系统不存在，请先手动录入课程，再批导课程学员");
                                    errList.add(importStudent);
                                    iterator.remove();
                                } else {
                                    if (courseNameCountMap.get(importStudent.getCourseName()) > 1) {
                                        importStudent.setErrReason("【1对1课程名称】在系统有多个重名课程，重名课程的学员请手动录入");
                                        errList.add(importStudent);
                                        iterator.remove();
                                    }
                                }
                            }
                        }
                        // 系统存在的学员，不允许批导
                        if(CollectionUtils.isNotEmpty(validList)){
                            // 先取所有课程id，用于查询<课程id,List<班级id>>
                            // 走到这里orgcourse一定不为空
                            List<Long> courseids = Lists.newArrayList();
                            Map<String, Long> courseNameIdMap = Maps.newHashMap();
                            for(OrgCourse orgCourse : orgCourses){
                                courseids.add(orgCourse.getId());
                                courseNameIdMap.put(orgCourse.getName(), orgCourse.getId());
                            }
                            // 查询所有<课程id,List<班级id>>，用户查询 <班级id,List<学员id>>
                            ArrayListMultimap<Long, Long> courseClassIdMap = orgCourseService.getCourseClassIdMap(orgId.longValue(), courseids);
                            List<Long> classIds = new ArrayList<Long>(courseClassIdMap.values());
                            // 查询<班级id,List<学员id>>
                            ArrayListMultimap<Long, Long> classStudentIdMap = orgStudentCourseService.getClassStudentIdMap(orgId.longValue(), classIds);
                            // <学员name，学员id>，学员name+手机号才是唯一key
                            // <学员手机号,学员id>
                            Map<String, Long> studentNameIdMap = Maps.newHashMap();
                            for(StudentInfoDto studentInfoDto : tempDtos){
                                studentNameIdMap.put(studentInfoDto.getName()+"_"+studentInfoDto.getMobile(), studentInfoDto.getStudentId());
                            }
                            // 课程id和学员id对应
                            ArrayListMultimap<Long, Long> courseStudentIdMap = ArrayListMultimap.create();
                            for(Long courseId : courseClassIdMap.keySet()){
                                List<Long> tempClassIds = courseClassIdMap.get(courseId);
                                if(CollectionUtils.isNotEmpty(tempClassIds)){
                                    for(Long tempClassId : tempClassIds){
                                        List<Long> studentIds = classStudentIdMap.get(tempClassId);
                                        if(CollectionUtils.isNotEmpty(studentIds)){
                                            courseStudentIdMap.putAll(courseId, studentIds);
                                        }
                                    }
                                }
                            }
                            
                            Iterator<ImportStudent> iterator = validList.iterator();
                            while (iterator.hasNext()) {
                                ImportStudent importStudent = iterator.next();
                                Long studentId = studentNameIdMap.get(importStudent.getStudentName()+"_"+importStudent.getStudentMobile());
                                // 已有学员校验
                                if(studentId != null){
                                    Long courseId = courseNameIdMap.get(importStudent.getCourseName());
                                    if(courseId != null){
                                        List<Long> tempStudentIds = courseStudentIdMap.get(courseId);
                                        if(CollectionUtils.isNotEmpty(tempStudentIds)){
                                            if(tempStudentIds.contains(studentId)){
                                                importStudent.setErrReason("系统中该课程已有该学员，无法导入");
                                                errList.add(importStudent);
                                                iterator.remove();
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    } catch (Exception e) {
                        log.error("valid upload student error~!", e);
                        setTaskStatusError(taskId);
                        return;
                    }
                    log.debug("*********************validstudentlist:{}", validList);
                    log.debug("*********************invalidstudentlist:{}", errList);
                    // 把正确的和错误的数据，存入缓存
                    cacheStudentData(taskId, validList, errList);

                    // 设置任务完成
                    setTaskStatusProcessed(taskId, validList.size(), errList.size());
                } catch (Throwable t) {
                    log.error("upload student error!", t);
                    setTaskStatusError(taskId);
                }
            }
        });

        return taskId;
    }

    @Override
    public List<ImportStudent> getValid1V1StudentList(final String taskId) {
        return redisTemplate.execute(new RedisCallback<List<ImportStudent>>() {

            @Override
            public List<ImportStudent> doInRedis(RedisConnection connection) throws DataAccessException {
                List<ImportStudent> data = Lists.newArrayList();
                Long size = redisTemplate.opsForList().size(getValidDataKey(taskId));
                if (size != null && size > 0) {
                    for (int i = 0; i < size; i++) {
                        try {
                            data.add(JacksonUtil.str2Obj(redisTemplate.opsForList().rightPop(getValidDataKey(taskId)),
                                ImportStudent.class));
                        } catch (Exception e) {
                            log.error("getValidStudentList getValidList error!", e);
                        }
                    }
                }
                log.debug("getValidList :{}", data);
                if (CollectionUtils.isNotEmpty(data)) {
                    for (ImportStudent importStudent : data) {
                        redisTemplate.opsForList().leftPush(getValidDataKey(taskId),
                            JacksonUtil.obj2Str(importStudent));
                    }
                }
                connection.expire(getValidDataKey(taskId).getBytes(), EXPIRE);
                return data;
            }
        });
    }

    @Override
    public List<ImportStudent> getInValid1V1StudentList(final String taskId) {
        return redisTemplate.execute(new RedisCallback<List<ImportStudent>>() {

            @Override
            public List<ImportStudent> doInRedis(RedisConnection connection) throws DataAccessException {
                List<ImportStudent> data = Lists.newArrayList();
                Long size = redisTemplate.opsForList().size(getErrDataKey(taskId));
                if (size != null && size > 0) {
                    for (int i = 0; i < size; i++) {
                        try {
                            data.add(JacksonUtil.str2Obj(redisTemplate.opsForList().rightPop(getErrDataKey(taskId)),
                                ImportStudent.class));
                        } catch (Exception e) {
                            log.error("getInValidStudentList getValidList error!", e);
                        }
                    }
                }
                log.debug("getInValidStudentList:{}", data);
                if (CollectionUtils.isNotEmpty(data)) {
                    for (ImportStudent importStudent : data) {
                        redisTemplate.opsForList().leftPush(getErrDataKey(taskId), JacksonUtil.obj2Str(importStudent));
                    }
                }
                connection.expire(getValidDataKey(taskId).getBytes(), EXPIRE);
                return data;
            }
        });
    }

    @Override
    public void save1V1Student(final String taskId) {
        final List<ImportStudent> importStudents = getValid1V1StudentList(taskId);
        final Integer orgId = TianxiaoPCContext.getOrgId();
        final Integer cascadeId = TianxiaoPCContext.getTXCascadeId();
        if (CollectionUtils.isEmpty(importStudents)) {
            setTaskStatusProcessed(taskId, 0, null);
            return;
        }
        if (orgId == null) {
            log.warn("saveStudents error!orgId is null");
            return;
        }
        boolean isProcessing = isTaskProcessing(orgId);

        if (isProcessing) {
            throw new BussinessException(CommonErrorCode.PERMISSION_DENY, "上一个任务仍在处理中，请等待完成后，再次上传");
        }
        threadPoolTaskExecutor.execute(new Runnable() {

            @Override
            public void run() {
                try {

                    initOrgTask(orgId);
                    setTaskStatusProcessing(taskId);

                    List<String> courseNames = Lists.newArrayList();
                    List<String> studentMobiles = Lists.newArrayList();
                    for (ImportStudent importStudent : importStudents) {
                        courseNames.add(importStudent.getCourseName());
                        studentMobiles.add(importStudent.getStudentMobile());
                    }
                    Map<String, OrgCourse> orgCourseMap =
                        orgCourseService.getAllOrgCoursesMapByNames(orgId.longValue(), courseNames);
                    ArrayListMultimap<String, StudentInfoDto> studentMap =
                        orgStudentService.getStudentMap(orgId.longValue(), studentMobiles);
                    if (CollectionUtils.isNotEmpty(importStudents)) {
                        for (final ImportStudent importStudent : importStudents) {
                            try {
                                OrgCourse orgCourse = orgCourseMap.get(importStudent.getCourseName());

                                // 不存在orgstudent，创建
                                // 不存在orgstudentclass，创建
                                // 系统异常时，把当前数据从validlist删除，再插入errlist，修改错误和正确数目
                                StudentInfoDto orgStudent = null;
                                List<StudentInfoDto> dtos = studentMap.get(importStudent.getStudentMobile());
                                if (CollectionUtils.isNotEmpty(dtos)) {
                                    for (StudentInfoDto dto : dtos) {
                                        if (importStudent.getStudentMobile().equals(dto.getMobile())
                                            && importStudent.getStudentName().equals(dto.getName())) {
                                            log.debug("*************can find dto:{}", dto);
                                            orgStudent = dto;
                                        }
                                    }
                                }
                                if (orgStudent == null) {
                                    orgStudent = new StudentInfoDto();
                                    orgStudent.setMobile(importStudent.getStudentMobile().toString());
                                    orgStudent.setName(importStudent.getStudentName().toString());
                                    orgStudent.setAddCascadeId(cascadeId == null ? 0 : cascadeId);
                                    OrgStudentAddresponseDto addDto = orgStudentService
                                        .addStudentIgnoreExistMobile(orgStudent, null, null, orgId.longValue());
                                    orgStudent.setStudentId(addDto.getUserId());
                                }
                                log.debug("**************orgStudent:{}", orgStudent);
                                // 创建1V1班,学员进班
                                Integer lessonCount = importStudent.getLessonCount() > ErpBizConf.MAX_1V1_LESSONCOUNT
                                    ? ErpBizConf.MAX_1V1_LESSONCOUNT : importStudent.getLessonCount();
                                courseClassService.createClassFor1V1(orgId.longValue(), null,orgCourse.getId(),
                                    orgStudent.getStudentId(), lessonCount);

                            } catch (final BussinessException be) {
                                log.warn("save importStudent error!,importStudent:{}", importStudent, be);
                                importStudent.setErrReason(be.getMessage());
                                appendErrorData(taskId, importStudent);
                            } catch (Exception e) {
                                log.error("save importStudent error!,importStudent:{}", importStudent, e);
                                // 错误追加
                                importStudent.setErrReason("系统无法插入此数据，请联系天校工作人员为您排查");
                                appendErrorData(taskId, importStudent);
                            }
                        }
                    }

                    setTaskStatusProcessed(taskId, null, null);
                } catch (Throwable t) {
                    log.error("save 1v1 student error!", t);
                    setTaskStatusError(taskId);
                } finally {
                    clearOrgTask(orgId);
                }
            }
        });
    }

}
