
 /**
 * Baijiahulian.com Inc.
 * Copyright (c) 2014-2016 All Rights Reserved.
 */

package com.baijia.tianxiao.biz.erp.service.impl;

import com.baijia.tianxiao.biz.erp.dto.OrgCourseInfoPcDto;
import com.baijia.tianxiao.biz.erp.dto.response.schedule.CommonInfoTipsDto;
import com.baijia.tianxiao.biz.erp.dto.response.schedule.CourseBaseInfoDto;
import com.baijia.tianxiao.biz.erp.dto.response.schedule.CourseNameAndIdDto;
import com.baijia.tianxiao.biz.erp.dto.response.schedule.VipClassDetailDto;
import com.baijia.tianxiao.biz.erp.service.ErpCourseService;
import com.baijia.tianxiao.biz.erp.service.ErpStudentService;
import com.baijia.tianxiao.constant.AvatarConstants;
import com.baijia.tianxiao.dal.enums.CourseTypeEnum;
import com.baijia.tianxiao.dal.org.constant.DeleteStatus;
import com.baijia.tianxiao.dal.org.dao.*;
import com.baijia.tianxiao.dal.org.po.*;
import com.baijia.tianxiao.dto.IdNameDto;
import com.baijia.tianxiao.dto.PersonBaseDto;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.filter.TianxiaoMContext;
import com.baijia.tianxiao.filter.TianxiaoPCContext;
import com.baijia.tianxiao.sal.course.dto.OrgCourseInfoDto;
import com.baijia.tianxiao.sal.course.dto.response.TeacherResponseDto;
import com.baijia.tianxiao.sal.course.enums.CourseConsumeRuleEnum;
import com.baijia.tianxiao.sal.course.service.CourseStudentService;
import com.baijia.tianxiao.sal.course.service.CourseTeacherService;
import com.baijia.tianxiao.sal.course.service.OrgCourseService;
import com.baijia.tianxiao.sal.organization.constant.CascadeType;
import com.baijia.tianxiao.sal.organization.constant.DeviceType;
import com.baijia.tianxiao.sal.organization.constant.TXPermissionConst;
import com.baijia.tianxiao.sal.organization.org.service.TxAccountPermissionService;
import com.baijia.tianxiao.sal.room.dto.ClassRoomDto;
import com.baijia.tianxiao.sal.room.service.ClassRoomService;
import com.baijia.tianxiao.sal.student.enums.StudentErrorCode;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.BaseUtils;
import com.baijia.tianxiao.util.json.JacksonUtil;
import com.beust.jcommander.internal.Lists;
import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.Transformer;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.lang.reflect.InvocationTargetException;
import java.util.List;
import java.util.Set;

/**
 * @title ErpCourseServiceImpl
 * @desc TODO
 * @author zhangbing
 * @date 2016年8月25日
 * @version 1.0
 */
@Service
@Slf4j
public class ErpCourseServiceImpl implements ErpCourseService {

    // 默认班主任为：未设置
    private static final String DEFAULT_CLASS_MASTER = "未设置";

    @Resource
    private OrgCourseDao orgCourseDao;
    @Resource
    private CourseTeacherService courseTeacherService;
    @Resource
    private TXCascadeAccountDao cascadeAccountDao;
    @Resource
    private ClassRoomService classRoomService;
    @Resource
    private TXCascadeCredentialDao credentialDao;
    @Resource
    private OrgCourseRoomDao orgCourseRoomDao;
    @Resource
    private OrgCourseTeacherDao orgCourseTeacherDao;
    @Resource
    private OrgStudentCourseDao orgStudentCourseDao;
    @Resource
    private OrgClassLessonDao orgClassLessonDao;
    @Resource
    private OrgCourseService orgCourseService;
    @Resource
    private ErpStudentService erpStudentService;
    @Resource
    private OrgCourseConsumeRuleDao orgCourseConsumeRuleDao;
    @Resource
    private OrgAccountDao orgAccountDao;
    @Resource
    private CourseStudentService courseStudentService;
    @Resource
    private OrgInfoDao orgInfoDao;
    @Resource
    private TxAccountPermissionService permissionService;
    @Resource
    private OrgLessonSignDao orgLessonSignDao;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public CourseBaseInfoDto getCourseDetailById(Long orgId, Long courseId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "illegal orgId");
        Preconditions.checkArgument(courseId != null && courseId > 0, "illegal courseId");
        CourseBaseInfoDto orgCourseInfoDto = new CourseBaseInfoDto();

        OrgCourse course = null;
        if (courseId != null && courseId > 0) {
            course = this.orgCourseDao.getById(courseId);
            if (course != null) {
                orgCourseInfoDto.setCourseNumber(course.getNumber());
                orgCourseInfoDto.setCourseId(courseId);
                orgCourseInfoDto.setCourseName(course.getName());
                orgCourseInfoDto.setCoursePrice(course.getPrice());
                orgCourseInfoDto.setAddress(course.getAddress());
                orgCourseInfoDto.setStartTime(course.getBeginTime());
                orgCourseInfoDto.setEndTime(course.getEndTime());

                // 总课次
                orgCourseInfoDto.setTotalLessons(course.getFreq()==null?0:course.getFreq());
                orgCourseInfoDto.setTeacherInfo(this.getCourseTeacherTips(orgId, courseId));
                orgCourseInfoDto.setClassMaster(getAccountName(orgId, course.getCascadeId()));
                orgCourseInfoDto.setRoomInfo(this.getCourseRoomTips(orgId, courseId));
                orgCourseInfoDto.setFinishedLessons(this.getFinishedLessonCount(orgId, courseId));
                orgCourseInfoDto.setStudentCount(this.getCourseStudentCount(orgId, courseId));

                //获取课消规则信息
                String consumRuleName = "未设置课消规则";
                OrgCourseConsumeRule courseConsumRule = orgCourseConsumeRuleDao.getRuleByCourseId(orgId, courseId);
                if (courseConsumRule != null) {
                    orgCourseInfoDto.setConsumRuleId(courseConsumRule.getId());
                    int consumVal = courseConsumRule.getRuleValue();
                    consumRuleName = CourseConsumeRuleEnum.getRuleDescByValue(consumVal);
                }
                orgCourseInfoDto.setConsumeRuleName(consumRuleName);
            }
        }

        log.info("ErpCourseServiceImpl.getCourseDetailById.orgId:{}, courseId:{}, data:{}", orgId, courseId, orgCourseInfoDto);
        return orgCourseInfoDto;
    }

    @Override
	public VipClassDetailDto getVipClassDetailById(Long orgId, Long courseId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "illegal orgId");
        Preconditions.checkArgument(courseId != null && courseId > 0, "illegal courseId");
        VipClassDetailDto vipClassDetailDto = new VipClassDetailDto();
        OrgCourse classInfo = orgCourseDao.getById(courseId);//班级信息
        if(classInfo == null || classInfo.getIsClass() != CourseTypeEnum.IS_CLASS_TRUE.getCode() || classInfo.getIsDel() == DeleteStatus.DELETED.getValue()) {
        	throw new BussinessException(CommonErrorCode.NOT_FOUND, "班级不存在或已被删除");
        }
        OrgCourse orgCourseInfo = orgCourseDao.getById(classInfo.getParentId());//课程信息
//        if(orgCourseInfo == null || orgCourseInfo.getIsCourse() != CourseTypeEnum.IS_COURSE_TRUE.getCode() || orgCourseInfo.getIsDel() == DeleteStatus.DELETED.getValue()) {
//        	throw new BussinessException(CommonErrorCode.NOT_FOUND, "班级所属的课程不存在或已被删除");
//        }
        vipClassDetailDto.setCourseId(classInfo.getId());//班级id
        vipClassDetailDto.setClassName(classInfo.getName());//班级名称
        vipClassDetailDto.setTeacherInfo(getCourseTeacherTips(orgId, classInfo.getId()));//老师信息
        vipClassDetailDto.setRoomInfo(getCourseRoomTips(orgId, classInfo.getId()));//教室信息
        vipClassDetailDto.setClassMaster(getAccountName(orgId, classInfo.getCascadeId()));//班主任信息
        vipClassDetailDto.setFinishedLessons(this.getFinishedVipLessonCount(orgId, courseId, orgCourseInfo.getId()));//已上课次
        vipClassDetailDto.setTotalLessons(classInfo.getFreq()==null ? 0 : classInfo.getFreq());//总课节数（课次）
        vipClassDetailDto.setCoursePrice(classInfo.getPrice()==null ? 0 : classInfo.getPrice());//课程单价
        vipClassDetailDto.setCreateTime(classInfo.getCreateTime());//创建时间
        vipClassDetailDto.setStudentCount(getCourseStudentCount(orgId, courseId));//班级人数
        vipClassDetailDto.setStudentStatus(vipClassDetailDto.getStudentCount() == 0 ? 1 : 0);//班级状态   班级人数为0表示已退班
//        Double totalPrice = orgStudentCourseDao.getVipClassTotalPrice(orgId, courseId);
//        vipClassDetailDto.setTotalPrice(totalPrice == null ? 0 : totalPrice);//课程总价
        vipClassDetailDto.setTotalPrice(0.0);//课程总价
        vipClassDetailDto.setAddress(classInfo.getAddress());
        //课消规则  备注：不按班级设置课消，按课程设置课消
        OrgCourseConsumeRule courseConsumRule = orgCourseConsumeRuleDao.getRuleByCourseId(orgId, orgCourseInfo.getId());
        if (courseConsumRule != null) {
            int consumeVal = courseConsumRule.getRuleValue();
        	vipClassDetailDto.setConsumeRuleId(consumeVal);//课消规则编码
            vipClassDetailDto.setConsumeRuleName(CourseConsumeRuleEnum.getRuleDescByValue(consumeVal));//课消规则描述语
        } else {
        	throw new BussinessException(CommonErrorCode.NOT_FOUND, "课消规则未设置");
        }
		return vipClassDetailDto;
	}

    /**
     *
     * @param orgId ： 机构ID
     * @param courseId ： 班级id
     * @param parentCourseId ： 班级对应的课程的课程id
     * @return
     */
    private Integer getFinishedVipLessonCount(Long orgId, Long courseId, Long parentCourseId) {
    	OrgCourseConsumeRule orgCourseConsumeRule = orgCourseConsumeRuleDao.getRuleByCourseId(orgId, parentCourseId);
    	int ruleValue = orgCourseConsumeRule.getRuleValue();
    	int count = orgLessonSignDao.getVipLessonSignCount(orgId, courseId, ruleValue);
		return count;
	}

	/**
     * 根据cdb.org_course.cascade_id获取班级班主任名称
     * @param orgId
     * @param cascadeId
     * @return
     */
    private String getAccountName(Long orgId, Integer cascadeId) {
        if (cascadeId != null && cascadeId > 0) {
            TXCascadeAccount cascadeAccount = cascadeAccountDao.getById(cascadeId);
            if (cascadeAccount == null || cascadeAccount.getIsdel() == 1 || cascadeAccount.getStatus() == 1) {
                log.warn("[Wechat] Message TXCascadeAccount is invalidate.cascadeAccount={}", cascadeAccount);
            } else {
                TXCascadeCredential credential =
                    credentialDao.getTXCascadeCredentialByCredentialId(cascadeAccount.getCredentialId());
                if (credential != null) {
                    return credential.getName() == null ? credential.getMobile() : credential.getName();
                } else {
                    log.error("[Wechat] Message TXCascadeCredential is invalidate.credentialId="
                        + cascadeAccount.getCredentialId());
                }
            }
        } else if (cascadeId == 0) {
            OrgInfo orgInfo = orgInfoDao.getOrgInfo(orgId.intValue(), "contacts");
            return (orgInfo != null && StringUtils.isNotBlank(orgInfo.getContacts())) ? orgInfo.getContacts() : "";
        }
        return DEFAULT_CLASS_MASTER;
    }

    /**
     * 获取课程教师相关信息
     * @param orgId
     * @param courseId
     * @return
     */
    private CommonInfoTipsDto getCourseRoomTips(Long orgId, Long courseId) {
        List<ClassRoomDto> data = this.classRoomService.listOrgCourseRooms(orgId, courseId);
        CommonInfoTipsDto tips = null;
        if (CollectionUtils.isNotEmpty(data)) {
            tips = new CommonInfoTipsDto();
            int count = data.size();
            Set<String> names = BaseUtils.getPropertiesList(data, "roomName");
            tips.setCount(count);
            tips.setNames(names);
        }
        log.info("getCourseRoomTips.tips:{}", tips);
        return tips;
    }

    /**
     * 获取课程老师相关信息
     * @param orgId
     * @param courseId
     * @return
     */
    private CommonInfoTipsDto getCourseTeacherTips(Long orgId, Long courseId) {
        List<TeacherResponseDto> data = this.courseTeacherService.listCourseTeacher(orgId, courseId);
        CommonInfoTipsDto tips = null;
        if (CollectionUtils.isNotEmpty(data)) {
            tips = new CommonInfoTipsDto();
            int count = data.size();
            Set<String> names = BaseUtils.getPropertiesList(data, "teacherName");
            tips.setCount(count);
            tips.setNames(names);
        }
        log.info("getCourseTeacherTips.tips:{}", tips);
        return tips;
    }

    /**
     * 获取班级学生数
     * @param orgId
     * @param courseId
     * @return
     */
    private int getCourseStudentCount(Long orgId, Long courseId) {
        // 学生数量
        List<Long> studentIds = this.orgStudentCourseDao.getStudents(orgId, courseId, 0);
        return CollectionUtils.isNotEmpty(studentIds) ? studentIds.size() : 0;
    }

    /**
     * 查询已上完的课节数
     * @param orgId
     * @param courseId
     * @return
     */
    private int getFinishedLessonCount(Long orgId, Long courseId) {
        int finishedLessonCount = this.orgClassLessonDao.getFinishLessonCount(orgId, courseId, DeleteStatus.NORMAL.getValue());
        return finishedLessonCount;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long saveOrUpdateCourse(Long orgId, OrgCourseInfoPcDto course) {
        Long courseId = this.orgCourseService.saveOrUpdateCourse(orgId, course, true, true);
        if (courseId == null || courseId <= 0) {
            throw new BussinessException(CommonErrorCode.SYSTEM_ERROR, "保存课程失败");
        }

        List<Long> roomIds = Lists.newArrayList();
        List<Long> studentIds = Lists.newArrayList();
        try {
            if (StringUtils.isNoneBlank(course.getRoomIds())) {
                roomIds = JacksonUtil.str2List(course.getRoomIds(), Long.class);
            }
            if (StringUtils.isNoneBlank(course.getStudentIds())) {
                studentIds = JacksonUtil.str2List(course.getStudentIds(), Long.class);
            }
        } catch (Exception e) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "教室、学生参数错误");
        }

        // 保存教室
        this.classRoomService.saveCourseRooms(orgId, courseId, roomIds);

        // 保存学生
        this.erpStudentService.saveCourseStudents(orgId, courseId, studentIds);

        // 保存(分配)班主任,参数cascadeId:-1清空班主任；0主账号；null无分配班主任权限，班主任不变；>0子账号
        if (course.getCascadeId() != null) {
            Integer curCascadeId = TianxiaoPCContext.getTXCascadeId();
            // 子账号，需要判断是否有分配班主任的权限
            if (curCascadeId != null && curCascadeId > 0) {
                boolean hasPermission = this.permissionService.checkPermission(orgId, curCascadeId, DeviceType.PC, TXPermissionConst.DISTRIBUTE_CLASSMANAGER.getpCode());
                if (!hasPermission) {
                    throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "没有分配班主任的权限");
                }
            }

            OrgCourse orgCourse = this.orgCourseDao.getByCourseId(courseId, "id", "cascadeId");
            if (course.getCascadeId() == 0) {
                orgCourse.setCascadeId(0);
            } else if (course.getCascadeId() < 0) {
                orgCourse.setCascadeId(-1);
            } else {
                TXCascadeAccount cascadeAccount = this.cascadeAccountDao.getByIdAndOrgId(course.getCascadeId().intValue(), orgId.intValue());
                if (cascadeAccount != null && cascadeAccount.getCredentialId() != null && cascadeAccount.getCredentialId() > 0) {
                    TXCascadeCredential txCascadeCredential = this.credentialDao.getTXCascadeCredentialByCredentialId(cascadeAccount.getCredentialId());
                    if (txCascadeCredential == null) {
                        throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "子帐号异常");
                    } else {
                        orgCourse.setCascadeId(cascadeAccount.getId());
                    }
                } else {
                    throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "子帐号异常");
                }
            }
            this.orgCourseDao.update(orgCourse, false, "cascadeId");
        }
        return courseId;
    }

    @SuppressWarnings("unchecked")
    @Override
    public List<CourseNameAndIdDto> fuzzyQuery(Long orgId, String courseName, Integer type, PageDto pageDto) {
        List<Long> courseIds = getCurrentUserCoursesIds();

        // 如果是员工账号,只能查询自己是班主任的课程
        // 如果是校长、主管并且主管拥有排课权限，查询所有课程
        if (CollectionUtils.isEmpty(courseIds)) {
            if (TianxiaoMContext.getTXCascadeId() != null) {
                TXCascadeAccount txCascadeAccount = cascadeAccountDao.getById(TianxiaoMContext.getTXCascadeId());
                if (txCascadeAccount == null) {
                    throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "子帐号不存在");
                }
                if (txCascadeAccount.getAccountType() == CascadeType.STAFF.getValue()) {
                    return Lists.newArrayList();
                }
            }
        }
        OrgAccount orgAccount = orgAccountDao.getById(TianxiaoPCContext.getOrgId());
        List<OrgCourse> courses = this.orgCourseDao.fuzzyQueryByCourseType(orgAccount.getNumber(), courseName, courseIds, type, pageDto);
//        Iterator<OrgCourse> it = courses.iterator();
//        while(it.hasNext()) {
//        	OrgCourse course = it.next();
//        	if(course.getIsCourse() == 1 && course.getCourseType() == 2) {
//        		it.remove();
//        	}
//        }
        List<CourseNameAndIdDto> data = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(courses)) {
            data = (List<CourseNameAndIdDto>) CollectionUtils.collect(courses, new Transformer() {

                @Override
                public Object transform(Object input) {
                    CourseNameAndIdDto dto = new CourseNameAndIdDto();
                    OrgCourse course = (OrgCourse)input;
                    dto.setId(course.getId());
                    dto.setName(course.getName());
                    dto.setCourseType(course.getCourseType());
                    return dto;
                }
            });
        }
        return data;
    }

    /**
     * 获取当前登录用户的课程IDS
     * @return
     */
    private List<Long> getCurrentUserCoursesIds() {
        List<Long> courseIds = Lists.newArrayList();
        // 登录帐号为子帐号，并且是员工帐号，并且排课权限打开时，只能查看自己作为班主任的课节
        // 涉及到分页，这里取courseIds时，按照时间倒序，测试时需要关注数据是否取全
        if (TianxiaoMContext.getTXCascadeId() != null) {
            TXCascadeAccount txCascadeAccount = cascadeAccountDao.getById(TianxiaoMContext.getTXCascadeId());
            if (txCascadeAccount == null) {
                throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "子帐号不存在");
            }
            if (txCascadeAccount.getAccountType() == CascadeType.STAFF.getValue()) {
                log.debug("before clear courseIds:{}", courseIds);
                List<Long> tempIds = orgCourseDao.getCourseIdsByCascadeId(TianxiaoMContext.getTXCascadeId(),null,CourseTypeEnum.IS_CLASS_TRUE.getCode(),null);
                log.debug("user cascadeId={},courseIds={}",TianxiaoMContext.getTXCascadeId(),tempIds);
                // bugfix 此处根据课程权限获取用户课节的逻辑有问题，如果用户指定查询一门课的课节会查出用户课程权限下的所有课节，故调整逻辑
                if(CollectionUtils.isEmpty(courseIds)){
                    courseIds.addAll(tempIds);
                }else{
                    // 如果是子帐号，并且指定了课程，那么需要判断课程是否在这个用户的权限中
                    if(!tempIds.containsAll(courseIds)){
                        courseIds.clear();
                    }
                }
            }
        }
        return courseIds;
    }

    @Override
    public OrgCourseInfoPcDto getOrgCourseInfoPc(Long orgId, Long courseId) throws IllegalAccessException, InvocationTargetException {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(courseId != null && courseId > 0, "courseId is illegal");
        OrgAccount orgAccount = this.orgAccountDao.getById(orgId, "isDel", "mobile");
        OrgInfo orgInfo = this.orgInfoDao.getOrgInfo(orgId.intValue(), "contacts");
        if (orgAccount == null || orgAccount.getIsDel() == DeleteStatus.DELETED.getValue() || orgInfo == null) {
            throw new BussinessException(StudentErrorCode.ORG_NOT_EXIST);
        }

        OrgCourse orgCourse = this.orgCourseDao.getByCourseId(courseId, "id", "cascadeId");
        if (orgCourse == null) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "机构id错误");
        }

        OrgCourseInfoPcDto courseDto = new OrgCourseInfoPcDto();
        OrgCourseInfoDto courseInfo = this.orgCourseService.getOrgCourseInfo(orgId, courseId);
        if (courseInfo == null) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "参数错误");
        } else {
                BeanUtils.copyProperties(courseDto, courseInfo);

                // 教室信息重新封装，用于前端统一处理
                List<ClassRoomDto> rooms = this.classRoomService.listOrgCourseRooms(orgId, courseId);
                List<IdNameDto> roomList = Lists.newArrayList();
                for (ClassRoomDto room: rooms) {
                    IdNameDto roomDto = new IdNameDto();
                    roomDto.setId(room.getRoomId().intValue());
                    roomDto.setName(room.getRoomName());
                    roomList.add(roomDto);
                }
                courseDto.setRooms(roomList);
                courseDto.setTotalLessons(courseInfo.getFreq() == null ? 0 : courseInfo.getFreq());//课节数量


                Integer lessonCount = orgClassLessonDao.getLessonCount(orgId, courseId, 0);
                courseDto.setScheduleLessons(lessonCount == null ? 0 : lessonCount);//已排课节数
                // 学生
                List<PersonBaseDto> students = this.courseStudentService.getCourseStudentsBase(orgId, courseId);
                courseDto.setStudents(students);
                // 老师信息重新封装，用于前端统一处理
                List<TeacherResponseDto> teachers = courseInfo.getTeachers();
                List<PersonBaseDto> teacherList = Lists.newArrayList();
                if (CollectionUtils.isNotEmpty(teachers)) {
                    for (TeacherResponseDto teacher : teachers) {
                        PersonBaseDto teacherDto = new PersonBaseDto();
                        teacherDto.setId(teacher.getTeacherId());
                        teacherDto.setName(teacher.getTeacherName());
                        teacherDto.setAvatarUrl(teacher.getAvatar());
                        teacherList.add(teacherDto);
                    }
                }
                courseDto.setCourseTeachers(teacherList);

                // 班主任
                if (orgCourse.getCascadeId() == null || orgCourse.getCascadeId() == 0) {
                    // 主帐号
                    courseDto.setCascadeId(0l);
                    courseDto.setCascadeName(orgInfo.getContacts());
                } else if (orgCourse.getCascadeId() == -1) {
                    courseDto.setCascadeId(-1l);
                } else {
                    TXCascadeAccount cascadeAccount = this.cascadeAccountDao.getByIdAndOrgId(orgCourse.getCascadeId(), orgId.intValue());
                    //子账号为null的处理
                    if(cascadeAccount == null){
                        courseDto.setCascadeId(0l);
                        courseDto.setCascadeName(orgInfo.getContacts());
                    }
                    
                    
                    else if (cascadeAccount.getCredentialId() != null && cascadeAccount.getCredentialId() > 0) {
                        TXCascadeCredential cascadeCredential = this.credentialDao.getTXCascadeCredentialByCredentialId(cascadeAccount.getCredentialId());
                        courseDto.setCascadeId(cascadeAccount.getId().longValue());
                        if (cascadeCredential != null) {
                            courseDto.setCascadeName(cascadeCredential.getName());
                            if (StringUtils.isNoneBlank(cascadeCredential.getAvatar())) {
                                courseDto.setCascadeAvatarUrl(cascadeCredential.getAvatar());
                            } else {
                                courseDto.setCascadeAvatarUrl(AvatarConstants.CONSULT_AVATAR_URL);
                            }
                        }
                    }
                }
        }
        return courseDto;
    }
}

    