package com.baijia.tianxiao.biz.erp.teacherCenter.service;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.beanutils.PropertyUtilsBean;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.DataFormat;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.ss.util.RegionUtil;

import com.baijia.tianxiao.biz.erp.dto.response.LessonTimeTableResponseDto;
import com.baijia.tianxiao.biz.erp.dto.response.exportCourse.CourseStudent;
import com.baijia.tianxiao.biz.erp.dto.response.exportCourse.DayScheduleDto;
import com.baijia.tianxiao.biz.erp.dto.response.exportCourse.ExportCourseInfoDto;
import com.baijia.tianxiao.biz.erp.dto.response.exportCourse.LessonScheduleDto;
import com.baijia.tianxiao.biz.erp.teacherCenter.dto.LessonMergeParams;
import com.baijia.tianxiao.dal.constant.ChargeUnit;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.excel.dto.ExportField;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.course.dto.response.OrgTeacherLessonInfoDto;
import com.baijia.tianxiao.sal.signup.dto.SignupCourseInfoDto;
import com.baijia.tianxiao.sal.signup.dto.response.OrgSingupInfoDto;
import com.baijia.tianxiao.util.date.DateUtil;

import lombok.extern.slf4j.Slf4j;

/**
 * Created with IntelliJ IDEA. User: Victor Weng Date: 16/5/19 Time: 上午11:52 To change this template use File | Settings
 * | File Templates.
 */
@Slf4j
public class ExcelCourseExportService {

    private static String HEADER_FORMAT = "@";

    public static final String EXPORT_DATE_FORMAT = "yyyy-MM-dd HH:mm:ss";

    public static final String UNIT_LESSON_MONEY_FORMULA = "IF(E%d>0,E%d*D%d/60,0)";

    public static final String LESSON_MONEY_FORMULA = "SUM(F%d:F%d)";

    private static final short CELL_HEIGHT = 2000;

    public static PropertyUtilsBean propertyUtilsBean = new PropertyUtilsBean();

    private static CellStyle createCellStyle(Workbook workBook, String format) {
        CellStyle cellStyle = workBook.createCellStyle();
        DataFormat dataFormat = workBook.createDataFormat();
        cellStyle.setAlignment(CellStyle.ALIGN_CENTER);
        cellStyle.setVerticalAlignment(CellStyle.VERTICAL_CENTER);
        cellStyle.setDataFormat(dataFormat.getFormat(format));

        cellStyle.setBorderBottom(CellStyle.BORDER_THIN);
        cellStyle.setBorderLeft(CellStyle.BORDER_THIN);
        cellStyle.setBorderRight(CellStyle.BORDER_THIN);
        cellStyle.setBorderTop(CellStyle.BORDER_THIN);

        return cellStyle;
    }

    public static Sheet createSheet(Workbook workbook, String sheetName, List<String> titles,
        Map<String, CellStyle> cellStyleMap, List<ExportField> exportFields, List<LessonTimeTableResponseDto> datas,
        Date startDate) {

        Sheet sheet = workbook.createSheet(sheetName);
        int columnSize = 0;// 总列数
        for (ExportField field : exportFields) {
            columnSize += field.getExportFieldSize();
        }
        int startTime = 5;// 开始时间
        createTitleRow(sheet, cellStyleMap, titles, columnSize);
        int row = titles != null ? titles.size() : 0;
        createHeaderRow(sheet, cellStyleMap, exportFields, row++);
        List<LessonMergeParams> mergeList = new ArrayList<LessonMergeParams>();
        for (LessonTimeTableResponseDto ltt : datas) {
            // 哪一列显示
            int index = DateUtil.getDateDiff(startDate, ltt.getStartTime()) + 1;
            // 那一行开始显示
            int startHourDiff = Math.round((float) DateUtil.getMinuteDiff(startDate, ltt.getStartTime()) / 60)
                - startTime - (index - 1) * 24;
            // 哪一行结束显示
            int endHourDiff = Math.round((float) DateUtil.getMinuteDiff(startDate, ltt.getEndTime()) / 60) - startTime
                - (index - 1) * 24 - 1;
            if (endHourDiff < startHourDiff) {
                endHourDiff += 1;
            }
            LessonMergeParams mergeParam = new LessonMergeParams();
            mergeParam.setIndex(index);
            mergeParam.setStartHour(startHourDiff);
            mergeParam.setEndHour(endHourDiff);
            mergeParam.setCourseName(ltt.getCourseName());
            mergeList.add(mergeParam);
        }

        for (int j = 0; j < 24 - startTime; j++) {
            int rowNum = row;
            rowNum += j;

            Row courseRow = sheet.createRow(rowNum);
            createCell(courseRow, 0, (startTime + j) + ":00", cellStyleMap, exportFields.get(0));

            for (LessonMergeParams lessonMerge : mergeList) {
                if (j >= lessonMerge.getStartHour() && j <= lessonMerge.getEndHour()) {
                    createCell(courseRow, lessonMerge.getIndex(), lessonMerge.getCourseName(), cellStyleMap,
                        exportFields.get(lessonMerge.getIndex()));
                }
            }
        }
        for (LessonMergeParams lmr : mergeList) {
            // 合并下重复的行
            mergeCell(sheet, lmr.getStartHour() + row, lmr.getEndHour() + row, lmr.getIndex(), lmr.getIndex());
        }
        return sheet;
    }

    /**
     * 导出班级信息(包含班级学员)
     * 
     * @param workbook
     * @param sheetName
     * @param cellStyleMap
     * @param exportFields
     * @param courseInfo
     * @return
     */
    public static void createCourseSheet(Workbook workbook, String sheetName, Map<String, CellStyle> cellStyleMap,
        List<ExportField> exportFields, ExportCourseInfoDto courseInfo) {
        Sheet sheet = workbook.createSheet(sheetName);
        int row = 0;
        // 创建表头行
        createHeaderRow(sheet, cellStyleMap, exportFields, row++);
        List<Object> courseInfoValueList = courseInfo.getCourseValueList();
        // 学生数为0时也需要填充班级基本信息
        for (int rowIndex = 0; rowIndex <= courseInfo.getStudentCount(); rowIndex++) {
            Row courseRow = sheet.createRow(row++);
            int columnIndex = 0;

            if (rowIndex == 0 || rowIndex < courseInfo.getStudentCount()) {
                // 填充班级基本信息(所有学员相同)
                for (; columnIndex < courseInfoValueList.size(); columnIndex++) {
                    createCell(courseRow, columnIndex, courseInfoValueList.get(columnIndex), cellStyleMap,
                        exportFields.get(columnIndex));
                }

                List<Object> studentValueList = null;
                if (courseInfo.getStudentCount() == 0) {
                    // 学生人数为空时学员信息填充""
                    studentValueList = CourseStudent.getDefaultValueList();
                } else {
                    studentValueList = courseInfo.getStudentList().get(rowIndex).getStudentValueList(courseInfo.getChargeUnit());
                }
                int studentValueIndex = 0;
                // 接班级信息后面填充学员信息
                for (; columnIndex < exportFields.size(); columnIndex++) {
                    createCell(courseRow, columnIndex, studentValueList.get(studentValueIndex++), cellStyleMap,
                        exportFields.get(columnIndex));
                }
            }
        }
        // 学生数>1时需要合并班级基本信息
        if (courseInfo.getStudentCount() > 1) {
            for (int mergeIndex = 0; mergeIndex < courseInfoValueList.size(); mergeIndex++) {
                mergeCell(sheet, 1, courseInfo.getStudentCount(), mergeIndex, mergeIndex);
            }
        }
    }

    /**
     * 导出一周的课表
     * 
     * @param workbook
     * @param sheetName
     * @param cellStyleMap
     * @param exportFields
     * @param data
     */
    public static void createScheduleSheet(Workbook workbook, String sheetName, Map<String, CellStyle> cellStyleMap,
        List<ExportField> exportFields, List<DayScheduleDto> data) {
        Sheet sheet = workbook.createSheet(sheetName);
        // 一周内单日最大课节数,用于创建EXCEL单元格填充空的内容单元格数量
        int maxColumnCount = 0;
        for (DayScheduleDto dayScheduleDto : data) {
            if (maxColumnCount < dayScheduleDto.getLessonCount()) {
                maxColumnCount = dayScheduleDto.getLessonCount();
            }
        }
        int row = 0;
        // 创建表头行
        createHeaderRow(sheet, cellStyleMap, exportFields, row++);

        // 内容换行样式
        CellStyle newLineStyle = createCellStyle(workbook, "@");
        newLineStyle.setWrapText(true);
        newLineStyle.setAlignment(CellStyle.ALIGN_LEFT);

        // 存储excel行,用于在对应的excel行按列创建cell
        Map<Integer, Row> lessonRowMap = new HashMap<>();
        for (int lessonIndex = 0; lessonIndex < maxColumnCount; lessonIndex++) {
            Row lessonRow = sheet.createRow(row++);
            lessonRow.setHeight(CELL_HEIGHT);
            lessonRowMap.put(lessonIndex, lessonRow);
        }

        int columnIndex = 0;
        for (DayScheduleDto dayScheduleDto : data) {
            List<LessonScheduleDto> lessons = dayScheduleDto.getLessons();
            int size = dayScheduleDto.getLessonCount();
            for (int index = 0; index < maxColumnCount; index++) {
                String value = "";
                Row lessonRow = lessonRowMap.get(index);
                if (index < size) {
                    LessonScheduleDto lessonScheduleDto = lessons.get(index);
                    value = lessonScheduleDto.getScheduleContent();
                }
                Cell cell = createCell(lessonRow, columnIndex, value, cellStyleMap, exportFields.get(columnIndex));
                cell.setCellStyle(newLineStyle);

            }
            columnIndex++;
        }
    }

    public static Sheet createSignupSheet(Workbook workbook, String sheetName, List<String> titles,
        Map<String, CellStyle> cellStyleMap, List<ExportField> exportFields, List<OrgSingupInfoDto> list) {

        Sheet sheet = workbook.createSheet(sheetName);
        int columnSize = 0;// 总列数
        for (ExportField field : exportFields) {
            columnSize += field.getExportFieldSize();
        }
        createTitleRow(sheet, cellStyleMap, titles, columnSize);
        int row = titles != null ? titles.size() : 0;
        createHeaderRow(sheet, cellStyleMap, exportFields, row++);
        List<LessonMergeParams> mergeList = new ArrayList<LessonMergeParams>();
        int i = 0;
        for (OrgSingupInfoDto dto : list) {
            int rowNum = row;
            rowNum += i;
            log.debug("--------------status={}", dto.getStatus());
            if (dto.getCourseInfos() == null || dto.getCourseInfos().size() == 0) {
                Row signupRow = sheet.createRow(rowNum);
                createCell(signupRow, 0, " ", cellStyleMap, exportFields.get(0));
                createCell(signupRow, 1, " ", cellStyleMap, exportFields.get(1));
                createCell(signupRow, 2, dto.getRealTotalMoney(), cellStyleMap, exportFields.get(2));
                createCell(signupRow, 3, " ", cellStyleMap, exportFields.get(3));
                createCell(signupRow, 4, " ", cellStyleMap, exportFields.get(4));
                createCell(signupRow, 5, " ", cellStyleMap, exportFields.get(5));
                createCell(signupRow, 6, "-", cellStyleMap, exportFields.get(6));
                createCell(signupRow, 7, " ", cellStyleMap, exportFields.get(7));
                createCell(signupRow, 8, dto.getTransferClassMoney(), cellStyleMap, exportFields.get(8));
                createCell(signupRow, 9, dto.getTotalPrice(), cellStyleMap, exportFields.get(9));
                createCell(signupRow, 10, dto.getTransferRefundMoney(), cellStyleMap, exportFields.get(10));
                createCell(signupRow, 11, DateUtil.getStrByDateFormate(dto.getUpdateTime(), "yyyy-MM-dd hh:MM:ss"),
                    cellStyleMap, exportFields.get(11));
                String statusStr =
                    dto.getStatus().intValue() == 1 ? "已完成" : dto.getStatus().intValue() == 2 ? "待收款" : "待报名";
                createCell(signupRow, 12, statusStr, cellStyleMap, exportFields.get(12));
                createCell(signupRow, 13, dto.getCascadeIdStr(), cellStyleMap, exportFields.get(13));

                i++;
            } else {
                int j = 0;
                for (SignupCourseInfoDto signupCourseInfoDto : dto.getCourseInfos()) {
                    Row signupRow = sheet.createRow(rowNum + j);
                    Number timeCount =
                        getCount(signupCourseInfoDto.getChargeUnit(), signupCourseInfoDto.getLessonCount());
                    createCell(signupRow, 0, dto.getStudentName(), cellStyleMap, exportFields.get(0));
                    createCell(signupRow, 1, dto.getStudentMobile(), cellStyleMap, exportFields.get(1));
                    createCell(signupRow, 2, dto.getRealTotalMoney(), cellStyleMap, exportFields.get(2));
                    createCell(signupRow, 3, signupCourseInfoDto.getOrgCourseName(), cellStyleMap, exportFields.get(3));
                    createCell(signupRow, 4, signupCourseInfoDto.getOriginPrice(), cellStyleMap, exportFields.get(4));
                    createCell(signupRow, 5, timeCount, cellStyleMap, exportFields.get(5));
                    createCell(signupRow, 6, signupCourseInfoDto.getPreferential(), cellStyleMap, exportFields.get(6));
                    createCell(signupRow, 7, signupCourseInfoDto.getPayPrice(), cellStyleMap, exportFields.get(7));
                    createCell(signupRow, 8, signupCourseInfoDto.getTransferClassDeduction(), cellStyleMap,
                        exportFields.get(8));
                    if (dto.getStatus().intValue() == 2) {
                        createCell(signupRow, 9, "--", cellStyleMap, exportFields.get(9));
                    } else {
                        createCell(signupRow, 9,
                            signupCourseInfoDto.getStudentPayPrice() + signupCourseInfoDto.getRealPayPrice(),
                            cellStyleMap, exportFields.get(9));
                    }
                    createCell(signupRow, 10, signupCourseInfoDto.getRefundAmount(), cellStyleMap,
                        exportFields.get(10));
                    createCell(signupRow, 11, DateUtil.getStrByDateFormate(dto.getUpdateTime(), "yyyy-MM-dd hh:MM:ss"),
                        cellStyleMap, exportFields.get(11));
                    String statusStr =
                        dto.getStatus().intValue() == 1 ? "已完成" : dto.getStatus().intValue() == 2 ? "待收款" : "待报名";
                    createCell(signupRow, 12, statusStr, cellStyleMap, exportFields.get(12));
                    createCell(signupRow, 13, dto.getCascadeIdStr(), cellStyleMap, exportFields.get(13));

                    j++;
                }
                i += j;
                if (j > 1) {
                    LessonMergeParams mergeParams = new LessonMergeParams();
                    mergeParams.setStartHour(rowNum);
                    mergeParams.setEndHour(rowNum + j - 1);
                    mergeList.add(mergeParams);
                }
            }
        }
        System.out.println(mergeList);
        for (LessonMergeParams lmr : mergeList) {
            // 合并下重复的行
            mergeCell(sheet, lmr.getStartHour(), lmr.getEndHour(), 0, 0);
            mergeCell(sheet, lmr.getStartHour(), lmr.getEndHour(), 1, 1);
            mergeCell(sheet, lmr.getStartHour(), lmr.getEndHour(), 2, 2);
            mergeCell(sheet, lmr.getStartHour(), lmr.getEndHour(), 9, 9);
            mergeCell(sheet, lmr.getStartHour(), lmr.getEndHour(), 10, 10);
            mergeCell(sheet, lmr.getStartHour(), lmr.getEndHour(), 11, 11);
        }
        return sheet;
    }

    /**
     * @return
     */
    private static Number getCount(int chargeUnit, int count) {
        if (chargeUnit == ChargeUnit.BY_MINUTE.getCode()) {
            return new BigDecimal(count).divide(new BigDecimal(60), 2, RoundingMode.HALF_UP).doubleValue();
        } else {
            return count;
        }
    }

    public static void main(String[] args) {
        System.out.println(getCount(2, 118));
    }

    private static void mergeCell(Sheet sheet, int firstRow, int lastRow, int firstColumn, int lastColumn) {
        CellRangeAddress rangeAddress = new CellRangeAddress(firstRow, lastRow, firstColumn, lastColumn);
        sheet.addMergedRegion(rangeAddress);
        try {
            RegionUtil.setBorderBottom(CellStyle.BORDER_THIN, rangeAddress, sheet, sheet.getWorkbook());
            RegionUtil.setBorderTop(CellStyle.BORDER_THIN, rangeAddress, sheet, sheet.getWorkbook());
            RegionUtil.setBorderLeft(CellStyle.BORDER_THIN, rangeAddress, sheet, sheet.getWorkbook());
            RegionUtil.setBorderRight(CellStyle.BORDER_THIN, rangeAddress, sheet, sheet.getWorkbook());
        } catch (Exception e) {
            log.warn("处理边框失败");
            e.printStackTrace();
        }
    }

    private static void createLessonRow(Row lessonRow, int baseColumn, OrgTeacherLessonInfoDto lessonInfo,
        List<ExportField> exportFields, Map<String, CellStyle> cellStyleMap) {
        baseColumn++;
        createCell(lessonRow, baseColumn, lessonInfo.getLessonNumberInfo(), cellStyleMap, exportFields.get(baseColumn));
        baseColumn++;
        createCell(lessonRow, baseColumn, lessonInfo.getRoomName(), cellStyleMap, exportFields.get(baseColumn));
        baseColumn++;
        createCell(lessonRow, baseColumn, lessonInfo.getStartTime(), cellStyleMap, exportFields.get(baseColumn));
        baseColumn++;
        createCell(lessonRow, baseColumn, lessonInfo.getLessonTime(), cellStyleMap, exportFields.get(baseColumn));
        baseColumn++;
        createCell(lessonRow, baseColumn, lessonInfo.getMinutes(), cellStyleMap, exportFields.get(baseColumn));
        baseColumn++;
        createCell(lessonRow, baseColumn, lessonInfo.getStudentCount(), cellStyleMap, exportFields.get(baseColumn));
        baseColumn++;
        createCell(lessonRow, baseColumn, lessonInfo.getSignCount(), cellStyleMap, exportFields.get(baseColumn));
        baseColumn++;
        createCell(lessonRow, baseColumn, lessonInfo.getLeaveCount(), cellStyleMap, exportFields.get(baseColumn));
        baseColumn++;
        createCell(lessonRow, baseColumn, lessonInfo.getAbsentCount(), cellStyleMap, exportFields.get(baseColumn));
        baseColumn++;
        createCell(lessonRow, baseColumn, lessonInfo.getUnsignCount(), cellStyleMap, exportFields.get(baseColumn));
        baseColumn++;
        createCell(lessonRow, baseColumn, lessonInfo.getSignRate(), cellStyleMap, exportFields.get(baseColumn));

    }

    private static Cell createCell(Row row, int column, Object value, Map<String, CellStyle> cellStyleMap,
        ExportField exportField) {
        Cell cell = row.createCell(column);
        String localFormat = null;
        value = value != null ? value : exportField.getDefaultVal();

        if (value != null) {
            if (value instanceof Date) {
                row.getSheet().setColumnWidth(cell.getColumnIndex(), 20 * 256);
                localFormat = EXPORT_DATE_FORMAT;
                cell.setCellValue((Date) value);
            } else {
                if (value instanceof Number && (((Number) value).longValue() < Integer.MAX_VALUE)) {
                    cell.setCellType(Cell.CELL_TYPE_NUMERIC);
                    if (value instanceof Double) {
                        localFormat = "0.00";
                        cell.setCellValue(((Number) value).doubleValue());
                    } else {
                        localFormat = "0";
                        cell.setCellValue(((Number) value).longValue());
                    }
                } else {
                    cell.setCellType(Cell.CELL_TYPE_STRING);
                    cell.setCellValue(value.toString());
                    localFormat = "@";
                }
            }
        } else {
            cell.setCellValue("");
        }
        cell.setCellStyle(getCellStyle(row.getSheet().getWorkbook(), cellStyleMap,
            exportField.getFormat() != null ? exportField.getFormat() : localFormat));
        return cell;
    }

    private static CellStyle getCellStyle(Workbook workbook, Map<String, CellStyle> cellStyleMap, String format) {
        if (!cellStyleMap.containsKey(format)) {
            cellStyleMap.put(format, createCellStyle(workbook, format));
        }
        return cellStyleMap.get(format);
    }

    /**
     * 递归创建出行头
     *
     * @param sheet
     * @param cellStyleMap
     * @param exportFields
     * @param row
     */
    private static void createHeaderRow(Sheet sheet, Map<String, CellStyle> cellStyleMap,
        List<ExportField> exportFields, int row) {
        Row headerRow = sheet.createRow(row);
        int column = 0;
        CellStyle style = getCellStyle(sheet.getWorkbook(), cellStyleMap, HEADER_FORMAT);
        for (ExportField field : exportFields) {
            column = createHeaderCell(headerRow, style, field, column);
        }
    }

    private static int createHeaderCell(Row row, CellStyle style, ExportField field, int column) {
        if (CollectionUtils.isEmpty(field.getChildren())) {
            row.getSheet().setColumnWidth(column, field.getWidth());
            Cell cell = row.createCell(column++);
            cell.setCellStyle(style);
            cell.setCellValue(field.getName());
        } else {
            for (ExportField child : field.getChildren()) {
                column = createHeaderCell(row, style, child, column);
            }
        }
        return column;
    }

    private static void createTitleRow(Sheet sheet, Map<String, CellStyle> cellStyleMap, List<String> titles,
        int headerSize) {
        int row = 0;
        // 补充头
        if (CollectionUtils.isNotEmpty(titles)) {
            for (String title : titles) {
                Row titleRow = sheet.createRow(row);
                Cell cell = titleRow.createCell(0);
                cell.setCellValue(title);
                cell.setCellStyle(getCellStyle(sheet.getWorkbook(), cellStyleMap, HEADER_FORMAT));
                mergeCell(sheet, row, row, 0, headerSize - 1);
                row++;
            }
        }
    }

    public static void exportExcel(HttpServletResponse response, Workbook workBook, String excelFileName) {
        if (StringUtils.isBlank(excelFileName)) {
            excelFileName = System.currentTimeMillis() + ".xlsx";
        }
        response.setContentType("application/vnd.ms-excel");
        try {
            response.setHeader("Content-Disposition",
                "attachment; filename=" + URLEncoder.encode(excelFileName, "utf-8"));
        } catch (UnsupportedEncodingException e1) {
            response.setHeader("Content-Disposition", "attachment; filename=" + excelFileName);
        }

        try {
            workBook.write(response.getOutputStream());
            response.getOutputStream().flush();
        } catch (IOException e) {
            log.error("export data catch error:", e);
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "导出数据处理失败");
        }
    }

    /**
     * 增加对Firefox浏览器下载文件乱码的处理
     * 
     * @param request
     * @param response
     * @param workBook
     * @param excelFileName
     */
    public static void exportExcel(HttpServletRequest request, HttpServletResponse response, Workbook workBook,
        String excelFileName) {
        if (StringUtils.isBlank(excelFileName)) {
            excelFileName = System.currentTimeMillis() + ".xlsx";
        }
        response.setContentType("application/vnd.ms-excel");
        try {
            String agent = request.getHeader("User-Agent");
            String encodeFileName = "";
            // IE系列浏览器使用Trident内核,不能使用"MSIE"判断,因为IE10及以下有"MSIE",IE11没有
            if (agent.contains("Trident")) {
                encodeFileName = URLEncoder.encode(excelFileName, "UTF-8");
            } else {
                // Firefox、Safari、其他浏览器
                encodeFileName = "\"" + new String(excelFileName.getBytes("UTF-8"), "iso-8859-1") + "\"";
            }
            response.setHeader("Content-Disposition", "attachment; filename=" + encodeFileName);
        } catch (UnsupportedEncodingException e1) {
            response.setHeader("Content-Disposition", "attachment; filename=\"" + excelFileName + "\"");
        }

        try {
            workBook.write(response.getOutputStream());
            response.getOutputStream().flush();
        } catch (IOException e) {
            log.error("export data catch error:", e);
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "导出数据处理失败");
        }
    }
}
