package com.baijia.tianxiao.biz.erp.kexiao.impl;

import com.baijia.tianxiao.biz.erp.kexiao.KexiaoComputeService;
import com.baijia.tianxiao.constant.KexiaoChangeEvent;
import com.baijia.tianxiao.constant.LessonStatus;
import com.baijia.tianxiao.dal.enums.CourseTypeEnum;
import com.baijia.tianxiao.dal.org.dao.*;
import com.baijia.tianxiao.dal.org.po.*;
import com.baijia.tianxiao.dal.signup.dao.OrgSignupCourseLessonDao;
import com.baijia.tianxiao.dal.signup.po.OrgSignupCourseLesson;
import com.baijia.tianxiao.dal.solr.dto.ClassLesson;
import com.baijia.tianxiao.sal.course.service.OrgSignupCourseLessonService;
import com.baijia.tianxiao.sal.kexiao.dto.*;
import com.baijia.tianxiao.sal.kexiao.utils.KexiaoUtil;
import com.baijia.tianxiao.util.CollectionHelper;
import com.baijia.tianxiao.util.ListUtil;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.gson.Gson;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;

/**
 * Created by liuxp on 16/12/29.
 */
@Service
@Slf4j
public class KexiaoComputeServiceImpl implements KexiaoComputeService{
    @Autowired
    private OrgCourseConsumeRuleDao courseConsumeRuleDao;
    @Autowired
    private OrgStudentKexiaoRecordDao kexiaoRecordDao;
    @Autowired
    private OrgStudentLessonDao studentLessonDao;
    @Autowired
    private OrgClassLessonDao classLessonDao;
    @Autowired
    private OrgSignupCourseLessonDao signupCourseLessonDao;
    @Autowired
    private OrgLessonSignDao lessonSignDao;
    @Autowired
    private OrgCourseDao courseDao;
    @Autowired
    private OrgSignupCourseLessonService signupCourseLessonService;

    public void computeKexiao(Long orgId, Collection<KexiaoChangeLog> logs) {
        List<KexiaoChangeLog> ruleChangeLogs = new ArrayList<>();
        List<KexiaoChangeLog> hisCompleteChangeLogs = new ArrayList<>();
        List<KexiaoChangeLog> batchSignChangeLogs = new ArrayList<>();
        List<KexiaoChangeLog> lessonStartChangeLogs = new ArrayList<>();
        List<KexiaoChangeLog> fastSignChangeLogs = new ArrayList<>();
        List<KexiaoChangeLog> delLessonChangeLogs = new ArrayList<>();
        List<KexiaoChangeLog> delStuLessonChangeLogs = new ArrayList<>();
        List<KexiaoChangeLog> cancelSignChangeLogs = new ArrayList<>();
        List<KexiaoChangeLog> stuLessonChangeLogs = new ArrayList<>();
        if (logs != null && logs.size() > 0) {
            for (KexiaoChangeLog changeLog : logs) {
                KexiaoChangeEvent event = KexiaoChangeEvent.getByType(changeLog.getEventType());
                switch (event) {
                    case KEXIAO_REUL:
                        ruleChangeLogs.add(changeLog);
                        break;
                    case START:
                        lessonStartChangeLogs.add(changeLog);
                        break;
                    case ADD_SIGNUP_INFO:
                        hisCompleteChangeLogs.add(changeLog);
                        break;
                    case BATCH_SIGN:
                        batchSignChangeLogs.add(changeLog);
                        break;
                    case FAST_SIGN:
                        fastSignChangeLogs.add(changeLog);
                        break;
                    case DEL_LESSON:
                        delLessonChangeLogs.add(changeLog);
                        break;
                    case DEL_STUDENT_LESSON:
                        delStuLessonChangeLogs.add(changeLog);
                        break;
                    case CANCEL_SIGN:
                        cancelSignChangeLogs.add(changeLog);
                        break;
                    case STUDENT_LESSON_CHANGE:
                        stuLessonChangeLogs.add(changeLog);
                        break;
                    default:
                        log.warn("[Kexiao] KexiaoChangeLog={}", changeLog);
                        throw new RuntimeException("KexiaoChangeEvent error");
                }
            }
        }
        if (ruleChangeLogs.size() > 0) {
            computeByKexiaoRule(orgId, ruleChangeLogs);
        }
        if (lessonStartChangeLogs.size() > 0) {
            computeByStaredTime(orgId, lessonStartChangeLogs);
        }
        if (batchSignChangeLogs.size() > 0) {
            computeByBatchSign(orgId, batchSignChangeLogs);
        }
        if (fastSignChangeLogs.size() > 0) {
            computeByFastSign(orgId, fastSignChangeLogs);
        }
        if (hisCompleteChangeLogs.size() > 0) {
            computeByCompleteSignUpInfo(orgId, hisCompleteChangeLogs);
        }
        if (delLessonChangeLogs.size() > 0) {
            deleteByDelLessonIds(orgId, delLessonChangeLogs);
        }
        if (delStuLessonChangeLogs.size() > 0) {
            deleteByDelStuLessonIds(orgId, delStuLessonChangeLogs);
        }
        if (cancelSignChangeLogs.size() > 0) {
            deleteByCancelSign(orgId, cancelSignChangeLogs);
        }
        if(stuLessonChangeLogs.size()>0){
            computeByStudentLessons(orgId,stuLessonChangeLogs);
        }
    }

    /**
     * 课消规则变化
     * @param orgId
     * @param logs
     */
    @Transactional
    public void computeByKexiaoRule(Long orgId, Collection<KexiaoChangeLog> logs) {
        log.info("[Kexiao] compute kexiao by rule.orgId={},logs={}", orgId, logs);
        if (logs != null) {
            Gson gson = new Gson();
            for (KexiaoChangeLog changeLog : logs) {
                RuleChangeInfo ruleChange = gson.fromJson(changeLog.getInfo(), RuleChangeInfo.class);
                List<OrgClassLesson> classLessons = classLessonDao.getLessonByCourseIds(Arrays.asList(ruleChange.getCourseId()), 0);
                Map<Long, OrgClassLesson> classLessonMap = CollectionHelper.toKeyMap(classLessons, "id");

                List<Long> lessonIds = ListUtil.toKeyList(classLessons, "id", OrgClassLesson.class);
                List<OrgStudentLesson> stuLessons = studentLessonDao.getOrgStudentLessonsByLessonIds(lessonIds);
                List<Long> userIds = ListUtil.toKeyList(stuLessons, "userId", OrgStudentLesson.class);
                if(lessonIds.isEmpty() || userIds.isEmpty()){
                    log.warn("[Kexiao] compute kexiao by rule.orgId={},lessonIds={},userIds={}", orgId,lessonIds, userIds);
                    continue;
                }
                Map<String, OrgSignupCourseLesson> signUpLessonsMap = signupCourseLessonDao.selectLessonInfoMap(orgId, lessonIds, userIds);
                List<OrgLessonSign> lessonSignList = lessonSignDao.getStudentLessonSign(changeLog.getOrgId(), null, lessonIds);
                Map<String, OrgLessonSign> signMap = new HashMap<>();
                if(lessonSignList!=null){
                    for (OrgLessonSign sign:lessonSignList){
                        signMap.put(getKey(sign),sign);
                    }
                }

                List<OrgStudentKexiaoRecord> records = new ArrayList<>();
                Set<Long> unFinishedLessonIds = new HashSet<>();
                if (ruleChange.getRuleValue() == 0) {
                    // 开课算课消
                    for (OrgStudentLesson studentLesson : stuLessons) {
                        OrgClassLesson classLesson = classLessonMap.get(studentLesson.getLessonId());
                        if (classLesson.getStartTime().compareTo(new Date()) <= 0) {// 计算课消
                            OrgSignupCourseLesson signUpLesson = signUpLessonsMap.get(getKey(studentLesson));
                            if(signUpLesson==null){
                                log.info("[Kexiao] Info is not complete.studentLesson={}",studentLesson);
                                continue;
                            }
                            OrgStudentKexiaoRecord record = constructKexiaoRecord(studentLesson, classLesson, signUpLesson, changeLog.getCreateTime());
                            records.add(record);
                        } else {// 不计算课消
                            unFinishedLessonIds.add(studentLesson.getId());
                        }
                    }
                } else {
                    // 签到算课消
                    for (OrgStudentLesson studentLesson : stuLessons) {
                        OrgClassLesson classLesson = classLessonMap.get(studentLesson.getLessonId());
                        OrgLessonSign lessonSign = signMap.get(getKey(studentLesson));
                        LessonStatus status = LessonStatus.UN_START;
                        if(lessonSign!=null){
                            status = KexiaoUtil.getKexiaoStatus(lessonSign.getStatus(), ruleChange.getRuleValue());
                        }
                        if (status == LessonStatus.FINISHED) {// 计算课消
                            OrgSignupCourseLesson signUpLesson = signUpLessonsMap.get(getKey(studentLesson));
                            if(signUpLesson!=null){
                                OrgStudentKexiaoRecord record = constructKexiaoRecord(studentLesson, classLesson, signUpLesson, changeLog.getCreateTime());
                                records.add(record);
                            }else {
                                log.warn("[Kexiao] signUpLesson is null.studentLesson={}",studentLesson);
                            }

                        } else {// 不计算课消
                            unFinishedLessonIds.add(studentLesson.getId());
                        }
                    }
                }
                Map<String,Object> condition = new HashMap<>();
                condition.put("studentLessonId",ListUtil.toKeyList(stuLessons,"id",OrgStudentLesson.class));
                kexiaoRecordDao.delByCondition(condition);
                kexiaoRecordDao.batchInsertRecords(records);
                if(lessonIds!=null && lessonIds.size()>0){
                    signupCourseLessonDao.batchUpdateKexiaoStatus(lessonIds,userIds,LessonStatus.UN_START.getStatus());
                }
                updateSignupCourseLessonKexiaoStatus(records);
            }
        }
    }

    @Transactional
    private void computeByBatchSign(Long orgId, Collection<KexiaoChangeLog> logs) {
        log.info("[Kexiao] compute batch sign.orgId={},logs={}", orgId, logs);
        Gson gson = new Gson();
        for (KexiaoChangeLog changeLog : logs) {
            BatchSignInfo extend = gson.fromJson(changeLog.getInfo(), BatchSignInfo.class);
            computeBySign(toFastSign(extend), orgId);
        }
    }

    private FastSignInfo toFastSign(BatchSignInfo batchSign){
        Map<Long,Integer> stuStatus = new HashMap<>();
        for (Long userId:batchSign.getUserIds()){
            stuStatus.put(userId, batchSign.getStatus());
        }
        return new FastSignInfo(batchSign.getClassLessonIds(),stuStatus);
    }

    @Transactional
    private void computeByFastSign(Long orgId, Collection<KexiaoChangeLog> logs) {
        log.info("[Kexiao] compute fast sign.orgId={},logs={}", orgId, logs);
        Gson gson = new Gson();
        for (KexiaoChangeLog changeLog : logs) {
            FastSignInfo extend = gson.fromJson(changeLog.getInfo(), FastSignInfo.class);
            computeBySign(extend, orgId);
        }
    }

    /**
     * 补充学员报名信息
     * @param orgId
     * @param logs
     */
    @Transactional
    private void computeByCompleteSignUpInfo(Long orgId, Collection<KexiaoChangeLog> logs) {
        log.info("[Kexiao] compute complete signup info kexiao.orgId={},logs={}", orgId, logs);
        Gson gson = new Gson();
        for (KexiaoChangeLog changeLog : logs) {
            SignUpCourseInfo info = gson.fromJson(changeLog.getInfo(), SignUpCourseInfo.class);
            List<SignUpCourseInfo.SingUpCourseBase> baseList = info.getSingUpCourseList();
            Set<Long> courseIds = new HashSet<>();
            Set<Long> userIds = new HashSet<>();
            for (SignUpCourseInfo.SingUpCourseBase base:baseList){
                courseIds.add(base.getOrgCourseId());
                userIds.add(base.getUserId());
            }
            List<OrgCourse> courses = courseDao.getByIds(courseIds, "id", "parentId", "isCourse", "isClass");
            List<Long> classIds = ListUtil.toKeyList(courses, "id", OrgCourse.class);
            log.info("[Kexiao] courses={}",courses);
            List<OrgStudentLesson> lessons = studentLessonDao.getByCourseIdsUserIds(orgId, classIds, userIds);
            if(lessons==null || lessons.size()<1){
                log.info("[Kexiao] lessons={}",lessons);
                return;
            }
            Map<Long,List<OrgStudentLesson>> lessonMap = new HashMap<>();
            Set<Long> classLessonIds = new HashSet<>();
            for (OrgStudentLesson lesson:lessons){
                List<OrgStudentLesson> lessonList = lessonMap.get(lesson.getCourseId());
                if(lessonList==null){
                    lessonList = Lists.newArrayList();
                    lessonMap.put(lesson.getCourseId(),lessonList);
                }
                lessonList.add(lesson);
                classLessonIds.add(lesson.getLessonId());
            }
            List<OrgClassLesson> classLessonList = classLessonDao.getByIds(classLessonIds, "id", "startTime");
            final Map<Long,OrgClassLesson> classLessonMap = CollectionHelper.toIdMap(classLessonList);
            Set<Long> keySet = lessonMap.keySet();
            Date date = new Date();
            for (Long classId:keySet){
                List<OrgStudentLesson> list = lessonMap.get(classId);
                Collections.sort(list, new Comparator<OrgStudentLesson>() {
                    @Override
                    public int compare(OrgStudentLesson o1, OrgStudentLesson o2) {
                        OrgClassLesson lesson1 = classLessonMap.get(o1.getLessonId());
                        OrgClassLesson lesson2 = classLessonMap.get(o2.getLessonId());
                        if(lesson1==null || lesson2==null){
                            log.warn("[Kexiao] StudentLesson1={},StudentLesson2={}", o1, o2);
                            return -1;
                        }else {
                            return lesson1.getStartTime().compareTo(lesson2.getStartTime());
                        }
                    }
                });
                if(list!=null && list.size()>0) {
                    signupCourseLessonService.saveSignupCourseLessons(orgId, classId, list);
                    for (OrgStudentLesson studentLesson : list) {
                        studentLesson.setUpdateTime(date);
                        studentLessonDao.update(studentLesson);
                    }
                }
            }
        }
    }


    @Transactional
    private void computeBySign(FastSignInfo extend, Long orgId) {
        Collection<Long> classLessonIds = extend.getClassLessonIds();
        List<OrgStudentLesson> studentLessons = studentLessonDao.getOrgStudentLessonsByLessonIds(classLessonIds);

        BatchData data = getBatchData(orgId, studentLessons);

        List<OrgStudentKexiaoRecord> records = new ArrayList<>();
        Set<Long> studentLessonIds = Sets.newHashSet();
        List<UnionKey> keys = new ArrayList<>();
        for (OrgStudentLesson studentLesson : studentLessons) {
            OrgClassLesson classLesson = data.getClassLesson(studentLesson.getLessonId());
            OrgSignupCourseLesson signUpLesson = data.getSignUpCourseLesson(getKey(studentLesson));
            OrgCourseConsumeRule rule = data.getRuleByClassId(classLesson.getCourseId());
            if(signUpLesson==null){
                log.info("[Kexiao] Info is not complete.studentLesson={}", studentLesson);
                continue;
            }
            if (rule != null && rule.getRuleValue()>0) {
                Integer status = extend.getStuStatus().get(studentLesson.getUserId());
                if (status == null) {
                    continue;
                }
                studentLessonIds.add(studentLesson.getId());
                LessonStatus lessonStatus = KexiaoUtil.getKexiaoStatus(status, rule.getRuleValue());
                if (lessonStatus == LessonStatus.FINISHED) {
                    OrgStudentKexiaoRecord record = constructKexiaoRecord(studentLesson, classLesson, signUpLesson, classLesson.getStartTime());
                    OrgCourse course = data.getOrgClass(classLesson.getCourseId());
                    record.setCourseType(course.getCourseType());
                    records.add(record);
                } else {
                    keys.add(new UnionKey(studentLesson.getLessonId(),studentLesson.getUserId()));
                    log.info("[Kexiao] Not need to compute kexiao.orgId={},ruleId={},value={}", orgId, rule.getId(),
                            status);
                }
            } else {
                log.info("[Kexiao] Rule is not exist.orgId={},courseId={}", orgId, classLesson.getCourseId());
            }
        }
        log.info("[Kexiao] compute sign and time kexiao.studentLessonIds={}", studentLessonIds);
        if(studentLessonIds.size()>0) {
            Map<String, Object> condition = new HashMap<>();
            condition.put("studentLessonId", studentLessonIds);
            kexiaoRecordDao.delByCondition(condition);
        }
        updateSignupCourseLessonKexiaoStatus(keys,LessonStatus.UN_START.getStatus());
        if (records.size()>0) {
            kexiaoRecordDao.batchInsertRecords(records);
            updateSignupCourseLessonKexiaoStatus(records);
        }
    }

    /**
     * 根据签到或者开课时间计算课消
     *
     * @param orgId
     * @param logs
     */
    @Transactional
    public void computeByStaredTime(Long orgId, Collection<KexiaoChangeLog> logs) {
        log.info("[Kexiao] compute by startTime.orgId={},logs={}", orgId, logs);
        Gson gson = new Gson();
        List<UnionKey> keys = new ArrayList<>();
        for (KexiaoChangeLog changeLog : logs) {
            LessonStartInfo extend = gson.fromJson(changeLog.getInfo(), LessonStartInfo.class);
            Set<Long> stuLessonIds = new HashSet(extend.getStuLessonIds());
            List<OrgStudentLesson> studentLessons = studentLessonDao.getByIds(stuLessonIds);

            List<OrgStudentKexiaoRecord> records = new ArrayList<>();
            BatchData data = getBatchData(orgId,studentLessons);
            Set<Long> studentLessonIds = Sets.newHashSet();
            for (OrgStudentLesson studentLesson : studentLessons) {
                OrgClassLesson classLesson = data.getClassLesson(studentLesson.getLessonId());
                OrgSignupCourseLesson signUpLesson = data.getSignUpCourseLesson(getKey(studentLesson));
                if(signUpLesson==null){
                    log.info("[Kexiao] Info is not complete.studentLesson={}", studentLesson);
                    continue;
                }
                OrgCourseConsumeRule rule = data.getRuleByClassId(studentLesson.getCourseId());
                if ((rule == null || rule.getRuleValue() == 0)) {
                    studentLessonIds.add(studentLesson.getId());
                    if(extend.getStatus()==LessonStatus.FINISHED.getStatus()) {
                        OrgStudentKexiaoRecord record = constructKexiaoRecord(studentLesson, classLesson, signUpLesson, classLesson.getStartTime());
                        OrgCourse course = data.getOrgClass(classLesson.getCourseId());
                        record.setCourseType(course.getCourseType());
                        records.add(record);
                    }
                } else {
                    keys.add(new UnionKey(studentLesson.getLessonId(),studentLesson.getUserId()));
                    log.info("[Kexiao] Not need to compute kexiao.orgId={},rule={},value={}", orgId, rule,
                            extend.getStatus());
                }
            }
            if(studentLessonIds.size()>0) {
                Map<String, Object> condition = new HashMap<>();
                condition.put("studentLessonId", studentLessonIds);
                kexiaoRecordDao.delByCondition(condition);
            }
            updateSignupCourseLessonKexiaoStatus(keys,LessonStatus.UN_START.getStatus());
            if (records.size()>0) {
                kexiaoRecordDao.batchInsertRecords(records);
                updateSignupCourseLessonKexiaoStatus(records);
            }
        }
    }

    @Transactional
    public void computeByStudentLessons(Long orgId, Collection<KexiaoChangeLog> logs){
        log.info("[Kexiao] compute by startTime.orgId={},logs={}", orgId, logs);
        Gson gson = new Gson();
        for (KexiaoChangeLog changeLog : logs) {
            StudentLessonInfo extend = gson.fromJson(changeLog.getInfo(), StudentLessonInfo.class);
            Set<Long> stuLessonIds = new HashSet(extend.getStuLessonIds());
            List<OrgStudentLesson> studentLessons = studentLessonDao.getByIds(stuLessonIds);
            computeByStudentLessons(orgId, studentLessons);

        }
    }

    /**
     * 课节信息变更
     * @param orgId
     * @param studentLessons
     */
    @Transactional
    private void computeByStudentLessons(Long orgId, List<OrgStudentLesson> studentLessons) {
        log.info("[Kexiao] compute by studentLessons.orgId={},studentLessons={}", orgId, studentLessons);
        List<OrgStudentKexiaoRecord> records = new ArrayList<>();
        BatchData data = getBatchData(orgId, studentLessons);
        Set<Long> studentLessonIds = Sets.newHashSet();
        Set<OrgStudentLesson> delStuLessons = Sets.newHashSet();

        List<Long> userIds = Lists.newArrayList();
        List<Long> lessonIds = Lists.newArrayList();
        List<UnionKey> keys = new ArrayList<>();

        for (OrgStudentLesson studentLesson : studentLessons) {
            userIds.add(studentLesson.getUserId());
            lessonIds.add(studentLesson.getLessonId());
        }

        Map<String,OrgLessonSign> signMap = new HashMap<>();
        List<OrgLessonSign> signList = lessonSignDao.getByOrgIdCourseIdLessonIdsStudentIds(orgId, null, lessonIds, userIds);
        if(signList!=null){
            for (OrgLessonSign sign:signList){
                signMap.put(getKey(sign),sign);
            }
        }

        for (OrgStudentLesson studentLesson : studentLessons) {
            if(studentLesson.getDelStatus()==1){
                studentLessonIds.add(studentLesson.getId());//删除的课节不计算课消
                delStuLessons.add(studentLesson);
                continue;
            }
            OrgClassLesson classLesson = data.getClassLesson(studentLesson.getLessonId());
            OrgSignupCourseLesson signUpLesson = data.getSignUpCourseLesson(getKey(studentLesson));
            if(signUpLesson==null){
                log.info("[Kexiao] Info is not complete.studentLesson={}", studentLesson);
                continue;
            }
            OrgCourseConsumeRule rule = data.getRuleByClassId(studentLesson.getCourseId());
            if ((rule == null || rule.getRuleValue() == 0)) {
                studentLessonIds.add(studentLesson.getId());
                if(studentLesson.getStartStatus()==LessonStatus.FINISHED.getStatus()) {
                    OrgStudentKexiaoRecord record = constructKexiaoRecord(studentLesson, classLesson, signUpLesson, classLesson.getStartTime());
                    OrgCourse course = data.getOrgClass(classLesson.getCourseId());
                    record.setCourseType(course.getCourseType());
                    records.add(record);
                }else {
                    keys.add(new UnionKey(studentLesson.getLessonId(),studentLesson.getUserId()));
                }
            } else {
                OrgLessonSign sign = signMap.get(getKey(studentLesson));
                if (sign == null) {
                    continue;
                }
                Integer status = sign.getStatus();
                studentLessonIds.add(studentLesson.getId());
                LessonStatus lessonStatus = KexiaoUtil.getKexiaoStatus(status, rule.getRuleValue());
                if (lessonStatus == LessonStatus.FINISHED) {
                    OrgStudentKexiaoRecord record = constructKexiaoRecord(studentLesson, classLesson, signUpLesson, classLesson.getStartTime());
                    OrgCourse course = data.getOrgClass(classLesson.getCourseId());
                    record.setCourseType(course.getCourseType());
                    records.add(record);
                } else {
                    keys.add(new UnionKey(studentLesson.getLessonId(),studentLesson.getUserId()));
                    log.info("[Kexiao] Not need to compute kexiao.orgId={},ruleId={},value={}", orgId, rule.getId(),
                            status);
                }
            }
        }
        delSignupCourseLessons(delStuLessons);
        if(studentLessonIds.size()>0) {
            Map<String, Object> condition = new HashMap<>();
            condition.put("studentLessonId", studentLessonIds);
            kexiaoRecordDao.delByCondition(condition);
        }
        updateSignupCourseLessonKexiaoStatus(keys,LessonStatus.UN_START.getStatus());
        if (records.size()>0) {
            kexiaoRecordDao.batchInsertRecords(records);
            updateSignupCourseLessonKexiaoStatus(records);
        }
    }

    @Transactional
    public void deleteByDelLessonIds(Long orgId, Collection<KexiaoChangeLog> logs){
        log.info("[Kexiao] compute by startTime.orgId={},logs={}", orgId, logs);
        Gson gson = new Gson();
        for (KexiaoChangeLog changeLog : logs) {
            ClassLessonInfo info = gson.fromJson(changeLog.getInfo(), ClassLessonInfo.class);
            Map<String,Object> condition = new HashMap<>();
            condition.put("lessonId", info.getLessonIds());
            kexiaoRecordDao.delByCondition(condition);
        }
    }

    @Transactional
    public void deleteByDelStuLessonIds(Long orgId, Collection<KexiaoChangeLog> logs){
        log.info("[Kexiao] compute by startTime.orgId={},logs={}", orgId, logs);
        Gson gson = new Gson();
        for (KexiaoChangeLog changeLog : logs) {
            StudentLessonInfo info = gson.fromJson(changeLog.getInfo(), StudentLessonInfo.class);
            Map<String,Object> condition = new HashMap<>();
            condition.put("studentLessonId",info.getStuLessonIds());
            kexiaoRecordDao.delByCondition(condition);
        }
    }

    public void deleteByCancelSign(Long orgId, Collection<KexiaoChangeLog> logs){
        log.info("[Kexiao] compute by startTime.orgId={},logs={}", orgId, logs);
        Gson gson = new Gson();
        for (KexiaoChangeLog changeLog : logs) {
            StudentLessonInfo info = gson.fromJson(changeLog.getInfo(), StudentLessonInfo.class);
            Map<String,Object> condition = new HashMap<>();
            condition.put("studentLessonId",info.getStuLessonIds());
            kexiaoRecordDao.delByCondition(condition);
        }
    }


    private OrgStudentKexiaoRecord constructKexiaoRecord(OrgStudentLesson studentLesson, OrgClassLesson classLesson,
                                                         OrgSignupCourseLesson signUpLesson, Date markTime) {
        log.info("[Kexiao] constructKexiaoRecord.studentLesson={},classLesson={},signUpLesson={},markTime={}", studentLesson, classLesson,signUpLesson,markTime);
        OrgStudentKexiaoRecord record = new OrgStudentKexiaoRecord();
        record.setOrgId(studentLesson.getOrgId());
        record.setStudentLessonId(studentLesson.getId());
        record.setCourseId(signUpLesson.getCourseId());
        record.setDelStatus(0);
        record.setLessonId(signUpLesson.getLessonId());
        record.setAmount(signUpLesson.getAmount());
        record.setCreateTime(new Date());
        record.setLessonDuration(signUpLesson.getLessonDuration());
        record.setLessonType(studentLesson.getLessonType());
        record.setMarkTime(markTime);
        record.setStartTime(classLesson.getStartTime());
        record.setUserId(studentLesson.getUserId());
        record.setClassId(signUpLesson.getClassId());
        record.setChargeUnit(signUpLesson.getChargeUnit());
        record.setKexiaoDuration(studentLesson.getKexiaoDuration());
        return record;
    }

    private String getKey(OrgStudentLesson studentLesson){
        return studentLesson.getUserId()+"_"+studentLesson.getLessonId();
    }

    private String getKey(OrgLessonSign sign){
        return sign.getUserId()+"_"+sign.getLessonId();
    }

    @Data
    private static class BatchData{
        Map<Long,Long> classCourseMap = Maps.newHashMap();
        Map<Long,OrgCourse> classMap = Maps.newHashMap();
        Map<Long, OrgCourseConsumeRule> ruleMap = Maps.newHashMap();
        Map<Long, OrgClassLesson> classLessonMap = Maps.newHashMap();
        Map<String, OrgSignupCourseLesson> signUpLessonsMap = Maps.newHashMap();

        public OrgCourse getOrgClass(Long classId){
            return classMap.get(classId);
        }

        public OrgCourseConsumeRule getRuleByClassId(Long classId){
            Long courseId = getClassCourseMap().get(classId);
            return ruleMap.get(courseId);
        }

        public OrgClassLesson getClassLesson(Long lessonId){
            return classLessonMap.get(lessonId);
        }

        public OrgSignupCourseLesson getSignUpCourseLesson(String key){
            return signUpLessonsMap.get(key);
        }
    }

    private BatchData getBatchData(Long orgId,List<OrgStudentLesson> studentLessons){

        BatchData data = new BatchData();
        if(studentLessons==null || studentLessons.size()<1){
            return data;
        }
        Map<Long,Long> classCourseMap = Maps.newHashMap();
        Map<Long,OrgCourse> classMap = Maps.newHashMap();
        Map<Long, OrgCourseConsumeRule> ruleMap;
        Map<Long, OrgClassLesson> classLessonMap;
        Map<String, OrgSignupCourseLesson> signUpLessonsMap;

        Set<Long> classLessonIds = new HashSet<>();
        Set<Long> userIds = new HashSet<>();
        for (OrgStudentLesson studentLesson : studentLessons) {
            classLessonIds.add(studentLesson.getLessonId());
            userIds.add(studentLesson.getUserId());
        }
        List<OrgClassLesson> classLessons = classLessonDao.getByIds(classLessonIds);
        classLessonMap = CollectionHelper.toKeyMap(classLessons, "id");
        signUpLessonsMap = signupCourseLessonDao.selectLessonInfoMap(orgId,classLessonIds,userIds);

        List<Long> classIds = ListUtil.toKeyList(classLessons, "courseId", OrgClassLesson.class);
        List<OrgCourse> courses = courseDao.getByIds(classIds, "id", "parentId", "isCourse", "isClass", "courseType");
        for (OrgCourse course:courses){
            if(course.getIsClass()== CourseTypeEnum.IS_CLASS_TRUE.getCode() && course.getIsCourse()==CourseTypeEnum.IS_COURSE_FALSE.getCode()){
                classCourseMap.put(course.getId(),course.getParentId());
            }else {
                classCourseMap.put(course.getId(), course.getId());
            }
            classMap.put(course.getId(),course);
        }
        List<OrgCourseConsumeRule> rules = courseConsumeRuleDao.queryConsumRuleListByCourseIds(orgId, classCourseMap.values());
        ruleMap = CollectionHelper.toKeyMap(rules, "courseId");

        data.setClassLessonMap(classLessonMap);
        data.setSignUpLessonsMap(signUpLessonsMap);
        data.setClassMap(classMap);
        data.setClassCourseMap(classCourseMap);
        data.setRuleMap(ruleMap);

        return data;
    }

    private Map<Long,Long> getClassIdToCourseId(List<OrgCourse> courses){
        Map<Long,Long> classCourseMap = Maps.newHashMap();
        for (OrgCourse course:courses){
            if(course.getIsClass()== CourseTypeEnum.IS_CLASS_TRUE.getCode() && course.getIsCourse()==CourseTypeEnum.IS_COURSE_FALSE.getCode()){
                classCourseMap.put(course.getId(),course.getParentId());
            }else {
                classCourseMap.put(course.getId(), course.getId());
            }
        }
        return classCourseMap;
    }

    private void updateSignupCourseLessonKexiaoStatus(Collection<UnionKey> unionKeyList,int kexiaoStatus){
        if(unionKeyList==null || unionKeyList.size()<1){
            return;
        }
        Map<Long,List<Long>> lessonIdToUserIdList = new HashMap<>();
        for (UnionKey key:unionKeyList){
            List<Long> userIds = lessonIdToUserIdList.get(key.getLessonId());
            if(userIds==null){
                userIds = new ArrayList<>();
                lessonIdToUserIdList.put(key.getLessonId(),userIds);
            }
            userIds.add(key.getUserId());
        }
        Set<Long> keySet = lessonIdToUserIdList.keySet();
        for (Long key:keySet){
            List<Long> userIds = lessonIdToUserIdList.get(key);
            signupCourseLessonDao.batchUpdateKexiaoStatus(Arrays.asList(key),userIds,kexiaoStatus);
        }
    }

    private void updateSignupCourseLessonKexiaoStatus(List<OrgStudentKexiaoRecord> records){
        if(records!=null){
            Map<Long,Set<Long>> lessonIdToUserIds = new HashMap<>();
            for (OrgStudentKexiaoRecord record:records){
                Set<Long> userIds = lessonIdToUserIds.get(record.getLessonId());
                if(userIds==null){
                    userIds = new HashSet<>();
                    lessonIdToUserIds.put(record.getLessonId(),userIds);
                }
                userIds.add(record.getUserId());
            }
            Set<Long> keySet = lessonIdToUserIds.keySet();
            log.info("[Kexiao] lessonIds={},lessonIdToUserIds={}",keySet,lessonIdToUserIds);
            for (Long lessonId:keySet){
                Set<Long> userIds = lessonIdToUserIds.get(lessonId);
                if(userIds!=null && userIds.size()>0) {
                    signupCourseLessonDao.batchUpdateKexiaoStatus(Arrays.asList(lessonId), userIds, LessonStatus.FINISHED.getStatus());
                }
            }
        }
    }

    private void delSignupCourseLessons(Collection<OrgStudentLesson> stuLessons){
        log.info("[Kexiao] Del student lessons={}",stuLessons);
        if(stuLessons==null || stuLessons.isEmpty()){
            return;
        }
        Map<Long,List<Long>> lessonToUserIds = new HashMap<>();
        for (OrgStudentLesson lesson:stuLessons){
            List<Long> userIds = lessonToUserIds.get(lesson.getLessonId());
            if(userIds==null){
                userIds = new ArrayList<>();
                lessonToUserIds.put(lesson.getLessonId(),userIds);
            }
            userIds.add(lesson.getUserId());
        }

        Set<Long> keySet = lessonToUserIds.keySet();
        for (Long lessonId:keySet){
            List<Long> userIds = lessonToUserIds.get(lessonId);
            signupCourseLessonDao.batchDelStudentsFromLesson(lessonId,userIds);
        }
    }

    @Data
    class UnionKey{
        private long lessonId;
        private long userId;

        public UnionKey() {
        }

        public UnionKey(long lessonId, long userId) {
            this.lessonId = lessonId;
            this.userId = userId;
        }
    }
}
