
/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2016 All Rights Reserved.
 */

package com.baijia.tianxiao.biz.erp.service.impl;

import java.io.IOException;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.regex.Pattern;

import javax.annotation.Resource;

import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataAccessException;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import com.baijia.tianxiao.biz.erp.constant.UploadFileStatus;
import com.baijia.tianxiao.biz.erp.service.ErpUploadService;
import com.baijia.tianxiao.biz.erp.util.ErpUploadFileUtil;
import com.baijia.tianxiao.biz.erp.util.ErpUploadFileUtil.StudentFormat;
import com.baijia.tianxiao.biz.erp.vo.ImportCourse;
import com.baijia.tianxiao.biz.erp.vo.ImportStudent;
import com.baijia.tianxiao.constants.OrgCourseStatus;
import com.baijia.tianxiao.dal.org.constant.DeleteStatus;
import com.baijia.tianxiao.dal.org.dao.OrgClassLessonDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentLessonDao;
import com.baijia.tianxiao.dal.org.po.OrgClassLesson;
import com.baijia.tianxiao.dal.org.po.OrgCourse;
import com.baijia.tianxiao.dal.org.po.OrgStudentCourse;
import com.baijia.tianxiao.dal.org.po.OrgStudentLesson;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.filter.TianxiaoPCContext;
import com.baijia.tianxiao.sal.course.dto.OrgCourseInfoDto;
import com.baijia.tianxiao.sal.course.service.OrgCourseService;
import com.baijia.tianxiao.sal.course.service.impl.ImportOrgCourseProcessServiceImpl.CourseFormat;
import com.baijia.tianxiao.sal.student.api.OrgStudentCourseService;
import com.baijia.tianxiao.sal.student.api.OrgStudentService;
import com.baijia.tianxiao.sal.student.dto.StudentInfoDto;
import com.baijia.tianxiao.sal.student.dto.response.OrgStudentAddresponseDto;
import com.baijia.tianxiao.sal.upload.service.UploadFileReaderService;
import com.baijia.tianxiao.util.json.JacksonUtil;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import lombok.extern.slf4j.Slf4j;

/**
 * @title ErpUploadServiceImpl
 * @desc O(∩_∩)O~
 * @author caoliang
 * @date 2016年7月5日
 * @version 1.0
 */
@Slf4j
@Service
public class ErpUploadServiceImpl implements ErpUploadService {

    private static final String TASK_SUFFIX = "_task_tx";
    private static final String STATUS_SUFFIX = "_status_tx";
    private static final String VALID_LIST_SUFFIX = "_valid_list_tx";
    private static final String INVALID_LIST_SUFFIX = "_invalid_list_tx";
    private static final String VALID_LIST_SIZE_SUFFIX = "_valid_list_size_tx";
    private static final String INVALID_LIST_SIZE_SUFFIX = "_invalid_list_size_tx";
    private static final Long EXPIRE = 60 * 120L;

    @Autowired
    private StringRedisTemplate redisTemplate;

    @Autowired(required = false)
    private ThreadPoolTaskExecutor threadPoolTaskExecutor;
    @Resource
    private OrgCourseService orgCourseService;
    @Autowired
    private OrgClassLessonDao orgClassLessonDao;
    @Autowired
    private OrgStudentService orgStudentService;
    @Autowired
    private OrgStudentCourseService orgStudentCourseService;
    @Autowired
    private OrgStudentLessonDao orgStudentLessonDao;

    /**
     * 校验excel是否为系统模板 校验数据格式 正确数据存redis 错误数据存redis
     * 
     * @throws IOException
     */
    @Override
    public String uploadCourse(MultipartFile file) throws Exception {
        final Integer orgId = TianxiaoPCContext.getOrgId();
        boolean isProcessing = redisTemplate.execute(new RedisCallback<Boolean>() {

            @Override
            public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                return connection.exists(getTaskKey(orgId).getBytes());
            }
        });

        if (isProcessing) {
            throw new BussinessException(CommonErrorCode.PERMISSION_DENY, "上一个任务扔在处理中，请等待完成后，再次上传");
        }
        final String taskId = UUID.randomUUID().toString();
        final UploadFileReaderService uploadFileService = ErpUploadFileUtil.getUploadFileService(file);
        List<String> templateHeader = ErpUploadFileUtil.getTemplateHeader("import_course.xlsx");
        final List<String> headers = ErpUploadFileUtil.tranHeader(uploadFileService.readData());
        if (!headers.contains("错误原因")) {
            headers.add("错误原因");
        }
        // 和系统提供header一致
        if (!CollectionUtils.isEqualCollection(templateHeader, headers)) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "请使用系统模板");
        }
        // 校验过程丢线程池
        threadPoolTaskExecutor.execute(new Runnable() {

            @Override
            public void run() {
                try {
                    // 初始化任务状态
                    redisTemplate.execute(new RedisCallback<Boolean>() {

                        @Override
                        public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                            connection.setEx(getStatusKey(taskId).getBytes(), EXPIRE,
                                UploadFileStatus.PROCESSING.getCode().toString().getBytes());
                            return true;
                        }
                    });
                    // 一行一行校验数据，正确、错误都存
                    final List<ImportCourse> errList = Lists.newArrayList();
                    final List<ImportCourse> validList = Lists.newArrayList();
                    Object[] lineData = null;
                    headers.remove("错误原因");
                    try {
                        while ((lineData = uploadFileService.readData()) != null) {
                            boolean invalid = false;
                            ImportCourse importCourse = new ImportCourse();
                            for (int i = 0; i < headers.size(); i++) {
                                Object value = lineData[i];
                                CourseFormat courseFormat = ErpUploadFileUtil.colMap.get(headers.get(i));
                                if (value == null || StringUtils.isBlank(value.toString().trim())) {
                                    invalid = true;
                                    BeanUtils.setProperty(importCourse, "errReason", "【" + headers.get(i) + "】不能为空");
                                } else {
                                    BeanUtils.setProperty(importCourse, courseFormat.getFieldName(), value.toString().trim());
                                    // 取值不合法
                                    if (!Pattern.matches(courseFormat.getFormatRegex(), value.toString())) {
                                        BeanUtils.setProperty(importCourse, "errReason", courseFormat.getFormatMsg());
                                        invalid = true;
                                    }
                                }
                            }
                            if (invalid) {
                                errList.add(importCourse);
                            } else {
                                validList.add(importCourse);
                            }
                        }
                    } catch (Exception e) {
                        log.error("valid upload course error~!", e);
                        redisTemplate.execute(new RedisCallback<Boolean>() {

                            @Override
                            public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                                connection.setEx(getStatusKey(taskId).getBytes(), EXPIRE,
                                    UploadFileStatus.ERROR.getCode().toString().getBytes());
                                return true;
                            }
                        });
                        return;
                    }

                    // 把正确的和错误的数据，存入缓存
                    redisTemplate.execute(new RedisCallback<Boolean>() {

                        @Override
                        public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                            if (CollectionUtils.isNotEmpty(validList)) {
                                for (ImportCourse importCourse : validList) {
                                    redisTemplate.opsForList().leftPush(getValidDataKey(taskId),
                                        JacksonUtil.obj2Str(importCourse));
                                    // connection.lPush(getValidDataKey(taskId).getBytes(),
                                    // JacksonUtil.obj2Str(importCourse).getBytes());
                                }
                                connection.expire(getValidDataKey(taskId).getBytes(), EXPIRE);
                            }
                            if (CollectionUtils.isNotEmpty(errList)) {
                                for (ImportCourse importCourse : errList) {
                                    redisTemplate.opsForList().leftPush(getErrDataKey(taskId),
                                        JacksonUtil.obj2Str(importCourse));
                                    // connection.lPush(getErrDataKey(taskId).getBytes(),
                                    // JacksonUtil.obj2Str(importCourse).getBytes());
                                }
                                connection.expire(getErrDataKey(taskId).getBytes(), EXPIRE);
                            }
                            return true;
                        }
                    });

                    // 设置任务完成
                    redisTemplate.execute(new RedisCallback<Boolean>() {

                        @Override
                        public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                            connection.setEx(getStatusKey(taskId).getBytes(), EXPIRE,
                                UploadFileStatus.PROCESSED.getCode().toString().getBytes());
                            connection.setEx(getErrDataSizeKey(taskId).getBytes(), EXPIRE,
                                String.valueOf(errList.size()).getBytes());
                            connection.setEx(getValidDataSizeKey(taskId).getBytes(), EXPIRE,
                                String.valueOf(validList.size()).getBytes());
                            return true;
                        }
                    });
                } catch (Throwable t) {
                    log.error("upload course error!", t);
                    redisTemplate.execute(new RedisCallback<Boolean>() {

                        @Override
                        public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                            connection.del(getTaskKey(orgId).getBytes());
                            return true;
                        }
                    });
                }
            }
        });

        return taskId;
    }

    private final String getTaskKey(Integer orgId) {
        return orgId + TASK_SUFFIX;
    }

    private final String getStatusKey(String taskId) {
        return taskId + STATUS_SUFFIX;
    }

    private final String getErrDataKey(String taskId) {
        return taskId + INVALID_LIST_SUFFIX;
    }

    private final String getValidDataKey(String taskId) {
        return taskId + VALID_LIST_SUFFIX;
    }

    private final String getErrDataSizeKey(String taskId) {
        return taskId + INVALID_LIST_SIZE_SUFFIX;
    }

    private final String getValidDataSizeKey(String taskId) {
        return taskId + VALID_LIST_SIZE_SUFFIX;
    }

    @Override
    public List<ImportCourse> getValidCourseList(final String taskId) {
        return redisTemplate.execute(new RedisCallback<List<ImportCourse>>() {

            @Override
            public List<ImportCourse> doInRedis(RedisConnection connection) throws DataAccessException {
                List<ImportCourse> data = Lists.newArrayList();
                Long size = redisTemplate.opsForList().size(getValidDataKey(taskId));
                if (size != null && size > 0) {
                    for (int i = 0; i < size; i++) {
                        try {
                            data.add(JacksonUtil.str2Obj(redisTemplate.opsForList().rightPop(getValidDataKey(taskId)),
                                ImportCourse.class));
                        } catch (Exception e) {
                            e.printStackTrace();
                            log.error("uploadcourse getValidList error!", e);
                        }
                    }
                }
                if (CollectionUtils.isNotEmpty(data)) {
                    for (ImportCourse importCourse : data) {
                        redisTemplate.opsForList().leftPush(getValidDataKey(taskId), JacksonUtil.obj2Str(importCourse));
                    }
                }
                connection.expire(getValidDataKey(taskId).getBytes(), EXPIRE);
                return data;

            }
        });
    }

    @Override
    public List<ImportCourse> getInValidCourseList(final String taskId) {
        return redisTemplate.execute(new RedisCallback<List<ImportCourse>>() {

            @Override
            public List<ImportCourse> doInRedis(RedisConnection connection) throws DataAccessException {
                List<ImportCourse> data = Lists.newArrayList();
                Long size = redisTemplate.opsForList().size(getErrDataKey(taskId));
                if (size != null && size > 0) {
                    for (int i = 0; i < size; i++) {
                        try {
                            data.add(JacksonUtil.str2Obj(redisTemplate.opsForList().rightPop(getErrDataKey(taskId)),
                                ImportCourse.class));
                        } catch (Exception e) {
                            e.printStackTrace();
                            log.error("uploadcourse getValidList error!", e);
                        }
                    }
                }
                if (CollectionUtils.isNotEmpty(data)) {
                    for (ImportCourse importCourse : data) {
                        redisTemplate.opsForList().leftPush(getErrDataKey(taskId), JacksonUtil.obj2Str(importCourse));
                    }
                }
                connection.expire(getValidDataKey(taskId).getBytes(), EXPIRE);
                return data;

            }
        });
    }

    @Override
    public Map<String, String> getTaskStatus(final String taskId) {
        return redisTemplate.execute(new RedisCallback<Map<String, String>>() {

            @Override
            public Map<String, String> doInRedis(RedisConnection connection) throws DataAccessException {
                byte[] statusBytes = connection.get(getStatusKey(taskId).getBytes());
                byte[] invalidDataSizeBytes = connection.get(getErrDataSizeKey(taskId).getBytes());
                byte[] validDataSizeBytes = connection.get(getValidDataSizeKey(taskId).getBytes());
                String status = UploadFileStatus.PROCESSING.getCode().toString();
                Map<String, String> data = Maps.newHashMap();
                if (statusBytes != null) {
                    status = redisTemplate.getStringSerializer().deserialize(statusBytes);
                }
                if (invalidDataSizeBytes != null) {
                    data.put("invalidDataSize", redisTemplate.getStringSerializer().deserialize(invalidDataSizeBytes));
                }
                if (validDataSizeBytes != null) {
                    data.put("validDataSize", redisTemplate.getStringSerializer().deserialize(validDataSizeBytes));
                }
                data.put("status", status);
                data.put("taskId", taskId);
                return data;
            }
        });
    }

    @Override
    public void saveCourse(final String taskId) {
        final List<ImportCourse> importCourses = getValidCourseList(taskId);
        final Integer orgId = TianxiaoPCContext.getOrgId();
        if (orgId == null) {
            log.error("saveCourse error!orgId is null");
            return;
        }
        threadPoolTaskExecutor.execute(new Runnable() {

            @Override
            public void run() {
                try {
                    redisTemplate.execute(new RedisCallback<Boolean>() {

                        @Override
                        public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                            connection.setEx(getStatusKey(taskId).getBytes(), EXPIRE,
                                UploadFileStatus.PROCESSING.getCode().toString().getBytes());
                            connection.setEx(getTaskKey(orgId).getBytes(), EXPIRE, "1".getBytes());
                            return true;
                        }
                    });
                    if (CollectionUtils.isNotEmpty(importCourses)) {
                        for (final ImportCourse importCourse : importCourses) {
                            final OrgCourseInfoDto course = buildCourse(importCourse);
                            try {
                                if (course == null) {
                                    throw new Exception("非法值");
                                }
                                orgCourseService.saveOrUpdateCourse(orgId.longValue(), course, false, true);
                            } catch (Exception e) {
                                log.error("save importCourse error!,importCourse:{}", importCourse, e);
                                // 错误追加
                                redisTemplate.execute(new RedisCallback<Boolean>() {

                                    @Override
                                    public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                                        importCourse.setErrReason("系统错误，无法插入，请联系天校工作人员为您排查");
                                        connection.lPush(getErrDataKey(taskId).getBytes(),
                                            JacksonUtil.obj2Str(importCourse).getBytes());
                                        connection.expire(getErrDataKey(taskId).getBytes(), EXPIRE);
                                        connection.incr(getErrDataSizeKey(taskId).getBytes());
                                        connection.expire(getErrDataSizeKey(taskId).getBytes(), EXPIRE);
                                        connection.decr(getValidDataSizeKey(taskId).getBytes());
                                        connection.expire(getValidDataSizeKey(taskId).getBytes(), EXPIRE);
                                        return true;
                                    }
                                });
                            }
                        }
                    }

                    redisTemplate.execute(new RedisCallback<Boolean>() {

                        @Override
                        public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                            connection.setEx(getStatusKey(taskId).getBytes(), EXPIRE,
                                UploadFileStatus.PROCESSED.getCode().toString().getBytes());
                            return true;
                        }
                    });
                } catch (Throwable t) {
                    log.error("save couers error!", t);
                } finally {
                    redisTemplate.execute(new RedisCallback<Boolean>() {

                        @Override
                        public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                            connection.del(getTaskKey(orgId).getBytes());
                            return true;
                        }
                    });
                }

            }
        });

    }

    private OrgCourseInfoDto buildCourse(ImportCourse importCourse) {
        OrgCourseInfoDto course = new OrgCourseInfoDto();
        if (importCourse.getCourseName() == null || importCourse.getCoursePrice() == null
            || importCourse.getFreq() == null || importCourse.getMaxStudent() == null) {
            return null;
        }
        course.setCourseName(importCourse.getCourseName().toString());
        course.setCoursePrice(Double.valueOf(importCourse.getCoursePrice().toString()));
        course.setFreq(Integer.valueOf(importCourse.getFreq().toString()));
        course.setMaxStudent(Integer.valueOf(importCourse.getMaxStudent().toString()));
        // 上架
        course.setStatus(OrgCourseStatus.IN_PROGRESS.getCode());
        return course;
    }

    @Override
    public String uploadStudent(MultipartFile file) throws Exception {
        final Integer orgId = TianxiaoPCContext.getOrgId();
        boolean isProcessing = redisTemplate.execute(new RedisCallback<Boolean>() {

            @Override
            public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                return connection.exists(getTaskKey(orgId).getBytes());
            }
        });

        if (isProcessing) {
            throw new BussinessException(CommonErrorCode.PERMISSION_DENY, "上一个任务扔在处理中，请等待完成后，再次上传");
        }
        final String taskId = UUID.randomUUID().toString();
        final UploadFileReaderService uploadFileService = ErpUploadFileUtil.getUploadFileService(file);
        List<String> templateHeader = ErpUploadFileUtil.getTemplateHeader("import_student.xlsx");
        final List<String> headers = ErpUploadFileUtil.tranHeader(uploadFileService.readData());
        if (!headers.contains("错误原因")) {
            headers.add("错误原因");
        }
        // 和系统提供header一致
        if (!CollectionUtils.isEqualCollection(templateHeader, headers)) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "请使用系统模板");
        }
        // 校验过程丢线程池
        threadPoolTaskExecutor.execute(new Runnable() {

            @Override
            public void run() {
                try {
                    // 初始化任务状态
                    redisTemplate.execute(new RedisCallback<Boolean>() {

                        @Override
                        public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                            connection.setEx(getStatusKey(taskId).getBytes(), EXPIRE,
                                UploadFileStatus.PROCESSING.getCode().toString().getBytes());
                            return true;
                        }
                    });
                    // 一行一行校验数据，正确、错误都存
                    final List<ImportStudent> errList = Lists.newArrayList();
                    final List<ImportStudent> validList = Lists.newArrayList();
                    final List<String> validMobiles = Lists.newArrayList();
                    final List<String> validCourseNames = Lists.newArrayList();
                    final Map<String, ImportStudent> mobileMap = Maps.newHashMap();
                    Object[] lineData = null;
                    headers.remove("错误原因");
                    try {
                        while ((lineData = uploadFileService.readData()) != null) {
                            boolean invalid = false;
                            ImportStudent importStudent = new ImportStudent();
                            for (int i = 0; i < headers.size(); i++) {
                                Object value = lineData[i];
                                StudentFormat studentFormat = ErpUploadFileUtil.studentMap.get(headers.get(i));
                                if (studentFormat.getFieldName().equals("lessonCount")
                                    || studentFormat.getFieldName().equals("insertCount")) {
                                    if (value != null && StringUtils.isNoneBlank(value.toString().trim())) {
                                        if (!Pattern.matches(studentFormat.getFormatRegex(), value.toString())) {
                                            BeanUtils.setProperty(importStudent, "errReason",
                                                studentFormat.getFormatMsg());
                                            invalid = true;
                                        }
                                        BeanUtils.setProperty(importStudent, studentFormat.getFieldName(),
                                            value.toString().trim());
                                    }

                                } else if (value == null || StringUtils.isBlank(value.toString().trim())) {
                                    invalid = true;
                                    BeanUtils.setProperty(importStudent, "errReason", "【" + headers.get(i) + "】不能为空");
                                } else {
                                    BeanUtils.setProperty(importStudent, studentFormat.getFieldName(),
                                        value.toString().trim());
                                    // 取值不合法
                                    if (!Pattern.matches(studentFormat.getFormatRegex(), value.toString().trim())) {
                                        BeanUtils.setProperty(importStudent, "errReason", studentFormat.getFormatMsg());
                                        invalid = true;
                                    }
                                }
                                if (!invalid) {
                                    if (importStudent.getStudentMobile() != null
                                        && mobileMap.containsKey(importStudent.getStudentMobile())) {
                                        ImportStudent exist =
                                            mobileMap.get(importStudent.getStudentMobile().toString());
                                        if (!exist.getStudentName().toString()
                                            .equals(importStudent.getStudentName().toString())) {
                                            invalid = true;
                                            importStudent.setErrReason("与表格内其他行数据【手机号】相同，但姓名不同，无法导入");
                                        } else {
                                            if (exist.getCourseName() != null && importStudent.getCourseName() != null
                                                && exist.getCourseName().toString()
                                                    .equals(importStudent.getCourseName().toString())) {
                                                importStudent.setErrReason("该行与表格内其他行数据重复，无法导入");
                                                invalid = true;
                                            } else {
                                            }
                                        }
                                    }
                                }

                            }

                            if (invalid) {
                                errList.add(importStudent);
                            } else {
                                validList.add(importStudent);
                                mobileMap.put(importStudent.getStudentMobile().toString(), importStudent);
                                validMobiles.add(importStudent.getStudentMobile().toString());
                            }
                        }
                        // if validlist不为空
                        // 拿机构所有学生，放Map<mobile,name>
                        // 第一次遍历validList
                        // if 手机号在map里，但name不一致的，放errList里,remove
                        if (CollectionUtils.isNotEmpty(validList)) {
                            List<StudentInfoDto> tempDtos =
                                orgStudentService.listOrgStudentByMobiles(orgId.longValue(), validMobiles);
                            if (CollectionUtils.isNotEmpty(tempDtos)) {
                                Map<String, String> mobileNameMap = Maps.newHashMap();
                                for (StudentInfoDto dto : tempDtos) {
                                    mobileNameMap.put(dto.getMobile(), dto.getName());
                                }
                                Iterator<ImportStudent> iterator = validList.iterator();
                                while (iterator.hasNext()) {
                                    ImportStudent importStudent = iterator.next();
                                    if (mobileNameMap.containsKey(importStudent.getStudentMobile())) {
                                        if (!importStudent.getStudentName()
                                            .equals(mobileNameMap.get(importStudent.getStudentMobile()))) {
                                            importStudent.setErrReason("与系统内学员【手机号】相同，但姓名不同，无法导入");
                                            errList.add(importStudent);
                                            iterator.remove();
                                        } else {
                                            validCourseNames.add(importStudent.getCourseName());
                                        }
                                    }
                                    else{
                                        validCourseNames.add(importStudent.getCourseName());
                                    }
                                }
                            }
                        }
                        // if validlist不为空
                        // 把courseName存list，按照list取出所有OrgCourse
                        // 遍历List<OrgCourse>，得到 <CourseName,id> existCourse,
                        // 得到list<id>，拿到<courseid,List<OrgClassLesson>.size()>
                        // 遍历validlist，
                        // if 不在existCourse key中的，放errlist
                        // else 通过name拿到id，如果<courseid,List<OrgClassLesson>.size()>中没有，则没排课，放errlist。validlist.remove
                        if (CollectionUtils.isNotEmpty(validList)) {
                            Map<String, Long> courseNameIdMap = Maps.newHashMap();
                            Map<String, Integer> courseNameCountMap = Maps.newHashMap();
                            List<Long> courseIds = Lists.newArrayList();
                            List<OrgCourse> orgCourses =
                                orgCourseService.getAllOrgCoursesByNames(orgId.longValue(), validCourseNames);
                            if (CollectionUtils.isNotEmpty(orgCourses)) {
                                for (OrgCourse orgCourse : orgCourses) {
                                    courseNameIdMap.put(orgCourse.getName(), orgCourse.getId());
                                    if (courseNameCountMap.containsKey(orgCourse.getName())) {
                                        Integer count = courseNameCountMap.get(orgCourse.getName());
                                        courseNameCountMap.put(orgCourse.getName(), ++count);
                                    } else {
                                        courseNameCountMap.put(orgCourse.getName(), 1);
                                    }
                                    courseIds.add(orgCourse.getId());
                                }
                            }
                            Map<Long, Integer> idCountMap = Maps.newHashMap();
                            if (CollectionUtils.isNotEmpty(courseIds)) {
                                idCountMap = orgClassLessonDao.getCourseLessonCount(orgId.longValue(), courseIds, null,
                                    null, null);
                            }
                            Iterator<ImportStudent> iterator = validList.iterator();
                            while (iterator.hasNext()) {
                                ImportStudent importStudent = iterator.next();
                                if (!courseNameIdMap.containsKey(importStudent.getCourseName())) {
                                    importStudent.setErrReason("【班级名称】在系统不存在，请先手动录入班级，再批导班级学员");
                                    errList.add(importStudent);
                                    iterator.remove();
                                } else {
                                    if (courseNameCountMap.get(importStudent.getCourseName()) > 1) {
                                        importStudent.setErrReason("【班级名称】在系统有多个重名班级，重名班级的学员请手动录入");
                                        errList.add(importStudent);
                                        iterator.remove();
                                    } else {
                                        // 不排课也让导入
                                        // Long courseId = courseNameIdMap.get(importStudent.getCourseName());
                                        // if (!idCountMap.containsKey(courseId)) {
                                        // importStudent.setErrReason("【班级名称】对应班级尚未排课，请先排课");
                                        // errList.add(importStudent);
                                        // iterator.remove();
                                        // }
                                    }
                                }
                            }
                        }

                    } catch (Exception e) {
                        log.error("valid upload student error~!", e);
                        redisTemplate.execute(new RedisCallback<Boolean>() {

                            @Override
                            public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                                connection.setEx(getStatusKey(taskId).getBytes(), EXPIRE,
                                    UploadFileStatus.ERROR.getCode().toString().getBytes());
                                return true;
                            }
                        });
                        return;
                    }
                    log.debug("*********************validstudentlist:{}", validList);
                    log.debug("*********************invalidstudentlist:{}", errList);
                    // 把正确的和错误的数据，存入缓存
                    redisTemplate.execute(new RedisCallback<Boolean>() {

                        @Override
                        public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                            if (CollectionUtils.isNotEmpty(validList)) {
                                for (ImportStudent importStudent : validList) {
                                    redisTemplate.opsForList().leftPush(getValidDataKey(taskId),
                                        JacksonUtil.obj2Str(importStudent));
                                }
                                connection.expire(getValidDataKey(taskId).getBytes(), EXPIRE);
                                log.debug("************save validList:{}", validList);
                            }
                            if (CollectionUtils.isNotEmpty(errList)) {
                                for (ImportStudent importStudent : errList) {
                                    redisTemplate.opsForList().leftPush(getErrDataKey(taskId),
                                        JacksonUtil.obj2Str(importStudent));
                                }
                                connection.expire(getErrDataKey(taskId).getBytes(), EXPIRE);
                                log.debug("************save errList:{}", errList);
                            }
                            return true;
                        }
                    });

                    // 设置任务完成
                    redisTemplate.execute(new RedisCallback<Boolean>() {

                        @Override
                        public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                            connection.setEx(getStatusKey(taskId).getBytes(), EXPIRE,
                                UploadFileStatus.PROCESSED.getCode().toString().getBytes());
                            connection.setEx(getErrDataSizeKey(taskId).getBytes(), EXPIRE,
                                String.valueOf(errList.size()).getBytes());
                            connection.setEx(getValidDataSizeKey(taskId).getBytes(), EXPIRE,
                                String.valueOf(validList.size()).getBytes());
                            return true;
                        }
                    });
                } catch (Throwable t) {
                    log.error("upload student error!", t);
                    redisTemplate.execute(new RedisCallback<Boolean>() {

                        @Override
                        public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                            connection.del(getTaskKey(orgId).getBytes());
                            return true;
                        }
                    });
                }
            }
        });

        return taskId;

    }

    @Override
    public void saveStudents(final String taskId) {
        final List<ImportStudent> importStudents = getValidStudentList(taskId);
        final Integer orgId = TianxiaoPCContext.getOrgId();
        final Integer cascadeId = TianxiaoPCContext.getTXCascadeId();
        if (orgId == null) {
            log.warn("saveStudents error!orgId is null");
            return;
        }
        threadPoolTaskExecutor.execute(new Runnable() {

            @Override
            public void run() {
                try {
                    redisTemplate.execute(new RedisCallback<Boolean>() {

                        @Override
                        public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                            connection.setEx(getStatusKey(taskId).getBytes(), EXPIRE,
                                UploadFileStatus.PROCESSING.getCode().toString().getBytes());
                            connection.setEx(getTaskKey(orgId).getBytes(), EXPIRE, "1".getBytes());
                            return true;
                        }
                    });
                    if (CollectionUtils.isNotEmpty(importStudents)) {
                        for (final ImportStudent importStudent : importStudents) {
                            try {
                                List<OrgCourse> orgCourses = orgCourseService.getAllOrgCourses(orgId.longValue(),
                                    importStudent.getCourseName().toString());
                                OrgCourse orgCourse = orgCourses.get(0);
                                List<OrgClassLesson> orgClassLessons = orgClassLessonDao
                                    .querylessonsByStartTime(new Date(), orgCourse.getId(), orgId.longValue());
                                List<OrgClassLesson> endLessons =
                                    orgClassLessonDao.queryEndedlessons(orgCourse.getId(), orgId.longValue());
                                    // 不排课，也让导入
                                    // if (CollectionUtils.isEmpty(orgClassLessons)) {
                                    // throw new BussinessException(CommonErrorCode.PARAM_ERROR, "当前课程已没有未开始课节");
                                    // }

                                // 不存在orgstudent，创建
                                // 不存在orgstudentclass，创建
                                // 按照逻辑创建org_student_lesson
                                // 系统异常时，把当前数据从validlist删除，再插入errlist，修改错误和正确数目
                                StudentInfoDto orgStudent = null;
                                // orgStudentService.getOrgStudentByMobile(orgId.longValue(),
                                // importStudent.getStudentMobile().toString());
                                List<StudentInfoDto> dtos = orgStudentService.listOrgStudentByMobile(orgId.longValue(),
                                    importStudent.getStudentMobile().toString());
                                if (CollectionUtils.isNotEmpty(dtos)) {
                                    for (StudentInfoDto dto : dtos) {
                                        if (importStudent.getStudentMobile().equals(dto.getMobile())
                                            && importStudent.getStudentName().equals(dto.getName())) {
                                            log.debug("*************can find dto:{}", dto);
                                            orgStudent = dto;
                                        }
                                    }
                                }
                                if (orgStudent == null) {
                                    orgStudent = new StudentInfoDto();
                                    orgStudent.setMobile(importStudent.getStudentMobile().toString());
                                    orgStudent.setName(importStudent.getStudentName().toString());
                                    orgStudent.setAddCascadeId(cascadeId==null?0:cascadeId);
                                    // 时间设置为过去5分钟前,
//                                     orgStudent.setNextRemindTime(System.currentTimeMillis() - 5 * 60 * 1000);
                                    OrgStudentAddresponseDto addDto = orgStudentService
                                        .addStudentIgnoreExistMobile(orgStudent, null, null, orgId.longValue());
                                    orgStudent.setStudentId(addDto.getUserId());
                                }
                                log.debug("**************orgStudent:{}", orgStudent);

                                OrgStudentCourse orgStudentCourse = orgStudentCourseService
                                    .getBySidAndCid(orgId.longValue(), orgStudent.getStudentId(), orgCourse.getId());
                                if (orgStudentCourse == null) {
                                    orgStudentCourse = new OrgStudentCourse();
                                    orgStudentCourse.setCourseId(orgCourse.getId());
                                    orgStudentCourse.setCreateTime(new Date());
                                    orgStudentCourse.setDelStatus(DeleteStatus.NORMAL.getValue());
                                    orgStudentCourse.setOrgId(orgId.longValue());
                                    orgStudentCourse.setStudentMobile(orgStudent.getMobile());
                                    orgStudentCourse.setStudentName(orgStudent.getName());
                                    orgStudentCourse.setUserId(orgStudent.getStudentId());
                                    orgStudentCourse.setStatus(0);
                                    orgStudentCourseService.saveOrgStudentCourse(orgStudentCourse);
                                }
                                if (orgStudentCourse.getDelStatus() == DeleteStatus.DELETED.getValue()) {
                                    orgStudentCourse.setDelStatus(DeleteStatus.NORMAL.getValue());
                                    orgStudentCourseService.updateOrgStudentCourse(orgStudentCourse);
                                }

                                if (orgStudentCourse.getStatus() != 0) {
                                    orgStudentCourse.setStatus(0);
                                    orgStudentCourseService.updateOrgStudentCourse(orgStudentCourse);
                                }
                                
                                // 改为不排课，也让导入
                                if (CollectionUtils.isNotEmpty(orgClassLessons)) {
                                    // 学员从班级下一个待开课课次开排，直至班级结课
                                    // 这里看mrd比较好，pm找周希，rd找曹亮
                                    List<OrgStudentLesson> orgStudentLessons = Lists.newArrayList();
                                    if (importStudent.getInsertCount() == null
                                        && importStudent.getLessonCount() == null) {
                                        for (OrgClassLesson orgClassLesson : orgClassLessons) {
                                            OrgStudentLesson orgStudentLesson = new OrgStudentLesson();
                                            orgStudentLesson.setCreateTime(new Date());
                                            orgStudentLesson.setDelStatus(DeleteStatus.NORMAL.getValue());
                                            orgStudentLesson.setLessonId(orgClassLesson.getId());
                                            orgStudentLesson.setOrgId(orgId.longValue());
                                            orgStudentLesson.setStudentMobile(orgStudent.getMobile());
                                            orgStudentLesson.setStudentName(orgStudent.getName());
                                            orgStudentLesson.setUserId(orgStudent.getStudentId());
                                            orgStudentLessons.add(orgStudentLesson);
                                        }
                                    } else if (importStudent.getLessonCount() != null
                                        && importStudent.getInsertCount() == null) {
                                        // 总课次不为空，插入课次为空
                                        Integer lessonCount =
                                            Integer.valueOf(importStudent.getLessonCount().toString());
                                        if (lessonCount >= orgClassLessons.size()) {
                                            for (OrgClassLesson orgClassLesson : orgClassLessons) {
                                                OrgStudentLesson orgStudentLesson = new OrgStudentLesson();
                                                orgStudentLesson.setCreateTime(new Date());
                                                orgStudentLesson.setDelStatus(DeleteStatus.NORMAL.getValue());
                                                orgStudentLesson.setLessonId(orgClassLesson.getId());
                                                orgStudentLesson.setOrgId(orgId.longValue());
                                                orgStudentLesson.setStudentMobile(orgStudent.getMobile());
                                                orgStudentLesson.setStudentName(orgStudent.getName());
                                                orgStudentLesson.setUserId(orgStudent.getStudentId());
                                                orgStudentLessons.add(orgStudentLesson);
                                            }
                                        } else {
                                            for (int i = 0; i < lessonCount; i++) {
                                                OrgClassLesson orgClassLesson = orgClassLessons.get(i);
                                                OrgStudentLesson orgStudentLesson = new OrgStudentLesson();
                                                orgStudentLesson.setCreateTime(new Date());
                                                orgStudentLesson.setDelStatus(DeleteStatus.NORMAL.getValue());
                                                orgStudentLesson.setLessonId(orgClassLesson.getId());
                                                orgStudentLesson.setOrgId(orgId.longValue());
                                                orgStudentLesson.setStudentMobile(orgStudent.getMobile());
                                                orgStudentLesson.setStudentName(orgStudent.getName());
                                                orgStudentLesson.setUserId(orgStudent.getStudentId());
                                                orgStudentLessons.add(orgStudentLesson);
                                            }
                                        }
                                    } else if (importStudent.getLessonCount() == null
                                        && importStudent.getInsertCount() != null) {
                                        // 总课次为空，插入课次不为空
                                        Integer insertCount =
                                            Integer.valueOf(importStudent.getInsertCount().toString());
                                        if (insertCount > endLessons.size()) {
                                            int point = insertCount - endLessons.size() - 1;
                                            for (; point < orgClassLessons.size(); point++) {
                                                OrgClassLesson orgClassLesson = orgClassLessons.get(point);
                                                OrgStudentLesson orgStudentLesson = new OrgStudentLesson();
                                                orgStudentLesson.setCreateTime(new Date());
                                                orgStudentLesson.setDelStatus(DeleteStatus.NORMAL.getValue());
                                                orgStudentLesson.setLessonId(orgClassLesson.getId());
                                                orgStudentLesson.setOrgId(orgId.longValue());
                                                orgStudentLesson.setStudentMobile(orgStudent.getMobile());
                                                orgStudentLesson.setStudentName(orgStudent.getName());
                                                orgStudentLesson.setUserId(orgStudent.getStudentId());
                                                orgStudentLessons.add(orgStudentLesson);
                                            }
                                        } else {
                                            for (OrgClassLesson orgClassLesson : orgClassLessons) {
                                                OrgStudentLesson orgStudentLesson = new OrgStudentLesson();
                                                orgStudentLesson.setCreateTime(new Date());
                                                orgStudentLesson.setDelStatus(DeleteStatus.NORMAL.getValue());
                                                orgStudentLesson.setLessonId(orgClassLesson.getId());
                                                orgStudentLesson.setOrgId(orgId.longValue());
                                                orgStudentLesson.setStudentMobile(orgStudent.getMobile());
                                                orgStudentLesson.setStudentName(orgStudent.getName());
                                                orgStudentLesson.setUserId(orgStudent.getStudentId());
                                                orgStudentLessons.add(orgStudentLesson);
                                            }
                                        }
                                    } else if (importStudent.getLessonCount() != null
                                        && importStudent.getInsertCount() != null) {
                                        Integer insertCount =
                                            Integer.valueOf(importStudent.getInsertCount().toString());
                                        Integer lessonCount =
                                            Integer.valueOf(importStudent.getLessonCount().toString());
                                        int start = 0;
                                        int end = lessonCount;
                                        if (insertCount > endLessons.size()) {
                                            start = insertCount - endLessons.size() - 1;
                                            end = lessonCount + start;
                                        }

                                        if (end > orgClassLessons.size()) {
                                            end = orgClassLessons.size();
                                        }
                                        for (; start < end; start++) {
                                            OrgClassLesson orgClassLesson = orgClassLessons.get(start);
                                            OrgStudentLesson orgStudentLesson = new OrgStudentLesson();
                                            orgStudentLesson.setCreateTime(new Date());
                                            orgStudentLesson.setDelStatus(DeleteStatus.NORMAL.getValue());
                                            orgStudentLesson.setLessonId(orgClassLesson.getId());
                                            orgStudentLesson.setOrgId(orgId.longValue());
                                            orgStudentLesson.setStudentMobile(orgStudent.getMobile());
                                            orgStudentLesson.setStudentName(orgStudent.getName());
                                            orgStudentLesson.setUserId(orgStudent.getStudentId());
                                            orgStudentLessons.add(orgStudentLesson);
                                        }
                                    }
                                    if (CollectionUtils.isNotEmpty(orgStudentLessons)) {
                                        orgStudentLessonDao.saveAll(orgStudentLessons);
                                    }
                                }
                            } catch (final BussinessException be) {
                                log.warn("save importStudent error!,importStudent:{}", importStudent, be);
                                redisTemplate.execute(new RedisCallback<Boolean>() {

                                    @Override
                                    public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                                        importStudent.setErrReason(be.getMessage());
                                        redisTemplate.opsForList().leftPush(getErrDataKey(taskId),
                                            JacksonUtil.obj2Str(importStudent));
                                        connection.expire(getErrDataKey(taskId).getBytes(), EXPIRE);
                                        connection.incr(getErrDataSizeKey(taskId).getBytes());
                                        connection.expire(getErrDataSizeKey(taskId).getBytes(), EXPIRE);
                                        connection.decr(getValidDataSizeKey(taskId).getBytes());
                                        connection.expire(getValidDataSizeKey(taskId).getBytes(), EXPIRE);
                                        return true;
                                    }
                                });
                            } catch (Exception e) {
                                log.error("save importStudent error!,importStudent:{}", importStudent, e);
                                // 错误追加
                                redisTemplate.execute(new RedisCallback<Boolean>() {

                                    @Override
                                    public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                                        importStudent.setErrReason("系统无法插入此数据，请联系天校工作人员为您排查");
                                        redisTemplate.opsForList().leftPush(getErrDataKey(taskId),
                                            JacksonUtil.obj2Str(importStudent));
                                        connection.expire(getErrDataKey(taskId).getBytes(), EXPIRE);
                                        connection.incr(getErrDataSizeKey(taskId).getBytes());
                                        connection.expire(getErrDataSizeKey(taskId).getBytes(), EXPIRE);
                                        connection.decr(getValidDataSizeKey(taskId).getBytes());
                                        connection.expire(getValidDataSizeKey(taskId).getBytes(), EXPIRE);
                                        return true;
                                    }
                                });
                            }
                        }
                    }

                    redisTemplate.execute(new RedisCallback<Boolean>() {

                        @Override
                        public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                            connection.setEx(getStatusKey(taskId).getBytes(), EXPIRE,
                                UploadFileStatus.PROCESSED.getCode().toString().getBytes());
                            return true;
                        }
                    });
                } catch (Throwable t) {
                    log.error("save couers error!", t);
                } finally {
                    redisTemplate.execute(new RedisCallback<Boolean>() {

                        @Override
                        public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                            connection.del(getTaskKey(orgId).getBytes());
                            return true;
                        }
                    });
                }

            }
        });

    }

    @Override
    public List<ImportStudent> getValidStudentList(final String taskId) {
        return redisTemplate.execute(new RedisCallback<List<ImportStudent>>() {

            @Override
            public List<ImportStudent> doInRedis(RedisConnection connection) throws DataAccessException {
                List<ImportStudent> data = Lists.newArrayList();
                Long size = redisTemplate.opsForList().size(getValidDataKey(taskId));
                if (size != null && size > 0) {
                    for (int i = 0; i < size; i++) {
                        try {
                            data.add(JacksonUtil.str2Obj(redisTemplate.opsForList().rightPop(getValidDataKey(taskId)),
                                ImportStudent.class));
                        } catch (Exception e) {
                            e.printStackTrace();
                            log.error("getValidStudentList getValidList error!", e);
                        }
                    }
                }
                log.debug("getValidList :{}", data);
                if (CollectionUtils.isNotEmpty(data)) {
                    for (ImportStudent importStudent : data) {
                        redisTemplate.opsForList().leftPush(getValidDataKey(taskId),
                            JacksonUtil.obj2Str(importStudent));
                    }
                }
                connection.expire(getValidDataKey(taskId).getBytes(), EXPIRE);
                return data;

            }
        });
    }

    @Override
    public List<ImportStudent> getInValidStudentList(final String taskId) {
        return redisTemplate.execute(new RedisCallback<List<ImportStudent>>() {

            @Override
            public List<ImportStudent> doInRedis(RedisConnection connection) throws DataAccessException {
                List<ImportStudent> data = Lists.newArrayList();
                Long size = redisTemplate.opsForList().size(getErrDataKey(taskId));
                if (size != null && size > 0) {
                    for (int i = 0; i < size; i++) {
                        try {
                            data.add(JacksonUtil.str2Obj(redisTemplate.opsForList().rightPop(getErrDataKey(taskId)),
                                ImportStudent.class));
                        } catch (Exception e) {
                            e.printStackTrace();
                            log.error("getInValidStudentList getValidList error!", e);
                        }
                    }
                }
                log.debug("getInValidStudentList:{}", data);
                if (CollectionUtils.isNotEmpty(data)) {
                    for (ImportStudent importStudent : data) {
                        redisTemplate.opsForList().leftPush(getErrDataKey(taskId), JacksonUtil.obj2Str(importStudent));
                    }
                }
                connection.expire(getValidDataKey(taskId).getBytes(), EXPIRE);
                return data;

            }
        });
    }
}
