/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */

package com.baijia.tianxiao.biz.erp.service.impl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.baijia.tianxiao.biz.erp.dto.response.CourseListInfoDto;
import com.baijia.tianxiao.biz.erp.dto.response.studentCenter.WeiKeListInfoDto;
import com.baijia.tianxiao.biz.erp.service.ErpCourseListService;
import com.baijia.tianxiao.dal.org.dao.OrgAccountDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseDao.OrgCourseListParam;
import com.baijia.tianxiao.dal.org.dao.OrgCourseRoomDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseTeacherDao;
import com.baijia.tianxiao.dal.org.dao.TXCascadeAccountDao;
import com.baijia.tianxiao.dal.org.po.OrgAccount;
import com.baijia.tianxiao.dal.org.po.OrgCourse;
import com.baijia.tianxiao.dal.org.po.TXCascadeAccount;
import com.baijia.tianxiao.dto.FieldShow;
import com.baijia.tianxiao.dto.query.QueryCondition;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.filter.TianxiaoMContext;
import com.baijia.tianxiao.sal.course.dto.request.CourseListRequestDto;
import com.baijia.tianxiao.sal.course.dto.request.WeikeRequestDto;
import com.baijia.tianxiao.sal.course.dto.response.CourseListReponseDto;
import com.baijia.tianxiao.sal.course.dto.response.TeacherResponseDto;
import com.baijia.tianxiao.sal.course.service.CourseStudentService;
import com.baijia.tianxiao.sal.course.service.CourseTeacherService;
import com.baijia.tianxiao.sal.course.service.OrgCourseListService;
import com.baijia.tianxiao.sal.course.service.OrgLessonSignService;
import com.baijia.tianxiao.sal.course.service.OrgCourseListService.CourseListParam;
import com.baijia.tianxiao.sal.organization.constant.CascadeType;
import com.baijia.tianxiao.sal.organization.org.service.TxCascadeCredentialService;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.CollectorUtil;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import lombok.extern.slf4j.Slf4j;

/**
 * @author bjhl
 * @version 1.0
 * @title ErpCourseListServiceImpl
 * @desc TODO
 * @date 2015年12月25日
 */
@Slf4j
@Service
public class ErpCourseListServiceImpl implements ErpCourseListService {

    @Resource
    private OrgAccountDao orgAccountDao;

    @Resource
    private OrgCourseDao orgCourseDao;

    @Resource
    private OrgCourseTeacherDao orgCourseTeacherDao;

    @Resource
    private OrgCourseRoomDao orgCourseRoomDao;

    @Resource
    private OrgCourseListService orgCourseListService;

    @Resource
    private OrgLessonSignService orgLessonSignService;

    @Resource
    private CourseTeacherService courseTeacherService;

    @Resource
    private CourseStudentService courseStudentService;

    @Resource
    private TxCascadeCredentialService txCascadeCredentialService;
    
    @Autowired
    private TXCascadeAccountDao txCascadeAccountDao;

    private Long getOrgNumber(Long orgId) {
        OrgAccount account = this.orgAccountDao.getAccountById(orgId.intValue(), "number");
        if (account == null) {
            return null;
        }
        Integer number = account.getNumber();
        return number != null ? number.longValue() : null;
    }

    private Map<Long, List<String>> getTeachersOfCourses(Long orgId, Collection<Long> courseIds) {
        log.info("orgId={},courseIds={}", orgId, courseIds);
        Map<Long, List<String>> resultMap = Maps.newHashMap();
        Map<Long, List<Long>> courseTeachersMap = orgCourseTeacherDao.getTeacherMap(courseIds);
        Set<Long> teacherIdsSet = Sets.newHashSet();
        for (Long key : courseTeachersMap.keySet()) {
            teacherIdsSet.addAll(courseTeachersMap.get(key));
        }

        List<TeacherResponseDto> teacherDtoList = this.courseTeacherService.getTeachers(teacherIdsSet, orgId);
        Map<Long, String> teacherMap =
            CollectorUtil.collectMap(teacherDtoList, new Function<TeacherResponseDto, Long>() {
                @Override
                public Long apply(TeacherResponseDto from) {
                    return from.getTeacherId();
                }
            }, new Function<TeacherResponseDto, String>() {
                @Override
                public String apply(TeacherResponseDto from) {
                    return from.getTeacherName();
                }
            });

        for (Long courseId : courseIds) {
            List<Long> teacherIds = courseTeachersMap.get(courseId);
            List<String> teacherList = Lists.newArrayList();
            if (CollectionUtils.isNotEmpty(teacherIds)) {
                for (Long tid : teacherIds) {
                    teacherList.add(teacherMap.get(tid));
                }
                resultMap.put(courseId, teacherList);
            }
        }

        log.debug("resultMap={}", resultMap);
        return resultMap;
    }

    @Override
    public List<WeiKeListInfoDto> getWeikeList(Long orgId, WeikeRequestDto requestDto) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "illegal orgId");
        long l = System.currentTimeMillis();
        List<CourseListReponseDto> dtos = this.orgCourseListService.getWeikeList(orgId, requestDto);
        log.debug("************getWeikeList cost={}",System.currentTimeMillis()-l);
        if (CollectionUtils.isEmpty(dtos)) {
            return Collections.emptyList();
        }

        Collection<Long> courseIds = CollectorUtil.collect(dtos, new Function<CourseListReponseDto, Long>() {

            @Override
            public Long apply(CourseListReponseDto input) {
                return input.getOrgCourseId();
            }
        });

        Map<Long, Integer> stuCntMap = this.courseStudentService.getCourseStudentCntMap(orgId, courseIds, null);

        log.debug("org_id={},courseIds={},studentCntMap = {}", orgId, courseIds, stuCntMap);

        List<WeiKeListInfoDto> responseDto = Lists.newArrayList();
        for (CourseListReponseDto courseListReponseDto : dtos) {
            WeiKeListInfoDto dto = new WeiKeListInfoDto();
            buildWeiKeListInfoDto(courseListReponseDto, dto, stuCntMap);
            responseDto.add(dto);
        }
        return responseDto;
    }

    private void buildWeiKeListInfoDto(CourseListReponseDto courseListReponseDto, WeiKeListInfoDto WeiKeListInfoDto,
        Map<Long, Integer> stuCntMap) {
        WeiKeListInfoDto.setCourseName(courseListReponseDto.getCourseName());
        WeiKeListInfoDto.setCourseUrl(courseListReponseDto.getCourseUrl());
        WeiKeListInfoDto.setCoverUrl(courseListReponseDto.getCoverUrl());
        
        WeiKeListInfoDto.setMaxStudent(courseListReponseDto.getMaxStudent());
        WeiKeListInfoDto.setOrgCourseId(courseListReponseDto.getOrgCourseId());
        WeiKeListInfoDto.setStudentCount(stuCntMap.get(courseListReponseDto.getOrgCourseId()) == null ? 0
            : stuCntMap.get(courseListReponseDto.getOrgCourseId()));
        WeiKeListInfoDto.setPrice(courseListReponseDto.getPrice());
        WeiKeListInfoDto.setStatus(courseListReponseDto.getStatus());
        WeiKeListInfoDto.setCourseCount(courseListReponseDto.getFreq());
        WeiKeListInfoDto.setIntroduction(courseListReponseDto.getIntroduction());
        WeiKeListInfoDto.setBeginTime(courseListReponseDto.getStartTime());
        WeiKeListInfoDto.setEndTime(courseListReponseDto.getEndTime());
    }

    @Override
    public List<CourseListInfoDto> getCourseList(Long orgId, Date startTime, Date endTime, CourseListRequestDto params,
        PageDto pageDto) {
        Long orgNumber = getOrgNumber(orgId);
        String searchKey = params.getSearchKey();
        String teacherIds = params.getTeacherIds();
        String roomIds = params.getRoomIds();

        /**
         * 按教室筛选
         */
        List<Long> roomCourseIds = null;
        Set<Long> roomIdSet = Sets.newHashSet();
        if (StringUtils.isNoneBlank(roomIds)) {
            String[] arr = roomIds.split(",");
            for (String str : arr) {
                roomIdSet.add(Long.parseLong(str));
            }
            roomCourseIds = Lists.newArrayList();
            if (CollectionUtils.isNotEmpty(roomIdSet)) {
                roomCourseIds = this.orgCourseRoomDao.getcoursIdsByRoomIds(roomIdSet);
            }
        }

        /**
         * 按老师筛选
         */
        List<Long> teacherCourseIds = null;
        Set<Long> teacherIdSet = Sets.newHashSet();
        if (StringUtils.isNoneBlank(teacherIds)) {
            String[] arr = teacherIds.split(",");
            for (String str : arr) {
                teacherIdSet.add(Long.parseLong(str));
            }
            teacherCourseIds = Lists.newArrayList();
            if (CollectionUtils.isNotEmpty(teacherIdSet)) {
                teacherCourseIds = this.orgCourseTeacherDao.getCourseIdsByteacherIds(teacherIdSet);
            }
        }

        List<Long> courseIds = null;
        if (roomCourseIds == null && teacherCourseIds != null) {
            courseIds = new ArrayList<Long>();
            courseIds.addAll(teacherCourseIds);
        }
        if (teacherCourseIds == null && roomCourseIds != null) {
            courseIds = new ArrayList<Long>();
            courseIds.addAll(roomCourseIds);
        }
        if (teacherCourseIds != null && roomCourseIds != null) {
            courseIds = new ArrayList<Long>();
            for (Long tcId : teacherCourseIds) {
                for (Long rcid : roomCourseIds) {
                    if (tcId.longValue() == rcid.longValue()) {
                        courseIds.add(tcId);
                    }
                }
            }
        }

        OrgCourseListParam param = new OrgCourseListParam();
        param.setKey(searchKey);
        param.setStartTime(startTime);
        param.setEndTime(endTime);
        param.setPage(pageDto);
        param.setStatus(params.getStatus());
        param.setCascadeId(params.getCascadeId());

        Map<String, FieldShow> fieldShowMap = FieldShow.getColumnShowMap(CourseListInfoDto.class);
        if (StringUtils.isNotBlank(params.getOrderField())) {
            FieldShow fieldShow = fieldShowMap.get(params.getOrderField());
            String orderField = fieldShow != null ? fieldShow.getPoName() : null;
            param.setOrderField(orderField);
            param.setOrderType(params.getOrderType());
        }

        if (StringUtils.isBlank(params.getOrderField())) {
            param.setOrderField("createTime");
            param.setOrderType("DESC");
        }

        param.setQueryCondition(QueryCondition.andCondtions(params.getQueryConditions(), fieldShowMap));
        param.setIsDel(params.getIsDel());
        List<OrgCourse> courses = this.orgCourseDao.getCourseList(orgNumber, courseIds, param);
        List<CourseListReponseDto> dtos =
            this.orgCourseListService.getCourseList(courses, orgId, true, false, true, true, true, true, true);

        log.debug("course List==={}", dtos);

        // 获取课程的老师列表：
        Map<Long, List<String>> courseTeachersMap = null;
        if (params.getLoadTeacherList() != null && params.getLoadTeacherList() == true) {
            courseIds = Lists.newArrayList();
            for (OrgCourse course : courses) {
                courseIds.add(course.getId());
            }
            courseTeachersMap = getTeachersOfCourses(orgId, courseIds);
        }

        Map<Long, String> cascadeMap = txCascadeCredentialService.getByTxCasCadeIds(orgId);
        List<CourseListInfoDto> responseDto = Lists.newArrayList();
        for (CourseListReponseDto courseListReponseDto : dtos) {
            CourseListInfoDto dto = new CourseListInfoDto();
            buildCourseListInfoDto(courseListReponseDto, dto);
            if (courseTeachersMap != null) {
                List<String> teachers = courseTeachersMap.get(courseListReponseDto.getOrgCourseId());
                if (CollectionUtils.isNotEmpty(teachers)) {
                    dto.setTeacherNames(StringUtils.join(teachers, ","));
                }
            }
            dto.setCascadeIdStr(cascadeMap.get(dto.getCascadeId().longValue()));
            responseDto.add(dto);
        }
        return responseDto;
    }

    @Override
    public List<CourseListInfoDto> getCourseListWithAttendanceRate(Long orgId, String searchKey, PageDto pageDto) {
        Long orgNumber = getOrgNumber(orgId);

        List<OrgCourse> courses = null;
        if (TianxiaoMContext.getTXCascadeId() == null) {
            courses = this.orgCourseDao.getCourseList(orgNumber, null, searchKey, null, null, null, pageDto);
        }else{
            TXCascadeAccount txCascadeAccount = txCascadeAccountDao.getById(TianxiaoMContext.getTXCascadeId());
            if(txCascadeAccount == null){
                throw new BussinessException(CommonErrorCode.PARAM_ERROR, "子帐号不存在");
            }
            if(txCascadeAccount.getAccountType() == CascadeType.STAFF.getValue()){
                courses = this.orgCourseDao.getCourseList(txCascadeAccount.getId(), orgNumber, null, searchKey, null, null, null, pageDto);
            }else{
                courses = this.orgCourseDao.getCourseList(orgNumber, null, searchKey, null, null, null, pageDto);
            }
        }
        List<CourseListReponseDto> dtos = getCourseListWithAttendanceRate(courses, orgId);
        List<CourseListInfoDto> responseDto = Lists.newArrayList();
        Map<Long, String> cascadeMap = txCascadeCredentialService.getByTxCasCadeIds(orgId);
        for (CourseListReponseDto courseListReponseDto : dtos) {
            CourseListInfoDto dto = new CourseListInfoDto();
            buildCourseListInfoDto(courseListReponseDto, dto);
            dto.setCascadeIdStr(cascadeMap.get(dto.getCascadeId()));
            responseDto.add(dto);
        }
        return responseDto;
    }

    private List<CourseListReponseDto> getCourseListWithAttendanceRate(List<OrgCourse> courses, Long orgId) {
        log.info("courses={},orgId={}", courses, orgId);
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId not valid!");
        if (CollectionUtils.isEmpty(courses)) {
            return Collections.emptyList();
        }
        List<CourseListReponseDto> dtos =
            this.orgCourseListService.getCourseList(courses, orgId, true, false, false, true, false, false, false);
        List<Long> courseIds = Lists.newArrayList();
        for (OrgCourse course : courses) {
            courseIds.add(course.getId());
        }
        Map<Long, Double> rateMap = this.orgLessonSignService.getAttendanceRateOfCourses(orgId, courseIds);
        for (CourseListReponseDto dto : dtos) {
            Long courseId = dto.getOrgCourseId();
            Double attendanceRate = rateMap.get(courseId);
            if (attendanceRate != null) {
                dto.setAttendanceRate(attendanceRate);
            } else {
                dto.setAttendanceRate(0.00);
            }
        }
        return dtos;
    }

    private void buildCourseListInfoDto(CourseListReponseDto courseListReponseDto, CourseListInfoDto dto) {
        dto.setColor(courseListReponseDto.getColor());
        dto.setCourseName(courseListReponseDto.getCourseName());
        dto.setCourseUrl(courseListReponseDto.getCourseUrl());
        dto.setCoverUrl(courseListReponseDto.getCoverUrl());
        dto.setEndTime(courseListReponseDto.getEndTime());
        dto.setFreq(courseListReponseDto.getFreq());
        dto.setLessonCount(courseListReponseDto.getLessonCount());
        dto.setLessonFinish(courseListReponseDto.getLessonFinish());
        dto.setMaxStudent(courseListReponseDto.getMaxStudent());
        dto.setOrgCourseId(courseListReponseDto.getOrgCourseId());
        dto.setOrgCourseNumber(courseListReponseDto.getOrgCourseNumber());
        dto.setRoomCount(courseListReponseDto.getRoomCount());
        dto.setRoomNames(courseListReponseDto.getRoomNames());
        dto.setStudentNames(courseListReponseDto.getStudentNames());
        dto.setTeacherNames(courseListReponseDto.getTeacherNames());
        dto.setStartTime(courseListReponseDto.getStartTime());
        dto.setStudentCount(courseListReponseDto.getStudentCount());
        dto.setTeacherCount(courseListReponseDto.getTeacherCount());
        dto.setAttendanceRate(courseListReponseDto.getAttendanceRate());
        dto.setPrice(courseListReponseDto.getPrice());
        dto.setStatus(courseListReponseDto.getStatus());
        dto.setAddress(courseListReponseDto.getAddress());
        dto.setCreateTime(courseListReponseDto.getCreateTime());
        dto.setCascadeId(courseListReponseDto.getCascadeId());
        
    }

    @Override
    public CourseListInfoDto getCourseDetail(Long orgId, Long orgCourseId) {
        CourseListReponseDto dto = this.orgCourseListService.getCourseDetail(orgId, orgCourseId);
        CourseListInfoDto courseListInfoDto = new CourseListInfoDto();
        buildCourseListInfoDto(dto, courseListInfoDto);
        return courseListInfoDto;
    }

}
