
/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */
package com.baijia.tianxiao.biz.erp.service.impl;

import com.baijia.tianxiao.biz.erp.constant.CourseSignStatus;
import com.baijia.tianxiao.biz.erp.dto.LessonTime;
import com.baijia.tianxiao.biz.erp.dto.request.AddCourseLessonRequestDto;
import com.baijia.tianxiao.biz.erp.dto.request.EditLessonRequestDto;
import com.baijia.tianxiao.biz.erp.dto.request.FastSignInLessonsEditRequestDto;
import com.baijia.tianxiao.biz.erp.dto.request.ListLessonsRequestDto;
import com.baijia.tianxiao.biz.erp.dto.response.LessonResponseDto;
import com.baijia.tianxiao.biz.erp.dto.response.LessonTimeTableResponseDto;
import com.baijia.tianxiao.biz.erp.dto.response.ListLessonResponseDto;
import com.baijia.tianxiao.biz.erp.enums.BatchOperateType;
import com.baijia.tianxiao.biz.erp.service.CourseLessonService;
import com.baijia.tianxiao.biz.erp.teacherCenter.service.ExcelCourseExportService;
import com.baijia.tianxiao.consants.DataStatus;
import com.baijia.tianxiao.consants.UserRole;
import com.baijia.tianxiao.constant.CommentStatus;
import com.baijia.tianxiao.constant.SignStatus;
import com.baijia.tianxiao.constants.TianXiaoConstant;
import com.baijia.tianxiao.constants.UserRoleEnum;
import com.baijia.tianxiao.dal.org.dao.OrgAccountDao;
import com.baijia.tianxiao.dal.org.dao.OrgClassLessonDao;
import com.baijia.tianxiao.dal.org.dao.OrgClassRoomDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseTeacherDao;
import com.baijia.tianxiao.dal.org.dao.OrgInfoDao;
import com.baijia.tianxiao.dal.org.dao.OrgLessonCommentDao;
import com.baijia.tianxiao.dal.org.dao.OrgLessonSignDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentCourseDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentLessonDao;
import com.baijia.tianxiao.dal.org.dao.OrgTeacherLessonDao;
import com.baijia.tianxiao.dal.org.dao.TXCascadeAccountDao;
import com.baijia.tianxiao.dal.org.po.OrgAccount;
import com.baijia.tianxiao.dal.org.po.OrgClassLesson;
import com.baijia.tianxiao.dal.org.po.OrgClassRoom;
import com.baijia.tianxiao.dal.org.po.OrgCourse;
import com.baijia.tianxiao.dal.org.po.OrgCourseTeacher;
import com.baijia.tianxiao.dal.org.po.OrgLessonSign;
import com.baijia.tianxiao.dal.org.po.OrgStudentLesson;
import com.baijia.tianxiao.dal.org.po.OrgTeacherLesson;
import com.baijia.tianxiao.dal.org.po.TXCascadeAccount;
import com.baijia.tianxiao.dal.user.dao.TeacherDao;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.excel.dto.ExportField;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.filter.TianxiaoMContext;
import com.baijia.tianxiao.sal.comment.constant.CommentErrorCode;
import com.baijia.tianxiao.sal.course.dto.HeaderDto;
import com.baijia.tianxiao.sal.course.dto.response.OrgStudentsChooseListDto;
import com.baijia.tianxiao.sal.course.dto.response.TeacherResponseDto;
import com.baijia.tianxiao.sal.course.service.CourseStudentService;
import com.baijia.tianxiao.sal.course.service.CourseTeacherService;
import com.baijia.tianxiao.sal.course.service.OrgCourseListService;
import com.baijia.tianxiao.sal.course.service.OrgLessonSignService;
import com.baijia.tianxiao.sal.organization.constant.CascadeType;
import com.baijia.tianxiao.sal.room.dto.ClassRoomDto;
import com.baijia.tianxiao.sal.room.service.ClassRoomService;
import com.baijia.tianxiao.sal.student.api.OrgStudentCourseService;
import com.baijia.tianxiao.util.collection.CollectorUtil;
import com.baijia.tianxiao.util.date.DateUtil;
import com.baijia.tianxiao.util.date.TimeStamp;
import com.baijia.tianxiao.util.json.JacksonUtil;

import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;

import lombok.extern.slf4j.Slf4j;

/**
 * @author cxm
 * @version 1.0
 * @title CourseLessonServiceImpl
 * @desc TODO
 * @date 2015年12月28日
 */
@Service
@Slf4j
public class CourseLessonServiceImpl implements CourseLessonService {

    @Resource
    private OrgClassLessonDao orgClassLessonDao;

    @Resource
    private OrgTeacherLessonDao orgTeacherLessonDao;

    @Resource
    private OrgStudentLessonDao orgStudentLessonDao;

    @Resource
    private OrgLessonSignDao orgLessonSignDao;

    @Resource
    private OrgCourseDao orgCourseDao;

    @Resource
    private OrgStudentDao orgStudentDao;

    @Resource
    private TeacherDao teacherDao;

    @Resource
    private OrgClassRoomDao orgClassRoomDao;

    @Resource
    private OrgLessonCommentDao orgLessonCommentDao;

    @Resource
    private OrgCourseListService orgCourseListService;

    @Resource
    private CourseStudentService courseStudentService;
    

    @Resource
    private CourseTeacherService courseTeacherService;

    @Resource
    private ClassRoomService classRoomService;

    @Resource
    private OrgInfoDao orgInfoDao;
    
    @Resource
    private OrgStudentCourseDao orgStudentCourseDao;
    
    @Resource
    private OrgStudentCourseService orgStudentCourseService;

    @Autowired
    private TXCascadeAccountDao txCascadeAccountDao;

    @Autowired
    private OrgAccountDao orgAccountDao;
    
    @Resource
    private OrgCourseTeacherDao orgCourseTeacherDao;
    
    @Resource
    private OrgLessonSignService orgLessonSignService;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void addClassLessons(Long orgId, Date startDay, Date endDay, AddCourseLessonRequestDto dto) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(startDay != null, "start day can not be null");
        Preconditions.checkArgument(dto != null, "class lessons param is illegal");
        Preconditions.checkArgument(dto.getCourseId() != null && dto.getCourseId() > 0, "courseId is illegal");

        List<Calendar> calens =
            dto.getRepeatUnit().getAvaliableDays(startDay, dto.getRepeatCount(), dto.getRepeatRange());
        TimeStamp startTime = TimeStamp.parse(dto.getStartTime());
        TimeStamp endTime = TimeStamp.parse(dto.getEndTime());

        OrgClassLesson lesson = null;
        List<OrgClassLesson> lessons = Lists.newArrayList();
        int index = 1;
        for (Calendar cal : calens) {
            LessonTime lessonTime = new LessonTime(cal, startTime, endTime);
            lesson = new OrgClassLesson();
            lesson.setStartTime(lessonTime.getStartTime());
            lesson.setEndTime(lessonTime.getEndTime());
            lesson.setCourseId(dto.getCourseId());
            lesson.setCreateTime(new Date());
            lesson.setDelStatus(DataStatus.NORMAL.getValue());
            lesson.setLayoutId(0l);
            lesson.setRoomId(dto.getRoomId() != null ? dto.getRoomId() : 0l);
            lesson.setUpdateTime(new Date());
            lesson.setNumber(index);
            lesson.setOrgId(orgId);
            lessons.add(lesson);
            index++;

            // orgClassLessonDao.save(lesson);
        }
        log.info("save {} lessons into db,params:{}", lessons.size(), dto);
        orgClassLessonDao.saveAll(lessons, "startTime", "endTime", "courseId", "createTime", "delStatus", "layoutId",
            "roomId", "updateTime", "orgId");

        /**
         * 添加教室到课程
         */
        Set<Long> roomIds = Sets.newHashSet();
        roomIds.add(dto.getRoomId() != null ? dto.getRoomId() : 0l);
        classRoomService.addClassRoomCourse(orgId.longValue(), dto.getCourseId(), roomIds);

        // 如果原来有课节的话,需要重新安排课节编号
        resetLessonNumber(orgId, dto.getCourseId());
        for (OrgClassLesson classLesson : lessons) {
            courseStudentService.addStudentToLesson(orgId, classLesson.getCourseId(), classLesson.getId(),
                dto.getStudentIds(), true);
            Set<Long> teacherIds = Sets.newHashSet();
            if (dto.getTeacherId() != null && dto.getTeacherId() > 0) {
                teacherIds.add(dto.getTeacherId());
                Set<Long> tempTeacherIds = Sets.newHashSet();
                tempTeacherIds.addAll(teacherIds);
                courseTeacherService.addTeacherToCourse(orgId, classLesson.getCourseId(), teacherIds);// 先添加到课程
                courseTeacherService.addTeacherToLesson(orgId, classLesson.getCourseId(), classLesson.getId(),
                    tempTeacherIds);
            }
        }
    }
    
    
    

    @Override
    @Transactional(readOnly = true)
    public ListLessonResponseDto listLessons(Long orgId, ListLessonsRequestDto params) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Set<Long> courseIds = Sets.newHashSet();
        Set<Long> roomIds = Sets.newHashSet();
        // Set<Long> teacherIds = Sets.newHashSet();
        Set<Long> lessonIds = Sets.newHashSet();
        Map<Long, Long> lessonIdTeacherIdMap = new HashMap<>();
        List<OrgClassLesson> lessons = queryLesson(orgId, lessonIdTeacherIdMap, params, true);
        ListLessonResponseDto result = new ListLessonResponseDto();
        if (CollectionUtils.isNotEmpty(lessons)) {
            Collections.sort(lessons, new Comparator<OrgClassLesson>() {
                @Override
                public int compare(OrgClassLesson o1, OrgClassLesson o2) {
                    return o1.getNumber().compareTo(o2.getNumber());
                }
            });
            for (OrgClassLesson lesson : lessons) {
                courseIds.add(lesson.getCourseId());
                roomIds.add(lesson.getRoomId());
                lessonIds.add(lesson.getId());
            }

            Map<Long, OrgCourse> courseMap = Maps.newHashMap();
           // bugfix 这里如果校验是否需要课程名字，后面的组装对象的逻辑就无法通过判断课程是否删除来过滤无效的课节，所以注释掉，以确保程序正确
           // if (params.isNeedcourseName()) {
                courseMap = getOrgCourseMap(orgId, courseIds);
           // }
            Map<Long, String> lessonIdTeacherNameMap = Maps.newHashMap();
            if (params.isNeedTeacherName()) {
                lessonIdTeacherNameMap = getLessonIdTeacherName(orgId, lessonIds, lessonIdTeacherIdMap);
            }
            Map<Long, OrgClassRoom> classRoomMap = Maps.newHashMap();
            if (params.isNeedRoomName()) {
                classRoomMap = getOrgClassRoomMap(roomIds);
            }

            Map<Long, Integer> studentCountMap = orgStudentLessonDao.queryLessonStudentCountMap(orgId, lessonIds);
            Map<Long, Integer> signStudentCountMap = Maps.newHashMap();
            if (!params.isNeedStudentSignStatus() && params.isNeedSignInfo()) {
                signStudentCountMap = orgLessonSignDao.queryLessonStudentCountMap(orgId, lessonIds);
            }
            Map<Long, Integer> commentCount = Maps.newHashMap();
            Map<Long, Integer> lessonSignStatus = Maps.newHashMap();
            if (params.isNeedStudentSignStatus()) {
                Preconditions.checkArgument(params.getStudentId() != null && params.getStudentId() > 0, "学生ID不正确");
                Long userId = this.orgStudentDao.getUserId(params.getStudentId());

                commentCount = orgLessonCommentDao.getLessonCommentCountMap(lessonIds, orgId, userId, null,
                    UserRole.STUDENT.getRole(), false);

                List<OrgLessonSign> lessonSign = orgLessonSignDao.getStudentSign(orgId, userId,
                    UserRole.STUDENT.getRole(), lessonIds, "lessonId", "status");
                if (CollectionUtils.isNotEmpty(lessonSign)) {
                    lessonSignStatus = CollectorUtil.collectMap(lessonSign, new Function<OrgLessonSign, Long>() {
                        @Override
                        public Long apply(OrgLessonSign input) {
                            return input.getLessonId();
                        }
                    }, new Function<OrgLessonSign, Integer>() {
                        @Override
                        public Integer apply(OrgLessonSign input) {
                            return input.getStatus();
                        }
                    });
                    log.debug("student lesson sign:{}", lessonSignStatus);
                }
            }

            result = buildListLessonResponseDto(lessons, courseMap, lessonIdTeacherNameMap, classRoomMap,
                studentCountMap, signStudentCountMap, commentCount, lessonSignStatus);
        } else {
            result.setFirstLessonStartTime(params.getFirstLessonStartTime());
            result.setLastLessonStartTime(params.getLastLessonStartTime());
            if (params.getFirstLessonStartTime() == null && params.getLastLessonStartTime() == null) {
                result.setFirstLessonStartTime(DateUtil.getCurrentDate());
            }
        }
        return result;
    }
    @Override
    @Transactional(readOnly = true)
    public ListLessonResponseDto listSignInLessons(Long orgId, ListLessonsRequestDto params) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Set<Long> courseIds = Sets.newHashSet();
        Set<Long> roomIds = Sets.newHashSet();
        // Set<Long> teacherIds = Sets.newHashSet();
        Set<Long> lessonIds = Sets.newHashSet();
        Map<Long, Long> lessonIdTeacherIdMap = new HashMap<>();
        List<OrgClassLesson> lessons = querySignInLesson(orgId, lessonIdTeacherIdMap, params, true);
        ListLessonResponseDto result = new ListLessonResponseDto();
        if (CollectionUtils.isNotEmpty(lessons)) {
            Collections.sort(lessons, new Comparator<OrgClassLesson>() {
                @Override
                public int compare(OrgClassLesson o1, OrgClassLesson o2) {
                    return o1.getNumber().compareTo(o2.getNumber());
                }
            });
            for (OrgClassLesson lesson : lessons) {
                courseIds.add(lesson.getCourseId());
                roomIds.add(lesson.getRoomId());
                lessonIds.add(lesson.getId());
            }
            
            Map<Long, OrgCourse> courseMap = Maps.newHashMap();
            // bugfix 这里如果校验是否需要课程名字，后面的组装对象的逻辑就无法通过判断课程是否删除来过滤无效的课节，所以注释掉，以确保程序正确
            // if (params.isNeedcourseName()) {
            courseMap = getOrgCourseMap(orgId, courseIds);
            // }
            Map<Long, String> lessonIdTeacherNameMap = Maps.newHashMap();
            if (params.isNeedTeacherName()) {
                lessonIdTeacherNameMap = getLessonIdTeacherName(orgId, lessonIds, lessonIdTeacherIdMap);
            }
            Map<Long, OrgClassRoom> classRoomMap = Maps.newHashMap();
            if (params.isNeedRoomName()) {
                classRoomMap = getOrgClassRoomMap(roomIds);
            }
            
            Map<Long, Integer> studentCountMap = orgStudentLessonDao.queryLessonStudentCountMap(orgId, lessonIds);
            Map<Long, Integer> signStudentCountMap = Maps.newHashMap();
            if (!params.isNeedStudentSignStatus() && params.isNeedSignInfo()) {
                signStudentCountMap = orgLessonSignDao.queryLessonStudentCountMap(orgId, lessonIds);
            }
            Map<Long, Integer> commentCount = Maps.newHashMap();
            Map<Long, Integer> lessonSignStatus = Maps.newHashMap();
            if (params.isNeedStudentSignStatus()) {
                Preconditions.checkArgument(params.getStudentId() != null && params.getStudentId() > 0, "学生ID不正确");
                Long userId = this.orgStudentDao.getUserId(params.getStudentId());
                
                commentCount = orgLessonCommentDao.getLessonCommentCountMap(lessonIds, orgId, userId, null,
                    UserRole.STUDENT.getRole(), false);
                
                List<OrgLessonSign> lessonSign = orgLessonSignDao.getStudentSign(orgId, userId,
                    UserRole.STUDENT.getRole(), lessonIds, "lessonId", "status");
                if (CollectionUtils.isNotEmpty(lessonSign)) {
                    lessonSignStatus = CollectorUtil.collectMap(lessonSign, new Function<OrgLessonSign, Long>() {
                        @Override
                        public Long apply(OrgLessonSign input) {
                            return input.getLessonId();
                        }
                    }, new Function<OrgLessonSign, Integer>() {
                        @Override
                        public Integer apply(OrgLessonSign input) {
                            return input.getStatus();
                        }
                    });
                    log.debug("student lesson sign:{}", lessonSignStatus);
                }
            }
            
            result = buildListLessonResponseDto(lessons, courseMap, lessonIdTeacherNameMap, classRoomMap,
                studentCountMap, signStudentCountMap, commentCount, lessonSignStatus);
        } else {
            result.setFirstLessonStartTime(params.getFirstLessonStartTime());
            result.setLastLessonStartTime(params.getLastLessonStartTime());
            if (params.getFirstLessonStartTime() == null && params.getLastLessonStartTime() == null) {
                result.setFirstLessonStartTime(DateUtil.getCurrentDate());
            }
        }
        return result;
    }

    private Map<Long, String> getLessonIdTeacherName(Long orgId, Collection<Long> lessonIds,
        Map<Long, Long> lessonTeacherIdMap) {
        if (MapUtils.isEmpty(lessonTeacherIdMap)) {
            lessonTeacherIdMap = orgTeacherLessonDao.queryLessonTeacherIdMap(orgId, lessonIds);
        }
        Map<Long, String> teacherNameMap = teacherDao.getTeacherRealNameMap(lessonTeacherIdMap.values());
        Map<Long, String> lessonIdTeacherNameMap = new HashMap<>(lessonTeacherIdMap.size());
        for (Long lessonId : lessonTeacherIdMap.keySet()) {
            lessonIdTeacherNameMap.put(lessonId, teacherNameMap.get(lessonTeacherIdMap.get(lessonId)));
        }
        return lessonIdTeacherNameMap;
    }

    @SuppressWarnings("unchecked")
    private List<OrgClassLesson> queryLesson(Long orgId, Map<Long, Long> lessonIdTeacherIdMap,
        ListLessonsRequestDto params, boolean isTimeTable) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Set<Long> courseIds = Sets.newHashSet();
        Set<Long> roomIds = Sets.newHashSet();
        Set<Long> teacherIds = Sets.newHashSet();
        Set<Long> lessonIds = Sets.newHashSet();
        if (params.getCourseId() != null) {
            courseIds.add(params.getCourseId());
        }
        // 登录帐号为子帐号，并且是员工帐号，并且排课权限打开时，只能查看自己作为班主任的课节
        // 涉及到分页，这里取courseIds时，按照时间倒序，测试时需要关注数据是否取全
        if (TianxiaoMContext.getTXCascadeId() != null) {
            TXCascadeAccount txCascadeAccount = txCascadeAccountDao.getById(TianxiaoMContext.getTXCascadeId());
            if (txCascadeAccount == null) {
                throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "子帐号不存在");
            }
            if (txCascadeAccount.getAccountType() == CascadeType.STAFF.getValue()) {
                log.debug("before clear courseIds:{}", courseIds);
                List<Long> tempIds = orgCourseDao.getCourseIdsByCascadeId(TianxiaoMContext.getTXCascadeId());
                log.debug("user cascadeId={},courseIds={}",TianxiaoMContext.getTXCascadeId(),tempIds);
                // bugfix 此处根据课程权限获取用户课节的逻辑有问题，如果用户指定查询一门课的课节会查出用户课程权限下的所有课节，故调整逻辑
                if(CollectionUtils.isEmpty(courseIds)){
                    courseIds.addAll(tempIds);
                }else{
                    // 如果是子帐号，并且指定了课程，那么需要判断课程是否在这个用户的权限中
                    if(!tempIds.containsAll(courseIds)){
                        courseIds.clear();
                    }
                }
                
                log.debug("after set courseIds:{}", courseIds);
                if(CollectionUtils.isEmpty(courseIds)){
                    return Collections.EMPTY_LIST;
                }
                
            }
        }

        if (params.getRoomId() != null) {
            roomIds.add(params.getRoomId());
        }
        if (params.getTeacherId() != null) {
            teacherIds.add(params.getTeacherId());
        }
        if (params.getStudentId() != null) {
            lessonIds.addAll(courseStudentService.getStudentLessonIds(orgId, params.getStudentId()));
            if (CollectionUtils.isEmpty(lessonIds)) {
                return Collections.emptyList();
            }
        }

        lessonIdTeacherIdMap = queryLessonIdByTeacherId(orgId, teacherIds);
        if (CollectionUtils.isNotEmpty(teacherIds) && lessonIdTeacherIdMap.isEmpty()) {
            return Collections.emptyList();
        }
        lessonIds.addAll(lessonIdTeacherIdMap.keySet());

        Date startTime = params.getStartTime();
        Date endTime = params.getEndTime();
        if (isTimeTable) {
            startTime  = params.getFirstLessonStartTime();
            endTime = params.getLastLessonStartTime();
        }

        List<OrgClassLesson> lessons = orgClassLessonDao.queryLessons(orgId, courseIds, roomIds, lessonIds, startTime,
            endTime, params.getPageDto());
        return lessons;
    }
    @SuppressWarnings("unchecked")
    private List<OrgClassLesson> querySignInLesson(Long orgId, Map<Long, Long> lessonIdTeacherIdMap,
        ListLessonsRequestDto params, boolean isTimeTable) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Set<Long> courseIds = Sets.newHashSet();
        Set<Long> roomIds = Sets.newHashSet();
        Set<Long> teacherIds = Sets.newHashSet();
        Set<Long> lessonIds = Sets.newHashSet();
        if (params.getCourseId() != null) {
            courseIds.add(params.getCourseId());
        }
        // 登录帐号为子帐号，并且是员工帐号，并且排课权限打开时，只能查看自己作为班主任的课节
        // 涉及到分页，这里取courseIds时，按照时间倒序，测试时需要关注数据是否取全
        if (TianxiaoMContext.getTXCascadeId() != null) {
            TXCascadeAccount txCascadeAccount = txCascadeAccountDao.getById(TianxiaoMContext.getTXCascadeId());
            if (txCascadeAccount == null) {
                throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "子帐号不存在");
            }
            if (txCascadeAccount.getAccountType() == CascadeType.STAFF.getValue()) {
                log.debug("before clear courseIds:{}", courseIds);
                List<Long> tempIds = orgCourseDao.getCourseIdsByCascadeId(TianxiaoMContext.getTXCascadeId());
                log.debug("user cascadeId={},courseIds={}",TianxiaoMContext.getTXCascadeId(),tempIds);
                // bugfix 此处根据课程权限获取用户课节的逻辑有问题，如果用户指定查询一门课的课节会查出用户课程权限下的所有课节，故调整逻辑
                if(CollectionUtils.isEmpty(courseIds)){
                    courseIds.addAll(tempIds);
                }else{
                    // 如果是子帐号，并且指定了课程，那么需要判断课程是否在这个用户的权限中
                    if(!tempIds.containsAll(courseIds)){
                        courseIds.clear();
                    }
                }
                
                log.debug("after set courseIds:{}", courseIds);
                if(CollectionUtils.isEmpty(courseIds)){
                    return Collections.EMPTY_LIST;
                }
                
            }
        }
        
        if (params.getRoomId() != null) {
            roomIds.add(params.getRoomId());
        }
        if (params.getTeacherId() != null) {
            teacherIds.add(params.getTeacherId());
        }
        if (params.getStudentId() != null) {
            lessonIds.addAll(courseStudentService.getStudentLessonIds(orgId, params.getStudentId()));
            if (CollectionUtils.isEmpty(lessonIds)) {
                return Collections.emptyList();
            }
        }
        
        lessonIdTeacherIdMap = queryLessonIdByTeacherId(orgId, teacherIds);
        if (CollectionUtils.isNotEmpty(teacherIds) && lessonIdTeacherIdMap.isEmpty()) {
            return Collections.emptyList();
        }
        lessonIds.addAll(lessonIdTeacherIdMap.keySet());
        
        Date startTime = params.getStartTime();
        Date endTime = params.getEndTime();
        if (isTimeTable) {
            startTime  = params.getFirstLessonStartTime();
            endTime = params.getLastLessonStartTime();
        }
        
        List<OrgClassLesson> lessons = orgClassLessonDao.querySignInLessons(orgId, courseIds, roomIds, lessonIds, startTime,
            endTime, params.getPageDto(), params.getLessonId());
        return lessons;
    }

    @Override
    @Transactional(readOnly = true)
    public List<LessonTimeTableResponseDto> queryTimeTable(Long orgId, ListLessonsRequestDto params) {
        Preconditions.checkArgument(params.getStartTime() != null && params.getEndTime() != null,
            "start time or end time is null");
        Preconditions.checkArgument(params.getEndTime().after(params.getStartTime()), "time condition error");
        Set<Long> courseIds = Sets.newHashSet();
        Set<Long> roomIds = Sets.newHashSet();
        Map<Long, Long> lessonIdTeacherIdMap = new HashMap<>();
        List<OrgClassLesson> lessons = queryLesson(orgId, lessonIdTeacherIdMap, params, false);
        if (CollectionUtils.isNotEmpty(lessons)) {
            Set<Long> lessonIds = Sets.newHashSet();
            for (OrgClassLesson lesson : lessons) {
                courseIds.add(lesson.getCourseId());
                roomIds.add(lesson.getRoomId());
                lessonIds.add(lesson.getId());
            }
            if (lessonIdTeacherIdMap.isEmpty()) {
                lessonIdTeacherIdMap = orgTeacherLessonDao.queryLessonTeacherIdMap(orgId, lessonIds);
                log.debug("query lessonId teacherId map:{}by lessonIds:{}", lessonIdTeacherIdMap, lessonIds);
            }
            Map<Long, OrgCourse> courseMap = getOrgCourseMap(orgId, courseIds);
            Map<Long, String> teacherNameMap = teacherDao.getTeacherRealNameMap(lessonIdTeacherIdMap.values());
            Map<Long, OrgClassRoom> classRoomMap = getOrgClassRoomMap(roomIds);
            List<LessonTimeTableResponseDto> result = Lists.newArrayList();
            for (OrgClassLesson lesson : lessons) {
                Long teacherId = lessonIdTeacherIdMap.get(lesson.getId());
                //bufix 稍加修改，过滤掉已经删除的课程
                OrgCourse tempCourse = courseMap.get(lesson.getCourseId());
                if(tempCourse!=null && tempCourse.getIsDel()==0){
                    LessonTimeTableResponseDto dto = buildTimeTableDto(lesson, tempCourse,
                        classRoomMap.get(lesson.getRoomId()), teacherId, teacherNameMap.get(teacherId));
                    result.add(dto);
                }
            }
            return result;
        } else {
            return Collections.emptyList();
        }
    }

    private Map<Long, OrgClassRoom> getOrgClassRoomMap(Collection<Long> roomIds) {
        List<OrgClassRoom> classRooms = orgClassRoomDao.getByIds(roomIds, "id", "roomName", "roomSize");
        return CollectorUtil.collectMap(classRooms, new Function<OrgClassRoom, Long>() {
            @Override
            public Long apply(OrgClassRoom input) {
                return input.getId();
            }
        });
    }

    private Map<Long, OrgCourse> getOrgCourseMap(Long orgId, Collection<Long> courseIds) {
        List<OrgCourse> courseList = orgCourseDao.getByIds(courseIds, "id", "cover", "name", "color","isDel");
        
        Iterator<OrgCourse> courseIter = courseList.iterator();  
        while (courseIter.hasNext()) {  
            OrgCourse orgCourse = courseIter.next();  
            if(orgCourse.getIsDel()!=0){
                courseIter.remove();
            }
        }  
        
        // 批量设置一下颜色
        orgCourseListService.getCourseList(courseList, orgId, false, false, false, false, false, false, false);
        return CollectorUtil.collectMap(courseList, new Function<OrgCourse, Long>() {
            @Override
            public Long apply(OrgCourse input) {
                return input.getId();
            }
        });
    }

    private LessonTimeTableResponseDto buildTimeTableDto(OrgClassLesson lesson, OrgCourse course,
        OrgClassRoom classRoom, Long teacherId, String teacherName) {
        LessonTimeTableResponseDto dto = new LessonTimeTableResponseDto();
        dto.setCourseId(lesson.getCourseId());
        if (course != null) {
            dto.setCourseColor(course.getColor());
            dto.setCourseName(course.getName());
        }
        dto.setCreateTime(lesson.getCreateTime());
        dto.setEndTime(lesson.getEndTime());
        dto.setLessonId(lesson.getId());
        dto.setIndex(lesson.getNumber());
        dto.setRoomId(lesson.getRoomId());
        if (classRoom != null) {
            dto.setRoomName(classRoom.getRoomName());
        }
        dto.setStartTime(lesson.getStartTime());
        dto.setUpdateTime(lesson.getUpdateTime());
        dto.setTeacherId(teacherId);
        dto.setTeacherName(teacherName);
        return dto;
    }

    private ListLessonResponseDto buildListLessonResponseDto(List<OrgClassLesson> lessons,
        Map<Long, OrgCourse> courseMap, Map<Long, String> lessonIdTeacherNameMap, Map<Long, OrgClassRoom> classRoomMap,
        Map<Long, Integer> studentCountMap, Map<Long, Integer> signStudentCountMap, Map<Long, Integer> commentCount,
        Map<Long, Integer> lessonSignStatus) {
        ListLessonResponseDto result = new ListLessonResponseDto();
        if (CollectionUtils.isNotEmpty(lessons)) {
            Date now = new Date();
            for (OrgClassLesson lesson : lessons) {
                if (result.getFirstLessonStartTime() == null
                    || result.getFirstLessonStartTime().after(lesson.getStartTime())) {
                    result.setFirstLessonStartTime(lesson.getStartTime());
                }
                if (result.getLastLessonStartTime() == null
                    || result.getLastLessonStartTime().before(lesson.getStartTime())) {
                    result.setLastLessonStartTime(lesson.getStartTime());
                }

                LessonResponseDto dto = new LessonResponseDto();
                dto.setCourseId(lesson.getCourseId());
                dto.setIndex(lesson.getNumber());
                OrgCourse course = courseMap.get(lesson.getCourseId());
                // bugfix 课程不存在则过滤掉，解决删除课程，课节并未删除，在签到列表展示的问题
                if(course == null){
                    continue;
                }
                if (StringUtils.isNotBlank(course.getName())) {
                    dto.setCourseName(course.getName());
                }
                OrgClassRoom classRoom = classRoomMap.get(lesson.getRoomId());
                dto.setRoomId(lesson.getRoomId());
                if (classRoom != null) {
                    dto.setRoomName(classRoom.getRoomName());
                }
                dto.setTeacherName(lessonIdTeacherNameMap.get(lesson.getId()));
                dto.setLessonEndTime(lesson.getEndTime());
                dto.setLessonId(lesson.getId());
                dto.setLessonStartTime(lesson.getStartTime());
                CourseSignStatus signStatus = CourseSignStatus.HAS_NOT_STARTED;
                if (now.after(lesson.getEndTime())) {
                    // 设置课程结束标记
                    signStatus = CourseSignStatus.NO_SIGN;
                    dto.setIsOver(1);
                }
                if (now.after(lesson.getStartTime()) && now.before(lesson.getEndTime())) {
                    signStatus = CourseSignStatus.SIGNING;
                }
                Integer signCount = signStudentCountMap.get(lesson.getId());
                SignStatus stuSignStatus = SignStatus.getSignStatusByCode(lessonSignStatus.get(lesson.getId()));
                if (stuSignStatus != null) {
                    log.debug("lessonId:{},lessonSignStatus:{},status:{}", lesson.getId(), lessonSignStatus,
                        stuSignStatus);
                    dto.setSignStatusEnum(stuSignStatus);
                } else {
                    
                    if (signCount != null && signCount > 0) {
                        signStatus = CourseSignStatus.SIGNED;
                    }
                    dto.setCourseSignStatusEnum(signStatus);
                }

                Integer commentCnt = commentCount.get(lesson.getId());
                if (commentCnt != null && commentCnt > 0) {
                    dto.setCommentEnum(CommentStatus.COMMENTED);
                } else {
                    dto.setCommentEnum(
                        CommentStatus.getCommentStatusByTime(lesson.getStartTime(), lesson.getEndTime(), true));
                }

                Integer studentCount = studentCountMap.get(lesson.getId());
                dto.setSignCount(signCount != null ? signCount : 0);
                dto.setStudentCount(studentCount != null ? studentCount : 0);
                result.getList().add(dto);
            }
        }
        return result;

    }

    private Map<Long, Long> queryLessonIdByTeacherId(Long orgId, Collection<Long> teacherIds) {
        if (CollectionUtils.isEmpty(teacherIds)) {
            return Collections.emptyMap();
        }
        List<OrgTeacherLesson> teacherLessons =
            orgTeacherLessonDao.queryTeacherLessons(orgId, teacherIds, "lessonId", "teacherId");
        if (CollectionUtils.isNotEmpty(teacherLessons)) {
            Map<Long, Long> result = new HashMap<>();
            for (OrgTeacherLesson orgTeacherLesson : teacherLessons) {
                result.put(orgTeacherLesson.getLessonId(), orgTeacherLesson.getTeacherId());
            }
            return result;
        }
        return Collections.emptyMap();
    }

    @Override
    @Transactional(readOnly = true)
    public LessonTimeTableResponseDto lessonInfo(Long orgId, Long lessonId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(lessonId != null && lessonId > 0, "lessonId is illegal");
        OrgClassLesson lesson = orgClassLessonDao.getById(lessonId);
        Date now = new Date();
        if (lesson == null || !orgId.equals(lesson.getOrgId())) {
            log.warn("lesson id:{} is illegal,query lesson:{}", lessonId, lesson);
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "课节ID不正确");
        }
        LessonTimeTableResponseDto dto = new LessonTimeTableResponseDto();
        // set course info
        OrgCourse course = orgCourseDao.getByCourseId(lesson.getCourseId(), "id", "name", "color");
        if (course == null) {
            log.warn("can not found course by lesson:{}", lesson);
            throw new BussinessException(CommonErrorCode.SYSTEM_ERROR, "无法找到对应班级");
        }
        dto.setCourseColor(course.getColor());
        dto.setCourseId(course.getId());
        dto.setCourseName(course.getName());
        // set lesson info
        dto.setCreateTime(lesson.getCreateTime());
        dto.setEndTime(lesson.getEndTime());
        dto.setIndex(lesson.getNumber());
        dto.setLessonId(lesson.getId());
        CourseSignStatus signStatus = CourseSignStatus.HAS_NOT_STARTED;
        if (now.after(lesson.getStartTime())) {
            signStatus = CourseSignStatus.NO_SIGN;
            if (now.before(lesson.getEndTime())) {
                signStatus = CourseSignStatus.SIGNING;
            }
        }
        dto.setRoomId(lesson.getRoomId());
        dto.setSignStatusEnum(signStatus);
        dto.setStartTime(lesson.getStartTime());
        dto.setUpdateTime(lesson.getUpdateTime());
        return dto;
    }

    @Override
    @Transactional(readOnly = true)
    public LessonTimeTableResponseDto lessonDetail(Long orgId, Long lessonId, boolean needStudentSign) {
        LessonTimeTableResponseDto dto = lessonInfo(orgId, lessonId);
        // set student
        dto.setStudents(courseStudentService.getLessonStudent(orgId, lessonId, needStudentSign));
        for (OrgStudentsChooseListDto student : dto.getStudents()) {
            if (student.getSignStatus() != null && SignStatus.SIGNED.getCode() == student.getSignStatus()) {
                dto.increaseSignStudent();
            }
        }
        List<TeacherResponseDto> teacherDtos = courseTeacherService.getLessonTeacher(orgId, lessonId);
        if (CollectionUtils.isNotEmpty(teacherDtos)) {
            TeacherResponseDto teacherDto = teacherDtos.get(0);
            dto.setTeacherId(teacherDto.getTeacherId());
            dto.setTeacherName(teacherDto.getTeacherName());
        }
        // set room info
        if (dto.getRoomId() != null && dto.getRoomId() > 0) {
            OrgClassRoom classRoom = orgClassRoomDao.getById(dto.getRoomId(), "id", "roomName");
            dto.setRoomName(classRoom.getRoomName());
        }
        return dto;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void delLesson(Long orgId, Long lessonId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(lessonId != null && lessonId > 0, "lessonId is illegal");
        OrgClassLesson lesson = orgClassLessonDao.getById(lessonId);
        if (lesson == null || !orgId.equals(lesson.getOrgId())) {
            log.warn("lesson id:{} is illegal,query lesson:{}", lessonId, lesson);
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "课节ID不正确");
        }
        if (CollectionUtils.isNotEmpty(orgLessonCommentDao.getLessonComments(lessonId, null, null, "id"))) {
            throw new BussinessException(CommentErrorCode.LESSON_WITH_COMMENT);
        }
        if (CollectionUtils.isNotEmpty(
            orgLessonSignDao.getLessonSignList(orgId, null, lessonId, null, null, null, null, null, "id"))) {
            throw new BussinessException(CommentErrorCode.LESSON_WITH_SIGN);
        }
        courseStudentService.deleteStudentFromLesson(orgId, lessonId, null);
        courseTeacherService.delTeacherFromLesson(orgId, lessonId, null);

        orgClassLessonDao.delById(lessonId);
        resetLessonNumber(orgId, lesson.getCourseId());

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void editLesson(Long orgId, EditLessonRequestDto editDto) {
        Preconditions.checkNotNull(editDto, "param can not be null");
        editDto.validate();
        OrgClassLesson lesson = orgClassLessonDao.getById(editDto.getLessonId());
        if (lesson == null || !orgId.equals(lesson.getOrgId())) {
            log.warn("lesson id:{} is illegal,query lesson:{}", editDto.getLessonId(), lesson);
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "课节ID不正确");
        }
        lesson.setDelStatus(DataStatus.NORMAL.getValue());
        boolean needRearrange = false;
        if (!editDto.getEndTime().equals(lesson.getEndTime())
            || !editDto.getStartTime().equals(lesson.getStartTime())) {
            lesson.setEndTime(editDto.getEndTime());
            lesson.setStartTime(editDto.getStartTime());
            // 需要重新设置number
            needRearrange = true;
        }
        List<Long> teacherIds = Lists.newArrayList();
        if (editDto.getRoomId() == null) {
            editDto.setRoomId(0l);
        }
        if (editDto.getTeacherId() == null) {
            editDto.setTeacherId(0l);
        } else {
            teacherIds.add(editDto.getTeacherId());
        }
        lesson.setRoomId(editDto.getRoomId());
        lesson.setUpdateTime(new Date());

        orgClassLessonDao.update(lesson);
        if (needRearrange) {
            resetLessonNumber(orgId, lesson.getCourseId());
        }

        /**
         * 添加教室到课程
         */
        Set<Long> roomIds = Sets.newHashSet();
        roomIds.add(editDto.getRoomId() != null ? editDto.getRoomId() : 0l);
        classRoomService.addClassRoomCourse(orgId.longValue(), lesson.getCourseId(), roomIds);

        courseTeacherService.resetLessonTeacher(orgId, lesson.getCourseId(), lesson.getId(), teacherIds);
        // courseStudentService.resetLessonStudent(orgId, lesson.getCourseId(), lesson.getId(),
        // editDto.getStudentIds());
    }

    private void resetLessonNumber(Long orgId, Long courseId) {
        List<OrgClassLesson> lessons =
            orgClassLessonDao.queryLessons(orgId, Lists.newArrayList(courseId), null, null, null, null, null, "id");
        if (CollectionUtils.isNotEmpty(lessons)) {
            int index = 1;
            @SuppressWarnings("unchecked")
            Map<String, Number>[] params = new Map[lessons.size()];
            for (OrgClassLesson lesson : lessons) {
                Map<String, Number> param = new HashMap<>();
                param.put("id", lesson.getId());
                param.put("number", index);
                params[index - 1] = param;
                index++;
            }
            orgClassLessonDao.batchUpdateLessonNumber(params);
        }
    }

    @Override
    @Transactional(readOnly = true)
    public Long getTeacherId(Long orgId, Long lessonId) {
        OrgTeacherLesson teacherLesson = this.orgTeacherLessonDao.getByLessonId(orgId, lessonId);
        if (teacherLesson != null) {
            return teacherLesson.getTeacherId();
        }
        return null;
    }

    @Override
    @Transactional(readOnly = true)
    public List<Long> getLessonStudentUserIds(Long orgId, Collection<Long> lessonIds) {
        if (CollectionUtils.isEmpty(lessonIds)) {
            return Lists.newArrayList();
        }
        return this.orgStudentLessonDao.getUserIdsByLessonIds(orgId, lessonIds);
    }

    @Override
    public void export(HttpServletResponse response, Long orgId, ListLessonsRequestDto param,
        List<LessonTimeTableResponseDto> dtoList) {
        Preconditions.checkArgument(param.getStartTime() != null, "开始日期不能为空");
        Preconditions.checkArgument(param.getEndTime() != null, "结束日期不能为空");
        Preconditions.checkArgument(orgId != null && orgId.longValue() > 0L, "机构ID不能为空");
        String orgName = this.orgInfoDao.getOrgShortNameByOrgId(Integer.valueOf(orgId.intValue()));
        String startTimeStr = TianXiaoConstant.DAY_FORMAT.format(param.getStartTime());
        String endTimeStr = TianXiaoConstant.DAY_FORMAT.format(param.getEndTime());
        String fileName = orgName + "-课表统计-" + startTimeStr + "~" + endTimeStr + ".xlsx";
        SXSSFWorkbook workbook = null;
        List<Date> dates = DateUtil.getDateBetween(param.getStartTime(), param.getEndTime());

        try {
            workbook = new SXSSFWorkbook(200);
            HashMap cellStyleMap = new HashMap();
            List summaryFields = Lists.newArrayList();
            summaryFields.add(new ExportField(" ", "@", 5620));
            for (Date date : dates) {
                summaryFields.add(new ExportField(
                    "周" + DateUtil.getChineseWeekDay(date) + "(" + DateUtil.getDayStr(date) + ")", "@", 5120));
            }
            List titles = Lists.newArrayList();
            titles.add(startTimeStr + "~" + endTimeStr + "课表统计");

            ExcelCourseExportService.createSheet(workbook, "课表统计", titles, cellStyleMap, summaryFields, dtoList,
                param.getStartTime());
            ExcelCourseExportService.exportExcel(response, workbook, fileName);
            log.debug("course lesson export success");
        } catch (Exception var20) {
            log.error("export catch exception:", var20);
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "导出数据处理异常");
        } finally {
            try {
                workbook.close();
            } catch (IOException var19) {
                log.error("close workbook catch error:", var19);
            }

        }

    }

    public static void main(String[] args) throws IOException {
        String json = FileUtils.readFileToString(new File("/Users/wengshengli/Downloads/response.json"));
        List dtoList = JacksonUtil.str2List(json, LessonTimeTableResponseDto.class);
        Date startDate = new Date(1465747200682l);
        Date endDate = new Date(1466351999682l);
        System.out.println(DateUtil.getDateDiff(startDate, endDate));
        List<Date> dates = DateUtil.getDateBetween(startDate, endDate);
        SXSSFWorkbook workbook = new SXSSFWorkbook(200);
        HashMap cellStyleMap = new HashMap();
        List summaryFields = Lists.newArrayList();
        summaryFields.add(new ExportField("", "@", 5620));
        for (Date date : dates) {
            summaryFields.add(new ExportField(
                "周" + DateUtil.getChineseWeekDay(date) + "(" + DateUtil.getDayStr(date) + ")", "@", 5120));
        }
        List titles = Lists.newArrayList();
        titles.add("课节汇总表");
        ExcelCourseExportService.createSheet(workbook, "课节汇总", titles, cellStyleMap, summaryFields, dtoList, startDate);
        File excelFile = new File("/Users/wengshengli/Documents/test1.xlsx");
        workbook.write(new FileOutputStream(excelFile));
    }

    @Override
    public HeaderDto getHeaderInfo() {
        HeaderDto dto = new HeaderDto();
        Integer txCascadeAccountId = TianxiaoMContext.getTXCascadeId();
        if (txCascadeAccountId == null) {
            return dto;
        }
        OrgAccount orgAccount = orgAccountDao.getAccountById(TianxiaoMContext.getOrgId());
        if (orgAccount == null) {
            log.info("orgAccount doesn't exist! orgAccountId:{}", TianxiaoMContext.getOrgId());
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "您无权使用此功能");
        }
        TXCascadeAccount txCascadeAccount = txCascadeAccountDao.getById(txCascadeAccountId);
        if (txCascadeAccount == null) {
            log.info("txCascadeAccount doesn't exist! txCascadeAccountId:{}", txCascadeAccountId);
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "您无权使用此功能");
        }
        // 员工显示自己的班级
        // 主管显示机构的班级
        List<Long> courseIds = Lists.newArrayList();
        Integer notSignCount = 0;
        if (txCascadeAccount.getAccountType() == CascadeType.STAFF.getValue()) {
            courseIds = orgCourseDao.getCourseIdsByCascadeId(txCascadeAccountId);
            if(CollectionUtils.isNotEmpty(courseIds)){
                notSignCount = orgClassLessonDao.getLessonCount(orgAccount.getId(), courseIds, new Date());
            }
        }
        if (txCascadeAccount.getAccountType() == CascadeType.MANAGER.getValue()) {
            notSignCount = orgClassLessonDao.getLessonCount(orgAccount.getId(), courseIds, new Date());
            courseIds = orgCourseDao.getCourseIds(orgAccount.getNumber().longValue(), null);
        }
        Integer totalCount = courseIds.size();
        dto.setClassCount(totalCount);
        dto.setSignCount(notSignCount);
        return dto;
    }

    @Override
    @Transactional(readOnly = true)
    public ListLessonResponseDto  queryCourseLessons(Long orgId,ListLessonsRequestDto params){
        Map<Long, Long> lessonIdTeacherIdMap = new HashMap<>();
        List<OrgClassLesson> lessons = queryLesson(orgId, lessonIdTeacherIdMap, params, true);
        List<Long> courseIds = new ArrayList<Long>();
        Set<Long> roomIds = Sets.newHashSet();
        Set<Long> lessonIds = Sets.newHashSet();
        ListLessonResponseDto result = new ListLessonResponseDto();
        if (CollectionUtils.isNotEmpty(lessons)) {
            Collections.sort(lessons, new Comparator<OrgClassLesson>() {
                @Override
                public int compare(OrgClassLesson o1, OrgClassLesson o2) {
                    return o1.getNumber().compareTo(o2.getNumber());
                }
            });
            for (OrgClassLesson lesson : lessons) {
                courseIds.add(lesson.getCourseId());
                roomIds.add(lesson.getRoomId());
                lessonIds.add(lesson.getId());
            }

            Map<Long, OrgCourse> courseMap = getOrgCourseMap(orgId, courseIds);
            

            Map<Long, Integer> studentCountMap = orgStudentLessonDao.queryLessonStudentCountMap(orgId, lessonIds);
            Map<Long, Integer> signStudentCountMap = Maps.newHashMap();
            if (!params.isNeedStudentSignStatus() && params.isNeedSignInfo()) {
                signStudentCountMap = orgLessonSignDao.queryLessonStudentCountMap(orgId, lessonIds);
            }
            Map<Long, Integer> commentCount = Maps.newHashMap();
            Map<Long, Integer> lessonSignStatus = Maps.newHashMap();
            if (params.isNeedStudentSignStatus()) {
                Preconditions.checkArgument(params.getStudentId() != null && params.getStudentId() > 0, "学生ID不正确");
                Long userId = this.orgStudentDao.getUserId(params.getStudentId());

                commentCount = orgLessonCommentDao.getLessonCommentCountMap(lessonIds, orgId, userId, null,
                    UserRole.STUDENT.getRole(), false);

                List<OrgLessonSign> lessonSign = orgLessonSignDao.getStudentSign(orgId, userId,
                    UserRole.STUDENT.getRole(), lessonIds, "lessonId", "status");
                if (CollectionUtils.isNotEmpty(lessonSign)) {
                    lessonSignStatus = CollectorUtil.collectMap(lessonSign, new Function<OrgLessonSign, Long>() {
                        @Override
                        public Long apply(OrgLessonSign input) {
                            return input.getLessonId();
                        }
                    }, new Function<OrgLessonSign, Integer>() {
                        @Override
                        public Integer apply(OrgLessonSign input) {
                            return input.getStatus();
                        }
                    });
                    log.debug("student lesson sign:{}", lessonSignStatus);
                }
            }

            Map<Long, String> lessonIdTeacherNameMap = Maps.newHashMap();
            Map<Long, OrgClassRoom> classRoomMap = Maps.newHashMap();
            result = buildListLessonResponseDto(lessons, courseMap, lessonIdTeacherNameMap, classRoomMap,
                studentCountMap, signStudentCountMap, commentCount, lessonSignStatus);
            
            Date now = new Date();
            int lastRow = result.getList().size()-1;
            result.setNewLessonId(result.getList().get(lastRow).getLessonId());
            for(LessonResponseDto lesson:result.getList()){
                if(now.before(lesson.getLessonEndTime())){
                    result.setNewLessonId(lesson.getLessonId());
                    break;
                }
            }
            
        } else {
            result.setFirstLessonStartTime(params.getFirstLessonStartTime());
            result.setLastLessonStartTime(params.getLastLessonStartTime());
            if (params.getFirstLessonStartTime() == null && params.getLastLessonStartTime() == null) {
                result.setFirstLessonStartTime(DateUtil.getCurrentDate());
            }
        }
        return result;
        
        
    }
    @Override
    @Transactional
    public void batchOperateLesson(Long orgId,Long courseId,List<Long> lessonIds,BatchOperateType operateType,String targetId){
      //  List<Long> leftLessonIds = orgClassLessonDao.getLeftLessonIds(orgId, courseId);
        
        /*if(!leftLessonIds.containsAll(lessonIds)){
            log.warn("leftlessonIds:{} is not contains lessonIds:{}", leftLessonIds, lessonIds);
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "只支持操作未开课的课节");
        }*/
        
        if (CollectionUtils.isNotEmpty(lessonIds) ) {
            
            if(operateType==BatchOperateType.del_lesson){
                delLesson(orgId,courseId,lessonIds,targetId);
            }
            else if(operateType==BatchOperateType.add_student){
                addStudent(orgId,courseId,lessonIds,targetId);
            }
            else if(operateType==BatchOperateType.del_student){
                delStudent(orgId,courseId,lessonIds,targetId);
            }
            else if(operateType==BatchOperateType.change_teacher){
                Long teacherId = Long.valueOf(targetId);
                changeTeacher(orgId,courseId,lessonIds,teacherId);
            }
            else if(operateType==BatchOperateType.change_classroom){
                Long roomId = Long.valueOf(targetId);
                changeClassroom(orgId,courseId,lessonIds,roomId);
            }
            else if(operateType==BatchOperateType.change_time){
                String[] time = targetId.split(",");
                String startTimeStr = time[0];
                String endTimeStr = time[1];
                changeTime(orgId,courseId,lessonIds,startTimeStr,endTimeStr);
            }
        }
        
    }
    
    
    private void delLesson(Long orgId,Long courseId,List<Long> lessonIds,String targetId){
        
        Map<Long, Integer> signCountMap = orgLessonSignDao.queryLessonStudentCountMap(orgId, lessonIds);
        
        for(Long lessonId:lessonIds){
            Integer tempCount = signCountMap.get(lessonId);
            if(tempCount != null && tempCount>0){
                throw new BussinessException(CommonErrorCode.PARAM_ERROR, "课程已签到，不能删除");
            }
        }
        
        orgClassLessonDao.delByIds(lessonIds);
        
        Map<String, Object> condition = Maps.newHashMap();
        condition.put("orgId", orgId);
        condition.put("lessonId", lessonIds);
        orgStudentLessonDao.delByCondition(condition);
        
        Map<String, Object> delCondition = new HashMap<>();
        delCondition.put("lessonId", lessonIds);
        orgTeacherLessonDao.delByCondition(delCondition);
        
        resetLessonNumber(orgId, courseId);
        
    }
    private void addStudent(Long orgId,Long courseId,List<Long> lessonIds,String targetId){
       // String[] array = targetId.split(",");
      /*  Collection<Long> studentIds = Lists.newArrayList();
        for(String idStr:array){
            studentIds.add(Long.valueOf(idStr));
        }*/
        
        List<Long> studentIds = null;
        try {
            studentIds = JacksonUtil.str2Obj(targetId, new TypeReference<List<Long>>() {
            });
        } catch (Exception e1) {
            log.warn("json class cast exception");
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "学生id转换一场");
        } 
       // courseStudentService.addStudentsToLessons(orgId, courseId, lessonIds, userIds, false);
        
        if (CollectionUtils.isEmpty(studentIds)) {
            log.warn("add student ids is empty");
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "学生id为空");
           // return;
        }
        if (true) {
            Map<Long, Long> userIdMap = orgStudentDao.getStudentIdUserIdMap(studentIds);
            if (MapUtils.isEmpty(userIdMap)) {
                log.warn("add student ids is empty");
                throw new BussinessException(CommonErrorCode.PARAM_ERROR, "学生id为空");
               // return;
            }
            studentIds = new ArrayList<Long>(userIdMap.values());
        }

      
        List<Long> courseStudents = orgStudentCourseDao.getStudents(orgId, courseId, 0);
        if (!courseStudents.containsAll(studentIds)) {
            studentIds.retainAll(courseStudents);
            log.warn("lesson studeng:{} is not all course student ids:{}", studentIds, courseStudents);
            
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "必须从课程学生中添加课节学生");
        }

        Map<Long, List<Long>> lessonStudentIds = orgStudentLessonDao.getLessonStudentList(lessonIds);
        List<OrgStudentLesson> delLessonStudent = orgStudentLessonDao.getLessonStudentListDelStatus(lessonIds);
        Map<String,OrgStudentLesson> lesseonStudentMap = Maps.newHashMap();
        for(OrgStudentLesson orgStudentLesson:delLessonStudent){
            String tempKey = orgStudentLesson.getUserId()+"|"+orgStudentLesson.getLessonId();
            lesseonStudentMap.put(tempKey, orgStudentLesson);
        }
        log.info("check orgStudentLessonDao.getLessonStudentList(lessonIds):{},result={}",lessonIds,lessonStudentIds);
        List<OrgStudentLesson> saveStudentLessons = Lists.newArrayList();
        List<OrgStudentLesson> updateStudentLessons = Lists.newArrayList();
        for( Long lessonId:lessonIds){
            //Map<Long, List<Long>> lessonStudentIds = orgStudentLessonDao.getLessonStudentList(Lists.newArrayList(lessonId));
            //List<Long> tempStudentId = lessonStudentIds.get(lessonId);
            List<Long> tempStudentIds = new ArrayList<Long>(studentIds);
            if (lessonStudentIds.containsKey(lessonId)) {
                
                List<Long> addedStuIds = lessonStudentIds.get(lessonId);
                if (CollectionUtils.isNotEmpty(addedStuIds)) {
                    tempStudentIds.removeAll(addedStuIds);
                }
            }

            
            OrgStudentLesson po = null;
            for (Long studentId : tempStudentIds) {
                po = new OrgStudentLesson();
                po.setCreateTime(new Date());
                po.setLessonId(lessonId);
                po.setOrgId(orgId);
                po.setUserId(studentId);
                String tempKey = studentId+"|"+lessonId;
                if(lesseonStudentMap.containsKey(tempKey)){
                    OrgStudentLesson tempStudentLesson = lesseonStudentMap.get(tempKey);
                    po.setId(tempStudentLesson.getId());
                    po.setDelStatus(0);
                    po.setStartStatus(0);
                    updateStudentLessons.add(po);
                }else{
                    saveStudentLessons.add(po);
                }
                
            }
            
        }
        
        log.info("save students :{} to lessonid:{}", saveStudentLessons, lessonIds);
        orgStudentLessonDao.saveAll(saveStudentLessons, "createTime", "lessonId", "orgId", "userId");
        
        for(OrgStudentLesson updateStudentLesson:updateStudentLessons){
            orgStudentLessonDao.update(updateStudentLesson, "createTime", "delStatus", "startStatus");
        }
        
        
    }
    
    private int delStudent(Long orgId,Long courseId,List<Long> lessonIds,String targetId){
        List<Long> studentIds = null;
        try {
            studentIds = JacksonUtil.str2Obj(targetId, new TypeReference<List<Long>>() {
            });
        } catch (Exception e1) {
            log.warn("json class cast exception");
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "学生id转换一场");
        } 
       // courseStudentService.addStudentsToLessons(orgId, courseId, lessonIds, userIds, false);
        
        if (CollectionUtils.isEmpty(studentIds)) {
            log.warn("add student ids is empty");
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "学生id为空");
           // return;
        }
        if (true) {
            Map<Long, Long> userIdMap = orgStudentDao.getStudentIdUserIdMap(studentIds);
            if (MapUtils.isEmpty(userIdMap)) {
                log.warn("add student ids is empty");
                throw new BussinessException(CommonErrorCode.PARAM_ERROR, "学生id为空");
               // return;
            }
            studentIds = new ArrayList<Long>(userIdMap.values());
        }
        
        
        List<Long> courseStudents = orgStudentCourseDao.getStudents(orgId, courseId, 0);
        
        if (!courseStudents.containsAll(studentIds)) {
            log.warn("lesson teacher:{} is not all course teacher ids:{}", studentIds, courseStudents);
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "必须从课程学生中删除课节学生");
        }
        
        
        Map<String,Object> countCondition = new HashMap<String,Object>();
        countCondition.put("orgId", orgId);
        countCondition.put("courseId", courseId);
        countCondition.put("lessonId", lessonIds);
        countCondition.put("userId",studentIds);
        countCondition.put("status", SignStatus.SIGNED.getCode());
        countCondition.put("userRole", UserRoleEnum.STUDENT.getCode());
        int count = orgLessonSignDao.countByCondition(countCondition, "userId", false);
        
        if(count >0){
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "已经签到的学生不能删除");
        }
        
        Map<String, Object> condition = Maps.newHashMap();
        condition.put("orgId", orgId);
        condition.put("lessonId", lessonIds);
        condition.put("userId", studentIds);
        return orgStudentLessonDao.delByCondition(condition);
    }
    
    private void changeTeacher(Long orgId,Long courseId,List<Long> lessonIds,Long teacherId){
        
       // Long teacherId = Long.valueOf(targetId);
        
        Map<String, Object> delCondition = new HashMap<>();
        delCondition.put("lessonId", lessonIds);
        int result = orgTeacherLessonDao.delByCondition(delCondition);
        log.info("del from org teacher lesson by lessonId:{},result:{}", lessonIds, result);
        
        List<OrgTeacherLesson> saveLessonTeachers = Lists.newArrayList();
        OrgTeacherLesson po = null;
        for (Long lessonId:lessonIds) {
            po = new OrgTeacherLesson();
            po.setCreateTime(new Date());
            po.setLessonId(lessonId);
            po.setOrgId(orgId);
            po.setTeacherId(teacherId);
            saveLessonTeachers.add(po);
        }
        log.info("save teachers :{} to lessonId:{}", saveLessonTeachers, lessonIds);
        orgTeacherLessonDao.saveAll(saveLessonTeachers, "createTime", "lessonId", "orgId", "teacherId");
        
        List<OrgCourseTeacher> courseTeachers = orgCourseTeacherDao.getTeacher(orgId, courseId);
        Set<Long> userIds = Sets.newHashSet();
        for (OrgCourseTeacher courseTeacher : courseTeachers) {
            userIds.add(courseTeacher.getUserId());
        }
        
        if(!userIds.contains(teacherId)){
            List<Long> teacherIds  = new ArrayList<Long>();
            teacherIds.add(teacherId);
            courseTeacherService.addTeacherToCourse(orgId, courseId, teacherIds);
        }
        
    }
    
    private void changeClassroom(Long orgId,Long courseId,List<Long> lessonIds,Long roomId){
       // Long roomId = Long.valueOf(targetId);
        
        orgClassLessonDao.updateLessonClassRoom(orgId, lessonIds,roomId);
        
        List<ClassRoomDto> roomList = classRoomService.listOrgCourseRooms(orgId, courseId);
        List<Long> currentRoomIds = new ArrayList<Long>();
        for(ClassRoomDto dto:roomList){
            currentRoomIds.add(dto.getRoomId());
        }
        
        if(!currentRoomIds.contains(roomId)){
            List<Long> newCourseIds = new ArrayList<Long>();
            newCourseIds.add(roomId);
            classRoomService.addClassRoomCourse(orgId, courseId, newCourseIds);
        }
        
    }
    
    private void changeTime(Long orgId,Long courseId,List<Long> lessonIds,String startTimeStr,String endTimeStr){
        
        TimeStamp startTime = TimeStamp.parse(startTimeStr);
        TimeStamp endTime = TimeStamp.parse(endTimeStr);
        
        List<OrgClassLesson> list = orgClassLessonDao.getByIds(lessonIds);
        
        for(OrgClassLesson lesson:list){
            
            Calendar cal = Calendar.getInstance();
            cal.setTime(lesson.getStartTime());
            
            LessonTime lessonTime = new LessonTime(cal, startTime, endTime);
            
            Map<String,Object> map = new HashMap<String,Object>();
            map.put("id", lesson.getId());
            map.put("startTime", lessonTime.getStartTime());
            map.put("endTime", lessonTime.getEndTime());
            
            this.orgClassLessonDao.update(map, "startTime","endTime");
          
        }
        resetLessonNumber(orgId, courseId);
    }
    
    @Override
    @Transactional(rollbackFor = Exception.class)
    public List<Long> addClassLessonsBatchForFastSignIn(Long orgId, Long courseId,Integer addLessonTimes,Set<Long> studentIds) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
       
        Preconditions.checkArgument(courseId != null && courseId > 0, "courseId is illegal");
        
        Long defaultTeacherId = 0L;
        //快速签到的排课不给默认老师
        /*List<TeacherResponseDto> courseTeachers = courseTeacherService.listCourseTeacher(orgId, courseId);
        if(courseTeachers.size()>0){
            defaultTeacherId = courseTeachers.get(0).getTeacherId();
        }*/
        Long defaultRoomId = 0L;
        
        //快速签到的排课不给默认的教室
        /*List<ClassRoomDto> courseRooms = classRoomService.listOrgCourseRooms(orgId, courseId);
        if(courseRooms.size()>0){
            defaultRoomId = courseRooms.get(0).getRoomId();
        }*/

        OrgClassLesson lesson = null;
        List<OrgClassLesson> lessons = Lists.newArrayList();
        Date todayStartTime = DateUtil.getStartOfDay(new Date());
        Date startTime = DateUtil.getDiffDateTime(todayStartTime, 6, Calendar.HOUR_OF_DAY);
        Date endTime = DateUtil.getDiffDateTime(todayStartTime, 7, Calendar.HOUR_OF_DAY);
        
        for (int i=0;i<addLessonTimes;i++) {
            lesson = new OrgClassLesson();
            lesson.setStartTime(startTime);
            lesson.setEndTime(endTime);
            lesson.setCourseId(courseId);
            lesson.setCreateTime(new Date());
            lesson.setDelStatus(DataStatus.NORMAL.getValue());
            lesson.setLayoutId(0l);
            lesson.setRoomId(defaultRoomId !=null?defaultRoomId:0L);
            lesson.setUpdateTime(new Date());
            lesson.setNumber(i+1);
            lesson.setOrgId(orgId);
            lessons.add(lesson);
        
        }
        log.info("save {} lessons into db,params:{}", lessons.size(),"courseId="+courseId );
        orgClassLessonDao.saveAll(lessons, "startTime", "endTime", "courseId", "createTime", "delStatus", "layoutId",
            "roomId", "updateTime", "orgId");

       

        // 如果原来有课节的话,需要重新安排课节编号
        resetLessonNumber(orgId, courseId);
        List<Long> newLessonsIds = Lists.newArrayList();
        for (OrgClassLesson classLesson : lessons) {
            newLessonsIds.add(classLesson.getId());
            courseStudentService.addStudentToLesson(orgId, classLesson.getCourseId(), classLesson.getId(),
                studentIds, true);
            if(defaultTeacherId !=null && defaultTeacherId>0){
                List<Long> tearcherIds = Lists.newArrayList();
                tearcherIds.add(defaultTeacherId);
                courseTeacherService.addTeacherToLesson(orgId, classLesson.getCourseId(), classLesson.getId(),
                    tearcherIds);
            }
            
        }
        
        
        
        return newLessonsIds;
    }
    
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void batchEditLessonsForFastSignIn(FastSignInLessonsEditRequestDto dto){
      
        List<Long> lessonIds = dto.getLessonIds();
        Long orgId = dto.getOrgId();
        Long courseId = dto.getCourseId();
        Long roomId = dto.getRoomId();
        Long teacherId = dto.getTeacherId();
        String startTimeStr = dto.getStartTime();
        String endTimeStr = dto.getEndTime();
        if (CollectionUtils.isNotEmpty(lessonIds)) {

            if(roomId!=null &&roomId >0){
                changeClassroom(orgId, courseId, lessonIds, roomId);
            }
            if(teacherId !=null && teacherId >0){
                changeTeacher(orgId, courseId, lessonIds, teacherId);
            }
            if(StringUtils.isNotBlank(startTimeStr) && StringUtils.isNotBlank(endTimeStr)){
                changeTime(orgId, courseId, lessonIds, startTimeStr, endTimeStr);
                
            }
            
            

        }
        
    }
    
    @Override
    @Transactional(rollbackFor = Exception.class)
    public List<LessonResponseDto> findLessonBaseInfoByLessonIds(Long orgId,Long courseId,List<Long> lessonIds){
        List<OrgClassLesson> lessons = orgClassLessonDao.queryLessons(orgId, null, null, lessonIds, null,
        null, null);
        
        return buildLessonBaseInfoList(orgId,courseId,lessons);
        
    }

    @Override
    @Transactional
    public List<LessonResponseDto> buildLessonBaseInfoList(Long orgId,Long courseId,List<OrgClassLesson> lessons){
        
        List<Long> lessonIds = Lists.newArrayList();
        for(OrgClassLesson lesson:lessons){
            lessonIds.add(lesson.getId());
        }
        
        Set<Long> roomIds = new HashSet<Long>();
        
        for(OrgClassLesson lesson:lessons){
            if(lesson.getRoomId() != null && lesson.getRoomId() >0){
                roomIds.add(lesson.getRoomId());
            }
        }
        Map<Long,Long> lessonTeacherIdMap = orgTeacherLessonDao.queryLessonTeacherIdMap(orgId, lessonIds);
        
        Map<Long, String> teacherNameMap = teacherDao.getTeacherRealNameMap(lessonTeacherIdMap.values());
        Map<Long, String> lessonIdTeacherNameMap = new HashMap<>(lessonTeacherIdMap.size());
        for (Long lessonId : lessonTeacherIdMap.keySet()) {
            lessonIdTeacherNameMap.put(lessonId, teacherNameMap.get(lessonTeacherIdMap.get(lessonId)));
        }
        
        Map<Long, OrgClassRoom> lessonRoomMap = getOrgClassRoomMap(roomIds);
        
        List<LessonResponseDto> dtoList = new ArrayList<LessonResponseDto>();
        for(OrgClassLesson lesson:lessons){
            LessonResponseDto dto = new LessonResponseDto();
            dto.setCourseId(lesson.getCourseId());
            dto.setLessonId(lesson.getId());
            dto.setIndex(lesson.getNumber());
            Long roomId = lesson.getRoomId();
            
            OrgClassRoom classRoom = lessonRoomMap.get(roomId);
            if(classRoom!=null){
                dto.setRoomId(roomId);
                dto.setRoomName(classRoom.getRoomName());
            }
            
            Long teacherId = lessonTeacherIdMap.get(lesson.getId());
            if(teacherId!=null && teacherId >0){
                dto.setTeacherId(teacherId);
                dto.setTeacherName(teacherNameMap.get(teacherId));
            }
            
            dto.setLessonStartTime(lesson.getStartTime());
            dto.setLessonEndTime(lesson.getEndTime());
            
            dtoList.add(dto);
        }
        
        return dtoList;
        
    }
    
}
