package com.baijia.tianxiao.biz.marketing.referral.service.impl;

import com.baijia.tianxiao.biz.marketing.referral.service.BizReferralService;
import com.baijia.tianxiao.biz.marketing.utils.PictureUtil;
import com.baijia.tianxiao.dal.activity.dao.referral.GiftInfoDao;
import com.baijia.tianxiao.dal.activity.dao.referral.ReferralInfoDao;
import com.baijia.tianxiao.dal.activity.po.referral.GiftInfo;
import com.baijia.tianxiao.dal.activity.po.referral.ReferralInfo;
import com.baijia.tianxiao.sal.marketing.activity.dto.ShareDto;
import com.baijia.tianxiao.sal.marketing.commons.constants.Config;
import com.baijia.tianxiao.sal.marketing.commons.enums.ConstantEnums;
import com.baijia.tianxiao.sal.marketing.commons.enums.EmailType;
import com.baijia.tianxiao.sal.marketing.commons.enums.TemplateTypeCategory;
import com.baijia.tianxiao.sal.marketing.commons.service.RedisService;
import com.baijia.tianxiao.sal.marketing.commons.utils.ActivityMailSender;
import com.baijia.tianxiao.sal.marketing.commons.utils.Mail;
import com.baijia.tianxiao.sal.marketing.commons.utils.TupleUtil;
import com.baijia.tianxiao.sal.marketing.commons.utils.TwoTuple;
import com.baijia.tianxiao.sal.marketing.referral.dto.BrokerDto;
import com.baijia.tianxiao.sal.marketing.referral.dto.CommissionDto;
import com.baijia.tianxiao.sal.marketing.referral.dto.CommissionListDto;
import com.baijia.tianxiao.sal.marketing.referral.dto.ReferralCustomer;
import com.baijia.tianxiao.sal.marketing.referral.dto.ReferralOrgInfo;
import com.baijia.tianxiao.sal.marketing.referral.dto.ReferralRequest;
import com.baijia.tianxiao.sal.marketing.referral.dto.ReferralResponse;
import com.baijia.tianxiao.sal.marketing.referral.service.ReferralRecordService;
import com.baijia.tianxiao.sal.marketing.referral.service.ReferralService;
import com.baijia.tianxiao.sal.organization.org.dto.OrgInfoSimpleDto;
import com.baijia.tianxiao.sal.organization.org.service.OrgInfoService;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.GenericsUtils;

import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import lombok.extern.slf4j.Slf4j;

/**
 * @author Rezar
 * @createDate :Mar 3, 2016 2:28:59 PM
 * @desc :
 */
@Slf4j
@Service
public class BizReferralServiceImpl implements BizReferralService {

    @Autowired
    private ReferralService referralService;
    @Autowired
    private OrgInfoService orgInfoService;
    @Autowired
    private GiftInfoDao giftInfoDao;
    @Autowired
    private ReferralInfoDao referralInfoDao;
    @Autowired
    private ReferralRecordService referralRecordService;
    @Autowired
    private RedisService redisService;

    @Override
    public ShareDto getShareInfo(Long activityId, Long orgId) {
        ShareDto dto = new ShareDto();

        ReferralInfo referralInfo = this.referralInfoDao.getReferralInfo(activityId, orgId);
        if (referralInfo == null) {
            return dto;
        }
        dto.setTitle(referralInfo.getTitle());
        try {
            String logo = orgInfoService.getOrgInfo(orgId).getLogo();
            dto.setImageUrl(PictureUtil.handlePictureSize(logo));
        } catch (Exception e) {
            log.error("[ActivityShareService ] ", e);
        }
        dto.setContent(ConstantEnums.REFERRAL_SLOGAN.value());
        dto.setShareUrl(getUrl(activityId));
        return dto;
    }

    private String getUrl(long activityId) {
        return Config.DOMAIN + "/referral.html?activityId=" + activityId;
    }

    /**
     * 渲染 转介绍 活动
     */
    @Override
    public ReferralResponse renderActivity(ReferralRequest request) {
        ReferralResponse rr = this.referralService.findDetail(request);
        if (rr == null) {
            return null;
        }
        try {
            OrgInfoSimpleDto orgInfo = this.orgInfoService.getOrgInfo(rr.getOrgId());
            //微官网连接
            String weiPage = ConstantEnums.ORG_WEI_PAGE.value().trim() + orgInfo.getOrgNumber();
            rr.setOrgWeiPage(weiPage);
            
            // 需要查询到机构主页的链接
            String logo = orgInfo.getLogo();
            String shortName = orgInfo.getShortName();
            Long orgNum = orgInfo.getOrgNumber();
//            String homePage = ConstantEnums.ORG_HOME_PAGE_URL.value().trim() + orgNum;
            rr.setOrgLogo(logo);
            rr.setOrgName(shortName);
            //pm要求点击二维码跳转到微官网上 将 homePage改为weiPage
            rr.setOrgQrCodeUrl(weiPage);
            rr.setOrgNumber(orgInfo.getOrgNumber());
            List<GiftInfo> giftInfos = this.giftInfoDao.selectGifts(request.getActivityId());
            rr.setGifts(giftInfos);

        } catch (Exception e) {
            log.error("[error while render ]", e);
        }
        return rr;
    }

    /**
     * 
     * 导出佣金结算列表
     * 
     * @param request
     * @return
     */
    @Override
    public TwoTuple<Integer, String> exportToPayAmountList(ReferralRequest request) {
        final Long activityId = request.getActivityId();
        String orgName = findOrg(request);
        Long orgId = request.getOrgId();
        if (GenericsUtils.isNullOrEmpty(orgName)) {
            log.info("can not find an org with org_ID {}", request.getOrgId());
            return TupleUtil.tuple(400, "机构不存在");
        }
        final String email = request.getEmail();
        String settleMentedMonth = request.getSettleMentedMonth();
        String needSettleMentedMonth = request.getNeedSettleMentedMonth();
        return createToPayAmountListMail(orgId, activityId, settleMentedMonth, needSettleMentedMonth, orgName, email);
    }

    private TwoTuple<Integer, String> createToPayAmountListMail(Long orgId, Long activityId, String settleMentedMonth,
        String needSettleMentedMonth, String orgName, String email) {
        PageDto pageDto = new PageDto();
        pageDto.setPageNum(1);

        // get all record
        List<CommissionDto> retListOver = getReferralRecord(activityId, settleMentedMonth, 1, pageDto);
        pageDto.setPageNum(1);
        List<CommissionDto> retListNotOver = getReferralRecord(activityId, needSettleMentedMonth, 2, pageDto);

        log.info("retListOver is : {} ", retListOver);

        log.info("retListNotOver are :{} ", retListNotOver);

        if (GenericsUtils.isNullOrEmpty(retListOver) && GenericsUtils.isNullOrEmpty(retListNotOver)) {
            log.info("没有结果数据");
            return TupleUtil.tuple(400, "暂无结果数据");
        }

        // check the send limit
        TwoTuple<Integer, String> checkLimit = checkLimit(orgId, activityId);
        if (checkLimit != null) {
            return checkLimit;
        }
        String subject = "转介绍佣金结算列表导出";
        String content = "HI " + orgName + "<br/> 附件里是您申请导出的转介绍佣金结算列表，请您查收 " + SUFFIX;
        Mail mail = ActivityMailSender.createMail(email, null, subject, content);
        if (GenericsUtils.notNullAndEmpty(retListOver)) {
            List<Map<String, String>> rows = GetRows(retListOver);
            ActivityMailSender.addAttache(mail, rows, settleMentedMonth + "_1_" + ".xls");
        }
        if (GenericsUtils.notNullAndEmpty(retListNotOver)) {
            List<Map<String, String>> rows2 = GetRows(retListNotOver);
            ActivityMailSender.addAttache(mail, rows2, needSettleMentedMonth + "_2_.xls");
        }

        ActivityMailSender.sendMail(email, mail, true);
        return TupleUtil.tuple(TupleUtil.OK, "邮件发送成功");
    }

    private List<Map<String, String>> GetRows(List<CommissionDto> retList) {
        List<Map<String, String>> rows = new ArrayList<>();
        for (CommissionDto record : retList) {
            Map<String, String> rowMap = new LinkedHashMap<>();
            String brokerName = record.getBrokerName();
            String phone = record.getBrokerPhone();
            String amountNum = record.getAccountNum();
            String amount = record.getAmount();
            String settleStatus = record.getSettleTime() == null ? "待结算" : "已结算";
            rowMap.put("经纪人", brokerName);
            rowMap.put("电话", phone);
            rowMap.put("支付宝账户", amountNum);
            rowMap.put("待付金额", amount);
            rowMap.put("支付状态", settleStatus);
            rows.add(rowMap);
        }
        return rows;
    }

    private List<CommissionDto> getReferralRecord(Long activityId, String month, Integer status, PageDto pageDto) {
        List<CommissionDto> retList = new ArrayList<>();
        CommissionListDto commissionListByMonth =
            this.referralRecordService.getCommissionListByMonth(activityId, month, status, pageDto);
        retList.addAll(commissionListByMonth.getList());
        while (commissionListByMonth.getHasMore() > 0) {
            pageDto.setPageNum(pageDto.getPageNum() + 1);
            commissionListByMonth =
                this.referralRecordService.getCommissionListByMonth(activityId, month, status, pageDto);
            retList.addAll(commissionListByMonth.getList());
        }
        return retList;
    }

    @Override
    public TwoTuple<Integer, String> exportCustomerList(ReferralRequest request) {
        final Long activityId = request.getActivityId();
        String orgName = findOrg(request);
        Long orgId = request.getOrgId();
        if (GenericsUtils.isNullOrEmpty(orgName)) {
            log.info("can not find an org with org_ID {}", request.getOrgId());
            return TupleUtil.tuple(400, "机构不存在");
        }
        final String email = request.getEmail();
        return createCustomerListMail(orgId, activityId, orgName, email);
    }

    private TwoTuple<Integer, String> createCustomerListMail(Long orgId, Long activityId, String orgName,
        String email) {
        ReferralInfo referralInfo = this.referralInfoDao.getReferralInfo(activityId, orgId);
        if (referralInfo == null) {
            return TupleUtil.tuple(TupleUtil.NO_OK, "活动不存在");
        }

        List<ReferralCustomer> customerList = this.referralRecordService.getCustomerListByActivityId(activityId, null);

        if (GenericsUtils.isNullOrEmpty(customerList)) {
            return TupleUtil.tuple(TupleUtil.NO_OK, "暂无客户数据");
        }

        TwoTuple<Integer, String> checkLimit = checkLimit(orgId, activityId);
        if (checkLimit != null) {
            return checkLimit;
        }
        final List<Map<String, String>> rows = new ArrayList<>();
        for (ReferralCustomer customer : customerList) {
            Map<String, String> rowMap = new LinkedHashMap<>();
            String name = customer.getReferralName();
            String phone = customer.getReferralPhone();
            String brokerName = customer.getBrokerName();
            String brokerPhone = customer.getBrokerPhone();
            String desc = customer.getReferralDesc();
            rowMap.put("客户姓名", name);
            rowMap.put("客户电话", phone);
            rowMap.put("经纪人姓名", brokerName);
            rowMap.put("经纪人手机号", brokerPhone);
            rowMap.put("推荐时间", DateFormatUtils.format(new Date(customer.getCreateTime()), "yyyy-MM-dd HH:mm:ss"));
            rowMap.put("推荐理由", desc);
            rows.add(rowMap);
        }
        String subject = "转介绍客户列表导出";
        String content = "HI " + orgName + "<br/> 附件里是您申请导出的转介绍客户列表，请您查收 " + SUFFIX;
        ActivityMailSender.sendMail(email, subject, content, activityId, rows);
        return TupleUtil.tuple(TupleUtil.OK, "邮件发送成功");
    }

    /**
     * 校验邮件发送次数的上限
     * 
     * @param orgId
     * @param activityId
     * @return
     */
    private TwoTuple<Integer, String> checkLimit(Long orgId, Long activityId) {
        boolean isNotLimited = redisService.addOrgEmailCount(orgId, activityId,
            TemplateTypeCategory.REFERRAL_TYPE.getType(), EmailType.BROKERS);
        if (!isNotLimited) {
            log.info("次数已达上限，请明天再发");
            return TupleUtil.tuple(TupleUtil.NO_OK, "今日已发送" + ConstantEnums.EMAIL_COUNT_LIMIT.value() + "次，请明日再试");
        }
        return null;
    }

    @Override
    public TwoTuple<Integer, String> exportBrokerList(ReferralRequest request) {
        final Long activityId = request.getActivityId();
        String orgName = findOrg(request);
        if (GenericsUtils.isNullOrEmpty(orgName)) {
            log.info("can not find an org with org_ID {}", request.getOrgId());
            return TupleUtil.tuple(400, "机构不存在");
        }
        final String email = request.getEmail();
        return createBrokerListMail(request.getOrgId(), activityId, orgName, email);
    }

    public String findOrg(ReferralRequest request) {
        Long orgId = request.getOrgId();
        OrgInfoSimpleDto simple = null;
        try {
            simple = orgInfoService.getOrgInfo(orgId);
        } catch (Exception e) {
            return null;
        }
        final String orgName = simple.getShortName();
        return orgName;
    }

    /**
     * 生成创建经济人数据
     *
     * @param activityId
     * @param orgName
     * @param email
     * @return
     */
    private TwoTuple<Integer, String> createBrokerListMail(Long orgId, Long activityId, String orgName, String email) {
        ReferralInfo referralInfo = this.referralInfoDao.getReferralInfo(activityId, orgId);
        if (referralInfo == null) {
            return TupleUtil.tuple(TupleUtil.NO_OK, "活动不存在");
        }

        List<BrokerDto> result = this.referralRecordService.getBrokerList(activityId, null);

        if (GenericsUtils.isNullOrEmpty(result)) {
            return TupleUtil.tuple(TupleUtil.NO_OK, "暂无经纪人数据");
        }

        checkLimit(orgId, activityId);
        final List<Map<String, String>> rows = new ArrayList<>();
        for (BrokerDto broker : result) {
            Map<String, String> rowMap = new LinkedHashMap<>();
            String name = broker.getBrokerName();
            String phone = broker.getBrokerPhone();
            // String accountType = BrokerDto.getBrokerAccDesc(broker.getAccountType());
            String accountNum = broker.getAccountNum();
            rowMap.put("经纪人姓名", name);
            rowMap.put("经纪人手机号", phone);
            rowMap.put("经纪人支付宝账号", accountNum);
            rows.add(rowMap);
        }
        String subject = "转介绍经纪人列表导出";
        String content = "HI " + orgName + "<br/> 附件里是您申请导出的转介绍经纪人列表，请您查收 " + SUFFIX;
        ActivityMailSender.sendMail(email, subject, content, activityId, rows);
        return TupleUtil.tuple(TupleUtil.OK, "邮件发送成功");
    }

    private static final String SUFFIX = "<br/><br/> 此邮件由系统自动发送，不需要回复";

    @Override
    public ReferralOrgInfo findReferralOrgInfo(Long activityId) {
        ReferralOrgInfo dto = null;
        ReferralInfo referralInfo = this.referralInfoDao.getReferralInfo(activityId, null);
        if (referralInfo == null) {
            return dto;
        }
        try {
            OrgInfoSimpleDto orgInfo = orgInfoService.getOrgInfo(referralInfo.getOrgId());
            if (orgInfo == null) {
                return null;
            }
            dto = new ReferralOrgInfo();
            String logo = orgInfo.getLogo();
            String name = orgInfo.getShortName();
            String qrCode = ConstantEnums.ORG_HOME_PAGE_URL.value().trim() + orgInfo.getOrgNumber();
            dto.setOrgLogo(logo);
            dto.setOrgName(name);
            dto.setOrgQrCode(qrCode);
        } catch (Exception e) {
            log.error("[BizReferralService ] ", e);
        }
        return dto;
    }

}
