
/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2016 All Rights Reserved.
 */

package com.baijia.tianxiao.biz.marketing.article.service.impl;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Map;

import com.baijia.tianxiao.enums.RedisKeyEnums;
import org.apache.commons.lang.math.RandomUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.baijia.tianxiao.biz.marketing.article.service.ArticleSyncService;
import com.baijia.tianxiao.biz.marketing.utils.SynchMailSender;
import com.baijia.tianxiao.dal.activity.dao.article.ArticleBaseInfoDao;
import com.baijia.tianxiao.dal.activity.po.ArticleBaseInfo;
import com.baijia.tianxiao.dal.activity.po.ArticleDetail;
import com.baijia.tianxiao.dal.article.dao.BisArticleInfoDao;
import com.baijia.tianxiao.dal.article.dao.BisChannelInfoDao;
import com.baijia.tianxiao.dal.article.dao.BisFeedChannelDao;
import com.baijia.tianxiao.dal.article.po.BisArticleInfo;
import com.baijia.tianxiao.dal.article.po.BisChannelInfo;
import com.baijia.tianxiao.dal.article.po.BisFeedChannel;
import com.baijia.tianxiao.sal.marketing.article.enums.TouTiaoArticleEnum;
import com.baijia.tianxiao.sal.marketing.commons.service.RedisService;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.GenericsUtils;
import com.baijia.tianxiao.util.ListUtil;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import lombok.extern.slf4j.Slf4j;

/**
 * @say little Boy, don't be sad.
 * @name Rezar
 * @time Aug 10, 2016
 * @desc
 */
@Service
@Slf4j
public class ArticleSyncServiceImpl implements ArticleSyncService {

    @Autowired
    private BisChannelInfoDao bisChannelInfoDao;
    @Autowired
    private BisFeedChannelDao bisFeedChannelDao;
    @Autowired
    private BisArticleInfoDao bisArticleInfoDao;
    @Autowired(required = false)
    private ArticleBaseInfoDao articleBaseInfoDao;
    @Autowired
    private RedisService redisService;

    public static final Long ONE_DAY = (24 * 3600 * 1000L);

    /*
     * (non-Javadoc)
     * 
     * @desc 同步头条文章到当前mongoDb库中
     * 
     * @see com.baijia.tianxiao.biz.marketing.article.service.ArticleSyncService#syncArticle()
     */
    @Override
    public void syncArticle() {
        log.debug(" begin to syncArticles ");
        List<String> typeDescs = TouTiaoArticleEnum.getTypeDescs();
        if (GenericsUtils.isNullOrEmpty(typeDescs)) {
            return;
        }
        // 这里的代码写的有意思? 一个包含一个不包含....
        List<BisChannelInfo> listBisChannelInfos = this.bisChannelInfoDao.listBisChannelInfos(typeDescs);
        List<BisChannelInfo> listOtherBisChannelInfos = this.bisChannelInfoDao.listOtherBisChannelInfos(typeDescs);
        listBisChannelInfos.addAll(listOtherBisChannelInfos);
        log.info("find all bisChannelInfos are : {} ", listBisChannelInfos);
        if (GenericsUtils.isNullOrEmpty(listBisChannelInfos)) {
            return;
        }
        PageDto pageDto = new PageDto();
        pageDto.setPageSize(1000);
        Map<String, Integer> typeCounts = Maps.newHashMap();
        for (BisChannelInfo bisChannelInfo : listBisChannelInfos) {
            log.info("begin to sync type : {} ", bisChannelInfo);
            Long curSysnTime = getPreSyncTime(bisChannelInfo.getName());
            Long channelId = bisChannelInfo.getId();
            List<BisFeedChannel> listFeedIdsWithPublistTime =
                this.bisFeedChannelDao.listFeedIdsWithPublistTime(channelId, curSysnTime, pageDto);
            if (GenericsUtils.isNullOrEmpty(listFeedIdsWithPublistTime)) {
                log.info("can not find any feedChannel with channel type : {} ", bisChannelInfo.getName());
                continue;
            }
            List<Long> articleIds = ListUtil.toKeyList(listFeedIdsWithPublistTime, "feedId", BisFeedChannel.class);
            try {
                int count =
                    searchArticleAndReStore(TouTiaoArticleEnum.getCategory(bisChannelInfo.getName()), articleIds);
                typeCounts.put(bisChannelInfo.getName(), count);
            } catch (Exception e) {
                log.error("can not insert docs for type : {} cause by : {} ", bisChannelInfo.getName(), e);
            }
            reSetLastSyncTime(bisChannelInfo.getName(),
                listFeedIdsWithPublistTime.get(listFeedIdsWithPublistTime.size() - 1));
        }
        SynchMailSender.sendNotifyMail(typeCounts);
    }

    /**
     * @param bisFeedChannel
     */
    private void reSetLastSyncTime(String name, BisFeedChannel bisFeedChannel) {
        String key = createSpecialKey(name);
        this.redisService.setKey(key, bisFeedChannel.getPublishTime().getTime());
        log.info("reset last sync time to : {} and the type name is : {} ",
            getStrByDateFormate(new Date(bisFeedChannel.getPublishTime().getTime()), "yyyy-MM-dd HH:mm:ss"), name);
    }

    /**
     * @param name
     * @return
     */
    private Long getPreSyncTime(String name) {
        String key = createSpecialKey(name);
        Long syncTime = redisService.getKeyValue(key);
        if (syncTime == null) {
            log.info("pre sync time is not exists ,and will use current time to startup sync task");
            syncTime = new Date().getTime() - 10 * ONE_DAY;
        }
        return syncTime;
    }

    // 测试使用 , 正式发布后，需要确定一个唯一不变的键值保证定时更新任务的正确性
    // private String randomKey = RandomUtil.str(5);

    /**
     * @param name
     * @return
     */
    private String createSpecialKey(String name) {
        String format = RedisKeyEnums.HD.SYNC_ONELINE_ARTICLE.getRedisKey();
        return String.format(format, name);
    }

    /**
     * @param category
     * @param listFeedIdsWithPublistTime
     */
    private int searchArticleAndReStore(Integer category, List<Long> listFeedIdsWithPublistTime) {
        log.info("begin to search all article ,and the length of ids is : {}  ", listFeedIdsWithPublistTime.size());
        List<BisArticleInfo> listBisArticleInfos =
            this.bisArticleInfoDao.listBisArticleInfos(listFeedIdsWithPublistTime);
        log.info("find all articles's size is : {} ", listBisArticleInfos.size());
        if (GenericsUtils.notNullAndEmpty(listBisArticleInfos)) {
            List<ArticleBaseInfo> articles = toArticles(category, listBisArticleInfos);
            this.articleBaseInfoDao.insertArticles(articles);
            return listBisArticleInfos.size();
        }
        return 0;
    }

    /**
     * @param category 当前文章所属的分类
     * @param listBisArticleInfos
     * @return
     */
    private List<ArticleBaseInfo> toArticles(Integer category, List<BisArticleInfo> listBisArticleInfos) {
        List<ArticleBaseInfo> articles = Lists.newArrayListWithCapacity(listBisArticleInfos.size());
        for (BisArticleInfo bisArticle : listBisArticleInfos) {
            ArticleDetail articleDetail = new ArticleDetail();
            articleDetail.setAbstractContent(bisArticle.getSummary());
            String coverUrl = bisArticle.getCoverUrl();
            if (GenericsUtils.isNullOrEmpty(coverUrl)) {
                continue;
            }
            String[] coverUrls = coverUrl.split(",");
            articleDetail.setArticleImg(coverUrls[0]);
            String title = bisArticle.getTitle();
            title = title.replaceAll("\\s+", "");
            articleDetail.setArticleTitle(title);
            articleDetail.setArticleUrl(bisArticle.getOriArticleUrl());
            String author = bisArticle.getAuthor();
            author = GenericsUtils.isNullOrEmpty(author) ? "天校" : author;
            articleDetail.setAuthor(author);
            articleDetail.setContent(bisArticle.getContent());
            articleDetail.setCreateTime(bisArticle.getCreateTime().getTime());
            String pushTime = getStrByDateFormate(bisArticle.getPublishTime(), "yyyy-MM-dd HH:mm:ss");
            articleDetail.setPushTime(pushTime);
            int baseCount = RandomUtils.nextInt(9000);
            String thumbUpCount = String.valueOf(baseCount);
            articleDetail.setThumbUpCount(thumbUpCount);
            int visitCountBase = baseCount + RandomUtils.nextInt(9000);
            String visitCount = String.valueOf(visitCountBase);
            articleDetail.setVisitCount(visitCount);
            articleDetail.setCategory(category);
            articles.add(articleDetail);
        }
        return articles;
    }

    public static String getStrByDateFormate(Date date, String formate) {
        if (date == null) {
            return "";
        }
        SimpleDateFormat dateformat1 = new SimpleDateFormat(formate);
        String str = dateformat1.format(date);
        return str;
    }

}
