package cn.kinyun.wework.sdk.api.license;

import cn.kinyun.wework.sdk.entity.ErrorCode;
import cn.kinyun.wework.sdk.entity.license.order.*;
import cn.kinyun.wework.sdk.exception.WeworkException;
import lombok.NonNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;
import java.text.MessageFormat;

/**
 * @author: lwl
 * @date: 2023/5/15
 * @description: 接口调用许可 -- 订单管理
 *
 * @link https://developer.work.weixin.qq.com/document/path/95644
 **/
@Component

public class LicenseOrderApi {
    @Autowired
    @Qualifier("weworkRestTemplate")
    private RestTemplate restTemplate;

    /**
     * 下单购买账号
     */
    @Value("${qyapi.license.create_new_order}")
    private String licenseCreateNewOrderUrl;


    /**
     * 创建续期任务
     */
    @Value("${qyapi.license.create_renew_order_job}")
    private String licenseCreateRenewOrderJobUrl;


    /**
     * 提交续期订单
     */
    @Value("${qyapi.license.submit_order_job}")
    private String licenseSubmitOrderJobUrl;


    /**
     * 获取订单列表
     */
    @Value("${qyapi.license.list_order}")
    private String licenseListOrderUrl;


    /**
     * 获取订单详情
     */
    @Value("${qyapi.license.get_order}")
    private String licenseGetOrderUrl;


    /**
     * 获取订单中的账号列表
     */
    @Value("${qyapi.license.list_order_account}")
    private String licenseListOrderAccountUrl;


    /**
     * 取消订单
     */
    @Value("${qyapi.license.cancel_order}")
    private String licenseCancelOrderUrl;

    /**
     * 创建多企业新购任务
     */
    @Value("${qyapi.license.create_new_order_job}")
    private String licenseCreateNewOrderJobUrl;

    /**
     * 提交多企业新购订单
     */
    @Value("${qyapi.license.submit_new_order_job}")
    private String licenseSubmitNewOrderJobUrl;

    /**
     * 获取多企业新购订单提交结果
     */
    @Value("${qyapi.license.new_order_job_result}")
    private String licenseNewOrderJobResultUrl;

    /**
     * 获取多企业订单详情
     */
    @Value("${qyapi.license.get_union_order}")
    private String licenseGetUnionOrderUrl;

    /**
     * 下单购买账号
     *
     * @return 订单号
     * @link https://developer.work.weixin.qq.com/document/path/95644
     */
    public String createNewOrder(@NonNull String accessToken, @NonNull CreateNewOrderReq req) throws WeworkException {
        String url = MessageFormat.format(licenseCreateNewOrderUrl, accessToken);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<CreateNewOrderReq> httpEntity = new HttpEntity<>(req, headers);
        ResponseEntity<OrderIdResp> forEntity = restTemplate.postForEntity(url, httpEntity, OrderIdResp.class);
        OrderIdResp result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return result.getOrderId();
    }

    /**
     * 创建续期任务
     *
     * @link https://developer.work.weixin.qq.com/document/path/95646
     */
    public CreateRenewOrderJobResp createRenewOrderJob(@NonNull String accessToken, @NonNull CreateRenewOrderJobReq req) throws WeworkException {
        String url = MessageFormat.format(licenseCreateRenewOrderJobUrl, accessToken);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<CreateRenewOrderJobReq> httpEntity = new HttpEntity<>(req, headers);
        ResponseEntity<CreateRenewOrderJobResp> forEntity = restTemplate.postForEntity(url, httpEntity, CreateRenewOrderJobResp.class);
        CreateRenewOrderJobResp result = forEntity.getBody();
        WeworkException.isSuccess(result);
        return result;
    }

    /**
     * 提交续期订单
     *
     * @link https://developer.work.weixin.qq.com/document/path/95646
     */
    public String submitOrderJob(@NonNull String accessToken, @NonNull SubmitOrderJobReq req) throws WeworkException {
        String url = MessageFormat.format(licenseCreateNewOrderUrl, accessToken);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<SubmitOrderJobReq> httpEntity = new HttpEntity<>(req, headers);
        ResponseEntity<OrderIdResp> forEntity = restTemplate.postForEntity(url, httpEntity, OrderIdResp.class);
        OrderIdResp result = forEntity.getBody();
        WeworkException.isSuccess(result);
        return result.getOrderId();
    }

    /**
     * 获取订单列表
     *
     * @link https://developer.work.weixin.qq.com/document/path/95647
     **/
    public ListOrderResp listOrder(@NonNull String accessToken, @NonNull ListOrderReq req) throws WeworkException {
        String url = MessageFormat.format(licenseListOrderUrl, accessToken);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<ListOrderReq> httpEntity = new HttpEntity<>(req, headers);
        ResponseEntity<ListOrderResp> forEntity = restTemplate.postForEntity(url, httpEntity, ListOrderResp.class);
        ListOrderResp result = forEntity.getBody();
        WeworkException.isSuccess(result);
        return result;
    }

    /**
     * 获取订单详情
     *
     * @link https://developer.work.weixin.qq.com/document/path/95647
     **/
    public GetOrderResp getOrder(@NonNull String accessToken, @NonNull String orderId) throws WeworkException {
        OrderIdDto req = new OrderIdDto();
        req.setOrderId(orderId);

        String url = MessageFormat.format(licenseGetOrderUrl, accessToken, orderId);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<OrderIdDto> httpEntity = new HttpEntity<>(headers);
        ResponseEntity<GetOrderResp> forEntity = restTemplate.postForEntity(url, httpEntity, GetOrderResp.class);
        GetOrderResp result = forEntity.getBody();
        WeworkException.isSuccess(result);
        return result;
    }

    /**
     * 获取订单中的账号列表
     *
     * @link https://developer.work.weixin.qq.com/document/path/95649
     **/
    public ListOrderAccountResp listOrderAccount(@NonNull String accessToken, @NonNull ListOrderAccountReq req) throws WeworkException {
        String url = MessageFormat.format(licenseListOrderAccountUrl, accessToken);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<ListOrderAccountReq> httpEntity = new HttpEntity<>(req, headers);
        ResponseEntity<ListOrderAccountResp> forEntity = restTemplate.postForEntity(url, httpEntity, ListOrderAccountResp.class);
        ListOrderAccountResp result = forEntity.getBody();
        WeworkException.isSuccess(result);
        return result;
    }

    /**
     * 取消订单
     *
     * @link https://developer.work.weixin.qq.com/document/path/96106
     **/
    public void cancelOrder(@NonNull String accessToken, @NonNull CancelOrderReq req) throws WeworkException{
        String url = MessageFormat.format(licenseCancelOrderUrl, accessToken);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<CancelOrderReq> httpEntity = new HttpEntity<>(req, headers);
        ResponseEntity<ErrorCode> forEntity = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);
        ErrorCode result = forEntity.getBody();
        WeworkException.isSuccess(result);
    }

    /**
     * 创建多企业新购任务
     *
     * @link https://developer.work.weixin.qq.com/document/path/98892
     **/
    public CreateNewOrderJobResp createNewOrderJob(@NonNull String accessToken, @NonNull CreateNewOrderJobReq req) throws WeworkException {
        String url = MessageFormat.format(licenseCreateNewOrderJobUrl, accessToken);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<CreateNewOrderJobReq> httpEntity = new HttpEntity<>(req,headers);
        ResponseEntity<CreateNewOrderJobResp> forEntity = restTemplate.postForEntity(url, httpEntity, CreateNewOrderJobResp.class);
        CreateNewOrderJobResp result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return result;
    }

    /**
     * 提交多企业新购订单
     *
     * @link https://developer.work.weixin.qq.com/document/path/98892
     **/
    public String submitNewOrderJob(@NonNull String accessToken, @NonNull SubmitNewOrderJobReq req) throws WeworkException{
        String url = MessageFormat.format(licenseSubmitNewOrderJobUrl, accessToken);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<SubmitNewOrderJobReq> httpEntity = new HttpEntity<>(req,headers);
        ResponseEntity<JobIdResp> forEntity = restTemplate.postForEntity(url, httpEntity, JobIdResp.class);
        JobIdResp result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return result.getJobId();
    }

    /**
     * 获取多企业新购订单提交结果
     *
     * @link https://developer.work.weixin.qq.com/document/path/98892
     **/
    public NewOrderJobResultResp getNewOrderJob(@NonNull String accessToken, @NonNull String jobId) throws WeworkException{
        JobIdDto req = new JobIdDto();
        req.setJobId(jobId);

        String url = MessageFormat.format(licenseNewOrderJobResultUrl, accessToken);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<JobIdDto> httpEntity = new HttpEntity<>(req,headers);
        ResponseEntity<NewOrderJobResultResp> forEntity = restTemplate.postForEntity(url, httpEntity, NewOrderJobResultResp.class);
        NewOrderJobResultResp result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return result;
    }

    /**
     * 获取多企业订单详情
     *
     * @link https://developer.work.weixin.qq.com/document/path/98893
     **/
    public GetUnionOrderResp getUnionOrder(@NonNull String accessToken, @NonNull GetUnionOrderReq req) throws WeworkException{
        String url = MessageFormat.format(licenseGetUnionOrderUrl, accessToken);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<GetUnionOrderReq> httpEntity = new HttpEntity<>(req,headers);
        ResponseEntity<GetUnionOrderResp> forEntity = restTemplate.postForEntity(url, httpEntity, GetUnionOrderResp.class);
        GetUnionOrderResp result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return result;
    }



}
