package cn.kinyun.wework.sdk.api.external;


import cn.kinyun.wework.sdk.annotation.GenAsync;
import cn.kinyun.wework.sdk.annotation.GenNullable;
import cn.kinyun.wework.sdk.entity.ErrorCode;
import cn.kinyun.wework.sdk.entity.external.massmsg.MassMsgListParams;
import cn.kinyun.wework.sdk.entity.external.massmsg.MassMsgListResp;
import cn.kinyun.wework.sdk.entity.external.massmsg.MassMsgParams;
import cn.kinyun.wework.sdk.entity.external.massmsg.MassMsgResp;
import cn.kinyun.wework.sdk.entity.external.massmsg.MassMsgTaskResult;
import cn.kinyun.wework.sdk.entity.external.massmsg.MassMsgUserTask;
import cn.kinyun.wework.sdk.entity.external.welcome.WelcomeMsg;
import cn.kinyun.wework.sdk.exception.WeworkException;

import com.google.common.collect.Maps;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;
import java.util.HashMap;
import java.util.Map;

import lombok.NonNull;

/**
 * 群发消息推送
 */
@Component
public class MassMsgPushApi {
    @Autowired
    @Qualifier("weworkRestTemplate")
    private RestTemplate restTemplate;

    /**
     * 创建企业群发
     */
    @Value("${qyapi.msgpush.add_massmsg_push}")
    private String addMassMsgPushUrl;

    /**
     * 获取企业的全部群发记录
     */
    @Value("${qyapi.msgpush.get_massmsg_list}")
    private String getMassMsgListUrl;

    /**
     * 获取群发成员发送任务列表
     */
    @Value("${qyapi.msgpush.get_user_massmsg_list}")
    private String getUserMassMsgListUrl;

    /**
     * 查询群发任务结果
     */
    @Value("${qyapi.msgpush.get_user_massmsg_result}")
    private String getMassMsgResultUrl;

    /**
     * 发送新客户欢迎语
     */
    @Value("${qyapi.msgpush.send_replymsg}")
    private String sendReplyMsgUrl;

    /**
     * 撤回应用消息
     */
    @Value("${qyapi.message.recall}")
    private String wwMessageRecall;

    /**
     * @param accessToken
     * @param params
     * @desc 企业跟第三方应用可通过此接口添加企业群发消息的任务并通知成员发送给相关客户或客户群
     * @link https://open.work.weixin.qq.com/api/doc/90000/90135/92135
     */
    @GenAsync
    public MassMsgResp massMsgSend(@NonNull String accessToken, @NonNull MassMsgParams params) throws WeworkException {
        String url = MessageFormat.format(addMassMsgPushUrl, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<MassMsgParams> httpEntity = new HttpEntity<>(params, headers);

        ResponseEntity<MassMsgResp> forEntity = restTemplate.postForEntity(url, httpEntity, MassMsgResp.class);

        MassMsgResp result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return result;
    }

    /**
     * @param accessToken
     * @param params
     * @desc 企业和第三方应用可通过此接口获取企业与成员的群发记录
     * @link https://open.work.weixin.qq.com/api/doc/90000/90135/93338
     */
    public MassMsgListResp getMassMsgList(@NonNull String accessToken, @NonNull MassMsgListParams params) throws WeworkException {
        String url = MessageFormat.format(getMassMsgListUrl, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<MassMsgListParams> httpEntity = new HttpEntity<>(params, headers);

        ResponseEntity<MassMsgListResp> forEntity = restTemplate.postForEntity(url, httpEntity, MassMsgListResp.class);

        MassMsgListResp result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return result;
    }

    /**
     * @param accessToken
     * @param msgId
     * @param limit
     * @param cursor
     * @desc 获取群发成员发送任务列表
     * @link https://open.work.weixin.qq.com/api/doc/90000/90135/93338
     */
    public MassMsgUserTask getMassMsgUserTask(@NonNull String accessToken, @NonNull String msgId, @GenNullable Integer limit, @GenNullable String cursor) throws WeworkException {
        String url = MessageFormat.format(getUserMassMsgListUrl, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        if (limit == null)
            limit = 50;
        Map<String, Object> params = new HashMap<>();
        params.put("msgid", msgId);
        params.put("limit", limit);
        params.put("cursor", cursor);
        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(params, headers);

        ResponseEntity<MassMsgUserTask> forEntity = restTemplate.postForEntity(url, httpEntity, MassMsgUserTask.class);

        MassMsgUserTask result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return result;
    }

    /**
     * @param accessToken
     * @param msgId 群发消息的id
     * @param userId 发送成员userid
     * @param limit
     * @param cursor
     * @desc 获取企业群发成员执行结果
     * @link https://open.work.weixin.qq.com/api/doc/90000/90135/93338
     */
    public MassMsgTaskResult getMassMsgTaskResult(@NonNull String accessToken, @NonNull String msgId,
                                                  @NonNull String userId, @GenNullable Integer limit,
                                                  @GenNullable String cursor) throws WeworkException {
        String url = MessageFormat.format(getMassMsgResultUrl, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        if (limit == null)
            limit = 50;
        Map<String, Object> params = new HashMap<>();
        params.put("msgid", msgId);
        params.put("userid", userId);
        params.put("limit", limit);
        params.put("cursor", cursor);
        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(params, headers);

        ResponseEntity<MassMsgTaskResult> forEntity = restTemplate.postForEntity(url, httpEntity, MassMsgTaskResult.class);

        MassMsgTaskResult result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return result;
    }

    /**
     * 发送新客户欢迎语
     * 
     * @param accessToken
     * @param params
     * @link https://open.work.weixin.qq.com/api/doc/90000/90135/92137
     */
    @GenAsync
    public void sendWelcomeMsg(@NonNull String accessToken, @NonNull WelcomeMsg params) throws WeworkException {
        String url = MessageFormat.format(sendReplyMsgUrl, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<WelcomeMsg> httpEntity = new HttpEntity<>(params, headers);

        ResponseEntity<ErrorCode> forEntity = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = forEntity.getBody();
        WeworkException.isSuccess(result);
    }



    /**
     * @param accessToken
     * @param msgId
     * @desc 撤回应用消息
     * @link https://developer.work.weixin.qq.com/document/path/94867
     */
    @GenAsync
    public ErrorCode recallMsg(@NonNull String accessToken, @NonNull String msgId) throws WeworkException {
        String url = MessageFormat.format(wwMessageRecall, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        Map<String, Object> params = Maps.newHashMap();
        params.put("msgid", msgId);

        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(params, headers);
        ResponseEntity<ErrorCode> forEntity = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return  result;
    }
}
