package cn.kinyun.wework.sdk.api.external;

import cn.kinyun.wework.sdk.annotation.*;
import cn.kinyun.wework.sdk.entity.ErrorCode;
import cn.kinyun.wework.sdk.entity.external.croptag.AddCorpTagParams;
import cn.kinyun.wework.sdk.entity.external.croptag.AddCorpTagResp;
import cn.kinyun.wework.sdk.entity.external.croptag.CorpTagGroup;
import cn.kinyun.wework.sdk.entity.external.croptag.CorpTagList;
import cn.kinyun.wework.sdk.entity.external.croptag.MarkTagParams;
import cn.kinyun.wework.sdk.entity.external.croptag.UpdateCorpTagParams;
import cn.kinyun.wework.sdk.exception.WeworkException;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import lombok.NonNull;

/**
 * 客户标签管理
 */
@Component
public class ExternalTagApi {
    @Autowired
    @Qualifier("weworkRestTemplate")
    private RestTemplate restTemplate;

    /**
     * 获取企业标签库
     */
    @Value("${qyapi.externalcontact.get_corp_tag_list}")
    private String corpTagListUrl;

    /**
     * 添加企业客户标签
     */
    @Value("${qyapi.externalcontact.add_corp_tag}")
    private String addCorpTagUrl;

    /**
     * 编辑企业客户标签或者标签组
     */
    @Value("${qyapi.externalcontact.edit_corp_tag}")
    private String editCorpTagUrl;

    /**
     * 删除企业客户标签
     */
    @Value("${qyapi.externalcontact.del_corp_tag}")
    private String delCorpTagUrl;

    /**
     * 编辑客户企业标签
     */
    @Value("${qyapi.externalcontact.mark_tag}")
    private String markTagUrl;

    /**
     * 获取企业标签列表
     * 
     * <p>若tag_id和group_id均为空，则返回所有标签。</p>
     * <p>同时传递tag_id和group_id时，忽略tag_id，仅以group_id作为过滤条件。</p>
     * 
     * @param accessToken
     * @param groupIds 要查询的标签组id，返回该标签组以及其下的所有标签信息
     * @param tagIds 要查询的标签id
     * @link https://open.work.weixin.qq.com/api/doc/90000/90135/92117
     */
    public List<CorpTagGroup> list(@NonNull String accessToken,//
                                   @GenNullable Collection<String> groupIds,
                                   @GenNullable Collection<String> tagIds) throws WeworkException {
        String url = MessageFormat.format(corpTagListUrl, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        Map<String, Object> params = new HashMap<>();
        params.put("group_id", groupIds);
        params.put("tag_id", tagIds);

        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(params, headers);

        ResponseEntity<CorpTagList> forEntity = restTemplate.postForEntity(url, httpEntity, CorpTagList.class);

        CorpTagList result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return result.getTagGroupList();
    }

    /**
     * 添加企业客户标签
     * 
     * @param accessToken
     * @param params 添加企业客户标签
     * @link https://open.work.weixin.qq.com/api/doc/90000/90135/92117
     */
    @GenAsync
    public AddCorpTagResp add(@NonNull String accessToken, @NonNull AddCorpTagParams params) throws WeworkException {
        String url = MessageFormat.format(addCorpTagUrl, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<AddCorpTagParams> httpEntity = new HttpEntity<>(params, headers);

        ResponseEntity<AddCorpTagResp> forEntity = restTemplate.postForEntity(url, httpEntity, AddCorpTagResp.class);

        AddCorpTagResp result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return result;
    }

    /**
     * 编辑企业客户标签
     * 
     * @param accessToken
     * @param params
     * @link https://open.work.weixin.qq.com/api/doc/90000/90135/92117
     */
    @GenAsync
    public void update(@NonNull String accessToken, @NonNull UpdateCorpTagParams params) throws WeworkException {
        String url = MessageFormat.format(editCorpTagUrl, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<UpdateCorpTagParams> httpEntity = new HttpEntity<>(params, headers);

        ResponseEntity<ErrorCode> forEntity = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = forEntity.getBody();
        WeworkException.isSuccess(result);
    }

    /**
     * 删除企业客户标签
     * 
     * @param accessToken
     * @param groupIds 标签组id
     * @param tagIds 标签id
     * @link https://open.work.weixin.qq.com/api/doc/90000/90135/92117
     */
    @GenAsync
    public void delete(@NonNull String accessToken, //
                       @GenNullable Collection<String> groupIds, //
                       @GenNullable Collection<String> tagIds) throws WeworkException {
        String url = MessageFormat.format(delCorpTagUrl, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        Map<String, Object> params = new HashMap<>();
        params.put("group_id", groupIds);
        params.put("tag_id", tagIds);

        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(params, headers);

        ResponseEntity<ErrorCode> forEntity = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = forEntity.getBody();
        WeworkException.isSuccess(result);
    }

    /**
     * 编辑客户企业标签
     * 
     * @param accessToken
     * @param params
     * @link https://open.work.weixin.qq.com/api/doc/90000/90135/92118
     */
    @GenAsync
    @GenRetry
    @GenLock("\"ExternalTagClient:markTag:\" + corpId")
    public void markTag(@NonNull String accessToken, @NonNull MarkTagParams params) throws WeworkException {
        String url = MessageFormat.format(markTagUrl, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<MarkTagParams> httpEntity = new HttpEntity<>(params, headers);

        ResponseEntity<ErrorCode> forEntity = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = forEntity.getBody();
        WeworkException.isSuccess(result);
    }
}
