/**
 * kuaikeguanjia.com Inc. Copyright (c) 2014-2021 All Rights Reserved.
 */
package cn.kinyun.wework.sdk.api;

import cn.kinyun.wework.sdk.annotation.GenIgnore;
import cn.kinyun.wework.sdk.entity.oauth2.UserDetail;
import cn.kinyun.wework.sdk.entity.oauth2.Userdetail3rd;
import cn.kinyun.wework.sdk.entity.oauth2.Userinfo;
import cn.kinyun.wework.sdk.entity.oauth2.Userinfo3rd;
import cn.kinyun.wework.sdk.enums.SnsapiScope;
import cn.kinyun.wework.sdk.exception.WeworkException;
import com.google.common.collect.Maps;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;
import java.util.Map;

import lombok.NonNull;

/**
 * @title WwOauth2Api
 * @author yanmaoyuan
 * @date 2021年2月25日
 * @version 1.0
 */
@Component
@SuppressWarnings("java:S2293")
public class WwOauth2Api {

    @Autowired
    @Qualifier("weworkRestTemplate")
    private RestTemplate restTemplate;

    /**
     * 构造第三方应用oauth2链接和构造企业oauth2链接
     * 第三方应用的和企业内部的是一样的
     */
    @Value("${qyapi.oauth2.suite_authorize}")
    private String wwSuiteAuthorizeUrl;

    /**
     * 获取访问用户身份
     * 第三方应用
     */
    @Value("${qyapi.oauth2.get_userinfo_3rd}")
    private String wwGetUserinfo3rd;

    /**
     * 获取访问用户身份
     * 企业内部应用
     */
    @Value("${qyapi.oauth2.get_userinfo}")
    private String getWwGetUserinfo;

    /**
     * 获取访问用户敏感信息
     */
    @Value("${qyapi.oauth2.get_userdetail_3rd}")
    private String wwGetUserdetail3rd;

    @Value("${qyapi.oauth2.get_userinfo_detail}")
    private String getUserdetail;

    /**
     * 
     * @param appId
     * @param redirectUri
     * @param scope
     * @param state
     * @return
     */
    @GenIgnore
    public String getSuiteAuthorizeUrl(String appId, String redirectUri, SnsapiScope scope, String state, String agentId) {
        String url = MessageFormat.format(wwSuiteAuthorizeUrl, appId, redirectUri, scope.getValue(), state);
        if (agentId != null) {
            url = url + "&agentid=" + agentId + "#wechat_redirect";
        }
        return url;
    }

    /**
     * 获取访问用户身份 第三方应用
     * 
     * @param suiteAccessToken
     * @param code 通过成员授权获取到的code，最大为512字节。每次成员授权带上的code将不一样，code只能使用一次，5分钟未被使用自动过期。
     * @return
     * @throws WeworkException
     * @see https://open.work.weixin.qq.com/api/doc/90001/90143/91121
     */
    public Userinfo3rd getUserinfo3rd(@NonNull String suiteAccessToken, @NonNull String code) throws WeworkException {
        String url = MessageFormat.format(wwGetUserinfo3rd, suiteAccessToken, code);

        ResponseEntity<Userinfo3rd> response = restTemplate.getForEntity(url, Userinfo3rd.class);

        Userinfo3rd result = response.getBody();
        WeworkException.isSuccess(result);
        return result;
    }

    /**
     * 获取访问用户身份 企业内部的
     * @param accessToken
     * @param code 	通过成员授权获取到的code，最大为512字节。每次成员授权带上的code将不一样，code只能使用一次，5分钟未被使用自动过期。
     * @link https://open.work.weixin.qq.com/api/doc/90000/90135/91023
     */
    public Userinfo getUserinfo(@NonNull String accessToken, @NonNull String code) throws WeworkException {
        String url = MessageFormat.format(getWwGetUserinfo, accessToken, code);

        ResponseEntity<Userinfo> responseEntity = restTemplate.getForEntity(url, Userinfo.class);

        Userinfo result = responseEntity.getBody();
        WeworkException.isSuccess(result);
        return result;
    }

    /**
     * 获取访问用户敏感信息
     * 
     * @param suiteAccessToken
     * @param userTicket 成员票据
     * @return
     * @throws WeworkException
     * @see https://open.work.weixin.qq.com/api/doc/90001/90143/91122
     */
    public Userdetail3rd getUserdetail3rd(@NonNull String suiteAccessToken, @NonNull String userTicket) throws WeworkException {
        String url = MessageFormat.format(wwGetUserdetail3rd, suiteAccessToken);
        
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        Map<String, Object> params = Maps.newHashMap();
        params.put("user_ticket", userTicket);

        HttpEntity<Map<String, Object>> request = new HttpEntity<Map<String, Object>>(params, headers);
        ResponseEntity<Userdetail3rd> response = restTemplate.postForEntity(url, request, Userdetail3rd.class);

        Userdetail3rd result = response.getBody();
        WeworkException.isSuccess(result);
        return result;
    }

    public UserDetail getUserdetail(@NonNull String accessToken, @NonNull String userTicket) throws WeworkException {
        String url = MessageFormat.format(getUserdetail, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        Map<String, Object> params = Maps.newHashMap();
        params.put("user_ticket", userTicket);

        HttpEntity<Map<String, Object>> request = new HttpEntity<Map<String, Object>>(params, headers);
        ResponseEntity<UserDetail> response = restTemplate.postForEntity(url, request, UserDetail.class);

        UserDetail result = response.getBody();
        WeworkException.isSuccess(result);
        return result;
    }
}
